<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class AIRS_Ajax {

    private static $_instance = null;

    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }
    
    private function __construct() {
        add_action( 'wp_ajax_airs_generate_summary', [ $this, 'generate_summary' ] );
        add_action( 'wp_ajax_airs_fetch_reviews', [ $this, 'fetch_reviews' ] );
        add_action( 'wp_ajax_airs_delete_all_summaries', [ $this, 'delete_all_summaries' ] );
        add_action( 'wp_ajax_airs_delete_selected_summaries', [ $this, 'delete_selected_summaries' ] );

    }

    private function check_permissions() {
        if ( ! check_ajax_referer( 'airs_ajax_nonce', 'nonce', false ) || ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Permission denied.' ], 403 );
        }
    }
    
    public function generate_summary() {
        $this->check_permissions();

        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
        if ( ! $post_id || get_post_type( $post_id ) !== 'listing' ) {
            wp_send_json_error( [ 'message' => 'Invalid Listing ID.' ] );
        }

        $result = AIRS_Core::instance()->generate_summary( $post_id );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( [ 'message' => $result->get_error_message() ] );
        } else {
            wp_send_json_success( [ 'message' => 'Summary generated successfully for ' . get_the_title($post_id) ] );
        }
    }

    public function fetch_reviews() {
        $this->check_permissions();

        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
        if ( ! $post_id || get_post_type( $post_id ) !== 'listing' ) {
            wp_send_json_error( [ 'message' => 'Invalid Listing ID.' ] );
        }

        $reviews = AIRS_Core::instance()->get_reviews_for_listing( $post_id );

        if ( empty( $reviews ) ) {
            wp_send_json_error( [ 'message' => 'No reviews found for Listing ID ' . $post_id ] );
        }

        wp_send_json_success( [
            'count' => count($reviews),
            'reviews' => $reviews,
        ] );
    }


    public function delete_all_summaries() {
        $this->check_permissions();

        global $wpdb;
        $table_name = $wpdb->prefix . 'ai_review_summaries';

        // TRUNCATE is faster than DELETE for emptying an entire table.
        $result = $wpdb->query( "TRUNCATE TABLE {$table_name}" );

        if ( false === $result ) {
            wp_send_json_error( [ 'message' => 'An error occurred while trying to delete the summaries.' ] );
        }

        wp_send_json_success( [ 'message' => 'All AI review summaries have been successfully deleted.' ] );
    }

        /**
     * Handles the AJAX request to delete summaries for selected listings.
     * This is triggered by the "Delete Summaries for Selected" button.
     */
    public function delete_selected_summaries() {
        // 1. Security Check: Verify nonce and user permissions.
        $this->check_permissions();

        // 2. Input Validation: Ensure we received an array of post IDs.
        if ( empty($_POST['post_ids']) || !is_array($_POST['post_ids']) ) {
            wp_send_json_error( [ 'message' => 'No listings were selected for deletion.' ] );
        }

        // 3. Sanitize Input: Make sure every item in the array is a valid integer.
        $post_ids = array_map( 'intval', $_POST['post_ids'] );

        // 4. Prepare for Database Query: Create the correct number of '%d' placeholders for the IN clause.
        // This is the safe way to build a dynamic IN clause with $wpdb->prepare.
        $ids_placeholder = implode( ',', array_fill( 0, count( $post_ids ), '%d' ) );

        global $wpdb;
        $table_name = $wpdb->prefix . 'ai_review_summaries';

        // 5. Build the Database Query: Prepare the final, safe SQL statement.
        $query = $wpdb->prepare( "DELETE FROM {$table_name} WHERE listing_id IN ({$ids_placeholder})", $post_ids );
        
        // 6. Execute the Query: Run the deletion and get the number of rows affected.
        $deleted_rows = $wpdb->query( $query );

        // 7. Error Handling: Check if the database query itself failed.
        // Note: $deleted_rows will be 0 if no matching rows were found, which is not an error.
        // It will be `false` only if there was a MySQL error.
        if ( false === $deleted_rows ) {
            wp_send_json_error( [ 'message' => 'A database error occurred during deletion.' ] );
        }

        // 8. Send Success Response: Return a success message and the list of IDs that were processed.
        // The JavaScript will use `deleted_ids` to update the UI by removing the checkmarks.
        wp_send_json_success( [
            'message' => sprintf( 'Successfully deleted %d summaries.', $deleted_rows ),
            'deleted_ids' => $post_ids
        ] );
    }
}