<?php
class LDS_Admin_Menu {

    public function __construct() {
        add_action( 'admin_menu', [ $this, 'add_plugin_pages' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_map_scripts' ] );
    }

    public function add_plugin_pages() {
        add_menu_page(
            'Listeo Data Scraper',
            'Listeo Scraper',
            'manage_options',
            'listeo-data-scraper',
            [ $this, 'create_import_page' ],
            'dashicons-cloud-upload',
            6
        );

        add_submenu_page(
            'listeo-data-scraper',
            'Import Listings',
            'Import',
            'manage_options',
            'listeo-data-scraper',
            [ $this, 'create_import_page' ]
        );

        add_submenu_page(
            'listeo-data-scraper',
            'Scraper Settings',
            'Settings',
            'manage_options',
            'lds-settings',
            [ $this, 'create_settings_page' ]
        );
    }

    public function create_import_page() {
        // Get current settings
        $limit = (int) get_option('lds_import_limit', 20);
        $photo_enabled = (bool) get_option('lds_enable_photo_import', 0);
        $photo_limit = $photo_enabled ? (int) get_option('lds_photo_import_limit', 0) : 0;
        $lang_setting = get_option('lds_description_language', 'site-default');
        $ai_enabled = (bool) get_option('lds_enable_ai_descriptions', 1);
        // Removed default listing type setting dependency
    
        // Determine language display text
        if ($lang_setting === 'site-default') {
            $wp_lang_name = \Locale::getDisplayLanguage(get_locale(), 'en');
            $language_display = "Site Default (" . esc_html($wp_lang_name) . ")";
        } else {
            $language_display = esc_html($lang_setting);
        }

        // AI status display
        $ai_status = $ai_enabled ? 'Enabled' : 'Disabled (Fast Mode)';
        $ai_status_class = $ai_enabled ? 'enabled' : 'disabled';
        ?>
        <div class="wrap">
            <h1 style="margin-bottom: 10px;">Import Listings</h1>
            <p style="color: #7f8c8d; margin-bottom: 25px;">Enter a search query and location to import listings from Google Places.</p>
    
            <div class="lds-main-container">
                
                <!-- Import Form Card -->
                <div class="lds-import-card">
                    <h3 class="lds-card-header">🚀 Import New Listings</h3>
                    <div class="lds-card-body">
                        <form id="lds-import-form">
                            <?php wp_nonce_field( 'lds_import_nonce', 'lds_nonce' ); ?>
                            
                            <!-- Search Examples -->
                            <div class="lds-examples-box">
                                <div class="lds-examples-title">Search Examples</div>
                                <ul class="lds-examples-list">
                                    <li><strong>Business Type:</strong> "Car Repair" → <strong>Location:</strong> "New York, NY"</li>
                                    <li><strong>Business Type:</strong> "Italian Restaurants" → <strong>Location:</strong> "10001"</li>
                                </ul>
                            </div>
                            
                            <div class="lds-form-group">
                                <label for="lds_query">Business Type / Service</label>
                                <input 
                                    type="text" 
                                    id="lds_query" 
                                    name="lds_query" 
                                    class="lds-form-input" 
                                    required 
                                    placeholder="e.g., Plumbers, Restaurants, Hair Salons" 
                                />
                                <div class="lds-form-description">What type of business are you looking for?</div>
                            </div>

                            <div class="lds-form-group">
                                <label for="lds_location">Location</label>
                                
                                <!-- Location Mode Toggle -->
                                <?php 
                                $map_enabled = 1; // Always enabled - map mode is default
                                $api_key = get_option('lds_google_api_key');
                                if ($map_enabled && !empty($api_key)): ?>
                                <div class="lds-location-mode" style="margin-bottom: 10px;">
                                    <button type="button" class="lds-mode-btn active" data-mode="text">📍 Text</button>
                                    <button type="button" class="lds-mode-btn" data-mode="map">🗺️ Map</button>
                                </div>
                                <!-- Search Method Information -->
                                <div class="lds-search-method-info" style="margin-bottom: 15px; padding: 12px; background: #f8f9fa; border-radius: 6px;">
                                    <div id="lds-text-search-info">
                                        <strong>📍 Text Search:</strong> <span style="color: #0073aa;">Uses broader understanding and context.</span> Works well with natural language (e.g., "kawiarnia", "coffee shop near university") and finds businesses globally.
                                    </div>
                                    <div id="lds-map-search-info" style="display: none;">
                                        <strong>🗺️ Map Search:</strong> <span style="color: #0073aa;">Location-based with literal keyword matching.</span> Best for specific areas with precise keywords - if you will search for "coffee shop" but coffee shop you expect to import do not have such keyword in the name then they won't be fetched.
                                    </div>
                                </div>
                                <?php elseif ($map_enabled && empty($api_key)): ?>
                                <div style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 10px; border-radius: 4px; margin-bottom: 15px; color: #856404;">
                                    <strong>Map Mode Unavailable:</strong> Google API Key is required. Please set it in <a href="<?php echo admin_url('admin.php?page=lds-settings'); ?>">Settings</a>.
                                </div>
                                <?php endif; ?>
                                
                                <!-- Text Mode (Default) -->
                                <div id="lds-text-mode" class="lds-location-input-mode">
                                    <input 
                                        type="text" 
                                        id="lds_location" 
                                        name="lds_location" 
                                        class="lds-form-input" 
                                        placeholder="e.g., New York, NY or 10001 or Times Square" 
                                    />
                                    <div class="lds-form-description">Enter city, state, zip code, or landmark</div>
                                </div>
                                
                                <!-- Map Mode -->
                                <?php if ($map_enabled && !empty($api_key)): ?>
                                <div id="lds-map-mode" class="lds-location-input-mode" style="display: none;">
                                    <!-- Address Search with Geolocate Button -->
                                    <div class="lds-map-search" style="margin-bottom: 10px;">
                                        <div class="lds-location-input-wrapper" style="position: relative; display: flex; gap: 12px; align-items: stretch;">
                                            <div style="position: relative; flex: 1;">
                                                <input type="text" id="lds-address-search" placeholder="Type address (e.g., London)" style="width: 100%; padding-right: 120px;" class="lds-form-input">
                                                <button type="button" id="lds-use-location-btn" class="lds-geolocate-overlay-btn">
                                                    Geolocate me
                                                </button>
                                            </div>
                                            <button type="button" id="lds-search-btn" class="lds-search-button">
                                                Search
                                            </button>
                                        </div>
                                    </div>

                                    <!-- Radius Selector -->
                                    <div class="lds-radius-selector" style="margin-bottom: 10px;">
                                        <div class="lds-radius-slider-container">
                                            <div class="lds-custom-slider">
                                                <div class="lds-slider-track"></div>
                                                <div class="lds-slider-progress"></div>
                                                <div class="lds-slider-thumb" id="lds-slider-thumb"></div>
                                            </div>
                                            <input type="hidden" id="lds-radius" name="lds_radius" value="2">
                                            <div class="lds-radius-display">
                                                <span id="lds-radius-value-display">2</span> km
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Selected Location Display -->
                                    <div id="lds-selected-location" style="margin-bottom: 10px;">
                                        <div style="display: flex; align-items: center; gap: 10px; flex-wrap: wrap;">
                                            <div>
                                                <strong>Selected Location:</strong> <span id="lds-location-text">Click on map to select a location</span>
                                            </div>

                                            <button type="button" id="lds-save-default-location" class="lds-save-default-btn" style="display: none;">
                                                Save as Default
                                            </button>
                                        </div>
                                    </div>

                                    <!-- Google Map Container -->
                                    <div id="lds-google-map" style="height: 400px; width: 100%; border: 1px solid #ddd; border-radius: 4px; margin-bottom: 10px;"></div>

                                    <!-- Hidden fields for coordinates -->
                                    <input type="hidden" id="lds-lat" name="lds_lat">
                                    <input type="hidden" id="lds-lng" name="lds_lng">
                                    <input type="hidden" id="lds-radius-value" name="lds_radius">
                                    <input type="hidden" id="lds-search-mode" name="lds_search_mode" value="text">
                                </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="lds-form-group">
                                <label for="lds_category">Assign to Category <i style="color:#d31c1c;font-style: normal; font-weight: 400">(Required)</i></label>
                                <?php
                                $categories = get_terms([
                                    'taxonomy' => 'listing_category', 
                                    'hide_empty' => false,
                                    'hierarchical' => true,
                                    'orderby' => 'name',
                                    'order' => 'ASC'
                                ]);
                                if (is_wp_error($categories)) {
                                    echo '<p style="color: #e74c3c;">Could not load categories.</p>';
                                } elseif (empty($categories)) {
                                    echo '<p style="color: #f39c12;">No categories found. Please create categories in <a href="' . admin_url('edit-tags.php?taxonomy=listing_category') . '" target="_blank">Listings → Categories</a> first.</p>';
                                } else {
                                    echo '<select id="lds_category" name="lds_category" class="lds-form-select" required>';
                                    echo '<option value="">-- Select a Category --</option>';
                                    
                                    // Build hierarchical options for categories
                                    $this->build_hierarchical_category_options($categories);
                                    
                                    echo '</select>';
                                }
                                ?>
                                <div class="lds-form-description">The selected category will be assigned to all imported listings.</div>
                            </div>

                            <div class="lds-form-group">
                                <label for="lds_region">Assign to Region (Optional)</label>
                                <?php
                                // Try multiple possible region taxonomy names - prioritize region (confirmed from regions importer)
                                $region_taxonomy = null;
                                $possible_region_taxonomies = ['region', 'listing_region', 'regions', 'location', 'listing_location'];
                                
                                foreach ($possible_region_taxonomies as $tax_name) {
                                    if (taxonomy_exists($tax_name)) {
                                        $region_taxonomy = $tax_name;
                                        break;
                                    }
                                }
                                
                                if ($region_taxonomy) {
                                    $regions = get_terms([
                                        'taxonomy' => $region_taxonomy, 
                                        'hide_empty' => false,
                                        'hierarchical' => true,
                                        'orderby' => 'name',
                                        'order' => 'ASC'
                                    ]);
                                    
                                    if (is_wp_error($regions) || empty($regions)) {
                                        echo '<select id="lds_region" name="lds_region" class="lds-form-select" disabled>';
                                        echo '<option value="">No regions found</option>';
                                        echo '</select>';
                                        echo '<div class="lds-form-description" style="color: #e74c3c;">No regions are available in the "' . $region_taxonomy . '" taxonomy. You can create regions in your WordPress admin.</div>';
                                    } else {
                                        echo '<select id="lds_region" name="lds_region" class="lds-form-select">';
                                        echo '<option value="">-- No Region (Optional) --</option>';
                                        
                                        // Build hierarchical options
                                        $this->build_hierarchical_region_options($regions, $region_taxonomy);
                                        
                                        echo '</select>';
                                        echo '<div class="lds-form-description">Optionally assign a region to all imported listings.</div>';
                                    }
                                    
                                    // Add hidden field to store the detected taxonomy name
                                    echo '<input type="hidden" id="lds_region_taxonomy" name="lds_region_taxonomy" value="' . esc_attr($region_taxonomy) . '" />';
                                } else {
                                    echo '<select id="lds_region" name="lds_region" class="lds-form-select" disabled>';
                                    echo '<option value="">No region taxonomy found</option>';
                                    echo '</select>';
                                    echo '<div class="lds-form-description" style="color: #e74c3c;">No region taxonomy detected. Common names tried: ' . implode(', ', $possible_region_taxonomies) . '</div>';
                                }
                                ?>
                            </div>

                            <div class="lds-form-group">
                                <label for="lds_listing_type">Listing Type</label>
                                <?php
                                // Get available listing types from Listeo theme/plugin post meta
                                // First, try to get from a sample listing post to see what types are available
                                $sample_listing = get_posts([
                                    'post_type' => 'listing',
                                    'posts_per_page' => 1,
                                    'post_status' => 'publish',
                                    'fields' => 'ids'
                                ]);
                                
                                $listing_types = [];
                                if (!empty($sample_listing)) {
                                    // Get the listing type meta from an existing listing to see available options
                                    $sample_meta = get_post_meta($sample_listing[0], '_listing_type', true);
                                    
                                    // Common Listeo listing types based on the theme structure
                                    $listing_types = [
                                        'service' => 'Service',
                                        'rental' => 'Rental',
                                        'event' => 'Event',
                                        'classifieds' => 'Classifieds'
                                    ];
                                } else {
                                    // Fallback if no listings exist yet
                                    $listing_types = [
                                        'service' => 'Service',
                                        'rental' => 'Rental',
                                        'event' => 'Event',
                                        'classifieds' => 'Classifieds'
                                    ];
                                }
                                
                                echo '<select id="lds_listing_type" name="lds_listing_type" class="lds-form-select" required>';
                                foreach ($listing_types as $value => $label) {
                                    // Default to 'service' if no setting exists
                                    $selected = ($value === 'service') ? 'selected' : '';
                                    echo '<option value="' . esc_attr($value) . '" ' . $selected . '>' . esc_html($label) . '</option>';
                                }
                                echo '</select>';
                                ?>
                                <div class="lds-form-description">Choose the type of listing that best describes these businesses.</div>
                            </div>
                            
                            <!-- Manual Selection Checkbox -->
                            <div class="lds-form-group" style="margin-top: 20px; padding: 15px; background: #f8f9fa; border-radius: 4px;">
                                <label style="display: flex; align-items: center; cursor: pointer; margin-bottom: 0;">
                                    <input type="checkbox" id="lds_manual_selection" name="lds_manual_selection" style="margin-right: 8px;" />
                                    <span style="font-weight: normal;">Let me manually select places to be imported</span>
                                </label>
                                <div class="lds-form-description">When enabled, you can review and select which places to import after fetching from Google</div>
                            </div>
                            
                            <div class="lds-submit-area">
                                <button type="submit" class="lds-submit-button">
                                    <span class="button-text">Run Import</span>
                                    <span class="lds-spinner"></span>
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
    
                <!-- Current Settings Card -->
                <div class="lds-settings-card">
                    <h3 class="lds-settings-header">Current Settings</h3>
                    <div class="lds-settings-body">
                        <ul class="lds-settings-list">
                            <li><strong>Listings per Import:</strong> <?php echo $limit; ?></li>
                            <li><strong>Photos per Listing:</strong> <?php echo $photo_enabled ? $photo_limit : '0 (Photo import disabled)'; ?></li>
                            <li><strong>AI Description Language:</strong> <?php echo $language_display; ?></li>
                            <li><strong>AI Descriptions:</strong> <span class="lds-ai-status <?php echo $ai_status_class; ?>"><?php echo $ai_status; ?></span></li>
                        </ul>
                        
                        <p style="color: #7f8c8d; font-size: 13px; margin-top: 15px;">These values can be changed on the settings page.</p>
                        <a href="<?php echo admin_url('admin.php?page=lds-settings'); ?>" class="lds-settings-button">
                            Change Settings
                        </a>
                    </div>
                </div>
    
            </div>
    
            <!-- Import Results Area -->
            <div id="lds-import-results"></div>
        </div>
        

        <?php
    }

    public function create_settings_page() {
        ?>
        <div class="wrap">
            <h1 style="margin-bottom: 10px;">Listeo Data Scraper Settings</h1>
            <p style="color: #7f8c8d; margin-bottom: 25px;">Configure your import settings and API credentials.</p>
            
            <div style="background: #ffffff; border-radius: 6px; box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08); border: 1px solid #e1e5e9; overflow: hidden; max-width: 800px;">
                <h3 style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px 25px; margin: 0; font-size: 18px; font-weight: 600;">
                    Plugin Configuration
                </h3>
                <div style="padding: 25px;">
                    <form method="post" action="options.php">
                        <?php
                            settings_fields( 'lds_settings_group' );
                            do_settings_sections( 'lds-settings-page' );
                        ?>
                        <p style="margin-top: 20px;">
                            <input type="submit" name="submit" class="lds-submit-button" value="Save Settings" />
                        </p>
                    </form>
                </div>
            </div>
        </div>
        <?php
    }

    public function register_settings() {
        register_setting( 'lds_settings_group', 'lds_google_api_key' );
        register_setting( 'lds_settings_group', 'lds_import_limit' );

        add_settings_section( 'lds_main_section', 'API and Import Settings', null, 'lds-settings-page' );

        add_settings_field( 'lds_google_api_key', 'Google API Key', [ $this, 'render_api_key_field' ], 'lds-settings-page', 'lds_main_section' );
        add_settings_field( 'lds_import_limit', 'Listings to Import', [ $this, 'render_import_limit_field' ], 'lds-settings-page', 'lds_main_section' );
    }

    public function render_api_key_field() {
        $value = esc_attr( get_option('lds_google_api_key') );
        echo "<input type='text' name='lds_google_api_key' value='{$value}' class='lds-form-input' style='max-width: 400px;' />";
        echo "<p class='lds-form-description'>Enter your Google Places API key. Ensure the 'Places API' is enabled in your Google Cloud Console.</p>";
    }

    public function render_import_limit_field() {
        $value = esc_attr( get_option('lds_import_limit', 20) );
        echo "<input type='number' name='lds_import_limit' value='{$value}' min='1' max='500' class='lds-form-input' style='max-width: 120px;' />";
        echo "<p class='lds-form-description'>Max number of listings to import per query (1-500).</p>";
    }

    /**
     * Build hierarchical category options for the select dropdown
     * 
     * @param array $terms All category terms
     * @param int $parent_id Parent term ID (0 for top level)
     * @param string $indent Indentation for child terms
     */
    private function build_hierarchical_category_options($terms, $parent_id = 0, $indent = '') {
        foreach ($terms as $term) {
            if ($term->parent == $parent_id) {
                $display_name = $indent . $term->name;
                
                // Add count of child categories for parent categories
                if ($parent_id == 0) {
                    $child_count = $this->count_child_terms($terms, $term->term_id);
                    if ($child_count > 0) {
                        $display_name .= '';
                    }
                }
                
                echo '<option value="' . esc_attr($term->term_id) . '">';
                echo esc_html($display_name);
                echo '</option>';
                
                // Recursively add child terms with indentation
                $this->build_hierarchical_category_options($terms, $term->term_id, $indent . '&nbsp;&nbsp;&nbsp;&nbsp;');
            }
        }
    }

    /**
     * Build hierarchical region options for the select dropdown
     * 
     * @param array $terms All region terms
     * @param string $taxonomy_name The taxonomy name
     * @param int $parent_id Parent term ID (0 for top level)
     * @param string $indent Indentation for child terms
     */
    private function build_hierarchical_region_options($terms, $taxonomy_name, $parent_id = 0, $indent = '') {
        foreach ($terms as $term) {
            if ($term->parent == $parent_id) {
                $display_name = $indent . $term->name;
                
                // Add count of child terms for parent regions
                if ($parent_id == 0) {
                    $child_count = $this->count_child_terms($terms, $term->term_id);
                    if ($child_count > 0) {
                        $display_name .= '';
                    }
                }
                
                echo '<option value="' . esc_attr($term->term_id) . '" data-taxonomy="' . esc_attr($taxonomy_name) . '">';
                echo esc_html($display_name);
                echo '</option>';
                
                // Recursively add child terms with indentation
                $this->build_hierarchical_region_options($terms, $taxonomy_name, $term->term_id, $indent . '&nbsp;&nbsp;&nbsp;&nbsp;');
            }
        }
    }

    /**
     * Count child terms for a given parent
     * 
     * @param array $terms All terms
     * @param int $parent_id Parent term ID
     * @return int Number of child terms
     */
    private function count_child_terms($terms, $parent_id) {
        $count = 0;
        foreach ($terms as $term) {
            if ($term->parent == $parent_id) {
                $count++;
            }
        }
        return $count;
    }

    /**
     * Enqueue Google Maps API and map functionality scripts
     */
    public function enqueue_map_scripts($hook) {
        // Only load on our plugin pages
        if (strpos($hook, 'listeo-data-scraper') === false) {
            return;
        }

        $map_enabled = 1; // Always enabled - map mode is default
        if (!$map_enabled) {
            return;
        }

        $api_key = get_option('lds_google_api_key');
        if (empty($api_key)) {
            return;
        }

        // Enqueue our map functionality script
        wp_enqueue_script(
            'lds-map-js',
            LDS_PLUGIN_URL . 'assets/js/lds-map.js',
            ['jquery'],
            LDS_VERSION,
            false // Load in head, not footer
        );

        // Pass settings to JavaScript
        $map_settings = [
            'default_lat' => get_option('lds_default_map_center_lat', '51.5074'),
            'default_lng' => get_option('lds_default_map_center_lng', '-0.1278'),
            'default_zoom' => (int) get_option('lds_map_zoom_level', 12),
            'default_radius' => (float) get_option('lds_default_radius', 2.0),
            'ajax_url' => admin_url('admin-ajax.php')
        ];

        wp_localize_script('lds-map-js', 'ldsMapSettings', $map_settings);

        // Enqueue Google Maps API after our script
        wp_enqueue_script(
            'google-maps-api',
            "https://maps.googleapis.com/maps/api/js?key={$api_key}&libraries=places",
            ['lds-map-js'],
            null,
            false // Load in head, not footer
        );
    }
}