<?php
/**
 * Plugin Name: AI Chat & Search
 * Plugin URI: https://purethemes.net
 * Description: AI-powered semantic search and conversational chat with natural language queries
 * Version: 1.2
 * Author: PureThemes
 * Author URI: https://purethemes.net
 * License: GPL2
 * Text Domain: listeo-ai-search
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.3
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('LISTEO_AI_SEARCH_VERSION', '1.2');
define('LISTEO_AI_SEARCH_PLUGIN_URL', plugin_dir_url(__FILE__));
define('LISTEO_AI_SEARCH_PLUGIN_PATH', plugin_dir_path(__FILE__));

/**
 * Main plugin class
 */
class Listeo_AI_Search {
    
    /**
     * Single instance
     */
    private static $instance = null;
    
    /**
     * Search handler instance
     * 
     * @var Listeo_AI_Search_Search_Handler
     */
    private $search_handler;
    
    /**
     * Shortcode handler instance
     * 
     * @var Listeo_AI_Search_Shortcode_Handler
     */
    private $shortcode_handler;
    
    /**
     * Admin interface instance
     * 
     * @var Listeo_AI_Search_Admin_Interface
     */
    private $admin_interface;
    
    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Load dependencies first
        $this->load_dependencies();

        // Initialize AJAX handlers early (before init)
        $this->search_handler = new Listeo_AI_Search_Search_Handler();

        add_action('init', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
        // Initialize remaining components
        $this->shortcode_handler = new Listeo_AI_Search_Shortcode_Handler();
        $this->admin_interface = new Listeo_AI_Search_Admin_Interface();
        
        // Initialize background processor if available
        if (class_exists('Listeo_AI_Background_Processor')) {
            Listeo_AI_Background_Processor::init();
        }
        
        // Auto-process listings
        add_action('save_post', array($this, 'process_listing_on_save'), 10, 2);
        
        // Background processing
        add_action('listeo_ai_process_background_batches', array($this, 'process_background_batches'));
    }
    
    /**
     * Load all required class files
     */
    private function load_dependencies() {
        // Content extractors (must load before embedding manager)
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/content-extractors/class-content-extractor-factory.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/content-extractors/class-content-extractor-listing.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/content-extractors/class-content-extractor-post.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/content-extractors/class-content-extractor-page.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/content-extractors/class-content-extractor-product.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/content-extractors/class-content-extractor-default.php';

        // Core utility classes
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-utility-helper.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-result-formatter.php';

        // Pro features manager (loads first to provide hooks)
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-pro-manager.php';

        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-embedding-manager.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-database-manager.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-analytics.php';

        // Manual batch processor for large catalogs
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-manual-batch-processor.php';

        // Search engines
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/search/class-fallback-engine.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/search/class-ai-engine.php';

        // Main handlers
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-search-handler.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/frontend/class-shortcode-handler.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/admin/class-admin-interface.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/admin/class-universal-settings.php';
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-listeo-field-integration.php';

        // Chat API for conversational search
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-chat-api.php';

        // Chat history tracking
        // Load from free version unless Pro has already loaded it
        if (!class_exists('Listeo_AI_Search_Chat_History')) {
            require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-chat-history.php';
        }

        // Chat shortcode
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-chat-shortcode.php';

        // Floating chat widget
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-floating-chat-widget.php';

        // Listeo integration (conditional - only loads if Listeo theme/core active)
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-listeo-detection.php';

        if (file_exists(LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/integrations/class-listeo-integration.php')) {
            require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/integrations/class-listeo-integration.php';
            // Initialize Listeo integration if available
            if (Listeo_AI_Detection::is_listeo_available()) {
                new Listeo_AI_Integration();
            }
        }

        // WooCommerce integration (conditional - only loads if WooCommerce active)
        // Load on plugins_loaded to ensure WooCommerce is fully initialized
        if (file_exists(LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/integrations/class-woocommerce-integration.php')) {
            require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/integrations/class-woocommerce-integration.php';
            add_action('plugins_loaded', function() {
                if (class_exists('WooCommerce')) {
                    new Listeo_AI_WooCommerce_Integration();
                }
            }, 20); // Priority 20 to ensure WooCommerce is loaded
        }

        // Background processor (existing)
        if (file_exists(LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-background-processor.php')) {
            require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-background-processor.php';
        }

        // Plugin updater (self-hosted updates)
        require_once LISTEO_AI_SEARCH_PLUGIN_PATH . 'includes/class-updater.php';
    }
    
    /**
     * Enqueue scripts and styles
     */
    public function enqueue_scripts() {
        wp_enqueue_script(
            'listeo-ai-search',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/js/search.js',
            array('jquery'),
            LISTEO_AI_SEARCH_VERSION,
            true
        );
        
        // Get placeholder image from theme/core
        $placeholder_url = '';
        if (function_exists('get_listeo_core_placeholder_image')) {
            $placeholder = get_listeo_core_placeholder_image();
            if (is_numeric($placeholder)) {
                $placeholder_img = wp_get_attachment_image_src($placeholder, 'medium');
                if ($placeholder_img && isset($placeholder_img[0])) {
                    $placeholder_url = $placeholder_img[0];
                }
            } else {
                $placeholder_url = $placeholder;
            }
        }

        // Fallback to theme customizer
        if (empty($placeholder_url)) {
            $placeholder_id = get_theme_mod('listeo_placeholder_id');
            if ($placeholder_id) {
                $placeholder_img = wp_get_attachment_image_src($placeholder_id, 'medium');
                if ($placeholder_img && isset($placeholder_img[0])) {
                    $placeholder_url = $placeholder_img[0];
                }
            }
        }

        wp_localize_script('listeo-ai-search', 'listeoAiSearch', array(
            'ajax_url' => get_admin_url(get_current_blog_id(), 'admin-ajax.php'),
            'nonce' => wp_create_nonce('listeo_ai_search_nonce'),
            'debugMode' => (bool) get_option('listeo_ai_search_debug_mode', false), // Debug mode from settings
            'ai_enabled' => true, // AI search is always enabled
            'max_results' => intval(get_option('listeo_ai_search_max_results', 10)),
            'search_url' => get_post_type_archive_link('listing') ?: home_url(), // Use proper archive URL or fallback to home
            'default_thumbnail' => $placeholder_url, // Theme placeholder image
            'strings' => array(
                'searching' => __('Searching...', 'listeo-ai-search'),
                'no_results' => __('No results found.', 'listeo-ai-search'),
                'error' => __('Search error occurred.', 'listeo-ai-search'),
                'best_match' => __('Best Match', 'listeo-ai-search'),
                'type_keywords_first' => __('Type keywords first', 'listeo-ai-search'),
                'top_listing_singular' => __('Top 1 listing matching', 'listeo-ai-search'),
                'top_listings_plural' => __('Top %d listings matching', 'listeo-ai-search'),
                // Error messages for search
                'rateLimitError' => __('Too many searches. Please wait a moment and try again.', 'listeo-ai-search'),
                'apiUnavailable' => __('AI search temporarily unavailable. Using basic search instead.', 'listeo-ai-search'),
                'sessionExpired' => __('Session expired. Please refresh the page.', 'listeo-ai-search'),
                'fallbackNotice' => __('Showing regular search results instead.', 'listeo-ai-search'),
                // Stock status
                'inStock' => __('In Stock', 'listeo-ai-search'),
                'outOfStock' => __('Out of Stock', 'listeo-ai-search'),
            )
        ));
        
        wp_enqueue_style(
            'listeo-ai-search',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/css/search.css',
            array(),
            LISTEO_AI_SEARCH_VERSION
        );
    }
    
    /**
     * Process listing on save
     * 
     * @param int $post_id Post ID
     * @param WP_Post $post Post object
     */
    public function process_listing_on_save($post_id, $post) {
        Listeo_AI_Search_Database_Manager::process_listing_on_save($post_id, $post);
    }
    
    /**
     * Custom debug logging to debug.log
     * 
     * @param string $message Log message
     * @param string $level Log level (info, error, warning, debug)
     */
    public static function debug_log($message, $level = 'info') {
        // Only log if debug mode is enabled
        if (!get_option('listeo_ai_search_debug_mode', false)) {
            return;
        }
        
        // Use WordPress standard debug logging
        $timestamp = date('Y-m-d H:i:s');
        $formatted_message = "[{$timestamp}] [{$level}] AI Chat: {$message}";

        error_log($formatted_message);
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Initialize default settings (only if not already set)
        $this->initialize_default_settings();

        // Create database tables
        Listeo_AI_Search_Database_Manager::create_tables();

        // Create manual batch tracking table
        Listeo_AI_Search_Manual_Batch_Processor::create_batch_table();

        // Create chat history table (only if feature is enabled)
        if (get_option('listeo_ai_chat_history_enabled', 0)) {
            Listeo_AI_Search_Chat_History::create_table();
        }

        // Schedule bulk processing of existing listings (only if not in manual mode)
        if (!Listeo_AI_Search_Manual_Batch_Processor::should_use_manual_mode()) {
            wp_schedule_single_event(time() + 60, 'listeo_ai_bulk_process_listings');
        }

        // Schedule weekly chat history cleanup (runs every 7 days)
        if (!wp_next_scheduled('listeo_ai_cleanup_chat_history')) {
            wp_schedule_event(time(), 'weekly', 'listeo_ai_cleanup_chat_history');
        }
    }

    /**
     * Get default settings array
     * Centralized defaults for consistency across plugin
     *
     * @return array Default settings
     */
    public static function get_default_settings() {
        return array(
            // Search settings
            'listeo_ai_search_min_match_percentage' => 50,
            'listeo_ai_search_best_match_threshold' => 75,
            'listeo_ai_search_max_results' => 10,
            'listeo_ai_search_rate_limit_per_hour' => 200,
            'listeo_ai_search_batch_size' => 500,
            'listeo_ai_search_embedding_delay' => 5,

            // Chat settings
            'listeo_ai_chat_enabled' => 1,
            'listeo_ai_chat_name' => __('AI Assistant', 'listeo-ai-search'),
            'listeo_ai_chat_welcome_message' => __('Hello! How can I help you today?', 'listeo-ai-search'),
            'listeo_ai_chat_system_prompt' => '',
            'listeo_ai_chat_model' => 'gpt-5-chat-latest',
            'listeo_ai_chat_max_tokens' => 500,
            'listeo_ai_chat_temperature' => 0.5,
            'listeo_ai_chat_max_results' => 10,
            'listeo_ai_chat_hide_images' => 1,
            'listeo_ai_chat_require_login' => 0,
            'listeo_ai_chat_history_enabled' => 1,
            'listeo_ai_chat_retention_days' => 30,
            'listeo_ai_chat_terms_notice_enabled' => 0,
            'listeo_ai_chat_terms_notice_text' => '',
            'listeo_ai_chat_rate_limit_tier1' => 10,
            'listeo_ai_chat_rate_limit_tier2' => 30,
            'listeo_ai_chat_rate_limit_tier3' => 100,

            // Floating widget settings
            'listeo_ai_floating_chat_enabled' => 0,
            'listeo_ai_floating_button_icon' => 'default',
            'listeo_ai_floating_custom_icon' => 0,
            'listeo_ai_floating_welcome_bubble' => __('Hi! How can I help you?', 'listeo-ai-search'),
            'listeo_ai_floating_popup_width' => 390,
            'listeo_ai_floating_popup_height' => 600,
            'listeo_ai_floating_button_color' => '#222222',
            'listeo_ai_primary_color' => '#0073ee',

            // Checkbox settings
            'listeo_ai_search_ai_location_filtering_enabled' => 0,
            'listeo_ai_search_debug_mode' => 0,
            'listeo_ai_search_disable_cache' => 0,
            'listeo_ai_search_query_expansion' => 0,
            'listeo_ai_search_enable_analytics' => 1,
            'listeo_ai_search_suggestions_enabled' => 0
        );
    }

    /**
     * Initialize default settings
     * Only sets values if they don't exist in database
     */
    private function initialize_default_settings() {
        $defaults = self::get_default_settings();

        foreach ($defaults as $option_name => $default_value) {
            // Only add if option doesn't exist (won't overwrite existing values)
            if (get_option($option_name) === false) {
                add_option($option_name, $default_value);
            }
        }
    }

    /**
     * Process background batches
     */
    public function process_background_batches() {
        global $wpdb;
        
        // Enhanced debug logging
        if (get_option('listeo_ai_search_debug_mode', false)) {
            self::debug_log("Background batch processing started", 'info');
        }
        
        // Get one queued batch to process
        $batch_table = Listeo_AI_Search_Manual_Batch_Processor::get_batch_table_name();
        $batch = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$batch_table} WHERE status = 'queued' ORDER BY created_at ASC LIMIT 1"
        ), ARRAY_A);
        
        if (!$batch) {
            if (get_option('listeo_ai_search_debug_mode', false)) {
                self::debug_log("No queued batches found - background processing complete", 'info');
            }
            return; // No queued batches
        }
        
        if (get_option('listeo_ai_search_debug_mode', false)) {
            self::debug_log("Processing batch {$batch['batch_id']} with {$batch['total_listings']} listings", 'info');
        }
        
        try {
            // Process this batch
            $result = Listeo_AI_Search_Manual_Batch_Processor::process_single_batch($batch['batch_id'], null);
            
            if (get_option('listeo_ai_search_debug_mode', false)) {
                $status = $result['success'] ? 'SUCCESS' : 'FAILED';
                self::debug_log("Batch {$batch['batch_id']} processing {$status}: " . $result['message'], $result['success'] ? 'info' : 'error');
            }
            
        } catch (Exception $e) {
            if (get_option('listeo_ai_search_debug_mode', false)) {
                self::debug_log("CRITICAL ERROR processing batch {$batch['batch_id']}: " . $e->getMessage(), 'error');
                self::debug_log("Stack trace: " . $e->getTraceAsString(), 'error');
            }
            
            // Mark batch as failed
            $wpdb->update(
                $batch_table,
                array(
                    'status' => 'failed',
                    'error_message' => 'Critical error: ' . $e->getMessage(),
                    'updated_at' => current_time('mysql')
                ),
                array('batch_id' => $batch['batch_id']),
                array('%s', '%s', '%s'),
                array('%s')
            );
        }
        
        // Check if there are more queued batches
        $remaining_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$batch_table} WHERE status = 'queued'"
        ));
        
        if (get_option('listeo_ai_search_debug_mode', false)) {
            self::debug_log("Remaining queued batches: {$remaining_count}", 'info');
        }
        
        // Schedule next batch processing if there are more queued batches
        if ($remaining_count > 0) {
            wp_schedule_single_event(time() + 30, 'listeo_ai_process_background_batches');
            if (get_option('listeo_ai_search_debug_mode', false)) {
                self::debug_log("Scheduled next background batch processing in 30 seconds", 'info');
            }
        }
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clean up scheduled events
        wp_clear_scheduled_hook('listeo_ai_bulk_process_listings');
        wp_clear_scheduled_hook('listeo_ai_process_listing');
        wp_clear_scheduled_hook('listeo_ai_process_background_batches');
        wp_clear_scheduled_hook('listeo_ai_cleanup_chat_history');
    }
}

/**
 * Get chat strings for localization (shared by shortcode and floating widget)
 * Centralized in one place to avoid duplication
 *
 * @param string $welcome_message Custom welcome message
 * @return array Localized strings
 */
function listeo_ai_get_chat_strings($welcome_message = '') {
    if (empty($welcome_message)) {
        $welcome_message = __('Hello! How can I help you today?', 'listeo-ai-search');
    }

    return array(
        'placeholder' => __('Ask about listings...', 'listeo-ai-search'),
        'sendButton' => __('Send', 'listeo-ai-search'),
        'welcomeMessage' => $welcome_message,
        'loading' => __('Thinking...', 'listeo-ai-search'),
        'loadingConfig' => __('Please wait, loading configuration...', 'listeo-ai-search'),
        'gettingDetails' => __('Getting listing details...', 'listeo-ai-search'),
        'searchingDatabase' => __('Thinking...', 'listeo-ai-search'),
        'analyzingResults' => __('Analyzing...', 'listeo-ai-search'),
        'generatingAnswer' => __('Thinking...', 'listeo-ai-search'),
        'analyzingListing' => __('Analyzing listing...', 'listeo-ai-search'),
        'loadingButton' => __('Loading...', 'listeo-ai-search'),
        'talkAboutListing' => __('Talk about this listing', 'listeo-ai-search'),
        'listingNotFound' => __('Sorry, I couldn\'t find details for that listing.', 'listeo-ai-search'),
        'errorGettingDetails' => __('Error getting listing details.', 'listeo-ai-search'),
        'errorLoadingListing' => __('Error loading listing.', 'listeo-ai-search'),
        'failedLoadDetails' => __('Failed to load listing details.', 'listeo-ai-search'),
        'errorApiKey' => __('OpenAI API Key is not configured. Please contact the site administrator.', 'listeo-ai-search'),
        'errorConfig' => __('Failed to load chat configuration. Please try again later.', 'listeo-ai-search'),
        'errorGeneral' => __('Sorry, an error occurred. Please try again.', 'listeo-ai-search'),
        'chatDisabled' => __('AI Chat is currently disabled.', 'listeo-ai-search'),
        'bestMatch' => __('Best Match', 'listeo-ai-search'),
        'showMore' => __('Show more (%d)', 'listeo-ai-search'),
        'listingContextLoaded' => __('Listing context loaded! You can now ask me anything about', 'listeo-ai-search'),
        // Chat loading states
        'searchingListings' => __('Searching listings...', 'listeo-ai-search'),
        'searchingProducts' => __('Searching products...', 'listeo-ai-search'),
        'searchingSiteContent' => __('Searching site content...', 'listeo-ai-search'),
        'analyzingProducts' => __('Analyzing products...', 'listeo-ai-search'),
        'gettingProductDetails' => __('Getting product details...', 'listeo-ai-search'),
        'analyzingProduct' => __('Analyzing product...', 'listeo-ai-search'),
        'analyzingContent' => __('Analyzing content...', 'listeo-ai-search'),
        // Error messages
        'searchFailed' => __('Search failed. Please try again.', 'listeo-ai-search'),
        'productSearchFailed' => __('Product search failed. Please try again.', 'listeo-ai-search'),
        'unknownFunction' => __('Unknown function requested.', 'listeo-ai-search'),
        'contentNotFound' => __('Content not found or not published.', 'listeo-ai-search'),
        'errorGettingContent' => __('Error getting content details.', 'listeo-ai-search'),
        'productNotFound' => __('Product not found or not published.', 'listeo-ai-search'),
        'errorGettingProduct' => __('Error getting product details.', 'listeo-ai-search'),
        // Rate limit messages
        'rateLimitPrefix' => __('You\'ve reached the limit of', 'listeo-ai-search'),
        'rateLimitSuffix' => __('messages per', 'listeo-ai-search'),
        'rateLimitWait' => __('Please wait', 'listeo-ai-search'),
        'rateLimitBeforeTrying' => __('before trying again.', 'listeo-ai-search'),
        'minute' => __('minute', 'listeo-ai-search'),
        'minutes' => __('minutes', 'listeo-ai-search'),
        'second' => __('second', 'listeo-ai-search'),
        'seconds' => __('seconds', 'listeo-ai-search'),
        // Stock status
        'inStock' => __('In Stock', 'listeo-ai-search'),
        'outOfStock' => __('Out of Stock', 'listeo-ai-search')
    );
}

// Initialize plugin
Listeo_AI_Search::get_instance();

/**
 * Chat history cleanup cron job
 * Runs weekly to delete old chat records based on retention setting
 */
add_action('listeo_ai_cleanup_chat_history', function() {
    if (!class_exists('Listeo_AI_Search_Chat_History')) {
        return;
    }

    // Get retention days from settings (default: 30 days)
    $retention_days = get_option('listeo_ai_chat_retention_days', 30);

    // Run cleanup
    $deleted = Listeo_AI_Search_Chat_History::cleanup_old_records($retention_days);

    // Log if debug mode enabled
    if (get_option('listeo_ai_search_debug_mode', false) && $deleted !== false) {
        Listeo_AI_Search::debug_log("Chat history cleanup: Deleted {$deleted} records older than {$retention_days} days", 'info');
    }
});
