/**
 * Manual Batch Interface JavaScript
 * 
 * Handles the frontend interaction for the manual batch processing interface
 * 
 * @package Listeo_AI_Search
 * @since 1.2.0
 */

(function($) {
    'use strict';

    var ManualBatchInterface = {
        processing: false,
        stopRequested: false,
        progressPollingInterval: null,
        
        init: function() {
            this.bindEvents();
            this.updateButtonStates();
            this.updateProcessButtonText();
            this.startProgressPolling();
        },
        
        bindEvents: function() {
            var self = this;
            
            // Batch configuration
            $('#initialize-batches').on('click', this.initializeBatches.bind(this));
            $('#save-batch-settings').on('click', this.saveBatchSettings.bind(this));
            
            // Batch selection
            $('#batch-select-all').on('change', this.toggleAllBatches);
            $('.batch-checkbox').on('change', this.updateButtonStates.bind(this));
            
            // Bulk actions
            $('#select-all-batches').on('click', function() { self.selectBatches('all'); });
            $('#select-none-batches').on('click', function() { self.selectBatches('none'); });
            $('#select-ready-batches').on('click', function() { self.selectBatches('ready'); });
            $('#select-failed-batches').on('click', function() { self.selectBatches('failed'); });
            
            // Processing actions
            $('#process-selected-batches').on('click', this.processSelectedBatches.bind(this));
            $('#stop-processing').on('click', this.stopProcessing.bind(this));
            
            // Individual batch actions
            $(document).on('click', '.retry-batch', this.retryBatch.bind(this));
            $(document).on('click', '.show-error', this.showBatchError.bind(this));
            
            // Reset action
            $('#reset-all-batches').on('click', this.showResetConfirmation.bind(this));
            $('#confirm-reset').on('click', this.resetAllBatches.bind(this));
            $('#cancel-reset').on('click', this.hideResetConfirmation.bind(this));
        },
        
        initializeBatches: function() {
            var self = this;
            var batchSize = $('#batch-size').val();
            
            this.showStatus('info', 'Initializing batches...');
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'listeo_ai_manual_batch_action',
                    batch_action: 'initialize_batches',
                    batch_size: batchSize,
                    nonce: listeo_ai_search_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.showStatus('success', response.data.message);
                        // Reload the page to show new batches
                        setTimeout(function() {
                            window.location.reload();
                        }, 1500);
                    } else {
                        self.showStatus('error', response.data.message || 'Failed to initialize batches');
                    }
                },
                error: function() {
                    self.showStatus('error', 'AJAX error occurred while initializing batches');
                }
            });
        },
        
        saveBatchSettings: function() {
            var self = this;
            var batchSize = $('#batch-size').val();
            var delay = $('#batch-delay').val();
            // Background processing is always enabled in Safe Mode
            var backgroundProcessing = 1;
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'listeo_ai_manual_batch_action',
                    batch_action: 'save_settings',
                    batch_size: batchSize,
                    batch_delay: delay,
                    background_processing: backgroundProcessing,
                    nonce: listeo_ai_search_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.showStatus('success', 'Settings saved successfully');
                    } else {
                        self.showStatus('error', response.data.message || 'Failed to save settings');
                    }
                },
                error: function() {
                    self.showStatus('error', 'AJAX error occurred while saving settings');
                }
            });
        },
        
        updateProcessButtonText: function() {
            // Background processing is always enabled in Safe Mode
            var processButton = $('#process-selected-batches .button-text');
            processButton.html('Queue for Background Processing');
        },
        
        restoreButtonState: function() {
            var $button = $('#process-selected-batches');
            var $buttonText = $button.find('.button-text');
            var $buttonSpinner = $button.find('.button-spinner');
            
            $button.prop('disabled', false);
            $buttonSpinner.hide();
            $buttonText.show();
        },
        
        toggleAllBatches: function() {
            var isChecked = $(this).is(':checked');
            $('.batch-checkbox:not(:disabled)').prop('checked', isChecked);
            ManualBatchInterface.updateButtonStates();
        },
        
        selectBatches: function(type) {
            $('.batch-checkbox').prop('checked', false);
            
            switch(type) {
                case 'all':
                    $('.batch-checkbox:not(:disabled)').prop('checked', true);
                    break;
                case 'ready':
                    $('.batch-row[data-status="ready"] .batch-checkbox').prop('checked', true);
                    break;
                case 'failed':
                    $('.batch-row[data-status="failed"] .batch-checkbox, .batch-row[data-status="partial"] .batch-checkbox').prop('checked', true);
                    break;
            }
            
            this.updateButtonStates();
        },
        
        updateButtonStates: function() {
            var selectedCount = $('.batch-checkbox:checked').length;
            var $processButton = $('#process-selected-batches');
            var $buttonText = $processButton.find('.button-text');
            
            if (selectedCount > 0 && !this.processing) {
                $processButton.prop('disabled', false);
                $buttonText.text('🚀 Generate Embeddings (' + selectedCount + ' batch' + (selectedCount > 1 ? 'es' : '') + ')');
            } else {
                $processButton.prop('disabled', true);
                $buttonText.text('🚀 Generate Embeddings');
            }
        },
        
        processSelectedBatches: function() {
            if (this.processing) {
                return;
            }
            
            var selectedBatches = [];
            $('.batch-checkbox:checked').each(function() {
                selectedBatches.push($(this).val());
            });
            
            if (selectedBatches.length === 0) {
                this.showStatus('error', 'No batches selected for processing');
                return;
            }
            
            // Show loading state on button
            var $button = $('#process-selected-batches');
            var $buttonText = $button.find('.button-text');
            var $buttonSpinner = $button.find('.button-spinner');
            
            $button.prop('disabled', true);
            $buttonText.hide();
            $buttonSpinner.show();
            
            // Background processing is always enabled in Safe Mode
            this.queueSelectedBatches(selectedBatches);
        },
        
        queueSelectedBatches: function(selectedBatches) {
            var self = this;
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'listeo_ai_manual_batch_action',
                    batch_action: 'queue_batches',
                    batch_ids: selectedBatches,
                    nonce: listeo_ai_search_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.showStatus('success', response.data.message);
                        // Reload to show updated status
                        setTimeout(function() {
                            window.location.reload();
                        }, 2000);
                    } else {
                        self.showStatus('error', response.data.message || 'Failed to queue batches');
                        // Restore button state on error
                        self.restoreButtonState();
                    }
                },
                error: function() {
                    self.showStatus('error', 'AJAX error occurred while queueing batches');
                    // Restore button state on error
                    self.restoreButtonState();
                }
            });
        },
        
        startForegroundProcessing: function(selectedBatches) {
            this.processing = true;
            this.stopRequested = false;
            
            // Update UI for processing state
            $('#process-selected-batches').hide();
            $('#stop-processing').show();
            $('.batch-checkbox').prop('disabled', true);
            
            this.showStatus('info', 'Starting batch processing...');
            this.showProgressBar();
            
            this.processBatchSequentially(selectedBatches, 0);
        },
        
        processBatchSequentially: function(batchIds, currentIndex) {
            var self = this;
            
            if (this.stopRequested || currentIndex >= batchIds.length) {
                this.finishProcessing();
                return;
            }
            
            var batchId = batchIds[currentIndex];
            var delay = $('#batch-delay').val();
            
            // Update progress
            var progress = ((currentIndex / batchIds.length) * 100).toFixed(1);
            this.updateProgressBar(progress, 'Processing batch ' + (currentIndex + 1) + ' of ' + batchIds.length + '...');
            
            // Mark batch as processing in UI
            var $batchRow = $('.batch-row[data-batch-id="' + batchId + '"]');
            $batchRow.attr('data-status', 'processing');
            $batchRow.find('.batch-status').html('<span class="status-icon">🔄</span><span class="status-text">Processing</span>');
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'listeo_ai_manual_batch_action',
                    batch_action: 'process_batch',
                    batch_id: batchId,
                    batch_delay: delay,
                    nonce: listeo_ai_search_ajax.nonce
                },
                success: function(response) {
                    self.updateBatchRowAfterProcessing(batchId, response);
                    
                    // Continue with next batch
                    setTimeout(function() {
                        self.processBatchSequentially(batchIds, currentIndex + 1);
                    }, 1000); // Small delay between batches
                },
                error: function() {
                    self.updateBatchRowAfterProcessing(batchId, {
                        success: false,
                        data: { message: 'AJAX error during batch processing' }
                    });
                    
                    // Continue with next batch even if this one failed
                    setTimeout(function() {
                        self.processBatchSequentially(batchIds, currentIndex + 1);
                    }, 1000);
                }
            });
        },
        
        updateBatchRowAfterProcessing: function(batchId, response) {
            var $batchRow = $('.batch-row[data-batch-id="' + batchId + '"]');
            
            if (response.success) {
                var status = response.data.status || 'complete';
                var statusIcon = status === 'complete' ? '✓' : (status === 'partial' ? '⚙️' : '⚠️');
                var statusText = status.charAt(0).toUpperCase() + status.slice(1);
                
                $batchRow.attr('data-status', status);
                $batchRow.find('.batch-status').html(
                    '<span class="status-icon">' + statusIcon + '</span>' +
                    '<span class="status-text">' + statusText + '</span>'
                );
                
                // Update progress if available
                if (response.data.processed_count !== undefined) {
                    var processedCount = response.data.processed_count;
                    var totalCount = parseInt($batchRow.find('.batch-count').text());
                    var progressPercent = totalCount > 0 ? ((processedCount / totalCount) * 100).toFixed(1) : 0;
                    
                    $batchRow.find('.progress-fill-small').css('width', progressPercent + '%');
                    $batchRow.find('.progress-text-small').text(processedCount + '/' + totalCount + ' (' + progressPercent + '%)');
                }
                
            } else {
                $batchRow.attr('data-status', 'failed');
                $batchRow.find('.batch-status').html(
                    '<span class="status-icon">⚠️</span>' +
                    '<span class="status-text">Failed</span>'
                );
            }
            
            // Update batch actions
            this.updateBatchActions($batchRow);
        },
        
        updateBatchActions: function($batchRow) {
            var status = $batchRow.attr('data-status');
            var batchId = $batchRow.attr('data-batch-id');
            var $actionsCell = $batchRow.find('.batch-actions');
            
            $actionsCell.empty();
            
            if (status === 'failed' || status === 'partial') {
                $actionsCell.append(
                    '<button type="button" class="button button-small retry-batch" data-batch-id="' + batchId + '">Retry</button>'
                );
            }
        },
        
        stopProcessing: function() {
            this.stopRequested = true;
            this.showStatus('info', 'Stopping processing after current batch...');
        },
        
        finishProcessing: function() {
            this.processing = false;
            this.stopRequested = false;
            
            // Update UI back to normal state
            $('#stop-processing').hide();
            $('#process-selected-batches').show();
            $('.batch-checkbox').prop('disabled', false);
            
            this.hideProgressBar();
            this.updateButtonStates();
            this.refreshOverallProgress();
            
            var message = this.stopRequested ? 
                'Processing stopped by user' : 
                'Batch processing completed';
            
            this.showStatus('success', message);
        },
        
        retryBatch: function(e) {
            var self = this;
            var batchId = $(e.target).data('batch-id');
            var delay = $('#batch-delay').val();
            
            this.showStatus('info', 'Retrying batch ' + batchId + '...');
            
            // Mark batch as processing
            var $batchRow = $('.batch-row[data-batch-id="' + batchId + '"]');
            $batchRow.attr('data-status', 'processing');
            $batchRow.find('.batch-status').html('<span class="status-icon">🔄</span><span class="status-text">Processing</span>');
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'listeo_ai_manual_batch_action',
                    batch_action: 'retry_batch',
                    batch_id: batchId,
                    batch_delay: delay,
                    nonce: listeo_ai_search_ajax.nonce
                },
                success: function(response) {
                    self.updateBatchRowAfterProcessing(batchId, response);
                    
                    if (response.success) {
                        self.showStatus('success', 'Batch ' + batchId + ' retry completed');
                    } else {
                        self.showStatus('error', response.data.message || 'Batch retry failed');
                    }
                    
                    self.refreshOverallProgress();
                },
                error: function() {
                    self.updateBatchRowAfterProcessing(batchId, {
                        success: false,
                        data: { message: 'AJAX error during batch retry' }
                    });
                    self.showStatus('error', 'AJAX error occurred during batch retry');
                }
            });
        },
        
        showBatchError: function(e) {
            var batchId = $(e.target).data('batch-id');
            var $errorDetails = $('.batch-row[data-batch-id="' + batchId + '"] .error-details');
            
            if ($errorDetails.is(':visible')) {
                $errorDetails.hide();
                $(e.target).text('Show Error');
            } else {
                $errorDetails.show();
                $(e.target).text('Hide Error');
            }
        },
        
        showResetConfirmation: function() {
            $('#reset-confirmation-modal').show();
        },
        
        hideResetConfirmation: function() {
            $('#reset-confirmation-modal').hide();
        },
        
        resetAllBatches: function() {
            var self = this;
            
            this.hideResetConfirmation();
            this.showStatus('info', 'Resetting all batches and embeddings...');
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'listeo_ai_manual_batch_action',
                    batch_action: 'reset_all',
                    nonce: listeo_ai_search_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.showStatus('success', response.data.message);
                        // Reload the page after reset
                        setTimeout(function() {
                            window.location.reload();
                        }, 2000);
                    } else {
                        self.showStatus('error', response.data.message || 'Failed to reset batches');
                    }
                },
                error: function() {
                    self.showStatus('error', 'AJAX error occurred during reset');
                }
            });
        },
        
        showStatus: function(type, message) {
            var $container = $('#batch-status-messages');
            
            // Define colors for different message types
            var styles = {
                'success': {
                    background: '#d4edda',
                    border: '#c3e6cb', 
                    color: '#155724'
                },
                'info': {
                    background: '#d1ecf1',
                    border: '#bee5eb',
                    color: '#0c5460'
                },
                'warning': {
                    background: '#fff3cd',
                    border: '#ffeaa7',
                    color: '#856404'
                },
                'error': {
                    background: '#f8d7da',
                    border: '#f5c6cb',
                    color: '#721c24'
                }
            };
            
            var style = styles[type] || styles['info'];
            
            // Create a new notification div with custom styling
            var notificationHtml = '<div class="batch-status-message" style="background: ' + style.background + '; border: 1px solid ' + style.border + '; border-radius: 4px; padding: 12px; margin-bottom: 15px; color: ' + style.color + ';">' +
                '<p style="margin: 0;">' + message + '</p>' +
                '</div>';
            
            // Clear previous messages and add new one
            $container.html(notificationHtml);
            $container.show();
            
            // Auto-hide success messages after 5 seconds
            if (type === 'success') {
                setTimeout(function() {
                    $container.fadeOut();
                }, 5000);
            }
        },
        
        showProgressBar: function() {
            $('#processing-progress-bar').show();
        },
        
        hideProgressBar: function() {
            $('#processing-progress-bar').hide();
        },
        
        updateProgressBar: function(percent, text) {
            $('#processing-progress-fill').css('width', percent + '%');
            $('#processing-progress-text').text(text);
        },
        
        startProgressPolling: function() {
            var self = this;
            
            // Poll for progress updates every 30 seconds
            this.progressPollingInterval = setInterval(function() {
                self.refreshOverallProgress();
            }, 30000);
        },
        
        refreshOverallProgress: function() {
            var self = this;
            
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'listeo_ai_manual_batch_action',
                    batch_action: 'get_progress',
                    nonce: listeo_ai_search_ajax.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.updateOverallProgressDisplay(response.data);
                    }
                },
                error: function() {
                    // Silently fail for progress polling
                }
            });
        },
        
        updateOverallProgressDisplay: function(progressData) {
            // Update the main progress stats
            $('.progress-stats .stat-box:nth-child(2) .stat-number').text(progressData.processed_listings.toLocaleString());
            $('.progress-stats .stat-box:nth-child(3) .stat-number').text(progressData.remaining_listings.toLocaleString());
            $('.progress-stats .stat-box:nth-child(4) .stat-number').text(progressData.progress_percentage + '%');
            
            // Update the main progress bar
            $('.progress-fill').css('width', progressData.progress_percentage + '%');
            $('.progress-text').text(
                progressData.processed_listings.toLocaleString() + ' of ' + 
                progressData.total_listings.toLocaleString() + ' listings processed (' + 
                progressData.progress_percentage + '%)'
            );
            
            // Update batch summary if available
            if (progressData.batch_summary) {
                var summary = progressData.batch_summary;
                $('.batch-stat.ready .count').text(summary.ready);
                $('.batch-stat.processing .count').text(summary.processing);
                $('.batch-stat.complete .count').text(summary.complete);
                $('.batch-stat.failed .count').text(summary.failed);
                $('.batch-stat.partial .count').text(summary.partial);
            }
        }
    };
    
    // Initialize when document is ready
    $(document).ready(function() {
        // Only initialize if we're on the database tab with manual batch interface
        if ($('#batch-list-body').length > 0 || $('.manual-batch-overview').length > 0) {
            ManualBatchInterface.init();
        }
    });
    
    // Cleanup on page unload
    $(window).on('beforeunload', function() {
        if (ManualBatchInterface.progressPollingInterval) {
            clearInterval(ManualBatchInterface.progressPollingInterval);
        }
    });

})(jQuery);
