<?php
/**
 * Admin Interface Class
 * 
 * Handles WordPress admin settings and management interface
 * 
 * @package Listeo_AI_Search
 * @since 1.0.5
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Listeo_AI_Search_Admin_Interface {

    /**
     * Centralized Settings Registry - Single Source of Truth
     *
     * This registry eliminates duplication across 10+ locations in the codebase.
     * All settings metadata is defined here once and derived everywhere else.
     *
     * @return array Comprehensive settings configuration
     */
    private function get_settings_registry() {
        return array(
            // ============================================
            // API Configuration Settings
            // ============================================
            'listeo_ai_search_api_key' => array(
                'type' => 'text',
                'section' => 'api-config',
                'sanitize' => 'sanitize_text_field',
                'default' => '',
                'description' => 'OpenAI API Key'
            ),
            'listeo_ai_search_debug_mode' => array(
                'type' => 'checkbox',
                'section' => 'api-config',
                'sanitize' => 'intval',
                'default' => 0,
                'description' => 'Enable debug logging'
            ),
            'listeo_ai_search_disable_cache' => array(
                'type' => 'checkbox',
                'section' => 'api-config',
                'sanitize' => 'intval',
                'default' => 0,
                'description' => 'Disable search result caching'
            ),

            // ============================================
            // Quality & Threshold Settings
            // ============================================
            'listeo_ai_search_query_expansion' => array(
                'type' => 'checkbox',
                'section' => 'quality-thresholds',
                'sanitize' => 'intval',
                'default' => 0,
                'description' => 'Enable query expansion'
            ),
            'listeo_ai_search_min_match_percentage' => array(
                'type' => 'number',
                'section' => 'quality-thresholds',
                'sanitize' => 'intval',
                'default' => 50,
                'description' => 'Minimum match percentage'
            ),
            'listeo_ai_search_best_match_threshold' => array(
                'type' => 'number',
                'section' => 'quality-thresholds',
                'sanitize' => 'intval',
                'default' => 75,
                'description' => 'Best match threshold'
            ),
            'listeo_ai_search_max_results' => array(
                'type' => 'number',
                'section' => 'quality-thresholds',
                'sanitize' => 'intval',
                'default' => 10,
                'description' => 'Maximum search results'
            ),

            // ============================================
            // Search Suggestions Settings
            // ============================================
            'listeo_ai_search_suggestions_enabled' => array(
                'type' => 'checkbox',
                'section' => 'search-suggestions',
                'sanitize' => 'intval',
                'default' => 0,
                'description' => 'Enable search suggestions'
            ),
            'listeo_ai_search_suggestions_source' => array(
                'type' => 'select',
                'section' => 'search-suggestions',
                'sanitize' => 'sanitize_text_field',
                'default' => 'ai',
                'description' => 'Suggestions source'
            ),
            'listeo_ai_search_custom_suggestions' => array(
                'type' => 'textarea',
                'section' => 'search-suggestions',
                'sanitize' => 'sanitize_textarea_field',
                'default' => '',
                'description' => 'Custom suggestions (one per line)'
            ),

            // ============================================
            // Processing Settings
            // ============================================
            'listeo_ai_search_batch_size' => array(
                'type' => 'number',
                'section' => 'processing',
                'sanitize' => 'intval',
                'default' => 500,
                'description' => 'Batch processing size'
            ),
            'listeo_ai_search_embedding_delay' => array(
                'type' => 'number',
                'section' => 'processing',
                'sanitize' => 'intval',
                'default' => 5,
                'description' => 'Delay between embeddings (seconds)'
            ),
            'listeo_ai_search_rate_limit_per_hour' => array(
                'type' => 'number',
                'section' => 'processing',
                'sanitize' => 'intval',
                'default' => 200,
                'min' => 10,
                'max' => 10000,
                'description' => 'API rate limit per hour'
            ),

            // ============================================
            // Analytics Settings
            // ============================================
            'listeo_ai_search_enable_analytics' => array(
                'type' => 'checkbox',
                'section' => 'analytics',
                'sanitize' => 'intval',
                'default' => 1,
                'description' => 'Enable analytics tracking'
            ),

            // ============================================
            // AI Chat Configuration Settings
            // ============================================
            'listeo_ai_chat_system_prompt' => array(
                'type' => 'textarea',
                'section' => 'ai-chat-config',
                'sanitize' => 'sanitize_textarea_field',
                'default' => '',
                'description' => 'System prompt for AI chat'
            ),
            'listeo_ai_chat_enabled' => array(
                'type' => 'checkbox',
                'section' => 'ai-chat-config',
                'sanitize' => 'intval',
                'default' => 1,
                'description' => 'Enable AI chat'
            ),
            'listeo_ai_chat_name' => array(
                'type' => 'text',
                'section' => 'ai-chat-config',
                'sanitize' => 'sanitize_text_field',
                'default' => __('AI Assistant', 'listeo-ai-search'),
                'description' => 'Chat assistant name'
            ),
            'listeo_ai_chat_welcome_message' => array(
                'type' => 'wysiwyg',
                'section' => 'ai-chat-config',
                'sanitize' => 'wp_kses_post',
                'default' => __('Hello! How can I help you today?', 'listeo-ai-search'),
                'description' => 'Welcome message (supports HTML)'
            ),
            'listeo_ai_chat_model' => array(
                'type' => 'select',
                'section' => 'ai-chat-config',
                'sanitize' => 'sanitize_text_field',
                'default' => 'gpt-4o-mini',
                'description' => 'OpenAI model to use'
            ),
            'listeo_ai_chat_max_tokens' => array(
                'type' => 'number',
                'section' => 'ai-chat-config',
                'sanitize' => 'intval',
                'default' => 500,
                'description' => 'Maximum tokens per response'
            ),
            'listeo_ai_chat_temperature' => array(
                'type' => 'float',
                'section' => 'ai-chat-config',
                'sanitize' => 'floatval',
                'default' => 0.5,
                'min' => 0,
                'max' => 2,
                'description' => 'Temperature (0-2)'
            ),
            'listeo_ai_chat_max_results' => array(
                'type' => 'number',
                'section' => 'ai-chat-config',
                'sanitize' => 'intval',
                'default' => 10,
                'description' => 'Maximum results to include in context'
            ),
            'listeo_ai_chat_hide_images' => array(
                'type' => 'checkbox',
                'section' => 'ai-chat-config',
                'sanitize' => 'intval',
                'default' => 1,
                'description' => 'Hide images in chat results'
            ),
            'listeo_ai_chat_require_login' => array(
                'type' => 'checkbox',
                'section' => 'ai-chat-config',
                'sanitize' => 'intval',
                'default' => 0,
                'description' => 'Require login to use chat'
            ),
            'listeo_ai_chat_history_enabled' => array(
                'type' => 'checkbox',
                'section' => 'ai-chat-config',
                'sanitize' => 'intval',
                'default' => 1,
                'description' => 'Enable chat history'
            ),
            'listeo_ai_chat_retention_days' => array(
                'type' => 'number',
                'section' => 'ai-chat-config',
                'sanitize' => 'intval',
                'default' => 30,
                'description' => 'Chat history retention (days)'
            ),
            'listeo_ai_chat_terms_notice_enabled' => array(
                'type' => 'checkbox',
                'section' => 'ai-chat-config',
                'sanitize' => 'intval',
                'default' => 0,
                'description' => 'Show terms notice'
            ),
            'listeo_ai_chat_terms_notice_text' => array(
                'type' => 'wysiwyg',
                'section' => 'ai-chat-config',
                'sanitize' => 'wp_kses_post',
                'default' => '',
                'description' => 'Terms notice text (supports HTML)'
            ),
            'listeo_ai_chat_rate_limit_tier1' => array(
                'type' => 'number',
                'section' => 'ai-chat-config',
                'sanitize' => 'intval',
                'default' => 10,
                'description' => 'Rate limit tier 1 (logged out users)'
            ),
            'listeo_ai_chat_rate_limit_tier2' => array(
                'type' => 'number',
                'section' => 'ai-chat-config',
                'sanitize' => 'intval',
                'default' => 30,
                'description' => 'Rate limit tier 2 (logged in users)'
            ),
            'listeo_ai_chat_rate_limit_tier3' => array(
                'type' => 'number',
                'section' => 'ai-chat-config',
                'sanitize' => 'intval',
                'default' => 100,
                'description' => 'Rate limit tier 3 (premium users)'
            ),

            // ============================================
            // Floating Chat Widget Settings
            // ============================================
            'listeo_ai_floating_chat_enabled' => array(
                'type' => 'checkbox',
                'section' => 'floating-chat-widget',
                'sanitize' => 'intval',
                'default' => 0,
                'description' => 'Enable floating chat widget'
            ),
            'listeo_ai_floating_button_icon' => array(
                'type' => 'select',
                'section' => 'floating-chat-widget',
                'sanitize' => 'sanitize_text_field',
                'default' => 'default',
                'description' => 'Button icon style'
            ),
            'listeo_ai_floating_custom_icon' => array(
                'type' => 'number',
                'section' => 'floating-chat-widget',
                'sanitize' => 'intval',
                'default' => 0,
                'description' => 'Custom icon attachment ID'
            ),
            'listeo_ai_floating_welcome_bubble' => array(
                'type' => 'wysiwyg',
                'section' => 'floating-chat-widget',
                'sanitize' => 'wp_kses_post',
                'default' => __('Hi! How can I help you?', 'listeo-ai-search'),
                'description' => 'Welcome bubble text (supports HTML)'
            ),
            'listeo_ai_floating_popup_width' => array(
                'type' => 'number',
                'section' => 'floating-chat-widget',
                'sanitize' => 'intval',
                'default' => 390,
                'description' => 'Popup width (pixels)'
            ),
            'listeo_ai_floating_popup_height' => array(
                'type' => 'number',
                'section' => 'floating-chat-widget',
                'sanitize' => 'intval',
                'default' => 600,
                'description' => 'Popup height (pixels)'
            ),
            'listeo_ai_floating_button_color' => array(
                'type' => 'color',
                'section' => 'floating-chat-widget',
                'sanitize' => 'sanitize_hex_color',
                'default' => '#222222',
                'description' => 'Button background color'
            ),
            'listeo_ai_primary_color' => array(
                'type' => 'color',
                'section' => 'floating-chat-widget',
                'sanitize' => 'sanitize_hex_color',
                'default' => '#0073ee',
                'description' => 'Primary color for UI elements'
            ),

            // ============================================
            // Special/Internal Settings (not in standard forms)
            // ============================================
            'listeo_ai_search_enabled_types' => array(
                'type' => 'array',
                'section' => 'internal',
                'sanitize' => 'array_map_sanitize_text_field',
                'default' => array('listing'),
                'description' => 'Enabled post types for AI search'
            ),
        );
    }

    /**
     * Get all setting keys from registry
     *
     * @return array Array of setting keys
     */
    private function get_all_setting_keys() {
        return array_keys($this->get_settings_registry());
    }

    /**
     * Get settings grouped by section
     *
     * @return array Settings organized by section
     */
    private function get_section_settings() {
        $registry = $this->get_settings_registry();
        $sections = array();

        foreach ($registry as $key => $config) {
            if ($config['section'] !== 'internal') {
                $sections[$config['section']][] = $key;
            }
        }

        return $sections;
    }

    /**
     * Get checkbox fields, optionally filtered by section
     *
     * @param string|null $section Optional section filter
     * @return array Array of checkbox field keys
     */
    private function get_checkbox_fields($section = null) {
        $registry = $this->get_settings_registry();
        $checkboxes = array();

        foreach ($registry as $key => $config) {
            if ($config['type'] === 'checkbox') {
                if ($section === null || $config['section'] === $section) {
                    $checkboxes[] = $key;
                }
            }
        }

        return $checkboxes;
    }

    /**
     * Get section checkboxes mapping
     *
     * @return array Checkboxes organized by section
     */
    private function get_section_checkboxes() {
        $registry = $this->get_settings_registry();
        $sections = array();

        foreach ($registry as $key => $config) {
            if ($config['type'] === 'checkbox' && $config['section'] !== 'internal') {
                $sections[$config['section']][] = $key;
            }
        }

        return $sections;
    }

    /**
     * Sanitize a setting value based on registry configuration
     *
     * @param string $key Setting key
     * @param mixed $value Value to sanitize
     * @return mixed Sanitized value
     */
    private function sanitize_setting($key, $value) {
        $registry = $this->get_settings_registry();

        if (!isset($registry[$key])) {
            // Unknown setting, use basic sanitization
            return sanitize_text_field($value);
        }

        $config = $registry[$key];

        // Handle arrays
        if (is_array($value)) {
            return array_map('sanitize_text_field', $value);
        }

        // Handle special cases with min/max bounds
        if ($key === 'listeo_ai_search_rate_limit_per_hour') {
            $value = intval($value);
            return max($config['min'], min($config['max'], $value));
        }

        if ($key === 'listeo_ai_chat_temperature') {
            $value = floatval($value);
            return max($config['min'], min($config['max'], $value));
        }

        if ($key === 'listeo_ai_floating_button_color' || $key === 'listeo_ai_primary_color') {
            $sanitized = sanitize_hex_color($value);
            return empty($sanitized) ? $config['default'] : $sanitized;
        }

        // Apply sanitization callback
        $sanitize_callback = $config['sanitize'];

        // Handle special sanitization for WYSIWYG fields (allow HTML)
        if ($sanitize_callback === 'wp_kses_post') {
            return wp_kses_post(wp_unslash($value));
        }

        // Handle textarea fields (strip slashes)
        if ($sanitize_callback === 'sanitize_textarea_field') {
            return sanitize_textarea_field(wp_unslash($value));
        }

        // Standard sanitization
        return call_user_func($sanitize_callback, $value);
    }

    /**
     * Get default value for a setting
     *
     * @param string $key Setting key
     * @return mixed Default value
     */
    private function get_default_value($key) {
        $registry = $this->get_settings_registry();
        return isset($registry[$key]) ? $registry[$key]['default'] : null;
    }

    /**
     * Constructor
     */
    public function __construct() {
        // Ensure default settings exist (for existing installations)
        add_action('admin_init', array($this, 'ensure_default_settings'), 5);

        // Admin menu
        add_action('admin_menu', array($this, 'admin_menu'));

        // Settings
        add_action('admin_init', array($this, 'admin_init'));

        // Enqueue admin scripts
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));

        // AJAX handlers for settings
        add_action('wp_ajax_listeo_ai_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_listeo_ai_test_api_key', array($this, 'ajax_test_api_key'));
        add_action('wp_ajax_listeo_ai_clear_cache', array($this, 'ajax_clear_cache'));
        add_action('wp_ajax_listeo_ai_regenerate_embedding', array($this, 'ajax_regenerate_embedding'));
        add_action('wp_ajax_listeo_ai_load_chat_history', array($this, 'ajax_load_chat_history'));
        add_action('wp_ajax_listeo_ai_clear_chat_history', array($this, 'ajax_clear_chat_history'));

        // AJAX handler for manual batch processing
        add_action('wp_ajax_listeo_ai_manual_batch_action', array($this, 'ajax_manual_batch_action'));

        // Show debug mode notice if enabled
        if (get_option('listeo_ai_search_debug_mode', false)) {
            add_action('admin_notices', array($this, 'show_debug_mode_notice'));
        }
    }

    /**
     * Ensure default settings exist in database
     * This handles existing installations that were activated before defaults were added
     */
    public function ensure_default_settings() {
        // Check if defaults have been initialized (use a flag to avoid running every page load)
        $defaults_initialized = get_option('listeo_ai_defaults_initialized', false);

        if ($defaults_initialized) {
            return; // Defaults already set
        }

        // Get defaults from main plugin class (single source of truth)
        $defaults = Listeo_AI_Search::get_default_settings();

        foreach ($defaults as $option_name => $default_value) {
            // Only add if option doesn't exist
            if (get_option($option_name) === false) {
                add_option($option_name, $default_value);
            }
        }

        // Set flag so this doesn't run again
        update_option('listeo_ai_defaults_initialized', true);
    }
    
    /**
     * Check if Listeo theme is active
     *
     * @return bool True if Listeo or Listeo child theme is active
     */
    private function is_listeo_theme_active() {
        $current_theme = wp_get_theme();
        $theme_name = $current_theme->get('Name');
        $parent_theme = $current_theme->get('Template');

        // Check if current theme or parent theme is Listeo
        return (stripos($theme_name, 'listeo') !== false || stripos($parent_theme, 'listeo') !== false);
    }

    /**
     * Add admin menu
     */
    public function admin_menu() {
        add_menu_page(
            __('AI Chat & Search', 'listeo-ai-search'),           // Page title
            __('AI Chat & Search', 'listeo-ai-search'),                 // Menu title
            'manage_options',                                     // Capability
            'listeo-ai-search',                                   // Menu slug
            array($this, 'admin_page'),                          // Callback function
            'dashicons-search',                                   // Search icon
            30                                                    // Position (after Comments)
        );
    }

    /**
     * Initialize admin settings
     *
     * REFACTORED: Now uses centralized settings registry
     */
    public function admin_init() {
        // Register settings with explicit capability for multisite compatibility
        $settings_args = array(
            'type' => 'string',
            'sanitize_callback' => null, // We handle sanitization in AJAX handler
            'default' => null,
        );

        // Register all settings from the central registry
        foreach ($this->get_all_setting_keys() as $setting_key) {
            register_setting('listeo_ai_search_settings', $setting_key, $settings_args);
        }

        // Add allowed_options filter for multisite compatibility
        add_filter('allowed_options', array($this, 'add_allowed_options'));
    }

    /**
     * Add plugin options to allowed options list for multisite compatibility
     *
     * REFACTORED: Now uses centralized settings registry
     *
     * @param array $allowed_options Array of allowed options
     * @return array Modified array of allowed options
     */
    public function add_allowed_options($allowed_options) {
        $allowed_options['listeo_ai_search_settings'] = $this->get_all_setting_keys();
        return $allowed_options;
    }

    /**
     * Add hidden fields to preserve other settings when submitting a form
     *
     * REFACTORED: Now uses centralized settings registry
     *
     * @param array $exclude_fields Array of field names to exclude from hidden fields
     */
    private function add_hidden_fields_except($exclude_fields = array()) {
        $all_settings = $this->get_all_setting_keys();

        foreach ($all_settings as $setting) {
            if (!in_array($setting, $exclude_fields)) {
                $value = get_option($setting);
                if ($value !== false && $value !== '') {
                    // Handle different input types
                    if (is_array($value)) {
                        foreach ($value as $key => $sub_value) {
                            echo '<input type="hidden" name="' . esc_attr($setting) . '[' . esc_attr($key) . ']" value="' . esc_attr($sub_value) . '">';
                        }
                    } else {
                        echo '<input type="hidden" name="' . esc_attr($setting) . '" value="' . esc_attr($value) . '">';
                    }
                }
            }
        }
    }

    /**
     * AJAX handler for saving settings
     */
    public function ajax_save_settings() {
        // Debug logging for multisite
        if (is_multisite() && get_option('listeo_ai_search_debug_mode', false)) {
            error_log('[AI Chat Multisite Debug] AJAX save settings called');
            error_log('[AI Chat Multisite Debug] Blog ID: ' . get_current_blog_id());
            error_log('[AI Chat Multisite Debug] User ID: ' . get_current_user_id());
            error_log('[AI Chat Multisite Debug] User can manage_options: ' . (current_user_can('manage_options') ? 'yes' : 'no'));
        }

        // Verify nonce
        if (!check_ajax_referer('listeo_ai_search_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'listeo-ai-search')));
            return;
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'listeo-ai-search')));
            return;
        }

        // Get section being saved (to properly handle unchecked checkboxes)
        $section = isset($_POST['section']) ? sanitize_text_field($_POST['section']) : '';

        // REFACTORED: Get settings mapping from central registry
        $section_settings = $this->get_section_settings();
        $section_checkboxes = $this->get_section_checkboxes();
        $all_settings = $this->get_all_setting_keys();

        $updated_settings = array();

        // Handle unchecked checkboxes for current section
        if ($section && isset($section_checkboxes[$section])) {
            foreach ($section_checkboxes[$section] as $checkbox_field) {
                if (!isset($_POST[$checkbox_field])) {
                    // Checkbox not in POST = unchecked, set to 0
                    update_option($checkbox_field, 0);
                    $updated_settings[$checkbox_field] = 0;
                }
            }
        }

        // Process each setting
        foreach ($all_settings as $setting) {
            if (isset($_POST[$setting])) {
                // REFACTORED: Use centralized sanitization from registry
                $value = $this->sanitize_setting($setting, $_POST[$setting]);

                // Update the option
                $update_result = update_option($setting, $value);

                // Debug logging for multisite
                if (is_multisite() && get_option('listeo_ai_search_debug_mode', false)) {
                    error_log(sprintf(
                        '[AI Chat Multisite Debug] update_option(%s) result: %s',
                        $setting,
                        $update_result ? 'success' : 'failed'
                    ));
                }

                $updated_settings[$setting] = $value;
            }
            // NOTE: Do NOT set missing settings to 0
            // If a setting is not in POST, it means it's from a different tab/form
            // Leave those settings unchanged in the database
        }

        // Debug logging for multisite
        if (is_multisite() && get_option('listeo_ai_search_debug_mode', false)) {
            error_log('[AI Chat Multisite Debug] Total settings updated: ' . count($updated_settings));
        }

        // Cleanup: Ensure all checkbox values are proper integers (not strings)
        $this->cleanup_checkbox_values();

        // Handle chat history table creation/deletion based on setting
        $history_enabled = get_option('listeo_ai_chat_history_enabled', 0);
        if ($history_enabled && class_exists('Listeo_AI_Search_Chat_History')) {
            // Create table if it doesn't exist
            Listeo_AI_Search_Chat_History::create_table();
        }

        wp_send_json_success(array(
            'message' => __('Settings saved successfully!', 'listeo-ai-search'),
            'updated_settings' => $updated_settings
        ));
    }

    /**
     * Cleanup corrupted checkbox values (convert string '0' to integer 0)
     *
     * REFACTORED: Now uses centralized settings registry
     */
    private function cleanup_checkbox_values() {
        // Get all checkbox fields from the central registry
        $checkbox_fields = $this->get_checkbox_fields();

        foreach ($checkbox_fields as $field) {
            $value = get_option($field);
            // Convert string '0' or '1' to proper integers
            if ($value === '0' || $value === 0 || $value === false || $value === 'false') {
                update_option($field, 0);
            } elseif ($value === '1' || $value === 1 || $value === true || $value === 'true') {
                update_option($field, 1);
            }
        }
    }

    /**
     * AJAX handler for testing API key
     */
    public function ajax_test_api_key() {
        // Verify nonce
        if (!check_ajax_referer('listeo_ai_search_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'listeo-ai-search')));
            return;
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'listeo-ai-search')));
            return;
        }

        // Get API key from POST data or current settings
        $api_key = sanitize_text_field($_POST['api_key'] ?? get_option('listeo_ai_search_api_key', ''));
        
        if (empty($api_key)) {
            wp_send_json_error(array('message' => __('Please enter an API key first.', 'listeo-ai-search')));
            return;
        }

        try {
            // Test the API key by making a simple request
            $response = wp_remote_get('https://api.openai.com/v1/models', array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api_key,
                    'Content-Type' => 'application/json',
                ),
                'timeout' => 15,
            ));

            if (is_wp_error($response)) {
                wp_send_json_error(array('message' => __('Connection failed: ', 'listeo-ai-search') . $response->get_error_message()));
                return;
            }

            $response_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);

            if ($response_code === 200) {
                wp_send_json_success(array(
                    'message' => __('✅ API key is valid!', 'listeo-ai-search')
                ));
            } elseif ($response_code === 401) {
                wp_send_json_error(array('message' => __('❌ Invalid API key. Please check your key and try again.', 'listeo-ai-search')));
            } elseif ($response_code === 429) {
                wp_send_json_error(array('message' => __('⚠️ API key valid but rate limit exceeded. Try again in a moment.', 'listeo-ai-search')));
            } else {
                $error_body = json_decode($response_body, true);
                $error_message = isset($error_body['error']['message']) ? $error_body['error']['message'] : __('Unknown error', 'listeo-ai-search');
                wp_send_json_error(array('message' => sprintf(__('❌ API Error (%d): %s', 'listeo-ai-search'), $response_code, $error_message)));
            }

        } catch (Exception $e) {
            wp_send_json_error(array('message' => __('❌ Test failed: ', 'listeo-ai-search') . $e->getMessage()));
        }
    }
    
    /**
     * AJAX handler for clearing cache
     */
    public function ajax_clear_cache() {
        // Verify nonce
        if (!check_ajax_referer('listeo_ai_search_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'listeo-ai-search')));
            return;
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'listeo-ai-search')));
            return;
        }

        try {
            $cleared_count = 0;
            $cleared_types = array();

            // Clear search result cache (pattern: listeo_ai_search_*)
            $cache_keys = $this->get_cache_keys();
            foreach ($cache_keys as $key) {
                if (delete_transient($key)) {
                    $cleared_count++;
                }
            }
            if (count($cache_keys) > 0) {
                $cleared_types[] = sprintf(__('search results (%d)', 'listeo-ai-search'), count($cache_keys));
            }

            // Clear API health cache
            if (delete_transient('listeo_ai_api_health')) {
                $cleared_count++;
                $cleared_types[] = __('API health status', 'listeo-ai-search');
            }

            // Clear rate limit cache (current hour)
            $rate_limit_key = 'listeo_ai_rate_limit_' . date('Y-m-d-H');
            if (delete_transient($rate_limit_key)) {
                $cleared_count++;
                $cleared_types[] = __('rate limiting', 'listeo-ai-search');
            }

            // Clear usage tracking cache (current hour)
            $usage_key = 'listeo_ai_usage_' . date('Y-m-d-H');
            if (delete_transient($usage_key)) {
                $cleared_count++;
                $cleared_types[] = __('usage tracking', 'listeo-ai-search');
            }

            // Clear Google Places cache (if any exist)
            global $wpdb;
            $google_transients = $wpdb->get_col(
                "SELECT option_name FROM {$wpdb->options} 
                 WHERE option_name LIKE '_transient_listeo_google_places_%'"
            );
            foreach ($google_transients as $transient_name) {
                $key = str_replace('_transient_', '', $transient_name);
                if (delete_transient($key)) {
                    $cleared_count++;
                }
            }
            if (count($google_transients) > 0) {
                $cleared_types[] = sprintf(__('Google Places data (%d)', 'listeo-ai-search'), count($google_transients));
            }

            // Clear processing delay cache
            $processing_keys = $wpdb->get_col(
                "SELECT option_name FROM {$wpdb->options} 
                 WHERE option_name LIKE '_transient_listeo_ai_processing_delay_%'"
            );
            foreach ($processing_keys as $transient_name) {
                $key = str_replace('_transient_', '', $transient_name);
                if (delete_transient($key)) {
                    $cleared_count++;
                }
            }
            if (count($processing_keys) > 0) {
                $cleared_types[] = sprintf(__('processing delays (%d)', 'listeo-ai-search'), count($processing_keys));
            }

            $message = $cleared_count > 0 
                ? sprintf(__('✅ Cleared %d cache entries: %s', 'listeo-ai-search'), $cleared_count, implode(', ', $cleared_types))
                : __('ℹ️ No cache entries found to clear.', 'listeo-ai-search');

            wp_send_json_success(array('message' => $message));

        } catch (Exception $e) {
            wp_send_json_error(array('message' => __('❌ Clear cache failed: ', 'listeo-ai-search') . $e->getMessage()));
        }
    }

    /**
     * Get all cache keys used by the plugin
     */
    private function get_cache_keys() {
        global $wpdb;
        
        // Get all transients that start with our plugin prefix
        $cache_keys = $wpdb->get_col(
            "SELECT REPLACE(option_name, '_transient_', '') as cache_key 
             FROM {$wpdb->options} 
             WHERE option_name LIKE '_transient_listeo_ai_search_%'"
        );
        
        return $cache_keys;
    }
    
    /**
     * AJAX handler for regenerating single embedding
     */
    public function ajax_regenerate_embedding() {
        // Verify nonce
        if (!check_ajax_referer('listeo_ai_search_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'listeo-ai-search')));
            return;
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'listeo-ai-search')));
            return;
        }

        // Get listing ID
        $listing_id = absint($_POST['listing_id'] ?? 0);
        if (!$listing_id) {
            wp_send_json_error(array('message' => __('Please enter a valid listing ID.', 'listeo-ai-search')));
            return;
        }

        // Check if listing exists
        $listing = get_post($listing_id);
        if (!$listing || $listing->post_type !== 'listing') {
            wp_send_json_error(array('message' => sprintf(__('Listing ID %d not found or not a listing post type.', 'listeo-ai-search'), $listing_id)));
            return;
        }

        try {
            // Check if API key is configured
            $api_key = get_option('listeo_ai_search_api_key', '');
            if (empty($api_key)) {
                wp_send_json_error(array('message' => __('OpenAI API key is not configured. Please configure it in Settings first.', 'listeo-ai-search')));
                return;
            }

            // Regenerate the embedding
            $result = Listeo_AI_Search_Database_Manager::generate_single_embedding($listing_id);
            
            if ($result['success']) {
                $message = sprintf(
                    __('✅ Embedding regenerated successfully for listing "%s" (ID: %d). Processed %d characters.', 'listeo-ai-search'), 
                    esc_html($listing->post_title), 
                    $listing_id,
                    $result['chars_processed'] ?? 0
                );
                
                wp_send_json_success(array(
                    'message' => $message,
                    'listing_title' => $listing->post_title,
                    'listing_id' => $listing_id,
                    'chars_processed' => $result['chars_processed'] ?? 0,
                    'embedding_dimensions' => $result['embedding_dimensions'] ?? 0
                ));
            } else {
                wp_send_json_error(array('message' => sprintf(
                    __('❌ Failed to regenerate embedding for listing "%s" (ID: %d): %s', 'listeo-ai-search'),
                    esc_html($listing->post_title),
                    $listing_id,
                    $result['error'] ?? __('Unknown error', 'listeo-ai-search')
                )));
            }

        } catch (Exception $e) {
            wp_send_json_error(array('message' => sprintf(
                __('❌ Error regenerating embedding for listing ID %d: %s', 'listeo-ai-search'),
                $listing_id,
                $e->getMessage()
            )));
        }
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        // Only load on our settings page (updated hook for top-level menu)
        if ($hook !== 'toplevel_page_listeo-ai-search') {
            return;
        }
        
        // Enqueue CSS
        wp_enqueue_style(
            'listeo-ai-search-admin',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/css/admin-styles.css',
            [],
            LISTEO_AI_SEARCH_VERSION
        );

        // Enqueue Pro features CSS (locked features and upgrade prompts)
        wp_enqueue_style(
            'listeo-ai-search-admin-pro',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/css/admin-pro.css',
            array('listeo-ai-search-admin'),
            LISTEO_AI_SEARCH_VERSION
        );

        // Enqueue manual batch interface CSS if on database tab (needed for Safe Mode)
        $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'settings';
        if ($active_tab === 'database') {
            wp_enqueue_style(
                'listeo-ai-search-manual-batch', 
                LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/css/manual-batch-interface.css', 
                [], 
                LISTEO_AI_SEARCH_VERSION
            );
            
            wp_enqueue_script(
                'listeo-ai-search-manual-batch', 
                LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/js/manual-batch-interface.js', 
                array('jquery'), 
                LISTEO_AI_SEARCH_VERSION,
                true
            );
        }
        
        // Enqueue jQuery (it should already be loaded, but just to be sure)
        wp_enqueue_script('jquery');

        // Enqueue WordPress media uploader (for custom icon upload)
        wp_enqueue_media();

        // Enqueue WordPress color picker
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');

        // Get the admin AJAX URL (multisite compatible)
        $admin_ajax_url = get_admin_url(get_current_blog_id(), 'admin-ajax.php');

        // Add inline script with localized data
        wp_add_inline_script('jquery', '
            window.listeo_ai_search_ajax = {
                ajax_url: "' . esc_js($admin_ajax_url) . '",
                nonce: "' . wp_create_nonce('listeo_ai_search_nonce') . '"
            };
        ');

        // Initialize color picker
        wp_add_inline_script('wp-color-picker', '
            jQuery(document).ready(function($) {
                $(".airs-color-picker").wpColorPicker();
            });
        ');

        // Add inline CSS for AJAX form styling
        wp_add_inline_style('listeo-ai-search-admin', '
            .airs-form-message {
                margin-top: 15px;
                padding: 10px 15px;
                border-radius: 5px;
                border-left: 4px solid;
                background: #fff;
            }
            .airs-form-message.airs-alert-success {
                border-left-color: #46b450;
                background: #ecf7ed;
                color: #1e4620;
            }
            .airs-form-message.airs-alert-error {
                border-left-color: #dc3232;
                background: #fbeaea;
                color: #761919;
            }
            .button-spinner {
                display: inline-flex;
                align-items: center;
                gap: 8px;
            }
            .airs-ajax-form button[type="submit"]:disabled {
                opacity: 0.7;
                cursor: not-allowed;
            }
            .airs-api-test-result {
                padding: 10px 15px;
                border-radius: 4px;
                border: 1px solid;
                font-size: 14px;
                line-height: 1.4;
            }
            .airs-api-test-result.airs-api-test-success {
                border-color: #46b450;
                background: #ecf7ed;
                color: #1e4620;
            }
            .airs-api-test-result.airs-api-test-error {
                border-color: #dc3232;
                background: #fbeaea;
                color: #761919;
            }
            .airs-button-secondary {
                background: #f7f7f7;
                color: #555;
                border: 1px solid #ccc;
            }
            .airs-button-secondary:hover {
                background: #e9e9e9;
                border-color: #999;
            }
            .airs-cache-actions {
                margin-top: 10px;
            }
            .airs-cache-actions button {
                display: inline-flex;
                align-items: center;
                gap: 5px;
            }
            @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
            .spin {
                animation: spin 1s linear infinite;
            }
        ');
    }
    
    /**
     * Admin page content
     */
    public function admin_page() {
        $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'settings';
        ?>
        <div class="wrap airs-admin-wrap">
            <div class="airs-header">
                <div class="airs-header-content">
                    <div class="airs-header-icon">
                        🔍
                    </div>
                    <div class="airs-header-text">
                        <h1><?php _e('AI Chat & Search by Purethemes', 'listeo-ai-search'); ?></h1>
                        <p><?php _e('An AI chatbot that searches the website database and responds like a human assistant.', 'listeo-ai-search'); ?></p>
                    </div>
                </div>
            </div>

            <nav class="airs-nav-tab-wrapper nav-tab-wrapper">
                <a href="?page=listeo-ai-search&tab=settings"
                   class="nav-tab <?php echo $active_tab == 'settings' ? 'nav-tab-active' : ''; ?>">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M12 15.5A3.5 3.5 0 0 1 8.5 12A3.5 3.5 0 0 1 12 8.5a3.5 3.5 0 0 1 3.5 3.5a3.5 3.5 0 0 1-3.5 3.5m7.43-2.53c.04-.32.07-.64.07-.97c0-.33-.03-.66-.07-1l2.11-1.63c.19-.15.24-.42.12-.64l-2-3.46c-.12-.22-.39-.31-.61-.22l-2.49 1c-.52-.39-1.06-.73-1.69-.98l-.37-2.65A.506.506 0 0 0 14 2h-4c-.25 0-.46.18-.5.42l-.37 2.65c-.63.25-1.17.59-1.69.98l-2.49-1c-.22-.09-.49 0-.61.22l-2 3.46c-.13.22-.07.49.12.64L4.57 11c-.04.34-.07.67-.07 1c0 .33.03.65.07.97l-2.11 1.66c-.19.15-.25.42-.12.64l2 3.46c.12.22.39.3.61.22l2.49-1.01c.52.4 1.06.74 1.69.99l.37 2.65c.04.24.25.42.5.42h4c.25 0 .46-.18.5-.42l.37-2.65c.63-.26 1.17-.59 1.69-.99l2.49 1.01c.22.08.49 0 .61-.22l2-3.46c.12-.22.07-.49-.12-.64l-2.11-1.66Z" fill="currentColor"/>
                    </svg>
                    <?php _e('Settings', 'listeo-ai-search'); ?>
                </a>
                <a href="?page=listeo-ai-search&tab=ai-chat"
                   class="nav-tab <?php echo $active_tab == 'ai-chat' ? 'nav-tab-active' : ''; ?>">
                   💬
                    <?php _e('AI Chat', 'listeo-ai-search'); ?>
                </a>
                <a href="?page=listeo-ai-search&tab=database"
                   class="nav-tab <?php echo $active_tab == 'database' ? 'nav-tab-active' : ''; ?>">
                   🗄️
                    <?php _e('Data Training', 'listeo-ai-search'); ?>
                </a>
                <a href="?page=listeo-ai-search&tab=stats"
                   class="nav-tab <?php echo $active_tab == 'stats' ? 'nav-tab-active' : ''; ?>">
                   📊
                    <?php _e('Stats', 'listeo-ai-search'); ?>
                </a>
                <?php
                // Hook for Pro plugin to add additional tabs (e.g., License tab)
                do_action('listeo_ai_search_admin_nav_tabs', $active_tab);
                ?>
            </nav>

            <?php if ($active_tab == 'settings') : ?>
                <div class="airs-tab-content airs-settings-tab">
                    <?php $this->render_settings_tab(); ?>
                </div>
            <?php elseif ($active_tab == 'database') : ?>
                <div class="airs-tab-content airs-database-tab">
                    <?php $this->render_database_tab(); ?>
                </div>
            <?php elseif ($active_tab == 'stats') : ?>
                <div class="airs-tab-content airs-stats-tab">
                    <?php $this->render_stats_tab(); ?>
                </div>
            <?php elseif ($active_tab == 'ai-chat') : ?>
                <div class="airs-tab-content airs-ai-chat-tab">
                    <?php $this->render_ai_chat_tab(); ?>
                </div>
            <?php else : ?>
                <?php
                // Hook for Pro plugin to render custom tab content (e.g., License tab)
                do_action('listeo_ai_search_admin_tab_content', $active_tab);
                ?>
            <?php endif; ?>
        </div>
        
        <?php $this->render_admin_scripts(); ?>
        <?php
    }
    
    /**
     * Render settings tab
     */
    private function render_settings_tab() {
        ?>
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('API Configuration', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Configure your OpenAI API settings and basic search preferences.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="api-config">
                    <div class="airs-form-group">
                        <label for="listeo_ai_search_api_key" class="airs-label">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M12 1L3 5V11C3 16.55 6.84 21.74 12 23C17.16 21.74 21 16.55 21 11V5L12 1M12 7C13.4 7 14.8 8.6 14.8 10.5V11C15.4 11 16 11.4 16 12V16C16 16.6 15.6 17 15 17H9C8.4 17 8 16.6 8 16V12C8 11.4 8.4 11 9 11V10.5C9 8.6 10.6 7 12 7M12 8.2C11.2 8.2 10.2 9.2 10.2 10.5V11H13.8V10.5C13.8 9.2 12.8 8.2 12 8.2Z" fill="currentColor"/>
                            </svg>
                            <?php _e('OpenAI API Key', 'listeo-ai-search'); ?>
                        </label>
                        <input type="password" id="listeo_ai_search_api_key" name="listeo_ai_search_api_key" value="<?php echo esc_attr(get_option('listeo_ai_search_api_key')); ?>" class="airs-input" placeholder="sk-..." />
                        
                        <!-- Test API Key Button -->
                        <div class="airs-api-test-wrapper" style="margin-top: 10px;">
                            <button type="button" id="test-api-key" class="airs-button airs-button-secondary" style="font-size: 13px; padding: 8px 16px;">
                                <span class="button-text">
                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="margin-right: 6px;">
                                        <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z" fill="currentColor"/>
                                    </svg>
                                    <?php _e('Test API Key', 'listeo-ai-search'); ?>
                                </span>
                                <span class="button-spinner" style="display: none;">
                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                        <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                        <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                            <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                        </path>
                                    </svg>
                                    <?php _e('Testing...', 'listeo-ai-search'); ?>
                                </span>
                            </button>
                            <div id="api-test-result" class="airs-api-test-result" style="margin-top: 8px; display: none;"></div>
                        </div>
                        
                        <div class="airs-help-text">
                            <?php _e('Enter your OpenAI API key from the OpenAI Dashboard.', 'listeo-ai-search'); ?>
                            <br><a href="https://docs.purethemes.net/listeo/knowledge-base/how-to-create-open-ai-api-key/" target="_blank"><?php _e('How to create Open AI API key →', 'listeo-ai-search'); ?></a>
                        </div>
                    </div>

                    <div class="airs-form-group">
                        <label for="listeo_ai_search_rate_limit_per_hour" class="airs-label">
                            <?php _e('API Rate Limit (per hour)', 'listeo-ai-search'); ?>
                        </label>
                        <input type="number" id="listeo_ai_search_rate_limit_per_hour" name="listeo_ai_search_rate_limit_per_hour" value="<?php echo esc_attr(get_option('listeo_ai_search_rate_limit_per_hour', 1000)); ?>" min="100" max="10000" step="1" class="airs-input airs-input-small" />
                        <span><?php _e('API calls per hour', 'listeo-ai-search'); ?></span>
                        <div class="airs-help-text">
                            <?php _e('Maximum number of OpenAI API calls allowed per hour (includes embeddings, chat completions, and search operations).', 'listeo-ai-search'); ?>
                            <?php
                            // Show current usage if available
                            $rate_limit_key = 'listeo_ai_rate_limit_' . date('Y-m-d-H');
                            $current_calls = get_transient($rate_limit_key) ?: 0;
                            if ($current_calls > 0) {
                                echo '<br><span style="background-color: rgba(0, 123, 255, 0.1); padding: 3px 7px; display: inline-block; margin-top: 5px; border-radius: 3px; color: rgba(0, 123, 255, 1);">Current hour usage: <strong style="color: rgba(0, 123, 255, 1)">' . $current_calls . '</strong> calls</span>';
                            }
                            ?>
                        </div>
                    </div>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                                <?php _e('Saving...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-form-message" style="display: none;"></div>
                </form>
            </div>
        </div>

        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Search Quality & Thresholds', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Fine-tune search quality and result filtering parameters.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="quality-thresholds">
                    <div class="airs-form-row">
                        <div class="airs-form-group">
                            <label for="listeo_ai_search_min_match_percentage" class="airs-label">
                                <?php _e('Minimum Match Percentage', 'listeo-ai-search'); ?>
                            </label>
                            <input type="number" id="listeo_ai_search_min_match_percentage" name="listeo_ai_search_min_match_percentage" value="<?php echo esc_attr(get_option('listeo_ai_search_min_match_percentage', 65)); ?>" min="0" max="100" step="5" class="airs-input airs-input-small" />
                            <span>%</span>
                            <div class="airs-help-text">
                                <?php _e('Only show AI search results with match percentage above this threshold. Lower values show more results but may be less relevant.', 'listeo-ai-search'); ?>
                                <br>
                                <span style="color: #059669;">60%</span> = Balanced, 
                                <span style="color: #2980b9;">75%</span> = Quality focused, 
                                <span style="color: #f39c12;">40%</span> = Many less relevant results
                            </div>
                        </div>

                        <div class="airs-form-group">
                            <label for="listeo_ai_search_best_match_threshold" class="airs-label">
                                <?php _e('Best Match Badge Threshold', 'listeo-ai-search'); ?>
                            </label>
                            <input type="number" id="listeo_ai_search_best_match_threshold" name="listeo_ai_search_best_match_threshold" value="<?php echo esc_attr(get_option('listeo_ai_search_best_match_threshold', 75)); ?>" min="50" max="95" step="1" class="airs-input airs-input-small" />
                            <span>%</span>
                            <div class="airs-help-text">
                                <?php _e('Show "Best Match" badge for search results above this similarity percentage. Higher values make the badge more exclusive.', 'listeo-ai-search'); ?>
                                <br>
                                <span style="color: #27ae60;">75%</span> = Balanced, 
                                <span style="color: #2980b9;">85%</span> = More exclusive, 
                                <span style="color: #f39c12;">65%</span> = More badges
                            </div>
                        </div>

                        <?php if (class_exists('Listeo_AI_Detection') && Listeo_AI_Detection::is_listeo_available()): ?>
                        <div class="airs-form-group">
                            <label for="listeo_ai_search_max_results" class="airs-label">
                                <?php _e('Maximum AI Top Picks Results', 'listeo-ai-search'); ?>
                            </label>
                            <input type="number" id="listeo_ai_search_max_results" name="listeo_ai_search_max_results" value="<?php echo esc_attr(get_option('listeo_ai_search_max_results', 10)); ?>" min="3" max="50" step="1" class="airs-input airs-input-small" />
                            <span><?php _e('results', 'listeo-ai-search'); ?></span>
                            <div class="airs-help-text">
                                <?php _e('Maximum number of AI Top Picks results to display in search results dropdown.', 'listeo-ai-search'); ?>
                                <br>
                                <span style="color: #27ae60;">5</span> = Balanced (recommended),
                                <span style="color: #2980b9;">3</span> = Compact,
                                <span style="color: #f39c12;">20</span> = Comprehensive
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>

                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_search_enable_analytics" value="1" <?php checked(get_option('listeo_ai_search_enable_analytics'), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Search Analytics', 'listeo-ai-search'); ?>
                                <small><?php _e('Enable search analytics tracking', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                        <div class="airs-help-text">
                            <?php _e('When enabled, tracks search queries, response times, and usage patterns to help improve search performance. Data is anonymized and stored locally.', 'listeo-ai-search'); ?>
                            <br><strong><?php _e('Privacy:', 'listeo-ai-search'); ?></strong> IP addresses are anonymized, only tracks search patterns for optimization.
                        </div>
                        <?php if (get_option('listeo_ai_search_enable_analytics')): ?>
                            <div style="margin-top: 10px; padding: 10px; background: #e8f5e8; border-left: 4px solid #46b450; border-radius: 5px;">
                                <?php
                                if (class_exists('Listeo_AI_Search_Analytics')) {
                                    $analytics = Listeo_AI_Search_Analytics::get_analytics(7);
                                    echo '<strong>Last 7 days:</strong> ' . $analytics['total_searches'] . ' searches, ';
                                    echo $analytics['avg_response_time'] . 'ms avg response ';
                                }
                                ?>
                            </div>
                        <?php endif; ?>
                    </div>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                                <?php _e('Saving...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-form-message" style="display: none;"></div>
                </form>
            </div>
        </div>

        <?php if ($this->is_listeo_theme_active()) : ?>
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Search Suggestions', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Help users discover what they can search for with intelligent suggestions.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="search-suggestions">
                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_search_suggestions_enabled" value="1" <?php checked(get_option('listeo_ai_search_suggestions_enabled'), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Enable search suggestions', 'listeo-ai-search'); ?>
                                <small><?php _e('Show helpful search suggestions below the search input to guide users.', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                    </div>
                    
                    <div class="airs-form-group">
                        <label class="airs-label" style="font-weight: 600; margin-bottom: 8px; display: block;">
                            <?php _e('Suggestion Source:', 'listeo-ai-search'); ?>
                        </label>
                        <label class="airs-checkbox-label">
                            <input type="radio" name="listeo_ai_search_suggestions_source" value="top_searches" <?php checked(get_option('listeo_ai_search_suggestions_source', 'top_searches'), 'top_searches'); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text"><?php _e('Show top 5 most popular searches', 'listeo-ai-search'); ?></span>
                        </label>
                        <label class="airs-checkbox-label">
                            <input type="radio" name="listeo_ai_search_suggestions_source" value="top_searches_10" <?php checked(get_option('listeo_ai_search_suggestions_source', 'top_searches'), 'top_searches_10'); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text"><?php _e('Show top 10 most popular searches', 'listeo-ai-search'); ?></span>
                        </label>
                        <label class="airs-checkbox-label">
                            <input type="radio" name="listeo_ai_search_suggestions_source" value="custom" <?php checked(get_option('listeo_ai_search_suggestions_source', 'top_searches'), 'custom'); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text"><?php _e('Use custom suggestions (comma-separated)', 'listeo-ai-search'); ?></span>
                        </label>
                    </div>
                    
                    <div class="airs-form-group">
                        <label for="listeo_ai_search_custom_suggestions" class="airs-label">
                            <?php _e('Custom Suggestions (comma-separated):', 'listeo-ai-search'); ?>
                        </label>
                        <textarea id="listeo_ai_search_custom_suggestions" name="listeo_ai_search_custom_suggestions" rows="3" class="airs-input" placeholder="pet-friendly apartments, cozy cafes"><?php echo esc_textarea(get_option('listeo_ai_search_custom_suggestions', '')); ?></textarea>
                        <div class="airs-help-text">
                            <?php _e('Enter search suggestions separated by commas. These will be displayed when "custom suggestions" is selected above.', 'listeo-ai-search'); ?>
                            <br><strong><?php _e('Examples:', 'listeo-ai-search'); ?></strong>
                            pet-friendly apartments, cozy cafes with wifi, outdoor wedding venues, 24/7 gyms downtown
                        </div>
                    </div>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                                <?php _e('Saving...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-form-message" style="display: none;"></div>
                </form>
            </div>
        </div>

        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Search Refining', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Extra AI-powered search features and optimizations.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="ai-features">

                    <div class="airs-form-group"  style="background: #fff;
    padding: 20px;
    border-radius: 5px;
    border: 1px solid #e0e0e0;">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_search_query_expansion" value="1" <?php checked(get_option('listeo_ai_search_query_expansion', false), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('AI Query Expansion', 'listeo-ai-search'); ?>
                            </span>
                        </label>
                        <div class="airs-help-text">
                        <span style="color: #f00;">It’s an experimental feature. </span> <br>

                            <?php _e('Expands queries with related keywords to find more relevant results, but may return broader matches.', 'listeo-ai-search'); ?>
<br>
                            <strong><?php _e('Examples:', 'listeo-ai-search'); ?></strong><br>
                            ➡️ "car broken down" → auto repair, mechanic, garage<br>
                            ➡️ "need somewhere to sleep" → hotels, hostels, apartments<br>
                            ➡️ "romantic evening" → restaurants, theaters, spas<br>
                           <small style="color: #666; display: block; margin-top: 10px; font-size: 13px;"><?php _e('⚠️ Adds ~1s latency per search due to additional AI processing', 'listeo-ai-search'); ?></small>
                        </div>

                    </div>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                                <?php _e('Saving...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-form-message" style="display: none;"></div>
                </form>
            </div>
        </div>
        <?php endif; ?>

        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Developer & Debug Options', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Advanced options for development and troubleshooting.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="developer-debug">
                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_search_debug_mode" value="1" <?php checked(get_option('listeo_ai_search_debug_mode'), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Debug Mode', 'listeo-ai-search'); ?>
                                <small><?php _e('Enable debug logging to wp-content/debug.log', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                        <div class="airs-help-text"><?php _e('When enabled, detailed search information will be logged to help troubleshoot issues. Make sure WP_DEBUG_LOG is enabled in wp-config.php.', 'listeo-ai-search'); ?></div>
                    </div>

                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_search_disable_cache" value="1" <?php checked(get_option('listeo_ai_search_disable_cache'), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Disable Cache', 'listeo-ai-search'); ?>
                                <small><?php _e('Disable search result caching (for development)', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                        <div class="airs-help-text"><?php _e('When enabled, search results will not be cached. Useful during development to see fresh results every time. Disable for production to improve performance.', 'listeo-ai-search'); ?></div>
                        
                        <!-- Clear Cache Button -->
                        <div class="airs-cache-actions" style="margin-top: 10px;">
                            <button type="button" id="listeo-clear-cache-btn" class="button button-secondary">
                                <span class="dashicons dashicons-trash" style="vertical-align: middle; margin-right: 5px;"></span>
                                <?php _e('Clear All Cache', 'listeo-ai-search'); ?>
                            </button>
                            <span id="listeo-clear-cache-status" style="margin-left: 10px; font-weight: bold;"></span>
                        </div>
                    </div>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                                <?php _e('Saving...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-form-message" style="display: none;"></div>
                </form>
            </div>
        </div>

        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('🚀 Performance Optimization', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Memory optimization and batch processing settings for large datasets.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="performance-optimization">
                    <div class="airs-form-group">
                        <label for="listeo_ai_search_batch_size" class="airs-label">
                            <?php _e('Batch Size', 'listeo-ai-search'); ?>
                        </label>
                        <input type="number" id="listeo_ai_search_batch_size" name="listeo_ai_search_batch_size" value="<?php echo esc_attr(get_option('listeo_ai_search_batch_size', 500)); ?>" min="100" max="2000" step="100" class="airs-input airs-input-small" />
                        <span><?php _e('embeddings per batch', 'listeo-ai-search'); ?></span>
                        <div class="airs-help-text">
                            <?php _e('Number of embeddings to process in each batch. Lower values use less memory but may be slower. Higher values are faster but use more memory.', 'listeo-ai-search'); ?>
                            <br><strong><?php _e('Recommendations:', 'listeo-ai-search'); ?></strong>
                            <span style="color: #059669;">500</span> = Balanced (recommended), 
                            <span style="color: #2980b9;">300</span> = Low memory servers, 
                            <span style="color: #f39c12;">1000</span> = High performance servers
                        </div>
                    </div>

                    <div class="airs-form-group">
                        <label for="listeo_ai_search_embedding_delay" class="airs-label">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z" fill="currentColor"/>
                            </svg>
                            <?php _e('Embedding Generation Delay', 'listeo-ai-search'); ?>
                        </label>
                        <input type="number" id="listeo_ai_search_embedding_delay" name="listeo_ai_search_embedding_delay" value="<?php echo esc_attr(get_option('listeo_ai_search_embedding_delay', 5)); ?>" min="0" max="60" step="1" class="airs-input airs-input-small" />
                        <span><?php _e('minutes', 'listeo-ai-search'); ?></span>
                        <div class="airs-help-text">
                            <?php _e('Minimum delay between embedding regenerations for the same post type after edit. Prevents unnecessary API costs from rapid successive edits.', 'listeo-ai-search'); ?>
                            <br><strong><?php _e('Examples:', 'listeo-ai-search'); ?></strong> 
                            <span style="color: #059669;">5</span> = Balanced (recommended), 
                            <span style="color: #2980b9;">10</span> = Conservative, 
                            <span style="color: #f39c12;">0</span> = No throttling
                            <br><small><?php _e('💡 Tip: User can make multiple quick edits within the delay period without triggering extra API calls', 'listeo-ai-search'); ?></small>
                        </div>
                    </div>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                                <?php _e('Saving...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-form-message" style="display: none;"></div>
                </form>
            </div>
        </div>
        <?php
    }

    /**
     * Render database management tab
     */
    private function render_database_tab() {
        ?>
        <!-- Generate Embeddings Section -->
        <div class="airs-card airs-card-full-width">
            <div class="airs-card-header">
                <h3><?php _e('Select Content for Training', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Select content types to index and generate AI embeddings for better search results.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">

                <!-- STEP 1: Content Sources Selection -->
                <?php
                $universal_settings = new Listeo_AI_Search_Universal_Settings();
                $universal_settings->render_content_sources_cards();
                ?>

                <!-- STEP 2: Generation Controls -->

                <!-- Simple Training Interface -->
                <?php $this->render_simple_batch_interface(); ?>
            </div>
        </div>

        <!-- Database Actions Section -->
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Database Actions', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Use these tools to manage your AI search database.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">

                <div class="airs-form-group">
                    <label for="listing-id-input" class="airs-label">
                        <?php _e('Check Embedding', 'listeo-ai-search'); ?>
                    </label>
                    <div style="display: flex; gap: 10px; align-items: center;">
                        <input type="number" id="listing-id-input" placeholder="Enter Listing ID" class="airs-input airs-input-small" />
                        <button type="button" id="check-embedding" class="airs-button airs-button-secondary"><?php _e('Check Embedding', 'listeo-ai-search'); ?></button>
                    </div>
                    <div class="airs-help-text"><?php _e('Enter a listing ID to view its embedding data and processed content.', 'listeo-ai-search'); ?></div>
                </div>

                <div class="airs-form-group">
                    <label for="regenerate-listing-id-input" class="airs-label">
                        <?php _e('Regenerate Embedding', 'listeo-ai-search'); ?>
                    </label>
                    <div style="display: flex; gap: 10px; align-items: center;">
                        <input type="number" id="regenerate-listing-id-input" placeholder="Enter Listing ID" class="airs-input airs-input-small" />
                        <button type="button" id="regenerate-embedding" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Regenerate Embedding', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <span class="dashicons dashicons-update spin" style="vertical-align: middle;"></span>
                                <?php _e('Processing...', 'listeo-ai-search'); ?>
                            </span>
                        </button>
                    </div>
                    <div class="airs-help-text"><?php _e('Enter a listing ID to regenerate its embedding data. This will fetch fresh content and create a new embedding.', 'listeo-ai-search'); ?></div>
                    <div id="regenerate-embedding-result" style="margin-top: 10px; display: none;"></div>
                </div>

                <div class="airs-form-group">
                    <label for="delete-post-type-select" class="airs-label">
                        <?php _e('Delete Embeddings by Post Type', 'listeo-ai-search'); ?>
                    </label>
                    <div style="display: flex; gap: 10px; align-items: center;">
                        <select id="delete-post-type-select" class="airs-input airs-input-small" style="max-width: 200px;">
                            <option value=""><?php _e('Select post type...', 'listeo-ai-search'); ?></option>
                            <?php
                            // Get enabled post types
                            $enabled_post_types = Listeo_AI_Search_Database_Manager::get_enabled_post_types();
                            foreach ($enabled_post_types as $post_type) {
                                $post_type_obj = get_post_type_object($post_type);
                                if ($post_type_obj) {
                                    echo '<option value="' . esc_attr($post_type) . '">' . esc_html($post_type_obj->label) . ' (' . esc_html($post_type) . ')</option>';
                                }
                            }
                            ?>
                        </select>
                        <button type="button" id="delete-by-post-type" class="airs-button airs-button-danger"><?php _e('Delete Embeddings', 'listeo-ai-search'); ?></button>
                    </div>
                    <div class="airs-help-text"><?php _e('Delete all embeddings for a specific post type. This will not delete the posts themselves, only their embeddings.', 'listeo-ai-search'); ?></div>
                </div>

                <div class="airs-form-group">
                    <button type="button" id="clear-database" class="airs-button airs-button-danger"><?php _e('Clear All Embeddings', 'listeo-ai-search'); ?></button>
                    <div class="airs-help-text"><?php _e('Delete all embeddings. You will need to regenerate them after clearing.', 'listeo-ai-search'); ?></div>
                </div>
            </div>
        </div>

        <!-- Embedding Viewer Section -->
        <div id="embedding-viewer" class="airs-card" style="display: none;">
            <div class="airs-card-header">
                <h3><?php _e('Embedding Data', 'listeo-ai-search'); ?></h3>
            </div>
            <div class="airs-card-body">
                <div id="embedding-content"></div>
                <div class="airs-form-actions">
                    <button type="button" id="close-embedding" class="airs-button airs-button-secondary"><?php _e('Close', 'listeo-ai-search'); ?></button>
                </div>
            </div>
        </div>

        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Database Status', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Monitor your AI search database status and view embedding statistics.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <div id="database-status" style="margin: 20px 0;">
                    <div id="status-content">
                        <p><em><?php _e('Loading database status...', 'listeo-ai-search'); ?></em></p>
                    </div>
                    <div class="airs-form-actions">
                        <button type="button" id="refresh-status" class="airs-button airs-button-secondary"><?php _e('Refresh Status', 'listeo-ai-search'); ?></button>
                        <button type="button" id="test-ajax" class="airs-button airs-button-secondary"><?php _e('Test AJAX', 'listeo-ai-search'); ?></button>
                    </div>
                </div>
            </div>
        </div>

        <div id="action-result" class="airs-card" style="display: none;">
            <div class="airs-card-body">
                <div id="result-message-content" class="airs-alert"></div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render simple batch interface
     */
    private function render_simple_batch_interface() {
        ?>
        <div id="listing-detection-info" class="listing-detection-info">
            <div id="listing-count-text" class="listing-count-text">
                <?php _e('Loading content information...', 'listeo-ai-search'); ?>
            </div>
        </div>
        
        <div id="regeneration-controls">
            <div class="airs-form-actions">
                <button type="button" id="start-regeneration" style="zoom: 1.15" class="airs-button airs-button-primary"><?php _e('🚀 Start Training', 'listeo-ai-search'); ?></button>
                <button type="button" id="stop-regeneration" class="airs-button airs-button-secondary" style="display: none;"><?php _e('Stop', 'listeo-ai-search'); ?></button>
            </div>
        </div>
        
        <div id="regeneration-progress" style="display: none; margin-top: 20px; padding: 20px; background: #f8f8f8; border: 1px solid #ddd; border-radius: 6px; text-align: center;">
            <div style="font-size: 32px; margin-bottom: 10px; animation: spin 2s linear infinite; display: inline-block;">⚙️</div>
            <p style="margin: 0; font-size: 15px; color: #222; font-weight: 500;"><?php _e('Training in progress...', 'listeo-ai-search'); ?></p>
            <p style="margin: 5px 0 0 0; font-size: 13px; color: #666;"><?php _e('This may take a while depending on the number of items selected', 'listeo-ai-search'); ?></p>
        </div>
        <style>
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        </style>
        
        <div id="regeneration-log" style="display: none; margin-top: 15px;">
            <h4><?php _e('Progress Log:', 'listeo-ai-search'); ?></h4>
            <div id="log-content" class="airs-log"></div>
        </div>
        <?php
    }
    

    /**
     * Render stats tab
     */
    private function render_stats_tab() {
        ?>
        <!-- AI Chatbot Stats Section -->
        <?php
        $chat_stats = get_option('listeo_ai_chat_stats', array(
            'total_sessions' => 0,
            'user_messages' => 0
        ));

        // Chat history stats (if enabled)
        $history_enabled = get_option('listeo_ai_chat_history_enabled', 0);
        $history_stats_30d = null;

        // Pagination
        $page = isset($_GET['history_page']) ? max(1, intval($_GET['history_page'])) : 1;
        $per_page = 3;
        $offset = ($page - 1) * $per_page;

        if ($history_enabled && class_exists('Listeo_AI_Search_Chat_History')) {
            $history_stats_30d = Listeo_AI_Search_Chat_History::get_stats(30);
            $recent_conversations = Listeo_AI_Search_Chat_History::get_recent_conversations($per_page, $offset);

            // Get total count for pagination
            global $wpdb;
            $table_name = Listeo_AI_Search_Chat_History::get_table_name();
            $total_conversations = $wpdb->get_var("SELECT COUNT(DISTINCT conversation_id) FROM {$table_name}");
            $total_pages = ceil($total_conversations / $per_page);
        }
        ?>

        <!-- Chat History Section -->
        <?php if ($history_enabled): ?>
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('💬 Chat History (Last 30 Days)', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Detailed conversation tracking', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <?php if (!AI_Chat_Search_Pro_Manager::can_access_conversation_logs()): ?>
                    <!-- PRO LOCKED: Entire Chat History Section -->
                    <?php $this->render_chat_history_locked(); ?>
                <?php else: ?>
                    <!-- PRO ACTIVE: Show real stats -->
                    <?php if (is_array($history_stats_30d) && !empty($history_stats_30d)): ?>
                    <div class="airs-stats-boxes">
                        <!-- Total Conversations -->
                        <div class="airs-stat-box airs-stat-box-green">
                            <div class="airs-stat-number airs-stat-number-green">
                                <?php echo number_format(isset($history_stats_30d['total_conversations']) ? intval($history_stats_30d['total_conversations']) : 0); ?>
                            </div>
                            <div class="airs-stat-label airs-stat-label-green">
                                <?php _e('Conversations', 'listeo-ai-search'); ?>
                            </div>
                        </div>

                        <!-- Total Messages -->
                        <div class="airs-stat-box airs-stat-box-blue">
                            <div class="airs-stat-number airs-stat-number-blue">
                                <?php echo number_format(isset($history_stats_30d['total_messages']) ? intval($history_stats_30d['total_messages']) : 0); ?>
                            </div>
                            <div class="airs-stat-label airs-stat-label-blue">
                                <?php _e('Messages Exchanged', 'listeo-ai-search'); ?>
                            </div>
                        </div>

                        <!-- Avg Messages per Conversation -->
                        <div class="airs-stat-box airs-stat-box-orange">
                            <div class="airs-stat-number airs-stat-number-orange">
                                <?php echo isset($history_stats_30d['avg_per_conversation']) ? floatval($history_stats_30d['avg_per_conversation']) : 0; ?>
                            </div>
                            <div class="airs-stat-label airs-stat-label-orange">
                                <?php _e('Avg per Conversation', 'listeo-ai-search'); ?>
                            </div>
                        </div>
                    </div>
                    <?php else: ?>
                    <p><?php _e('No chat history data available yet. Start using the AI chat to see statistics here.', 'listeo-ai-search'); ?></p>
                    <?php endif; ?>

                    <!-- Recent Conversations -->
                    <?php if (!empty($recent_conversations)): ?>
                <div style="margin: 20px 0;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                        <div style="display: flex; align-items: center; gap: 15px;">
                            <div>
                                <h3 style="margin: 0;"><?php _e('Recent Conversations', 'listeo-ai-search'); ?></h3>
                                <p style="color: #666; margin: 5px 0 0 0;"><?php _e('Click on a conversation to view the full chat history', 'listeo-ai-search'); ?></p>
                            </div>
                        </div>
                        <div style="font-size: 13px; color: #666;">
                            <?php printf(__('Page %d of %d', 'listeo-ai-search'), $page, max(1, $total_pages)); ?>
                        </div>
                    </div>

                    <div id="listeo-history-conversations">
                    <?php foreach ($recent_conversations as $conv): ?>
                        <?php
                        $messages = Listeo_AI_Search_Chat_History::get_conversation($conv['conversation_id']);
                        $user_info = $conv['user_id'] ? get_userdata($conv['user_id']) : null;
                        ?>
                        <div style="background: #fff; border: 1px solid #ddd; border-radius: 5px; padding: 15px; margin-bottom: 15px;">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; border-bottom: 1px solid #eee; padding-bottom: 10px;">
                                <div>
                                    <strong><?php _e('Conversation ID:', 'listeo-ai-search'); ?></strong>
                                    <code style="background: #f5f5f5; padding: 2px 6px; border-radius: 3px;"><?php echo substr($conv['conversation_id'], 0, 16); ?>...</code>
                                </div>
                                <div style="text-align: right;">
                                    <div style="font-size: 12px; color: #666;">
                                        <?php if ($user_info): ?>
                                            👤 <?php echo esc_html($user_info->display_name); ?> (<?php echo esc_html($user_info->user_email); ?>)
                                        <?php else: ?>
                                            👤 <?php _e('Guest User', 'listeo-ai-search'); ?>
                                        <?php endif; ?>
                                    </div>
                                    <div style="font-size: 12px; color: #999;">
                                        🕒 <?php echo date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($conv['first_message_at'])); ?>
                                    </div>
                                </div>
                            </div>

                            <div style="font-size: 13px; color: #666; margin-bottom: 10px;">
                                💬 <?php echo $conv['message_count']; ?> <?php _e('messages', 'listeo-ai-search'); ?>
                                <?php if ($conv['first_message_at'] !== $conv['last_message_at']): ?>
                                    • <?php _e('Duration:', 'listeo-ai-search'); ?> <?php echo human_time_diff(strtotime($conv['first_message_at']), strtotime($conv['last_message_at'])); ?>
                                <?php endif; ?>
                            </div>

                            <details style="margin-top: 10px;">
                                <summary style="cursor: pointer; padding: 8px; background: #f9f9f9; border-radius: 3px; font-weight: 500;">
                                    <?php _e('View Messages', 'listeo-ai-search'); ?> (<?php echo count($messages); ?>)
                                </summary>
                                <div style="margin-top: 10px; padding: 10px; background: #fafafa; border-radius: 3px; max-height: 400px; overflow-y: auto;">
                                    <?php foreach ($messages as $msg): ?>
                                        <!-- User Message -->
                                        <div style="margin-bottom: 15px; padding: 10px; background: #e8f4ff; border-radius: 4px;">
                                            <div style="font-weight: bold; color: #1976d2; margin-bottom: 5px; font-size: 12px;">
                                                👤 <?php _e('User', 'listeo-ai-search'); ?>
                                                <span style="color: #999; font-weight: normal; margin-left: 10px;">
                                                    <?php echo date_i18n(get_option('time_format'), strtotime($msg['created_at'])); ?>
                                                </span>
                                            </div>
                                            <div style="color: #333; word-break: break-word;">
                                                <?php echo esc_html(trim($msg['user_message'])); ?>
                                            </div>
                                        </div>

                                        <!-- AI Response -->
                                        <div style="margin-bottom: 15px; padding: 10px; background: #ffffff; border-radius: 4px;">
                                            <div style="font-weight: bold; color: #666; margin-bottom: 5px; font-size: 12px;">
                                                🤖 <?php _e('AI Assistant', 'listeo-ai-search'); ?>
                                                <span style="color: #999; font-weight: normal; margin-left: 10px;">
                                                    <?php echo esc_html($msg['model_used']); ?>
                                                </span>
                                            </div>
                                            <div style="color: #333; word-break: break-word;">
                                                <?php echo wp_kses_post($msg['assistant_message']); ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </details>
                        </div>
                    <?php endforeach; ?>
                    </div>

                    <!-- Pagination -->
                    <div id="listeo-history-pagination">
                    <?php if ($total_pages > 1): ?>
                        <div style="display: flex; justify-content: center; gap: 10px; margin-top: 20px; padding: 20px 0; border-top: 1px solid #eee;">
                            <?php if ($page > 1): ?>
                                <button class="button listeo-history-page" data-page="<?php echo $page - 1; ?>">
                                    ← <?php _e('Previous', 'listeo-ai-search'); ?>
                                </button>
                            <?php endif; ?>

                            <div style="display: flex; gap: 5px; align-items: center;">
                                <?php
                                $range = 2;
                                $start = max(1, $page - $range);
                                $end = min($total_pages, $page + $range);

                                if ($start > 1): ?>
                                    <button class="button listeo-history-page" data-page="1">1</button>
                                    <?php if ($start > 2): ?>
                                        <span style="padding: 0 5px;">...</span>
                                    <?php endif; ?>
                                <?php endif; ?>

                                <?php for ($i = $start; $i <= $end; $i++): ?>
                                    <?php if ($i == $page): ?>
                                        <span class="button button-primary" style="cursor: default;"><?php echo $i; ?></span>
                                    <?php else: ?>
                                        <button class="button listeo-history-page" data-page="<?php echo $i; ?>"><?php echo $i; ?></button>
                                    <?php endif; ?>
                                <?php endfor; ?>

                                <?php if ($end < $total_pages): ?>
                                    <?php if ($end < $total_pages - 1): ?>
                                        <span style="padding: 0 5px;">...</span>
                                    <?php endif; ?>
                                    <button class="button listeo-history-page" data-page="<?php echo $total_pages; ?>"><?php echo $total_pages; ?></button>
                                <?php endif; ?>
                            </div>

                            <?php if ($page < $total_pages): ?>
                                <button class="button listeo-history-page" data-page="<?php echo $page + 1; ?>">
                                    <?php _e('Next', 'listeo-ai-search'); ?> →
                                </button>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                    </div>
                            <a href="#" id="clear-chat-history" class="airs-button airs-button-secondary" style="font-size: 13px; padding: 8px 16px; text-decoration: none; white-space: nowrap;">
                                <svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="margin-right: 6px;">
                                    <path d="M6 19c0 1.1.9 2 2 2h8c1.1 0 2-.9 2-2V7H6v12zM19 4h-3.5l-1-1h-5l-1 1H5v2h14V4z" fill="currentColor"/>
                                </svg>
                                <?php _e('Clear History', 'listeo-ai-search'); ?>
                            </a>
                    <script>
                    jQuery(document).ready(function($) {
                        $(document).on('click', '.listeo-history-page', function(e) {
                            e.preventDefault();
                            var page = $(this).data('page');
                            var $container = $('#listeo-history-conversations');
                            var $pagination = $('#listeo-history-pagination');

                            // Show loading
                            $container.html('<p style="text-align: center; padding: 40px;"><span class="spinner is-active" style="float: none; margin: 0;"></span> Loading...</p>');

                            $.ajax({
                                url: ajaxurl,
                                type: 'POST',
                                data: {
                                    action: 'listeo_ai_load_chat_history',
                                    nonce: '<?php echo wp_create_nonce('listeo_ai_search_nonce'); ?>',
                                    page: page
                                },
                                success: function(response) {
                                    if (response.success) {
                                        $container.html(response.data.conversations);
                                        $pagination.html(response.data.pagination);

                                        // Scroll to top of conversations
                                        $('html, body').animate({
                                            scrollTop: $container.offset().top - 100
                                        }, 300);
                                    } else {
                                        $container.html('<p style="color: #d63638; text-align: center; padding: 20px;">' + response.data.message + '</p>');
                                    }
                                },
                                error: function() {
                                    $container.html('<p style="color: #d63638; text-align: center; padding: 20px;">Failed to load conversations. Please try again.</p>');
                                }
                            });
                        });

                        // Clear History button handler
                        $(document).on('click', '#clear-chat-history', function(e) {
                            e.preventDefault();

                            if (!confirm('<?php _e('Are you sure you want to delete all chat history? This action cannot be undone.', 'listeo-ai-search'); ?>')) {
                                return;
                            }

                            var $button = $(this);
                            var originalHtml = $button.html();

                            // Show loading
                            $button.prop('disabled', true).html('<svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="margin-right: 6px;"><path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/><path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor"><animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/></path></svg> <?php _e('Clearing...', 'listeo-ai-search'); ?>');

                            $.ajax({
                                url: ajaxurl,
                                type: 'POST',
                                data: {
                                    action: 'listeo_ai_clear_chat_history',
                                    nonce: '<?php echo wp_create_nonce('listeo_ai_search_nonce'); ?>'
                                },
                                success: function(response) {
                                    if (response.success) {
                                        // Reload the page to show updated stats
                                        location.reload();
                                    } else {
                                        alert(response.data.message);
                                        $button.prop('disabled', false).html(originalHtml);
                                    }
                                },
                                error: function() {
                                    alert('<?php _e('Failed to clear chat history. Please try again.', 'listeo-ai-search'); ?>');
                                    $button.prop('disabled', false).html(originalHtml);
                                }
                            });
                        });
                    });
                    </script>
                </div>
                    <?php else: ?>
                    <p style="padding: 20px; text-align: center; color: #666;">
                        <?php _e('No conversations yet. Start using the AI chat to see history here.', 'listeo-ai-search'); ?>
                    </p>
                    <?php endif; ?>
                <?php endif; // End Pro check ?>
            </div>
        </div>
        <?php elseif (!$history_enabled): ?>
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('💬 Chat History', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Chat history tracking is currently disabled.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <p><?php _e('Enable "Chat History Tracking" in the AI Chat tab to start collecting conversation data for analytics.', 'listeo-ai-search'); ?></p>
                <p style="background: #fff3cd; padding: 15px; border-radius: 5px; border-left: 4px solid #ffc107;">
                    <strong><?php _e('Benefits:', 'listeo-ai-search'); ?></strong><br>
                    • Track popular questions and user needs<br>
                    • Monitor conversation quality and patterns<br>
                    • Identify most requested information<br>
                    • Improve chatbot responses over time
                </p>
            </div>
        </div>
        <?php endif; ?>

        <!-- Search Analytics Section -->
        <?php if (get_option('listeo_ai_search_enable_analytics') && class_exists('Listeo_AI_Search_Analytics')): ?>
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('🔍 Popular Search Queries', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Analytics of the keywords used by AI to provide responses to users.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <?php
                $analytics_7d = Listeo_AI_Search_Analytics::get_analytics(7);
                $analytics_30d = Listeo_AI_Search_Analytics::get_analytics(30);
                ?>

                <!-- Statistics Boxes -->
                <div class="airs-stats-boxes">
                    <!-- 7 Days Total Searches -->
                    <div class="airs-stat-box airs-stat-box-green">
                        <div class="airs-stat-number airs-stat-number-green">
                            <?php echo $analytics_7d['total_searches']; ?>
                        </div>
                        <div class="airs-stat-label airs-stat-label-green">
                            <?php _e('Total Searches in Last 7 Days', 'listeo-ai-search'); ?>
                        </div>
                    </div>

                    <!-- 30 Days Total Searches -->
                    <div class="airs-stat-box airs-stat-box-blue">
                        <div class="airs-stat-number airs-stat-number-blue">
                            <?php echo $analytics_30d['total_searches']; ?>
                        </div>
                        <div class="airs-stat-label airs-stat-label-blue">
                            <?php _e('Total Searches in Last 30 Days', 'listeo-ai-search'); ?>
                        </div>
                    </div>
                </div>

                <!-- Popular Search Queries -->
                <div class="airs-queries-box airs-queries-box-green">
                    <h3><?php _e('Last 7 Days', 'listeo-ai-search'); ?></h3>
                    <?php if (!empty($analytics_7d['popular_queries'])): ?>
                        <div class="airs-query-tags">
                            <?php foreach ($analytics_7d['popular_queries'] as $query => $count): ?>
                                <span class="airs-query-tag-green">
                                    <strong><?php echo esc_html($query); ?></strong> (<?php echo $count; ?>x)
                                </span>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <p><em><?php _e('No search queries recorded yet for the last 7 days.', 'listeo-ai-search'); ?></em></p>
                    <?php endif; ?>
                </div>

                <!-- Popular Search Queries - 30 Days -->
                <div class="airs-queries-box airs-queries-box-blue">
                    <h3><?php _e('Last 30 Days', 'listeo-ai-search'); ?></h3>
                    <?php if (!empty($analytics_30d['popular_queries'])): ?>
                        <div class="airs-query-tags">
                            <?php foreach ($analytics_30d['popular_queries'] as $query => $count): ?>
                                <span class="airs-query-tag-blue">
                                    <strong><?php echo esc_html($query); ?></strong> (<?php echo $count; ?>x)
                                </span>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <p><em><?php _e('No search queries recorded yet for the last 30 days.', 'listeo-ai-search'); ?></em></p>
                    <?php endif; ?>
                </div>

                <!-- Search Insights -->
                <div style="background: #f0f8ff; padding: 20px; border-radius: 5px; margin: 20px 0; border: 1px solid #cce7ff;">
                    <h3 style="margin-top: 0;"><?php _e('📈 Search Insights', 'listeo-ai-search'); ?></h3>
                    <div class="airs-help-text">
                        <?php _e('Use this popular query data to:', 'listeo-ai-search'); ?>
                        <br>• <?php _e('Optimize your listing content for popular search terms', 'listeo-ai-search'); ?>
                        <br>• <?php _e('Add more businesses in popular categories', 'listeo-ai-search'); ?>
                        <br>• <?php _e('Improve search suggestions based on actual user behavior', 'listeo-ai-search'); ?>
                        <br>• <?php _e('Identify trending topics and seasonal patterns', 'listeo-ai-search'); ?>
                    </div>
                </div>
                
                <!-- Analytics Actions -->
                <div>
                    <div class="airs-form-actions">
                        <button type="button" id="clear-analytics" class="airs-button airs-button-danger" onclick="return confirm('Are you sure? This will delete all analytics data.');">
                            <?php _e('Clear Analytics Data', 'listeo-ai-search'); ?>
                        </button>
                        <button type="button" id="refresh-analytics" class="airs-button airs-button-secondary" onclick="window.location.reload();">
                            <?php _e('Refresh Analytics', 'listeo-ai-search'); ?>
                        </button>
                    </div>
                    <div class="airs-help-text"><?php _e('Analytics data is automatically cleaned up after 1000 entries to prevent database bloat.', 'listeo-ai-search'); ?></div>
                </div>
            </div>
        </div>
        <?php else: ?>
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('📊 Search Analytics', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Analytics are currently disabled. Enable them in the Settings tab to start tracking search patterns.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <div style="background: #e2e3e5; padding: 40px 20px; border-radius: 5px; text-align: center;">
                    <h3><?php _e('📊 Search Analytics Disabled', 'listeo-ai-search'); ?></h3>
                    <p><?php _e('Enable search analytics in the Settings tab to track search patterns and performance.', 'listeo-ai-search'); ?></p>
                    <a href="?page=listeo-ai-search&tab=settings" class="airs-button airs-button-primary"><?php _e('Go to Settings', 'listeo-ai-search'); ?></a>
                </div>
            </div>
        </div>
        <?php endif; ?>
        <?php
    }

    /**
     * Render locked conversation logs (Pro feature)
     */
    private function render_conversation_logs_locked() {
        ?>
        <div class="ai-chat-pro-feature-locked" style="margin-top: 30px;">
            <!-- Blurred preview using actual conversation design -->
            <div class="preview-container preview-blurred">
                <div style="margin: 20px 0;">
                    <h3 style="margin: 0 0 15px 0;"><?php _e('Recent Conversations', 'listeo-ai-search'); ?></h3>

                    <?php
                    // Generate 3 dummy conversations
                    $dummy_conversations = array(
                        array('id' => 'a1b2c3d4e5f6g7h8', 'messages' => 8, 'user' => 'Guest User', 'duration' => '15 minutes'),
                        array('id' => 'x9y8z7w6v5u4t3s2', 'messages' => 5, 'user' => 'john.doe@example.com', 'duration' => '8 minutes'),
                        array('id' => 'm5n4o3p2q1r0s9t8', 'messages' => 12, 'user' => 'Guest User', 'duration' => '22 minutes'),
                    );

                    foreach ($dummy_conversations as $idx => $conv):
                        $time_ago = ($idx + 1) * 3; // 3, 6, 9 hours ago
                    ?>
                    <div style="background: #fff; border: 1px solid #ddd; border-radius: 5px; padding: 15px; margin-bottom: 15px;">
                        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; border-bottom: 1px solid #eee; padding-bottom: 10px;">
                            <div>
                                <strong><?php _e('Conversation ID:', 'listeo-ai-search'); ?></strong>
                                <code style="background: #f5f5f5; padding: 2px 6px; border-radius: 3px;"><?php echo substr($conv['id'], 0, 16); ?>...</code>
                            </div>
                            <div style="text-align: right;">
                                <div style="font-size: 12px; color: #666;">
                                    👤 <?php echo esc_html($conv['user']); ?>
                                </div>
                                <div style="font-size: 12px; color: #999;">
                                    🕒 <?php echo sprintf(__('%d hours ago', 'listeo-ai-search'), $time_ago); ?>
                                </div>
                            </div>
                        </div>

                        <div style="font-size: 13px; color: #666; margin-bottom: 10px;">
                            💬 <?php echo $conv['messages']; ?> <?php _e('messages', 'listeo-ai-search'); ?>
                            • <?php _e('Duration:', 'listeo-ai-search'); ?> <?php echo $conv['duration']; ?>
                        </div>

                        <details style="margin-top: 10px;">
                            <summary style="cursor: pointer; padding: 8px; background: #f9f9f9; border-radius: 3px; font-weight: 500;">
                                <?php _e('View Messages', 'listeo-ai-search'); ?> (<?php echo $conv['messages']; ?>)
                            </summary>
                            <div style="margin-top: 10px; padding: 10px; background: #fafafa; border-radius: 3px; max-height: 400px; overflow-y: auto;">
                                <!-- User Message -->
                                <div style="margin-bottom: 15px; padding: 10px; background: #e8f4ff; border-radius: 4px;">
                                    <div style="font-weight: bold; color: #1976d2; margin-bottom: 5px; font-size: 12px;">
                                        👤 <?php _e('User', 'listeo-ai-search'); ?>
                                        <span style="color: #999; font-weight: normal; margin-left: 10px;">
                                            <?php echo date('g:i a', strtotime('-' . ($time_ago + 1) . ' hours')); ?>
                                        </span>
                                    </div>
                                    <div style="color: #333; word-break: break-word;">
                                        <?php _e('Looking for a restaurant near downtown...', 'listeo-ai-search'); ?>
                                    </div>
                                </div>

                                <!-- AI Response -->
                                <div style="margin-bottom: 15px; padding: 10px; background: #ffffff; border-radius: 4px;">
                                    <div style="font-weight: bold; color: #666; margin-bottom: 5px; font-size: 12px;">
                                        🤖 <?php _e('AI Assistant', 'listeo-ai-search'); ?>
                                        <span style="color: #999; font-weight: normal; margin-left: 10px;">
                                            gpt-4o-mini
                                        </span>
                                    </div>
                                    <div style="color: #333; word-break: break-word;">
                                        <p><?php _e('I found several great restaurants downtown. Would you like to see Italian, Asian, or American cuisine options?', 'listeo-ai-search'); ?></p>
                                    </div>
                                </div>

                                <!-- More messages indicator -->
                                <div style="text-align: center; padding: 10px; color: #999; font-size: 12px;">
                                    ... <?php echo ($conv['messages'] - 2); ?> <?php _e('more messages', 'listeo-ai-search'); ?> ...
                                </div>
                            </div>
                        </details>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Overlay -->
            <div class="lock-overlay">
                <div class="lock-content">
                    <h3><?php _e('Chat History & Analytics', 'listeo-ai-search'); ?></h3>

                    <ul class="benefits-list">
                        <li><?php _e('Conversation statistics and metrics', 'listeo-ai-search'); ?></li>
                        <li><?php _e('Complete message history', 'listeo-ai-search'); ?></li>
                    </ul>

                    <a href="<?php echo esc_url(AI_Chat_Search_Pro_Manager::get_upgrade_url('conversation_logs')); ?>"
                       class="button button-primary button-hero" target="_blank">
                        <?php _e('Upgrade to Pro', 'listeo-ai-search'); ?>
                    </a>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render locked chat history (Pro feature) - includes stat boxes + conversations
     */
    private function render_chat_history_locked() {
        ?>
        <div class="ai-chat-pro-feature-locked">
            <!-- Blurred preview with dummy stat boxes -->
            <div class="preview-container preview-blurred">
                <div class="airs-stats-boxes">
                    <!-- Total Conversations -->
                    <div class="airs-stat-box airs-stat-box-green">
                        <div class="airs-stat-number airs-stat-number-green">
                            42
                        </div>
                        <div class="airs-stat-label airs-stat-label-green">
                            <?php _e('Conversations', 'listeo-ai-search'); ?>
                        </div>
                    </div>

                    <!-- Total Messages -->
                    <div class="airs-stat-box airs-stat-box-blue">
                        <div class="airs-stat-number airs-stat-number-blue">
                            287
                        </div>
                        <div class="airs-stat-label airs-stat-label-blue">
                            <?php _e('Messages Exchanged', 'listeo-ai-search'); ?>
                        </div>
                    </div>

                    <!-- Avg Messages per Conversation -->
                    <div class="airs-stat-box airs-stat-box-orange">
                        <div class="airs-stat-number airs-stat-number-orange">
                            6.8
                        </div>
                        <div class="airs-stat-label airs-stat-label-orange">
                            <?php _e('Avg per Conversation', 'listeo-ai-search'); ?>
                        </div>
                    </div>
                </div>

                <!-- Dummy conversations -->
                <div style="margin: 20px 0;">
                    <h3 style="margin: 0 0 15px 0;"><?php _e('Recent Conversations', 'listeo-ai-search'); ?></h3>

                    <?php
                    $dummy_conversations = array(
                        array('id' => 'a1b2c3d4e5f6g7h8', 'messages' => 8, 'user' => 'Guest User', 'duration' => '15 minutes'),
                        array('id' => 'x9y8z7w6v5u4t3s2', 'messages' => 5, 'user' => 'john.doe@example.com', 'duration' => '8 minutes'),
                        array('id' => 'm3n4o5p6q7r8s9t0', 'messages' => 12, 'user' => 'Guest User', 'duration' => '22 minutes'),
                    );

                    foreach ($dummy_conversations as $idx => $conv):
                        $time_ago = ($idx + 1) * 3;
                    ?>
                    <div style="background: #fff; border: 1px solid #ddd; border-radius: 5px; padding: 15px; margin-bottom: 15px;">
                        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; border-bottom: 1px solid #eee; padding-bottom: 10px;">
                            <div>
                                <strong><?php _e('Conversation ID:', 'listeo-ai-search'); ?></strong>
                                <code style="background: #f5f5f5; padding: 2px 6px; border-radius: 3px;"><?php echo substr($conv['id'], 0, 16); ?>...</code>
                            </div>
                            <div style="text-align: right;">
                                <div style="font-size: 12px; color: #666;">
                                    👤 <?php echo esc_html($conv['user']); ?>
                                </div>
                                <div style="font-size: 12px; color: #999;">
                                    🕒 <?php echo sprintf(__('%d hours ago', 'listeo-ai-search'), $time_ago); ?>
                                </div>
                            </div>
                        </div>

                        <div style="font-size: 13px; color: #666; margin-bottom: 10px;">
                            💬 <?php echo $conv['messages']; ?> <?php _e('messages', 'listeo-ai-search'); ?>
                            • <?php _e('Duration:', 'listeo-ai-search'); ?> <?php echo $conv['duration']; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Overlay -->
            <div class="lock-overlay">
                <div class="lock-content">
                    <h3><?php _e('Chat History & Analytics', 'listeo-ai-search'); ?></h3>

                    <ul class="benefits-list">
                        <li><?php _e('Conversation statistics and metrics', 'listeo-ai-search'); ?></li>
                        <li><?php _e('Complete message history', 'listeo-ai-search'); ?></li>
                    </ul>

                    <a href="<?php echo esc_url(AI_Chat_Search_Pro_Manager::get_upgrade_url('chat_history')); ?>"
                       class="button button-primary button-hero" target="_blank">
                        <?php _e('Upgrade to Pro', 'listeo-ai-search'); ?>
                    </a>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render AI Chat tab
     */
    private function render_ai_chat_tab() {
        global $wpdb;

        // Get current settings
        $system_prompt = get_option('listeo_ai_chat_system_prompt', '');
        $chat_enabled = get_option('listeo_ai_chat_enabled', 0);
        $chat_name = get_option('listeo_ai_chat_name', 'Assistant');
        $welcome_message = get_option('listeo_ai_chat_welcome_message', 'Hello! I can help you find restaurants, hotels, and services. What would you like to search for?');
        $model = get_option('listeo_ai_chat_model', 'gpt-5-chat-latest');
        $max_tokens = get_option('listeo_ai_chat_max_tokens', 500);
        $temperature = get_option('listeo_ai_chat_temperature', 0.5);

        ?>

        <!-- AI Chat Configuration -->
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('AI Chat Configuration', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Configure AI-powered conversational search for your listings. This allows users to search using natural language.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="ai-chat-config">

                    <!-- Enable Chat -->
                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_chat_enabled" value="1" <?php checked($chat_enabled, 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Enable AI Chat', 'listeo-ai-search'); ?>
                                <small><?php _e('Enable conversational AI chat interface for listing searches.', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                    </div>

                    <!-- Shortcode Usage -->
                    <div class="airs-form-group">
                        <label class="airs-label"><?php _e('Shortcode Usage', 'listeo-ai-search'); ?></label>

                        <code style="background: #f7f7f8; padding: 12px; border-radius: 4px; display: block; margin-bottom: 8px; font-size: 14px;">
                            [ai_chat]
                        </code>

                        <p class="airs-help-text"><?php _e('Available parameters: <code>height="600px"</code> <code>pictures="enabled"</code> or <code>pictures="disabled"</code>', 'listeo-ai-search'); ?></p>
                    </div>

                    <!-- Elementor Usage (Only show when Listeo theme is active) -->
                    <?php if (class_exists('Listeo_AI_Detection') && Listeo_AI_Detection::is_listeo_available()): ?>
                    <div class="airs-form-group">
                        <label class="airs-label"><?php _e('Elementor Usage', 'listeo-ai-search'); ?></label>
                        <p class="airs-help-text"><?php _e('Edit any page with Elementor and add <mark style="background: #ffba003b; color: #b48507; padding: 2px 5px; border-radius: 4px;">AI Chat</mark> widget', 'listeo-ai-search'); ?></p>
                    </div>
                    <?php endif; ?>

                    <!-- Chatbot Name -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_chat_name" class="airs-label">
                            <?php _e('Chatbot Name', 'listeo-ai-search'); ?>
                        </label>
                        <input type="text" id="listeo_ai_chat_name" name="listeo_ai_chat_name" value="<?php echo esc_attr($chat_name); ?>" class="airs-input" placeholder="Assistant" />
                        <p class="airs-help-text"><?php _e('The name displayed in the chat header. Default: "Assistant"', 'listeo-ai-search'); ?></p>
                    </div>

                    <!-- Welcome Message -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_chat_welcome_message" class="airs-label">
                            <?php _e('Welcome Message', 'listeo-ai-search'); ?>
                        </label>
                        <textarea id="listeo_ai_chat_welcome_message" name="listeo_ai_chat_welcome_message" class="airs-input" rows="3" placeholder="Hello! I can help you find restaurants, hotels, and services. What would you like to search for?"><?php echo esc_textarea($welcome_message); ?></textarea>
                        <p class="airs-help-text"><?php _e('The initial greeting message displayed when chat loads. HTML tags allowed (e.g., &lt;b&gt;, &lt;i&gt;, &lt;a&gt;, &lt;br&gt;).', 'listeo-ai-search'); ?></p>
                    </div>

                    <!-- Model Selection -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_chat_model" class="airs-label">
                            <?php _e('OpenAI Model', 'listeo-ai-search'); ?>
                        </label>
                        <select id="listeo_ai_chat_model" name="listeo_ai_chat_model" class="airs-input">
                            <option value="gpt-4.1-nano" <?php selected($model, 'gpt-4.1-nano'); ?>>GPT-4.1 Nano (Fastest & Cheapest)</option>
                            <option value="gpt-4.1-mini" <?php selected($model, 'gpt-4.1-mini'); ?>>GPT-4.1 Mini (Fast & Good)</option>
                            <option value="gpt-4.1" <?php selected($model, 'gpt-4.1'); ?>>GPT-4.1 (Smartest non-reasoning model)</option>
                            <option value="gpt-5-chat-latest" <?php selected($model, 'gpt-5-chat-latest'); ?>>GPT-5 (GPT-5 model used in ChatGPT)</option>
                        </select>
                        <p class="airs-help-text"><?php _e('Select the OpenAI model for chat responses. The better the model, the more accurate and context-aware the responses at a cost of time response.', 'listeo-ai-search'); ?></p>
                    </div>

                    <!-- Max Tokens -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_chat_max_tokens" class="airs-label">
                            <?php _e('Max Tokens', 'listeo-ai-search'); ?>
                        </label>
                        <input type="number" id="listeo_ai_chat_max_tokens" name="listeo_ai_chat_max_tokens" value="<?php echo esc_attr($max_tokens); ?>" class="airs-input" min="500" max="8000" step="100" />
                        <p class="airs-help-text"><?php _e('Maximum tokens for chat responses (300-1000). Default 500.', 'listeo-ai-search'); ?></p>
                    </div>

                    <!-- Temperature (Hidden - keeps default 0.7) -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_chat_temperature" class="airs-label">
                            <?php _e('Creativity', 'listeo-ai-search'); ?>
                        </label>
                        <input type="number" id="listeo_ai_chat_temperature" name="listeo_ai_chat_temperature" value="<?php echo esc_attr($temperature); ?>" class="airs-input" min="0" max="2" step="0.1" />
                        <p class="airs-help-text"><?php _e('Controls response creativity (0-1). Lower = more focused, Higher = more creative.', 'listeo-ai-search'); ?></p>
                    </div>

                    <!-- Maximum Results -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_chat_max_results" class="airs-label">
                            <?php _e('Maximum Results Displayed', 'listeo-ai-search'); ?>
                        </label>
                        <input type="number" id="listeo_ai_chat_max_results" name="listeo_ai_chat_max_results" value="<?php echo esc_attr(get_option('listeo_ai_chat_max_results', 10)); ?>" class="airs-input" min="5" max="50" step="1" />
                        <p class="airs-help-text"><?php _e('Maximum number of listings to display in chat results (5-50).  Default: 10', 'listeo-ai-search'); ?></p>
                    </div>

                    <!-- System Prompt -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_chat_system_prompt" class="airs-label">
                            <?php _e('Custom System Prompt (Additional Instructions)', 'listeo-ai-search'); ?>
                        </label>
                        <textarea id="listeo_ai_chat_system_prompt" name="listeo_ai_chat_system_prompt" rows="8" class="airs-input" maxlength="6000" placeholder="<?php _e('Add custom instructions about your website, business focus, special features...', 'listeo-ai-search'); ?>"><?php echo esc_textarea($system_prompt); ?></textarea>
                        <span id="system-prompt-counter" style="color: #666;"></span>
                        <p class="airs-help-text">
                            <?php _e('Add custom instructions that will be appended to the default system prompt. Use this to describe your website, special features, or guide the AI behavior.', 'listeo-ai-search'); ?>                        </p>
                        <script>
                            (function($) {
                                var textarea = $('#listeo_ai_chat_system_prompt');
                                var counter = $('#system-prompt-counter');

                                function updateCounter() {
                                    var length = textarea.val().length;
                                    var max = 6000;
                                    counter.text(length + ' / ' + max + ' characters');

                                    if (length > 5500) {
                                        counter.css('color', '#d63638');
                                    } else if (length > 4500) {
                                        counter.css('color', '#dba617');
                                    } else {
                                        counter.css('color', '#666');
                                    }
                                }

                                textarea.on('input', updateCounter);
                                updateCounter();
                            })(jQuery);
                        </script>

                    </div>

                    <!-- Hide Images in Results -->
                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_chat_hide_images" value="1" <?php checked(get_option('listeo_ai_chat_hide_images', 1), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Hide Images in Chat Results', 'listeo-ai-search'); ?>
                                <small><?php _e('Remove listing thumbnails from search results in chat for a cleaner, text-only interface.', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                    </div>

                    <!-- Require Login -->
                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_chat_require_login" value="1" <?php checked(get_option('listeo_ai_chat_require_login', 0), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Require Login to Use Chat', 'listeo-ai-search'); ?>
                                <small><?php _e('Only logged-in WordPress users can access the AI chat. Shortcode will show a login message, floating widget will be hidden.', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                    </div>

                    <!-- Chat History Tracking -->
                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_chat_history_enabled" value="1" <?php checked(get_option('listeo_ai_chat_history_enabled', 0), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Enable Chat History Tracking', 'listeo-ai-search'); ?>
                                <small><?php _e('Save user questions and AI responses for analytics. Chat history is automatically deleted after 30 days.', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                    </div>

                    <!-- Terms of Use Notice -->
                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" id="listeo_ai_chat_terms_notice_enabled" name="listeo_ai_chat_terms_notice_enabled" value="1" <?php checked(get_option('listeo_ai_chat_terms_notice_enabled', 0), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Show Terms of Use Notice', 'listeo-ai-search'); ?>
                                <small><?php _e('Display a terms notice below the chat input for privacy compliance.', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                    </div>

                    <!-- Terms Notice Text (Conditional - shown when checkbox is checked) -->
                    <div id="listeo_ai_chat_terms_notice_text_wrapper" style="display: <?php echo get_option('listeo_ai_chat_terms_notice_enabled', 0) ? 'block' : 'none'; ?>; margin-left: 30px;">
                        <label for="listeo_ai_chat_terms_notice_text" class="airs-label">
                            <?php _e('Terms Notice Text', 'listeo-ai-search'); ?>
                        </label>
                        <textarea id="listeo_ai_chat_terms_notice_text" name="listeo_ai_chat_terms_notice_text" class="airs-input" rows="2" placeholder="By using this chat, you agree to our Terms of Use and Privacy Policy"><?php echo esc_textarea(get_option('listeo_ai_chat_terms_notice_text', 'By using this chat, you agree to our <a href="/terms-of-use" target="_blank">Terms of Use</a> and <a href="/privacy-policy" target="_blank">Privacy Policy</a>')); ?></textarea>
                        <p class="airs-help-text"><?php _e('Text displayed below the chat input. HTML tags allowed (e.g., &lt;a href="/terms"&gt;Terms&lt;/a&gt;).', 'listeo-ai-search'); ?></p>
                    </div>

                    <script>
                    jQuery(document).ready(function($) {
                        // Toggle terms notice text field visibility
                        $('#listeo_ai_chat_terms_notice_enabled').on('change', function() {
                            if ($(this).is(':checked')) {
                                $('#listeo_ai_chat_terms_notice_text_wrapper').slideDown(300);
                            } else {
                                $('#listeo_ai_chat_terms_notice_text_wrapper').slideUp(300);
                            }
                        });
                    });
                    </script>

                    <?php /* Rate Limiting Section - Hidden from UI but functionality still active */ ?>

                    <?php $this->add_hidden_fields_except(array('listeo_ai_chat_system_prompt', 'listeo_ai_chat_enabled', 'listeo_ai_chat_model', 'listeo_ai_chat_max_tokens', 'listeo_ai_chat_temperature', 'listeo_ai_chat_name', 'listeo_ai_chat_welcome_message', 'listeo_ai_chat_max_results', 'listeo_ai_chat_hide_images', 'listeo_ai_chat_require_login', 'listeo_ai_chat_history_enabled', 'listeo_ai_chat_retention_days', 'listeo_ai_chat_terms_notice_enabled', 'listeo_ai_chat_terms_notice_text', 'listeo_ai_chat_rate_limit_tier1', 'listeo_ai_chat_rate_limit_tier2', 'listeo_ai_chat_rate_limit_tier3')); ?>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                            </span>
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Floating Chat Widget Settings -->
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e('Floating Chat Widget', 'listeo-ai-search'); ?></h3>
                <p><?php _e('Display a floating chat button on all pages that opens the AI chat in a popup.', 'listeo-ai-search'); ?></p>
            </div>
            <div class="airs-card-body">
                <form class="airs-ajax-form" data-section="floating-chat-widget">

                    <!-- Enable Floating Widget -->
                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="listeo_ai_floating_chat_enabled" value="1" <?php checked(get_option('listeo_ai_floating_chat_enabled', 0), 1); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e('Enable Floating Chat Widget', 'listeo-ai-search'); ?>
                                <small><?php _e('Show a floating chat button in the bottom-right corner of all pages.', 'listeo-ai-search'); ?></small>
                            </span>
                        </label>
                    </div>

                    <!-- Custom Icon Upload -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_floating_custom_icon" class="airs-label">
                            <?php _e('Custom Button Icon (SVG)', 'listeo-ai-search'); ?>
                        </label>
                        <?php
                        $custom_icon_id = get_option('listeo_ai_floating_custom_icon', 0);
                        $custom_icon_url = $custom_icon_id ? wp_get_attachment_image_url($custom_icon_id, 'thumbnail') : '';
                        ?>
                        <div class="airs-media-upload">
                            <input type="hidden" id="listeo_ai_floating_custom_icon" name="listeo_ai_floating_custom_icon" value="<?php echo esc_attr($custom_icon_id); ?>" />
                            <div class="airs-media-preview" id="listeo-custom-icon-preview">
                                <?php if ($custom_icon_url) : ?>
                                    <img src="<?php echo esc_url($custom_icon_url); ?>" alt="Custom icon" style="max-width: 60px; max-height: 60px; border-radius: 4px;" />
                                <?php else : ?>
                                    <div class="airs-media-placeholder" style="width: 60px; height: 60px; border: 2px dashed #ddd; border-radius: 4px; display: flex; align-items: center; justify-content: center; color: #999;">
                                        <i class="sl sl-icon-bubbles" style="font-size: 24px;"></i>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="airs-media-buttons" style="margin-top: 10px;">
                                <button type="button" class="airs-button airs-button-secondary" id="listeo-upload-custom-icon">
                                    <?php _e('Upload Icon', 'listeo-ai-search'); ?>
                                </button>
                                <?php if ($custom_icon_id) : ?>
                                    <button type="button" class="airs-button airs-button-secondary" id="listeo-remove-custom-icon" style="margin-left: 5px;">
                                        <?php _e('Remove', 'listeo-ai-search'); ?>
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>
                        <p class="airs-help-text"><?php _e('Upload a custom SVG icon (recommended size: 40x40px or 60x60px). If no custom icon is set, default chat bubble icon will be used.', 'listeo-ai-search'); ?></p>
                    </div>

                    <!-- Welcome Bubble Message -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_floating_welcome_bubble" class="airs-label">
                            <?php _e('Welcome Bubble Message', 'listeo-ai-search'); ?>
                        </label>
                        <input type="text" id="listeo_ai_floating_welcome_bubble" name="listeo_ai_floating_welcome_bubble" value="<?php echo esc_attr(get_option('listeo_ai_floating_welcome_bubble', __('Hi! How can I help you?', 'listeo-ai-search'))); ?>" class="airs-input" placeholder="<?php esc_attr_e('Hi! How can I help you?', 'listeo-ai-search'); ?>" />
                        <p class="airs-help-text"><?php _e('Short message displayed above the button on first visit. ', 'listeo-ai-search'); ?></p>
                    </div>

                    <!-- Popup Width -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_floating_popup_width" class="airs-label">
                            <?php _e('Popup Width (px)', 'listeo-ai-search'); ?>
                        </label>
                        <input type="number" id="listeo_ai_floating_popup_width" name="listeo_ai_floating_popup_width" value="<?php echo esc_attr(get_option('listeo_ai_floating_popup_width', 390)); ?>" class="airs-input" min="320" max="800" step="10" />
                        <p class="airs-help-text"><?php _e('Maximum width of the chat popup in pixels. Default: 390px. Range: 320-800px.', 'listeo-ai-search'); ?></p>
                    </div>

                    <!-- Popup Height -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_floating_popup_height" class="airs-label">
                            <?php _e('Popup Height (px)', 'listeo-ai-search'); ?>
                        </label>
                        <input type="number" id="listeo_ai_floating_popup_height" name="listeo_ai_floating_popup_height" value="<?php echo esc_attr(get_option('listeo_ai_floating_popup_height', 600)); ?>" class="airs-input" min="400" max="900" step="10" />
                        <p class="airs-help-text"><?php _e('Maximum height of the chat popup in pixels. Default: 600px. Range: 400-900px.', 'listeo-ai-search'); ?></p>
                    </div>

                    <!-- Button Color -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_floating_button_color" class="airs-label">
                            <?php _e('Buttons Color', 'listeo-ai-search'); ?>
                        </label>
                        <input type="text" id="listeo_ai_floating_button_color" name="listeo_ai_floating_button_color" value="<?php echo esc_attr(get_option('listeo_ai_floating_button_color', '#222222')); ?>" class="airs-input airs-color-picker" data-default-color="#222222" />
                        <p class="airs-help-text"><?php _e('Color for floating button, send button, and listing context button. Default: #222222', 'listeo-ai-search'); ?></p>
                    </div>

                    <!-- Primary Color -->
                    <div class="airs-form-group">
                        <label for="listeo_ai_primary_color" class="airs-label">
                            <?php _e('Primary Color', 'listeo-ai-search'); ?>
                        </label>
                        <input type="text" id="listeo_ai_primary_color" name="listeo_ai_primary_color" value="<?php echo esc_attr(get_option('listeo_ai_primary_color', '#0073ee')); ?>" class="airs-input airs-color-picker" data-default-color="#0073ee" />
                        <p class="airs-help-text"><?php _e('Primary color for links, user messages, and other UI elements. Default: #0073ee', 'listeo-ai-search'); ?></p>
                    </div>

                    <?php $this->add_hidden_fields_except(array('listeo_ai_floating_chat_enabled', 'listeo_ai_floating_custom_icon', 'listeo_ai_floating_welcome_bubble', 'listeo_ai_floating_popup_width', 'listeo_ai_floating_popup_height', 'listeo_ai_floating_button_color', 'listeo_ai_primary_color')); ?>

                    <div class="airs-form-actions">
                        <button type="submit" class="airs-button airs-button-primary">
                            <span class="button-text"><?php _e('Save Settings', 'listeo-ai-search'); ?></span>
                            <span class="button-spinner" style="display: none;">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12,1A11,11,0,1,0,23,12,11,11,0,0,0,12,1Zm0,19a8,8,0,1,1,8-8A8,8,0,0,1,12,20Z" opacity="0.25"/>
                                    <path d="M12,4a8,8,0,0,1,7.89,6.7A1.53,1.53,0,0,0,21.38,12h0a1.5,1.5,0,0,0,1.48-1.75,11,11,0,0,0-21.72,0A1.5,1.5,0,0,0,2.62,12h0a1.53,1.53,0,0,0,1.49-1.3A8,8,0,0,1,12,4Z" fill="currentColor">
                                        <animateTransform attributeName="transform" dur="0.75s" repeatCount="indefinite" type="rotate" values="0 12 12;360 12 12"/>
                                    </path>
                                </svg>
                            </span>
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <?php
    }

    /**
     * AJAX handler for manual batch processing actions
     */
    /**
     * AJAX handler for loading chat history with pagination
     */
    public function ajax_load_chat_history() {
        // Verify nonce
        if (!check_ajax_referer('listeo_ai_search_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'listeo-ai-search')));
            return;
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'listeo-ai-search')));
            return;
        }

        // PRO FEATURE CHECK: Conversation logs require Pro version
        if (!AI_Chat_Search_Pro_Manager::can_access_conversation_logs()) {
            wp_send_json_error(array(
                'message' => __('Conversation logs are a Pro feature. Please upgrade to access full chat history.', 'listeo-ai-search'),
                'upgrade_url' => AI_Chat_Search_Pro_Manager::get_upgrade_url('conversation_logs')
            ));
            return;
        }

        if (!class_exists('Listeo_AI_Search_Chat_History')) {
            wp_send_json_error(array('message' => __('Chat history class not found.', 'listeo-ai-search')));
            return;
        }

        $page = isset($_POST['page']) ? max(1, intval($_POST['page'])) : 1;
        $per_page = 3;
        $offset = ($page - 1) * $per_page;

        $recent_conversations = Listeo_AI_Search_Chat_History::get_recent_conversations($per_page, $offset);

        // Get total count for pagination
        global $wpdb;
        $table_name = Listeo_AI_Search_Chat_History::get_table_name();
        $total_conversations = $wpdb->get_var("SELECT COUNT(DISTINCT conversation_id) FROM {$table_name}");
        $total_pages = ceil($total_conversations / $per_page);

        // Build HTML
        ob_start();
        foreach ($recent_conversations as $conv) {
            $messages = Listeo_AI_Search_Chat_History::get_conversation($conv['conversation_id']);
            $user_info = $conv['user_id'] ? get_userdata($conv['user_id']) : null;
            ?>
            <div style="background: #fff; border: 1px solid #ddd; border-radius: 5px; padding: 15px; margin-bottom: 15px;">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; border-bottom: 1px solid #eee; padding-bottom: 10px;">
                    <div>
                        <strong><?php _e('Conversation ID:', 'listeo-ai-search'); ?></strong>
                        <code style="background: #f5f5f5; padding: 2px 6px; border-radius: 3px;"><?php echo substr($conv['conversation_id'], 0, 16); ?>...</code>
                    </div>
                    <div style="text-align: right;">
                        <div style="font-size: 12px; color: #666;">
                            <?php if ($user_info): ?>
                                👤 <?php echo esc_html($user_info->display_name); ?> (<?php echo esc_html($user_info->user_email); ?>)
                            <?php else: ?>
                                👤 <?php _e('Guest User', 'listeo-ai-search'); ?>
                            <?php endif; ?>
                        </div>
                        <div style="font-size: 12px; color: #999;">
                            🕒 <?php echo date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($conv['first_message_at'])); ?>
                        </div>
                    </div>
                </div>

                <div style="font-size: 13px; color: #666; margin-bottom: 10px;">
                    💬 <?php echo $conv['message_count']; ?> <?php _e('messages', 'listeo-ai-search'); ?>
                    <?php if ($conv['first_message_at'] !== $conv['last_message_at']): ?>
                        • <?php _e('Duration:', 'listeo-ai-search'); ?> <?php echo human_time_diff(strtotime($conv['first_message_at']), strtotime($conv['last_message_at'])); ?>
                    <?php endif; ?>
                </div>

                <details style="margin-top: 10px;">
                    <summary style="cursor: pointer; padding: 8px; background: #f9f9f9; border-radius: 3px; font-weight: 500;">
                        <?php _e('View Messages', 'listeo-ai-search'); ?> (<?php echo count($messages); ?>)
                    </summary>
                    <div style="margin-top: 10px; padding: 10px; background: #fafafa; border-radius: 3px; max-height: 400px; overflow-y: auto;">
                        <?php foreach ($messages as $msg): ?>
                            <!-- User Message -->
                            <div style="margin-bottom: 15px; padding: 10px; background: #e8f4ff; border-radius: 4px;">
                                <div style="font-weight: bold; color: #1976d2; margin-bottom: 5px; font-size: 12px;">
                                    👤 <?php _e('User', 'listeo-ai-search'); ?>
                                    <span style="color: #999; font-weight: normal; margin-left: 10px;">
                                        <?php echo date_i18n(get_option('time_format'), strtotime($msg['created_at'])); ?>
                                    </span>
                                </div>
                                <div style="color: #333; word-break: break-word;">
                                    <?php echo esc_html(trim($msg['user_message'])); ?>
                                </div>
                            </div>

                            <!-- AI Response -->
                            <div style="margin-bottom: 15px; padding: 10px; background: #ffffff; border-radius: 4px;">
                                <div style="font-weight: bold; color: #666; margin-bottom: 5px; font-size: 12px;">
                                    🤖 <?php _e('AI Assistant', 'listeo-ai-search'); ?>
                                    <span style="color: #999; font-weight: normal; margin-left: 10px;">
                                        <?php echo esc_html($msg['model_used']); ?>
                                    </span>
                                </div>
                                <div style="color: #333; word-break: break-word;">
                                    <?php echo wp_kses_post($msg['assistant_message']); ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </details>
            </div>
            <?php
        }
        $conversations_html = ob_get_clean();

        // Build pagination HTML
        ob_start();
        if ($total_pages > 1):
            $range = 2;
            $start = max(1, $page - $range);
            $end = min($total_pages, $page + $range);
            ?>
            <div style="display: flex; justify-content: center; gap: 10px; margin-top: 20px; padding: 20px 0; border-top: 1px solid #eee;">
                <?php if ($page > 1): ?>
                    <button class="button listeo-history-page" data-page="<?php echo $page - 1; ?>">
                        ← <?php _e('Previous', 'listeo-ai-search'); ?>
                    </button>
                <?php endif; ?>

                <div style="display: flex; gap: 5px; align-items: center;">
                    <?php if ($start > 1): ?>
                        <button class="button listeo-history-page" data-page="1">1</button>
                        <?php if ($start > 2): ?>
                            <span style="padding: 0 5px;">...</span>
                        <?php endif; ?>
                    <?php endif; ?>

                    <?php for ($i = $start; $i <= $end; $i++): ?>
                        <?php if ($i == $page): ?>
                            <span class="button button-primary" style="cursor: default;"><?php echo $i; ?></span>
                        <?php else: ?>
                            <button class="button listeo-history-page" data-page="<?php echo $i; ?>"><?php echo $i; ?></button>
                        <?php endif; ?>
                    <?php endfor; ?>

                    <?php if ($end < $total_pages): ?>
                        <?php if ($end < $total_pages - 1): ?>
                            <span style="padding: 0 5px;">...</span>
                        <?php endif; ?>
                        <button class="button listeo-history-page" data-page="<?php echo $total_pages; ?>"><?php echo $total_pages; ?></button>
                    <?php endif; ?>
                </div>

                <?php if ($page < $total_pages): ?>
                    <button class="button listeo-history-page" data-page="<?php echo $page + 1; ?>">
                        <?php _e('Next', 'listeo-ai-search'); ?> →
                    </button>
                <?php endif; ?>
            </div>
        <?php endif;
        $pagination_html = ob_get_clean();

        wp_send_json_success(array(
            'conversations' => $conversations_html,
            'pagination' => $pagination_html,
            'page' => $page,
            'total_pages' => $total_pages
        ));
    }

    /**
     * AJAX handler for clearing all chat history
     */
    public function ajax_clear_chat_history() {
        // Verify nonce
        if (!check_ajax_referer('listeo_ai_search_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'listeo-ai-search')));
            return;
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'listeo-ai-search')));
            return;
        }

        if (!class_exists('Listeo_AI_Search_Chat_History')) {
            wp_send_json_error(array('message' => __('Chat history class not found.', 'listeo-ai-search')));
            return;
        }

        global $wpdb;
        $table_name = Listeo_AI_Search_Chat_History::get_table_name();

        // Delete all records
        $deleted = $wpdb->query("DELETE FROM {$table_name}");

        if ($deleted === false) {
            wp_send_json_error(array('message' => __('Failed to clear chat history.', 'listeo-ai-search')));
            return;
        }

        wp_send_json_success(array(
            'message' => sprintf(__('Successfully deleted %d chat records.', 'listeo-ai-search'), $deleted),
            'deleted' => $deleted
        ));
    }

    public function ajax_manual_batch_action() {
        // Verify nonce
        if (!check_ajax_referer('listeo_ai_search_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => __('Security check failed.', 'listeo-ai-search')));
            return;
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'listeo-ai-search')));
            return;
        }
        
        $batch_action = sanitize_text_field($_POST['batch_action']);
        
        try {
            switch ($batch_action) {
                case 'initialize_batches':
                    $batch_size = isset($_POST['batch_size']) ? intval($_POST['batch_size']) : null;
                    $result = Listeo_AI_Search_Manual_Batch_Processor::initialize_batches($batch_size);
                    break;
                    
                case 'save_settings':
                    $batch_size = intval($_POST['batch_size']);
                    $batch_delay = intval($_POST['batch_delay']);
                    
                    update_option('listeo_ai_search_manual_batch_size', $batch_size);
                    update_option('listeo_ai_search_manual_batch_delay', $batch_delay);
                    // Background processing is always enabled in Safe Mode
                    update_option('listeo_ai_search_background_processing', 1);
                    
                    $result = array(
                        'success' => true,
                        'message' => __('Settings saved successfully', 'listeo-ai-search')
                    );
                    break;
                    
                case 'queue_batches':
                    $batch_ids = isset($_POST['batch_ids']) ? (array) $_POST['batch_ids'] : array();
                    $batch_ids = array_map('sanitize_text_field', $batch_ids);
                    $result = Listeo_AI_Search_Manual_Batch_Processor::queue_batches($batch_ids);
                    break;
                    
                case 'process_batch':
                    $batch_id = sanitize_text_field($_POST['batch_id']);
                    $delay = isset($_POST['batch_delay']) ? intval($_POST['batch_delay']) : null;
                    $result = Listeo_AI_Search_Manual_Batch_Processor::process_single_batch($batch_id, $delay);
                    break;
                    
                case 'process_selected_batches':
                    $batch_ids = array_map('sanitize_text_field', $_POST['batch_ids']);
                    $delay = isset($_POST['batch_delay']) ? intval($_POST['batch_delay']) : null;
                    $result = Listeo_AI_Search_Manual_Batch_Processor::process_selected_batches($batch_ids, $delay);
                    break;
                    
                case 'retry_batch':
                    $batch_id = sanitize_text_field($_POST['batch_id']);
                    $delay = isset($_POST['batch_delay']) ? intval($_POST['batch_delay']) : null;
                    $result = Listeo_AI_Search_Manual_Batch_Processor::retry_failed_batch($batch_id, $delay);
                    break;
                    
                case 'reset_all':
                    $result = Listeo_AI_Search_Manual_Batch_Processor::reset_all_batches();
                    break;
                    
                case 'get_progress':
                    $result = Listeo_AI_Search_Manual_Batch_Processor::get_processing_progress();
                    $result['success'] = true;
                    break;
                    
                case 'get_batch_status':
                    $batches = Listeo_AI_Search_Manual_Batch_Processor::get_all_batches();
                    $summary = Listeo_AI_Search_Manual_Batch_Processor::get_batch_summary();
                    $result = array(
                        'success' => true,
                        'batches' => $batches,
                        'summary' => $summary
                    );
                    break;
                    
                default:
                    wp_send_json_error(array('message' => __('Invalid batch action.', 'listeo-ai-search')));
                    return;
            }
            
            if (isset($result['success']) && $result['success']) {
                wp_send_json_success($result);
            } else {
                wp_send_json_error($result);
            }
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => $e->getMessage()));
        }
    }
    
    /**
     * Render admin JavaScript
     */
    private function render_admin_scripts() {
        ?>
        <script>
        jQuery(document).ready(function($) {
            console.log('Listeo AI Search admin script loaded');
            
            // Safe Mode Checkbox Handler
            $('#enable-safe-mode-checkbox').on('change', function() {
                if ($(this).is(':checked')) {
                    var confirmed = confirm('<?php _e('This will enable Safe Mode with advanced batch processing. Do you want to continue?', 'listeo-ai-search'); ?>');
                    if (confirmed) {
                        // Create and submit a form to enable Safe Mode
                        var form = $('<form method="post"></form>');
                        form.append('<?php echo wp_nonce_field('listeo_safe_mode_action', '_wpnonce', true, false); ?>');
                        form.append('<input type="hidden" name="enable_safe_mode" value="1">');
                        $('body').append(form);
                        form.submit();
                    } else {
                        // Uncheck the box if user cancels
                        $(this).prop('checked', false);
                    }
                }
            });
            
            // AJAX Settings Form Handler
            $('.airs-ajax-form').on('submit', function(e) {
                e.preventDefault();
                
                var $form = $(this);
                var $button = $form.find('button[type="submit"]');
                var $buttonText = $button.find('.button-text');
                var $buttonSpinner = $button.find('.button-spinner');
                var $message = $form.find('.airs-form-message');
                var section = $form.data('section');
                
                // Show loading state
                $button.prop('disabled', true);
                $buttonText.hide();
                $buttonSpinner.show();
                $message.hide();
                
                // Collect all current form values
                var formData = {};
                
                // Get all form inputs
                $form.find('input, textarea, select').each(function() {
                    var $input = $(this);
                    var name = $input.attr('name');
                    
                    if (name) {
                        if ($input.attr('type') === 'checkbox') {
                            formData[name] = $input.is(':checked') ? '1' : '0';
                        } else if ($input.attr('type') === 'radio') {
                            if ($input.is(':checked')) {
                                formData[name] = $input.val();
                            }
                        } else {
                            formData[name] = $input.val();
                        }
                    }
                });

                // NOTE: Do NOT collect values from other forms (hidden tabs)
                // Only save settings from the current form being submitted
                // Other settings remain unchanged in the database

                // Add AJAX parameters
                formData.action = 'listeo_ai_save_settings';
                formData.nonce = listeo_ai_search_ajax.nonce;
                formData.section = section;
                
                console.log('Saving settings for section:', section, formData);
                
                // Send AJAX request
                $.post(listeo_ai_search_ajax.ajax_url, formData)
                    .done(function(response) {
                        console.log('Settings save response:', response);

                        if (response.success) {
                            $message.removeClass('airs-alert-error').addClass('airs-alert-success')
                                   .html('<strong>✓ Success!</strong> ' + response.data.message)
                                   .show();

                            // Update all hidden fields across ALL forms with the values we just saved
                            // This prevents stale hidden fields from overwriting changes in other sections
                            $.each(formData, function(fieldName, fieldValue) {
                                if (fieldName !== 'action' && fieldName !== 'nonce' && fieldName !== 'section') {
                                    // Find this field as a hidden input in ALL other forms and update its value
                                    $('input[type="hidden"][name="' + fieldName + '"]').val(fieldValue);
                                }
                            });

                            // Hide success message after 3 seconds
                            setTimeout(function() {
                                $message.fadeOut();
                            }, 3000);
                        } else {
                            $message.removeClass('airs-alert-success').addClass('airs-alert-error')
                                   .html('<strong>✗ Error!</strong> ' + (response.data.message || 'Unknown error occurred'))
                                   .show();
                        }
                    })
                    .fail(function(xhr, status, error) {
                        console.error('AJAX Error:', xhr, status, error);
                        $message.removeClass('airs-alert-success').addClass('airs-alert-error')
                               .html('<strong>✗ Error!</strong> Connection failed: ' + error)
                               .show();
                    })
                    .always(function() {
                        // Restore button state
                        $button.prop('disabled', false);
                        $buttonText.show();
                        $buttonSpinner.hide();
                    });
            });
            
            // Test API Key Button Handler
            $('#test-api-key').on('click', function(e) {
                e.preventDefault();
                
                var $button = $(this);
                var $buttonText = $button.find('.button-text');
                var $buttonSpinner = $button.find('.button-spinner');
                var $result = $('#api-test-result');
                var apiKey = $('#listeo_ai_search_api_key').val().trim();
                
                if (!apiKey) {
                    $result.removeClass('airs-api-test-success').addClass('airs-api-test-error')
                          .html('❌ Please enter an API key first.')
                          .show();
                    return;
                }
                
                // Show loading state
                $button.prop('disabled', true);
                $buttonText.hide();
                $buttonSpinner.show();
                $result.removeClass('airs-api-test-success airs-api-test-error')
                       .html('Testing API connection...')
                       .show();
                
                // Send AJAX request
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_test_api_key',
                    api_key: apiKey,
                    nonce: listeo_ai_search_ajax.nonce
                })
                .done(function(response) {
                    console.log('API test response:', response);
                    
                    if (response.success) {
                        $result.removeClass('airs-api-test-error').addClass('airs-api-test-success')
                              .html(response.data.message)
                              .show();
                        
                        // Show details if available
                        if (response.data.details) {
                            $result.append('<br><small>' + response.data.details + '</small>');
                        }
                    } else {
                        $result.removeClass('airs-api-test-success').addClass('airs-api-test-error')
                              .html(response.data.message || 'Test failed')
                              .show();
                    }
                })
                .fail(function(xhr, status, error) {
                    console.error('API Test AJAX Error:', xhr, status, error);
                    $result.removeClass('airs-api-test-success').addClass('airs-api-test-error')
                          .html('❌ Connection failed: ' + error)
                          .show();
                })
                .always(function() {
                    // Reset button state
                    $button.prop('disabled', false);
                    $buttonText.show();
                    $buttonSpinner.hide();
                });
            });
            
            // Clear Cache Button Handler
            $('#listeo-clear-cache-btn').on('click', function(e) {
                e.preventDefault();
                
                var $button = $(this);
                var $status = $('#listeo-clear-cache-status');
                var originalText = $button.html();
                
                // Show loading state
                $button.prop('disabled', true)
                       .html('<span class="dashicons dashicons-update spin" style="vertical-align: middle; margin-right: 5px;"></span><?php _e("Clearing...", "listeo-ai-search"); ?>');
                $status.html('').removeClass('success error');
                
                // Send AJAX request
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_clear_cache',
                    nonce: listeo_ai_search_ajax.nonce
                })
                .done(function(response) {
                    if (response.success) {
                        $status.html(response.data.message)
                               .addClass('success')
                               .css('color', '#46b450');
                    } else {
                        $status.html(response.data.message || 'Clear cache failed')
                               .addClass('error')
                               .css('color', '#dc3232');
                    }
                })
                .fail(function(xhr, status, error) {
                    $status.html('❌ Connection failed: ' + error)
                           .addClass('error')
                           .css('color', '#dc3232');
                })
                .always(function() {
                    // Reset button state after 2 seconds
                    setTimeout(function() {
                        $button.prop('disabled', false).html(originalText);
                        $status.fadeOut(3000);
                    }, 2000);
                });
            });
            
            // Check if AJAX variables are available
            if (typeof listeo_ai_search_ajax === 'undefined') {
                console.error('AJAX variables not loaded properly');
                $('#status-content').html('<p style="color: red;">Error: AJAX configuration not loaded. Please refresh the page.</p>');
                return;
            }

            // Load initial status
            refreshDatabaseStatus();
            
            // Refresh status button
            $('#refresh-status').on('click', function() {
                refreshDatabaseStatus();
            });
            
            // Test AJAX button
            $('#test-ajax').on('click', function() {
                console.log('Testing AJAX connection...');
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_test'
                }, function(response) {
                    console.log('Test response:', response);
                    alert('AJAX Test Result: ' + JSON.stringify(response));
                }).fail(function(xhr, status, error) {
                    console.error('Test AJAX failed:', xhr, status, error);
                    alert('AJAX Test Failed: ' + error + ' (Status: ' + xhr.status + ')');
                });
            });
            
            // Database action buttons
            $('#clear-database').on('click', function() {
                if (confirm('Are you sure? This will delete all embeddings and cannot be undone.')) {
                    performDatabaseAction('clear_all', 'Clearing database...');
                }
            });

            // Delete embeddings by post type
            $('#delete-by-post-type').on('click', function() {
                var postType = $('#delete-post-type-select').val();
                if (!postType) {
                    alert('Please select a post type');
                    return;
                }

                var postTypeLabel = $('#delete-post-type-select option:selected').text();
                if (confirm('Are you sure you want to delete all embeddings for ' + postTypeLabel + '? This cannot be undone.')) {
                    performDatabaseActionWithPostType('delete_by_post_type', postType, 'Deleting embeddings for ' + postTypeLabel + '...');
                }
            });
            
            // Analytics action button
            $('#clear-analytics').on('click', function() {
                performAnalyticsAction('clear_analytics', 'Clearing analytics data...');
            });
            
            // Check embedding button
            $('#check-embedding').on('click', function() {
                var listingId = $('#listing-id-input').val();
                if (!listingId) {
                    alert('Please enter a listing ID');
                    return;
                }
                
                console.log('Checking embedding for listing ID:', listingId);
                checkEmbeddingData(listingId);
            });
            
            // Regenerate embedding button
            $('#regenerate-embedding').on('click', function() {
                var listingId = $('#regenerate-listing-id-input').val();
                if (!listingId || listingId <= 0) {
                    alert('Please enter a valid listing ID');
                    return;
                }
                
                var $button = $(this);
                var $buttonText = $button.find('.button-text');
                var $buttonSpinner = $button.find('.button-spinner');
                var $result = $('#regenerate-embedding-result');
                
                // Show loading state
                $button.prop('disabled', true);
                $buttonText.hide();
                $buttonSpinner.show();
                $result.removeClass('success error').html('').hide();
                
                console.log('Regenerating embedding for listing ID:', listingId);
                
                // Send AJAX request
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_regenerate_embedding',
                    listing_id: listingId,
                    nonce: listeo_ai_search_ajax.nonce
                })
                .done(function(response) {
                    console.log('Regenerate embedding response:', response);
                    
                    if (response.success) {
                        $result.removeClass('error').addClass('success')
                              .html(response.data.message)
                              .css('color', '#46b450')
                              .show();
                        
                        // Clear the input field on success
                        $('#regenerate-listing-id-input').val('');
                    } else {
                        $result.removeClass('success').addClass('error')
                              .html(response.data.message || 'Regeneration failed')
                              .css('color', '#dc3232')
                              .show();
                    }
                })
                .fail(function(xhr, status, error) {
                    console.error('Regenerate embedding AJAX error:', xhr, status, error);
                    $result.removeClass('success').addClass('error')
                          .html('❌ Connection failed: ' + error)
                          .css('color', '#dc3232')
                          .show();
                })
                .always(function() {
                    // Reset button state
                    $button.prop('disabled', false);
                    $buttonText.show();
                    $buttonSpinner.hide();
                });
            });
            
            // Close embedding viewer
            $('#close-embedding').on('click', function() {
                $('#embedding-viewer').hide();
            });
            
            function refreshDatabaseStatus() {
                $('#status-content').html('<p><em>Loading...</em></p>');

                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_manage_database',
                    database_action: 'get_stats',
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    console.log('Database status response:', response);
                    if (response.success) {
                        displayDatabaseStatus(response.data);
                        updateListingDetectionInfo(response.data);
                    } else {
                        $('#status-content').html('<p style="color: red;">Error: ' + (response.data || 'Unknown error') + '</p>');
                        $('#listing-count-text').text('Error loading listing information');
                    }
                }).fail(function(xhr, status, error) {
                    console.error('AJAX Error Details:', {
                        status: status,
                        error: error,
                        responseText: xhr.responseText,
                        url: listeo_ai_search_ajax.ajax_url,
                        readyState: xhr.readyState,
                        statusCode: xhr.status
                    });
                    $('#status-content').html('<p style="color: red;">AJAX error: ' + error + ' (Status: ' + xhr.status + ')</p>');
                });
            }
            
            function displayDatabaseStatus(data) {
                var html = '<table class="widefat">';
                
                if (data.error) {
                    html += '<tr><td colspan="2" style="color: red;"><strong>Error:</strong> ' + data.error + '</td></tr>';
                }
                
                html += '<tr><td><strong>Table Exists:</strong></td><td>' + (data.table_exists ? '✓ Yes' : '✗ No') + '</td></tr>';
                // html += '<tr><td><strong>Total:</strong></td><td>' + data.total_listings + '</td></tr>';
            html += '<tr><td><strong>Processed Embeddings:</strong></td><td>' + data.total_embeddings + '</td></tr>';
             //   html += '<tr><td><strong>Coverage:</strong></td><td>' + data.coverage_percentage + '%</td></tr>';
                html += '<tr><td><strong>Missing Embeddings:</strong></td><td>' + data.without_embeddings + '</td></tr>';
                html += '<tr><td><strong>Recent Activity (24h):</strong></td><td>' + data.recent_embeddings + '</td></tr>';
                html += '</table>';
                
                // Show recent embeddings if available
                if (data.recent_items && data.recent_items.length > 0) {
                    html += '<h4>Recent Embeddings (Last 10):</h4>';
                    html += '<table class="widefat" style="margin-top: 10px;">';
                    html += '<thead><tr><th>ID</th><th>Title</th><th>Type</th><th>Created</th></tr></thead><tbody>';
                    data.recent_items.forEach(function(item) {
                        html += '<tr>';
                        html += '<td><a href="#" class="embedding-link" data-id="' + item.listing_id + '">' + item.listing_id + '</a></td>';
                        html += '<td>' + (item.title || 'N/A') + '</td>';
                        html += '<td>' + (item.listing_type || 'N/A') + '</td>';
                        html += '<td>' + item.created_at + '</td>';
                        html += '</tr>';
                    });
                    html += '</tbody></table>';
                    html += '<p class="description"><em>Click on any ID to view its embedding data.</em></p>';
                } else {
                    html += '<p><em>No recent embeddings found.</em></p>';
                }
                
                // Show missing embeddings if available - but only if we already have some embeddings
                if (data.missing_items && data.missing_items.length > 0 && data.total_embeddings > 0) {
                    html += '<h4 style="margin-top: 30px;">Missing Embeddings:</h4>';
                    html += '<div style="margin: 10px 0;">';
                    html += '<button type="button" class="button button-secondary" id="select-all-missing">Select All</button> ';
                    html += '<button type="button" class="button button-secondary" id="deselect-all-missing">Deselect All</button> ';
                    html += '<button type="button" class="button button-primary" id="generate-selected-missing" style="margin-left: 10px;" disabled>Generate Selected</button>';
                    html += '</div>';
                    html += '<div style="max-height: 400px; overflow-y: auto; border: 1px solid #ddd; border-radius: 4px;">';
                    html += '<table class="widefat" style="margin: 0;">';
                    html += '<thead><tr><th style="width: 40px;"><input type="checkbox" id="missing-select-all"></th><th>ID</th><th>Title</th><th>Type</th><th>Last Modified</th><th>Action</th></tr></thead><tbody>';
                    data.missing_items.forEach(function(item) {
                        html += '<tr>';
                        html += '<td><input type="checkbox" class="missing-item-checkbox" data-id="' + item.listing_id + '"></td>';
                        html += '<td>' + item.listing_id + '</td>';
                        html += '<td>' + (item.title || 'N/A') + '</td>';
                        html += '<td>' + (item.listing_type || 'N/A') + '</td>';
                        html += '<td>' + item.created_at + '</td>';
                        html += '<td><button type="button" class="button button-primary button-small generate-embedding-btn" data-id="' + item.listing_id + '">Generate</button></td>';
                        html += '</tr>';
                    });
                    html += '</tbody></table>';
                    html += '</div>';
                    html += '<p class="description"><em>Click "Generate" for individual listings or select multiple and use "Generate Selected".</em></p>';
                } else if (data.without_embeddings > 0 && data.total_embeddings === 0) {
                    html += '<h4 style="margin-top: 30px;">Missing Embeddings:</h4>';
                    html += '<p><em>All ' + data.without_embeddings + ' listings are missing embeddings. Use the "Generate Structured Embeddings" tool above to process them in bulk.</em></p>';
                }
                
                $('#status-content').html(html);
                
                // Add click handlers for embedding links
                $('.embedding-link').on('click', function(e) {
                    e.preventDefault();
                    var listingId = $(this).data('id');
                    checkEmbeddingData(listingId);
                });
                
                // Add click handlers for generate embedding buttons
                $('.generate-embedding-btn').on('click', function(e) {
                    e.preventDefault();
                    var $button = $(this);
                    var listingId = $button.data('id');
                    generateSingleEmbedding(listingId, $button);
                });
                
                // Add handlers for select all/deselect all
                $('#select-all-missing, #missing-select-all').on('change click', function(e) {
                    var isChecked = $(this).prop('checked') || $(this).attr('id') === 'select-all-missing';
                    $('.missing-item-checkbox').prop('checked', isChecked);
                    updateGenerateSelectedButton();
                });
                
                $('#deselect-all-missing').on('click', function(e) {
                    $('.missing-item-checkbox').prop('checked', false);
                    $('#missing-select-all').prop('checked', false);
                    updateGenerateSelectedButton();
                });
                
                // Add handler for individual checkboxes
                $('.missing-item-checkbox').on('change', function() {
                    updateGenerateSelectedButton();
                    // Update select all checkbox state
                    var totalCheckboxes = $('.missing-item-checkbox').length;
                    var checkedCheckboxes = $('.missing-item-checkbox:checked').length;
                    $('#missing-select-all').prop('checked', checkedCheckboxes === totalCheckboxes);
                });
                
                // Add handler for generate selected
                $('#generate-selected-missing').on('click', function(e) {
                    e.preventDefault();
                    var selectedIds = [];
                    $('.missing-item-checkbox:checked').each(function() {
                        selectedIds.push($(this).data('id'));
                    });
                    
                    if (selectedIds.length > 0) {
                        generateBulkEmbeddings(selectedIds);
                    }
                });
            }
            
            function checkEmbeddingData(listingId) {
                $('#embedding-content').html('<p><em>Loading embedding data for listing ' + listingId + '...</em></p>');
                $('#embedding-viewer').show();
                
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_manage_database',
                    database_action: 'get_embedding',
                    listing_id: listingId,
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    console.log('Embedding response:', response);
                    if (response.success) {
                        displayEmbeddingData(response.data);
                    } else {
                        $('#embedding-content').html('<p style="color: red;">Error: ' + (response.data || 'Unknown error') + '</p>');
                    }
                }).fail(function(xhr, status, error) {
                    console.error('Embedding AJAX Error:', xhr, status, error);
                    $('#embedding-content').html('<p style="color: red;">AJAX error: ' + error + '</p>');
                });
            }
            
            function displayEmbeddingData(data) {
                var html = '<h4>Listing #' + data.listing_id + ': ' + (data.title || 'Unknown Title') + '</h4>';
                
                if (data.embedding_exists) {
                    html += '<p><strong>✓ Embedding exists</strong> (Created: ' + data.created_at + ')</p>';
                    
                    if (data.processed_content) {
                        html += '<h5>Processed Content:</h5>';
                        html += '<div style="background: white; padding: 10px; border: 1px solid #ddd; white-space: pre-wrap; font-family: monospace; font-size: 12px; max-height: 200px; overflow-y: auto;">';
                        html += data.processed_content;
                        html += '</div>';
                    }
                    
                    if (data.embedding_preview) {
                        html += '<h5>Embedding Vector (first 10 dimensions):</h5>';
                        html += '<div style="background: white; padding: 10px; border: 1px solid #ddd; font-family: monospace; font-size: 12px;">';
                        html += '[' + data.embedding_preview.join(', ') + '...]';
                        html += '</div>';
                        html += '<p class="description">Full embedding vector contains ' + data.vector_dimensions + ' dimensions.</p>';
                    }
                } else {
                    html += '<p><strong>✗ No embedding found</strong></p>';
                    html += '<p>This listing has not been processed for AI search yet.</p>';
                }
                
                $('#embedding-content').html(html);
            }
            
            function performDatabaseAction(actionType, loadingMessage) {
                $('#result-message-content').text(loadingMessage).removeClass('airs-alert-error').addClass('airs-alert-success');
                $('#action-result').show();

                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_manage_database',
                    database_action: actionType,
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    if (response.success) {
                        $('#result-message-content').text('Action completed successfully!');
                        setTimeout(refreshDatabaseStatus, 2000);
                    } else {
                        $('#result-message-content').text('Error: ' + (response.data || 'Unknown error')).removeClass('airs-alert-success').addClass('airs-alert-error');
                    }
                }).fail(function() {
                    $('#result-message-content').text('AJAX error occurred').removeClass('airs-alert-success').addClass('airs-alert-error');
                });
            }

            function performDatabaseActionWithPostType(actionType, postType, loadingMessage) {
                $('#result-message-content').text(loadingMessage).removeClass('airs-alert-error').addClass('airs-alert-success');
                $('#action-result').show();

                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_manage_database',
                    database_action: actionType,
                    post_type: postType,
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    if (response.success) {
                        var deletedCount = response.data.deleted_count || 0;
                        $('#result-message-content').text('Successfully deleted ' + deletedCount + ' embedding(s) for post type: ' + postType);
                        $('#delete-post-type-select').val(''); // Reset dropdown
                        setTimeout(refreshDatabaseStatus, 2000);
                    } else {
                        $('#result-message-content').text('Error: ' + (response.data || 'Unknown error')).removeClass('airs-alert-success').addClass('airs-alert-error');
                    }
                }).fail(function() {
                    $('#result-message-content').text('AJAX error occurred').removeClass('airs-alert-success').addClass('airs-alert-error');
                });
            }
            
            function performAnalyticsAction(actionType, loadingMessage) {
                $('#result-message-content').text(loadingMessage).removeClass('airs-alert-error').addClass('airs-alert-success');
                $('#action-result').show();
                
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_analytics',
                    analytics_action: actionType,
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    if (response.success) {
                        $('#result-message-content').text('Analytics data cleared successfully!');
                        setTimeout(function() { window.location.reload(); }, 2000);
                    } else {
                        $('#result-message-content').text('Error: ' + (response.data || 'Unknown error')).removeClass('airs-alert-success').addClass('airs-alert-error');
                    }
                }).fail(function() {
                    $('#result-message-content').text('AJAX error occurred').removeClass('airs-alert-success').addClass('airs-alert-error');
                });
            }
            
            // NEW: Structured embedding generation functionality
            $('#start-regeneration').on('click', function() {
                if (!confirm('This will generate embeddings for ALL listings using the structured format. This may take a while and will consume OpenAI API credits. Continue?')) {
                    return;
                }
                
                regenerationRunning = true;
                currentOffset = 0;
                totalListings = 0; // Reset total listings counter
                
                $('#start-regeneration').hide();
                $('#stop-regeneration').show();
                $('#regeneration-progress').show();
                $('#regeneration-log').show();

                logMessage('🚀 Starting structured embedding generation...');

                runRegenerationBatch();
            });
            
            var regenerationRunning = false;
            var currentOffset = 0;
            var totalListings = 0;

            $('#stop-regeneration').on('click', function() {
                regenerationRunning = false;
                $('#start-regeneration').show();
                $('#stop-regeneration').hide();
                logMessage('⏸️ Generation stopped by user.');
            });
            
            function runRegenerationBatch() {
                if (!regenerationRunning) return;

                var batchSize = 100; // Fixed batch size for optimal performance

                logMessage('📦 Processing batch starting at offset ' + currentOffset + '...');

                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_manage_database',
                    database_action: 'start_regeneration',
                    batch_size: batchSize,
                    start_offset: currentOffset,
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    if (response.success) {
                        var data = response.data;

                        // Set total listings from first response
                        if (data.total_listings && totalListings === 0) {
                            totalListings = data.total_listings;
                        }

                        // Update current offset, ensure it's a valid number
                        if (typeof data.next_offset !== 'undefined') {
                            currentOffset = data.next_offset;
                        }

                        // Log results
                        logMessage('✅ Batch completed: ' + (data.processed || 0) + ' items processed');

                        if (data.errors && data.errors.length > 0) {
                            data.errors.forEach(function(error) {
                                logMessage('⚠️ Error: ' + error, 'error');
                            });
                        }

                        // Check if we're complete
                        if (data.status === 'complete' || data.processed === 0) {
                            finishRegeneration('🎉 Generation completed successfully!');
                            refreshDatabaseStatus();
                        } else {
                            // Continue with next batch after delay
                            if (regenerationRunning) {
                                setTimeout(runRegenerationBatch, 1000);
                            }
                        }
                    } else {
                        logMessage('❌ Batch failed: ' + (response.data || 'Unknown error'), 'error');
                        finishRegeneration('❌ Generation failed.');
                    }
                }).fail(function(xhr, status, error) {
                    logMessage('❌ AJAX error: ' + error, 'error');
                    finishRegeneration('❌ Generation failed due to connection error.');
                });
            }
            
            function finishRegeneration(message) {
                regenerationRunning = false;
                $('#start-regeneration').show();
                $('#stop-regeneration').hide();
                $('#regeneration-progress').hide();
                logMessage(message);
            }
            
            function logMessage(message, type) {
                var timestamp = new Date().toLocaleTimeString();
                var colorClass = type === 'error' ? 'color: #ff6b6b;' : 'color: #51cf66;';
                
                $('#log-content').append(
                    '<div style="' + colorClass + '">[' + timestamp + '] ' + message + '</div>'
                );
                
                // Auto-scroll to bottom
                var logElement = $('#log-content')[0];
                logElement.scrollTop = logElement.scrollHeight;
            }
            
            function generateSingleEmbedding(listingId, $button) {
                // Update button state
                var originalText = $button.text();
                $button.text('Generating...').prop('disabled', true);
                
                $.post(listeo_ai_search_ajax.ajax_url, {
                    action: 'listeo_ai_manage_database',
                    database_action: 'generate_single',
                    listing_id: listingId,
                    nonce: listeo_ai_search_ajax.nonce
                }, function(response) {
                    if (response.success) {
                        $button.text('✓ Done').removeClass('button-primary').addClass('button-secondary');
                        // Remove the row after successful generation
                        setTimeout(function() {
                            $button.closest('tr').fadeOut(500, function() {
                                $(this).remove();
                                updateGenerateSelectedButton();
                                // Refresh the status to get updated counts
                                refreshDatabaseStatus();
                            });
                        }, 1500);
                    } else {
                        $button.text('✗ Failed').removeClass('button-primary').addClass('button-secondary');
                        alert('Failed to generate embedding: ' + (response.message || 'Unknown error'));
                        setTimeout(function() {
                            $button.text(originalText).prop('disabled', false).removeClass('button-secondary').addClass('button-primary');
                        }, 3000);
                    }
                }).fail(function(xhr, status, error) {
                    $button.text('✗ Error').removeClass('button-primary').addClass('button-secondary');
                    alert('AJAX error: ' + error);
                    setTimeout(function() {
                        $button.text(originalText).prop('disabled', false).removeClass('button-secondary').addClass('button-primary');
                    }, 3000);
                });
            }
            
            function updateGenerateSelectedButton() {
                var checkedCount = $('.missing-item-checkbox:checked').length;
                var $button = $('#generate-selected-missing');
                
                if (checkedCount > 0) {
                    $button.prop('disabled', false).text('Generate Selected (' + checkedCount + ')');
                } else {
                    $button.prop('disabled', true).text('Generate Selected');
                }
            }
            
            function updateListingDetectionInfo(data) {

            }
            
            function generateBulkEmbeddings(listingIds) {
                if (!listingIds || listingIds.length === 0) return;
                
                var $button = $('#generate-selected-missing');
                var originalText = $button.text();
                var totalIds = listingIds.length;
                var completedIds = 0;
                var failedIds = [];
                
                $button.text('Generating ' + totalIds + ' embeddings...').prop('disabled', true);
                
                // Process embeddings one by one to avoid API rate limits
                function processNext() {
                    if (listingIds.length === 0) {
                        // All done
                        var successCount = completedIds - failedIds.length;
                        var message = 'Completed: ' + successCount + ' success';
                        if (failedIds.length > 0) {
                            message += ', ' + failedIds.length + ' failed';
                        }
                        
                        $button.text(message).prop('disabled', false);
                        setTimeout(function() {
                            refreshDatabaseStatus();
                            $button.text('Generate Selected').prop('disabled', true);
                        }, 3000);
                        return;
                    }
                    
                    var currentId = listingIds.shift();
                    completedIds++;
                    
                    $button.text('Processing ' + completedIds + '/' + totalIds + '...');
                    
                    $.post(listeo_ai_search_ajax.ajax_url, {
                        action: 'listeo_ai_manage_database',
                        database_action: 'generate_single',
                        listing_id: currentId,
                        nonce: listeo_ai_search_ajax.nonce
                    }, function(response) {
                        if (response.success) {
                            // Remove the row for successful generation
                            $('.missing-item-checkbox[data-id="' + currentId + '"]').closest('tr').fadeOut(300, function() {
                                $(this).remove();
                            });
                        } else {
                            failedIds.push(currentId);
                        }
                        
                        // Continue with next after small delay
                        setTimeout(processNext, 1000);
                    }).fail(function() {
                        failedIds.push(currentId);
                        setTimeout(processNext, 1000);
                    });
                }
                
                processNext();
            }

            // Media Uploader for Custom Icon
            var customIconFrame;

            $('#listeo-upload-custom-icon').on('click', function(e) {
                e.preventDefault();

                // If the media frame already exists, reopen it.
                if (customIconFrame) {
                    customIconFrame.open();
                    return;
                }

                // Create the media frame.
                customIconFrame = wp.media({
                    title: '<?php _e('Select Custom Icon', 'listeo-ai-search'); ?>',
                    button: {
                        text: '<?php _e('Use this icon', 'listeo-ai-search'); ?>'
                    },
                    library: {
                        type: ['image/svg+xml']
                    },
                    multiple: false
                });

                // When an image is selected, run a callback.
                customIconFrame.on('select', function() {
                    var attachment = customIconFrame.state().get('selection').first().toJSON();

                    // Set the hidden input value to the attachment ID
                    $('#listeo_ai_floating_custom_icon').val(attachment.id);

                    // Update the preview
                    var imgHtml = '<img src="' + attachment.url + '" alt="Custom icon" style="max-width: 60px; max-height: 60px; border-radius: 4px;" />';
                    $('#listeo-custom-icon-preview').html(imgHtml);

                    // Show remove button if it doesn't exist
                    if ($('#listeo-remove-custom-icon').length === 0) {
                        $('.airs-media-buttons').append('<button type="button" class="airs-button airs-button-secondary" id="listeo-remove-custom-icon" style="margin-left: 5px;"><?php _e('Remove', 'listeo-ai-search'); ?></button>');
                    }
                });

                // Finally, open the modal
                customIconFrame.open();
            });

            // Remove custom icon
            $(document).on('click', '#listeo-remove-custom-icon', function(e) {
                e.preventDefault();

                // Clear the hidden input
                $('#listeo_ai_floating_custom_icon').val('');

                // Reset the preview to placeholder
                var placeholderHtml = '<div class="airs-media-placeholder" style="width: 60px; height: 60px; border: 2px dashed #ddd; border-radius: 4px; display: flex; align-items: center; justify-content: center; color: #999;"><i class="sl sl-icon-bubbles" style="font-size: 24px;"></i></div>';
                $('#listeo-custom-icon-preview').html(placeholderHtml);

                // Remove the remove button
                $(this).remove();
            });
        });
        </script>
        <?php
    }
    
    /**
     * Show debug mode notice
     */
    public function show_debug_mode_notice() {
        $screen = get_current_screen();
        if ($screen && $screen->id === 'toplevel_page_listeo-ai-search') {
            $debug_file = WP_CONTENT_DIR . '/debug.log';
            echo '<div class="notice notice-info">';
            echo '<p><strong>Debug Mode Active:</strong> Detailed logs are being written to <code>' . esc_html($debug_file) . '</code></p>';
            echo '<p>This is helpful for diagnosing batch processing issues. Disable debug mode when not needed to reduce log file size.</p>';
            echo '</div>';
        }
    }
}
