<?php
/**
 * Page Content Extractor
 *
 * Extracts content from WordPress 'page' post type for AI embeddings.
 *
 * @package Listeo_AI_Search
 * @since 2.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Listeo_AI_Content_Extractor_Page {

    /**
     * Extract content from page for embedding generation
     *
     * @param int $post_id Page ID
     * @return string Structured content for embedding
     */
    public function extract_content($post_id) {
        $post = get_post($post_id);

        if (!$post || $post->post_type !== 'page') {
            return '';
        }

        $structured_content = "";

        // Title
        $structured_content .= "TITLE: " . get_the_title($post_id) . ". ";

        // Content
        if (!empty($post->post_content)) {
            $content = strip_tags($post->post_content);
            $content = preg_replace('/\s+/', ' ', $content);
            $structured_content .= "CONTENT: " . trim($content) . ". ";
        }

        // Excerpt
        if (!empty($post->post_excerpt)) {
            $excerpt = strip_tags($post->post_excerpt);
            $structured_content .= "EXCERPT: " . trim($excerpt) . ". ";
        }

        // Parent page hierarchy (useful for context)
        if ($post->post_parent) {
            $parent_titles = array();
            $parent_id = $post->post_parent;

            while ($parent_id) {
                $parent = get_post($parent_id);
                if ($parent) {
                    $parent_titles[] = $parent->post_title;
                    $parent_id = $parent->post_parent;
                } else {
                    break;
                }
            }

            if (!empty($parent_titles)) {
                $structured_content .= "PARENT_PAGES: " . implode(' > ', array_reverse($parent_titles)) . ". ";
            }
        }

        // Featured image alt text
        $thumbnail_id = get_post_thumbnail_id($post_id);
        if ($thumbnail_id) {
            $alt_text = get_post_meta($thumbnail_id, '_wp_attachment_image_alt', true);
            if (!empty($alt_text)) {
                $structured_content .= "IMAGE_CONTEXT: " . $alt_text . ". ";
            }
        }

        // Page template (can provide context about page purpose)
        $template = get_page_template_slug($post_id);
        if ($template) {
            $template_name = basename($template, '.php');
            $structured_content .= "PAGE_TYPE: " . str_replace('-', ' ', $template_name) . ". ";
        }

        // Limit total length
        if (strlen($structured_content) > 8000) {
            $structured_content = substr($structured_content, 0, 8000);
        }

        return trim($structured_content);
    }
}
