<?php
/**
 * Post Content Extractor
 *
 * Extracts content from WordPress standard 'post' post type for AI embeddings.
 *
 * @package Listeo_AI_Search
 * @since 2.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Listeo_AI_Content_Extractor_Post {

    /**
     * Extract content from blog post for embedding generation
     *
     * @param int $post_id Post ID
     * @return string Structured content for embedding
     */
    public function extract_content($post_id) {
        $post = get_post($post_id);

        if (!$post || $post->post_type !== 'post') {
            return '';
        }

        $structured_content = "";

        // Title
        $structured_content .= "TITLE: " . get_the_title($post_id) . ". ";

        // Content
        if (!empty($post->post_content)) {
            $content = strip_tags($post->post_content);
            $content = preg_replace('/\s+/', ' ', $content);
            $structured_content .= "CONTENT: " . trim($content) . ". ";
        }

        // Excerpt
        if (!empty($post->post_excerpt)) {
            $excerpt = strip_tags($post->post_excerpt);
            $structured_content .= "EXCERPT: " . trim($excerpt) . ". ";
        }

        // Categories
        $categories = get_the_category($post_id);
        if (!empty($categories)) {
            $cat_names = array_map(function($cat) {
                return $cat->name;
            }, $categories);
            $structured_content .= "CATEGORIES: " . implode(', ', $cat_names) . ". ";
        }

        // Tags
        $tags = get_the_tags($post_id);
        if (!empty($tags)) {
            $tag_names = array_map(function($tag) {
                return $tag->name;
            }, $tags);
            $structured_content .= "TAGS: " . implode(', ', $tag_names) . ". ";
        }

        // Author
        $author_id = $post->post_author;
        $author = get_the_author_meta('display_name', $author_id);
        if ($author) {
            $structured_content .= "AUTHOR: " . $author . ". ";
        }

        // Featured image alt text (useful for context)
        $thumbnail_id = get_post_thumbnail_id($post_id);
        if ($thumbnail_id) {
            $alt_text = get_post_meta($thumbnail_id, '_wp_attachment_image_alt', true);
            if (!empty($alt_text)) {
                $structured_content .= "IMAGE_CONTEXT: " . $alt_text . ". ";
            }
        }

        // Limit total length
        if (strlen($structured_content) > 8000) {
            $structured_content = substr($structured_content, 0, 8000);
        }

        return trim($structured_content);
    }
}
