<?php
/**
 * Database Manager Class
 * 
 * Handles database operations and embedding storage
 * 
 * @package Listeo_AI_Search
 * @since 1.0.5
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Listeo_AI_Search_Database_Manager {
    
    /**
     * Get the embeddings table name
     * 
     * @return string Table name
     */
    public static function get_embeddings_table_name() {
        global $wpdb;
        return $wpdb->prefix . 'listeo_ai_embeddings';
    }
    
    /**
     * Create database tables
     */
    public static function create_tables() {
        global $wpdb;
        
        $table_name = self::get_embeddings_table_name();
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            listing_id bigint(20) NOT NULL,
            embedding longtext NOT NULL,
            content_hash varchar(32) NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY listing_id (listing_id),
            KEY content_hash (content_hash)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Create performance indexes for SQL pre-filtering
     * 
     * @return bool Success status
     */
    public static function create_performance_indexes() {
        global $wpdb;
        
        try {
            // Index for address-based location filtering
            $wpdb->query("
                CREATE INDEX IF NOT EXISTS idx_address_meta_search 
                ON {$wpdb->postmeta} (meta_key, meta_value(100)) 
                WHERE meta_key IN ('_address', '_city', '_region', '_country')
            ");
            
            // Index for listing type filtering  
            $wpdb->query("
                CREATE INDEX IF NOT EXISTS idx_post_type_status 
                ON {$wpdb->posts} (post_type, post_status)
            ");
            
            // Composite index for location + type filtering
            $wpdb->query("
                CREATE INDEX IF NOT EXISTS idx_location_type_search 
                ON {$wpdb->postmeta} (meta_key, post_id) 
                WHERE meta_key IN ('_address', '_city', '_region')
            ");
            
            return true;
            
        } catch (Exception $e) {
            error_log('Listeo AI Search - Index creation error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Compress embedding vector for storage using half-precision floats
     * 
     * @param array $embedding_vector Array of floating point values
     * @return string Compressed base64-encoded binary data
     */
    public static function compress_embedding_for_storage($embedding_vector) {
        if (empty($embedding_vector) || !is_array($embedding_vector)) {
            return '';
        }
        
        try {
            // Convert to half-precision floats (16-bit) for massive memory savings
            $packed = '';
            foreach ($embedding_vector as $value) {
                // Quantize to 16-bit signed integer for efficient storage
                // Range: -32768 to 32767, good enough for normalized embeddings (-1 to 1)
                $quantized = intval(round($value * 32767));
                $quantized = max(-32767, min(32767, $quantized)); // Clamp to valid range
                $packed .= pack('s', $quantized); // 's' = signed 16-bit little-endian
            }
            
            // Compress the packed data and encode as base64 for database storage
            return base64_encode(gzcompress($packed, 6)); // Level 6 = good compression vs speed balance
            
        } catch (Exception $e) {
            error_log('Listeo AI Search - Embedding compression error: ' . $e->getMessage());
            // Fallback to JSON if compression fails (backward compatibility)
            return json_encode($embedding_vector);
        }
    }
    
    /**
     * Decompress embedding from storage back to float array
     * 
     * @param string $compressed_data Compressed base64-encoded data or JSON string
     * @return array|false Embedding vector array or false on failure
     */
    public static function decompress_embedding_from_storage($compressed_data) {
        if (empty($compressed_data)) {
            return false;
        }
        
        try {
            // BACKWARD COMPATIBILITY: Check if it's JSON (legacy format)
            if (substr(trim($compressed_data), 0, 1) === '[') {
                // It's JSON - decode normally for backward compatibility
                return json_decode($compressed_data, true);
            }
            
            // It's compressed binary data - decompress it
            $packed = gzuncompress(base64_decode($compressed_data));
            if ($packed === false) {
                // Decompression failed, try JSON fallback
                return json_decode($compressed_data, true);
            }
            
            // Unpack 16-bit signed integers back to floats
            $quantized_values = unpack('s*', $packed);
            if ($quantized_values === false) {
                return false;
            }
            
            // Convert back to normalized float values
            $embedding = array();
            foreach ($quantized_values as $quantized) {
                $embedding[] = $quantized / 32767.0; // Convert back to -1.0 to 1.0 range
            }
            
            return $embedding;
            
        } catch (Exception $e) {
            error_log('Listeo AI Search - Embedding decompression error: ' . $e->getMessage());
            // Final fallback to JSON
            return json_decode($compressed_data, true);
        }
    }
    
    /**
     * Get database statistics
     * 
     * @return array Database statistics
     */
    public static function get_database_stats() {
        global $wpdb;
        
        $table_name = self::get_embeddings_table_name();
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") === $table_name;
        
        if (!$table_exists) {
            // Create table if it doesn't exist
            self::create_tables();
        }
        
        try {
            // Get total embeddings count
            $total_embeddings = $wpdb->get_var("SELECT COUNT(*) FROM {$table_name}") ?: 0;
            
            // Get recent embeddings (last 10)
            $recent_items = $wpdb->get_results("
                SELECT e.listing_id, 
                       p.post_title as title,
                       COALESCE(pm.meta_value, 'unknown') as listing_type,
                       e.created_at
                FROM {$table_name} e
                INNER JOIN {$wpdb->posts} p ON e.listing_id = p.ID
                LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_listing_type'
                WHERE p.post_status = 'publish'
                ORDER BY e.created_at DESC
                LIMIT 10
            ", ARRAY_A);
            
            // Get total published listings
            $total_listings = $wpdb->get_var("
                SELECT COUNT(*) 
                FROM {$wpdb->posts} 
                WHERE post_type = 'listing' AND post_status = 'publish'
            ") ?: 0;
            
            // Get listings without embeddings
            $without_embeddings = $wpdb->get_var("
                SELECT COUNT(*) 
                FROM {$wpdb->posts} p
                LEFT JOIN {$table_name} e ON p.ID = e.listing_id
                WHERE p.post_type = 'listing' 
                AND p.post_status = 'publish' 
                AND e.listing_id IS NULL
            ") ?: 0;
            
            // Get recent activity
            $recent_embeddings = $wpdb->get_var("
                SELECT COUNT(*) 
                FROM {$table_name} 
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
            ") ?: 0;
            
            // Calculate coverage percentage
            $coverage_percentage = $total_listings > 0 ? round(($total_embeddings / $total_listings) * 100, 1) : 0;
            
            return array(
                'table_exists' => $table_exists,
                'total_embeddings' => (int) $total_embeddings,
                'total_listings' => (int) $total_listings,
                'without_embeddings' => (int) $without_embeddings,
                'coverage_percentage' => $coverage_percentage,
                'recent_embeddings' => (int) $recent_embeddings,
                'recent_items' => $recent_items ?: array(),
                'missing_items' => self::get_missing_embeddings(10) // Get up to 10 missing
            );
            
        } catch (Exception $e) {
            error_log('Listeo AI Search - Database stats error: ' . $e->getMessage());
            return array(
                'error' => $e->getMessage(),
                'table_exists' => $table_exists,
                'total_embeddings' => 0,
                'total_listings' => 0,
                'without_embeddings' => 0,
                'coverage_percentage' => 0,
                'recent_embeddings' => 0,
                'recent_items' => array(),
                'missing_items' => array()
            );
        }
    }
    
    /**
     * Get listings that are missing embeddings
     * 
     * @param int $limit Number of missing embeddings to return
     * @return array Array of listing data without embeddings
     */
    public static function get_missing_embeddings($limit = 10) {
        global $wpdb;
        
        $table_name = self::get_embeddings_table_name();
        
        try {
            $missing_items = $wpdb->get_results($wpdb->prepare("
                SELECT p.ID as listing_id,
                       p.post_title as title,
                       COALESCE(pm.meta_value, 'unknown') as listing_type,
                       p.post_modified as created_at
                FROM {$wpdb->posts} p
                LEFT JOIN {$table_name} e ON p.ID = e.listing_id
                LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_listing_type'
                WHERE p.post_type = 'listing' 
                AND p.post_status = 'publish' 
                AND e.listing_id IS NULL
                ORDER BY p.post_modified DESC
                LIMIT %d
            ", $limit), ARRAY_A);
            
            return $missing_items ?: array();
            
        } catch (Exception $e) {
            error_log('Listeo AI Search - Missing embeddings error: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Generate embedding for a single listing (on-demand)
     * 
     * @param int $listing_id Listing ID to generate embedding for
     * @return array Result array with success/error status
     */
    public static function generate_single_embedding($listing_id) {
        // Validate listing
        $post = get_post($listing_id);
        if (!$post || $post->post_type !== 'listing' || $post->post_status !== 'publish') {
            return array(
                'success' => false,
                'error' => 'Invalid listing or listing not published'
            );
        }
        
        // Check API key
        $api_key = get_option('listeo_ai_search_api_key', '');
        if (empty($api_key)) {
            return array(
                'success' => false,
                'error' => 'No API key configured'
            );
        }
        
        try {
            global $wpdb;
            
            // Collect listing content to get character count
            $content = Listeo_AI_Background_Processor::collect_listing_content($listing_id);
            $chars_processed = strlen($content);
            
            // Generate embedding using embedding manager directly for better control
            $embedding_manager = new Listeo_AI_Search_Embedding_Manager();
            $embedding = $embedding_manager->generate_embedding($content, true);
            
            if (!$embedding) {
                return array(
                    'success' => false,
                    'error' => 'Failed to generate embedding via OpenAI API'
                );
            }
            
            $embedding_dimensions = count($embedding);
            
            // Store embedding in database
            $table_name = $wpdb->prefix . 'listeo_ai_embeddings';
            $content_hash = md5($content);
            
            $result = $wpdb->replace($table_name, array(
                'listing_id' => $listing_id,
                'embedding' => self::compress_embedding_for_storage($embedding),
                'content_hash' => $content_hash,
                'updated_at' => current_time('mysql')
            ));
            
            if ($result === false) {
                return array(
                    'success' => false,
                    'error' => 'Failed to store embedding in database'
                );
            }
            
            return array(
                'success' => true,
                'chars_processed' => $chars_processed,
                'embedding_dimensions' => $embedding_dimensions,
                'listing_title' => $post->post_title
            );
            
        } catch (Exception $e) {
            return array(
                'success' => false,
                'error' => $e->getMessage()
            );
        }
    }

    /**
     * Process listing for embedding generation
     * 
     * @param int $post_id Post ID
     * @param WP_Post $post Post object
     */
    public static function process_listing_on_save($post_id, $post) {
        // Debug: Log all save_post calls for listings
        if (get_option('listeo_ai_search_debug_mode', false)) {
            Listeo_AI_Search_Utility_Helper::debug_log("process_listing_on_save called for post {$post_id}, type: {$post->post_type}, status: {$post->post_status}");
        }
        
        // Only process listings
        if ($post->post_type !== 'listing' || $post->post_status !== 'publish') {
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("Skipping post {$post_id} - not a published listing", 'info');
            }
            return;
        }
        
        // Check if API key is configured
        $api_key = get_option('listeo_ai_search_api_key', '');
        if (empty($api_key)) {
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("Skipping post {$post_id} - no API key configured", 'warning');
            }
            return;
        }
        
        // Skip auto-processing if in manual mode (>20 listings) UNLESS Safe Mode is enabled
        // Safe Mode should still allow individual new listings to auto-generate embeddings
        $safe_mode_enabled = get_option('listeo_ai_search_safe_mode_enabled', 0);
        
        if (class_exists('Listeo_AI_Search_Manual_Batch_Processor') && 
            Listeo_AI_Search_Manual_Batch_Processor::should_use_manual_mode() && 
            !$safe_mode_enabled) {
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("Skipping auto-processing for post {$post_id} - manual batch mode is active (Safe Mode disabled)", 'info');
            }
            return;
        }
        
        // Check embedding generation throttling
        $delay_minutes = get_option('listeo_ai_search_embedding_delay', 5);
        if ($delay_minutes > 0) {
            $last_processing_key = 'listeo_ai_last_embedding_' . $post_id;
            $last_processing_time = get_transient($last_processing_key);
            
            if ($last_processing_time !== false) {
                // Still within throttle period - skip processing
                if (get_option('listeo_ai_search_debug_mode', false)) {
                    error_log("Listeo AI Search: Skipping embedding regeneration for listing {$post_id} - still within {$delay_minutes} minute throttle period");
                }
                return;
            }
            
            // Set throttle marker to prevent rapid successive calls
            set_transient($last_processing_key, time(), $delay_minutes * MINUTE_IN_SECONDS);
        }
        
        // Get current content for hashing
        $embedding_manager = new Listeo_AI_Search_Embedding_Manager($api_key);
        $content = $embedding_manager->get_listing_content_for_embedding($post_id);
        $content_hash = md5($content);
        
        global $wpdb;
        $table_name = self::get_embeddings_table_name();
        
        // Check if we already have an embedding with the same content hash
        $existing_hash = $wpdb->get_var($wpdb->prepare(
            "SELECT content_hash FROM {$table_name} WHERE listing_id = %d",
            $post_id
        ));
        
        // Debug: Log hash comparison details
        if (get_option('listeo_ai_search_debug_mode', false)) {
            Listeo_AI_Search_Utility_Helper::debug_log("Listing {$post_id} - existing_hash: " . var_export($existing_hash, true) . ", new_hash: {$content_hash}");
        }
        
        if ($existing_hash === $content_hash) {
            // Content hasn't changed, no need to regenerate
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("Skipping embedding regeneration for listing {$post_id} - content hash unchanged", 'info');
            }
            return;
        }
        
        // Debug log: Successfully passed all checks, proceeding with embedding generation
        if (get_option('listeo_ai_search_debug_mode', false)) {
            Listeo_AI_Search_Utility_Helper::debug_log("Scheduling embedding regeneration for listing {$post_id} - passed throttling and content hash checks", 'info');
        }
        
        // In Safe Mode, process immediately using the background processor method for optimal performance
        if ($safe_mode_enabled) {
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("Safe Mode: Processing individual listing {$post_id} immediately", 'info');
            }
            
            try {
                // Use the background processor's method directly for immediate processing
                if (class_exists('Listeo_AI_Background_Processor')) {
                    $result = Listeo_AI_Background_Processor::process_single_listing($post_id);
                    
                    if ($result && get_option('listeo_ai_search_debug_mode', false)) {
                        Listeo_AI_Search_Utility_Helper::debug_log("Safe Mode: Successfully processed listing {$post_id} immediately", 'info');
                    }
                }
            } catch (Exception $e) {
                if (get_option('listeo_ai_search_debug_mode', false)) {
                    Listeo_AI_Search_Utility_Helper::debug_log("Safe Mode: Error processing listing {$post_id}: " . $e->getMessage(), 'error');
                }
                // Fall back to background processing if immediate processing fails
            }
            return;
        }
        
        // Schedule background processing (original method for non-Safe Mode)
        if (class_exists('Listeo_AI_Background_Processor')) {
            if (get_option('listeo_ai_search_debug_mode', false)) {
                error_log("Listeo AI Search: Scheduling background processing for listing {$post_id}");
            }
            
            // Use WordPress action to trigger background processing
            wp_schedule_single_event(time(), 'listeo_ai_process_listing', array($post_id));
            
            if (get_option('listeo_ai_search_debug_mode', false)) {
                error_log("Listeo AI Search: Background processing scheduled for listing {$post_id}");
            }
        } else {
            if (get_option('listeo_ai_search_debug_mode', false)) {
                error_log("Listeo AI Search: Background processor class not found for listing {$post_id}");
            }
        }
    }
    
    /**
     * Get all embeddings for search with optional location filtering
     * 
     * @param string $listing_types Comma-separated listing types or 'all'
     * @param array $detected_locations Array of detected locations for filtering
     * @return array Database results with embeddings
     */
    public static function get_embeddings_for_search($listing_types = 'all', $detected_locations = array()) {
        global $wpdb;
        
        $table_name = self::get_embeddings_table_name();
        
        // Build location condition if locations detected
        $location_condition = '';
        if (!empty($detected_locations)) {
            $location_parts = array();
            foreach ($detected_locations as $location) {
                $safe_location = $wpdb->esc_like($location);
                $location_parts[] = "(
                    pm_address.meta_value LIKE '%{$safe_location}%' OR 
                    pm_city.meta_value LIKE '%{$safe_location}%' OR
                    pm_region.meta_value LIKE '%{$safe_location}%'
                )";
            }
            $location_condition = ' AND (' . implode(' OR ', $location_parts) . ')';
        }
        
        // Build type condition
        $type_condition = '';
        if ($listing_types !== 'all') {
            $types_array = array_map('trim', explode(',', $listing_types));
            $types_placeholders = implode(',', array_fill(0, count($types_array), '%s'));
            $type_condition = $wpdb->prepare(" AND p.post_type IN ($types_placeholders)", $types_array);
        }
        
        $query = "
            SELECT DISTINCT e.listing_id, e.embedding, p.post_title, p.post_status 
            FROM {$table_name} e 
            INNER JOIN {$wpdb->posts} p ON e.listing_id = p.ID 
            LEFT JOIN {$wpdb->postmeta} pm_address ON p.ID = pm_address.post_id AND pm_address.meta_key = '_address'
            LEFT JOIN {$wpdb->postmeta} pm_city ON p.ID = pm_city.post_id AND pm_city.meta_key = '_city'
            LEFT JOIN {$wpdb->postmeta} pm_region ON p.ID = pm_region.post_id AND pm_region.meta_key = '_region'
            WHERE p.post_status = 'publish'
            {$location_condition}
            {$type_condition}
        ";
        
        return $wpdb->get_results($query);
    }
    
    /**
     * Store embedding in database
     * 
     * @param int $listing_id Listing ID
     * @param array $embedding Embedding vector
     * @param string $content_hash Content hash
     * @return bool Success status
     */
    public static function store_embedding($listing_id, $embedding, $content_hash) {
        global $wpdb;
        
        $table_name = self::get_embeddings_table_name();
        
        // Use new compression method for storage efficiency
        $compressed_embedding = self::compress_embedding_for_storage($embedding);
        
        $result = $wpdb->replace(
            $table_name,
            array(
                'listing_id' => $listing_id,
                'embedding' => $compressed_embedding,
                'content_hash' => $content_hash,
                'updated_at' => current_time('mysql')
            ),
            array('%d', '%s', '%s', '%s')
        );
        
        return $result !== false;
    }
    
    /**
     * Get embedding by listing ID
     * 
     * @param int $listing_id Listing ID
     * @return array|null Embedding data or null if not found
     */
    public static function get_embedding_by_listing_id($listing_id) {
        global $wpdb;
        
        $table_name = self::get_embeddings_table_name();
        
        $result = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table_name} WHERE listing_id = %d",
            $listing_id
        ), ARRAY_A);
        
        if ($result && !empty($result['embedding'])) {
            $result['embedding'] = self::decompress_embedding_from_storage($result['embedding']);
        }
        
        return $result;
    }
    
    /**
     * Delete embedding by listing ID
     * 
     * @param int $listing_id Listing ID
     * @return bool Success status
     */
    public static function delete_embedding($listing_id) {
        global $wpdb;
        
        $table_name = self::get_embeddings_table_name();
        
        $result = $wpdb->delete(
            $table_name,
            array('listing_id' => $listing_id),
            array('%d')
        );
        
        return $result !== false;
    }
    
    /**
     * Clear all embeddings
     * 
     * @return bool Success status
     */
    public static function clear_all_embeddings() {
        global $wpdb;
        
        $table_name = self::get_embeddings_table_name();
        
        $result = $wpdb->query("TRUNCATE TABLE {$table_name}");
        
        return $result !== false;
    }
    
    /**
     * Get embeddings count for search with optional filtering (for batch planning)
     * 
     * @param string $listing_types Comma-separated listing types or 'all'
     * @param array $detected_locations Array of detected locations for filtering
     * @return int Total count of embeddings that would be returned
     */
    public static function count_embeddings_for_search($listing_types = 'all', $detected_locations = array()) {
        global $wpdb;
        
        $table_name = self::get_embeddings_table_name();
        
        // Build location condition if locations detected
        $location_condition = '';
        if (!empty($detected_locations)) {
            $location_parts = array();
            foreach ($detected_locations as $location) {
                $safe_location = $wpdb->esc_like($location);
                $location_parts[] = "(
                    pm_address.meta_value LIKE '%{$safe_location}%' OR 
                    pm_city.meta_value LIKE '%{$safe_location}%' OR
                    pm_region.meta_value LIKE '%{$safe_location}%'
                )";
            }
            $location_condition = ' AND (' . implode(' OR ', $location_parts) . ')';
        }
        
        // Build type condition
        $type_condition = '';
        if ($listing_types !== 'all') {
            $types_array = array_map('trim', explode(',', $listing_types));
            $types_placeholders = implode(',', array_fill(0, count($types_array), '%s'));
            $type_condition = $wpdb->prepare(" AND p.post_type IN ($types_placeholders)", $types_array);
        }
        
        $query = "
            SELECT COUNT(DISTINCT e.listing_id) as total_count
            FROM {$table_name} e 
            INNER JOIN {$wpdb->posts} p ON e.listing_id = p.ID 
            LEFT JOIN {$wpdb->postmeta} pm_address ON p.ID = pm_address.post_id AND pm_address.meta_key = '_address'
            LEFT JOIN {$wpdb->postmeta} pm_city ON p.ID = pm_city.post_id AND pm_city.meta_key = '_city'
            LEFT JOIN {$wpdb->postmeta} pm_region ON p.ID = pm_region.post_id AND pm_region.meta_key = '_region'
            WHERE p.post_status = 'publish'
            {$location_condition}
            {$type_condition}
        ";
        
        $result = $wpdb->get_var($query);
        return intval($result);
    }
    
    /**
     * Get embeddings in batches for memory-efficient processing
     * 
     * @param string $listing_types Comma-separated listing types or 'all'
     * @param array $detected_locations Array of detected locations for filtering
     * @param int $batch_size Number of embeddings per batch (default: 500)
     * @param int $offset Starting offset for this batch
     * @return array Database results with embeddings for this batch
     */
    public static function get_embeddings_batch($listing_types = 'all', $detected_locations = array(), $batch_size = 500, $offset = 0) {
        global $wpdb;
        
        $table_name = self::get_embeddings_table_name();
        
        // Build location condition if locations detected
        $location_condition = '';
        if (!empty($detected_locations)) {
            $location_parts = array();
            foreach ($detected_locations as $location) {
                $safe_location = $wpdb->esc_like($location);
                $location_parts[] = "(
                    pm_address.meta_value LIKE '%{$safe_location}%' OR 
                    pm_city.meta_value LIKE '%{$safe_location}%' OR
                    pm_region.meta_value LIKE '%{$safe_location}%'
                )";
            }
            $location_condition = ' AND (' . implode(' OR ', $location_parts) . ')';
        }
        
        // Build type condition
        $type_condition = '';
        if ($listing_types !== 'all') {
            $types_array = array_map('trim', explode(',', $listing_types));
            $types_placeholders = implode(',', array_fill(0, count($types_array), '%s'));
            $type_condition = $wpdb->prepare(" AND p.post_type IN ($types_placeholders)", $types_array);
        }
        
        $query = "
            SELECT DISTINCT e.listing_id, e.embedding, p.post_title, p.post_status 
            FROM {$table_name} e 
            INNER JOIN {$wpdb->posts} p ON e.listing_id = p.ID 
            LEFT JOIN {$wpdb->postmeta} pm_address ON p.ID = pm_address.post_id AND pm_address.meta_key = '_address'
            LEFT JOIN {$wpdb->postmeta} pm_city ON p.ID = pm_city.post_id AND pm_city.meta_key = '_city'
            LEFT JOIN {$wpdb->postmeta} pm_region ON p.ID = pm_region.post_id AND pm_region.meta_key = '_region'
            WHERE p.post_status = 'publish'
            {$location_condition}
            {$type_condition}
            ORDER BY e.listing_id
            LIMIT %d OFFSET %d
        ";
        
        return $wpdb->get_results($wpdb->prepare($query, $batch_size, $offset));
    }
}
