<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class AIRS_Core {

    private static $_instance = null;

    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Get overall rating for the listing from GooGenerate between 3 and {$total_points} points total:
- Include negative points ONLY if there are genuine, significant issues that would impact someone's decision
- If the overall rating is high, prioritize positive highlights and be very selective with negative ones
- If there are no significant negative points, include only positive points
- Never force or invent negative points just to meet a quota
- NEVER turn positive statements into negative points (e.g., "no problems" is positive, not negative)
- If customers say they had "no issues" or "no complaints", this means they're satisfied - don't twist this into a negative
- Let the content naturally determine the positive/negative balance

Each object in the array must have three keys:or Listeo ratings
     */
    private function get_overall_rating( $post_id ) {
        // First try to get Google rating from transient
        $transient_name = 'listeo_reviews_' . $post_id;
        $google_data = get_transient( $transient_name );
        
        if ( ! empty( $google_data ) && isset( $google_data['result']['rating'] ) ) {
            return (float) $google_data['result']['rating'];
        }
        
        // Fallback to Listeo's rating system
        $listeo_rating = get_post_meta( $post_id, 'listeo-avg-rating', true );
        if ( ! empty( $listeo_rating ) ) {
            return (float) $listeo_rating;
        }
        
        // Default neutral rating if no data available
        return 3.5;
    }

    /**
     * Get balanced sample of reviews based on overall rating
     */
    private function get_balanced_review_sample( $post_id, $all_reviews, $overall_rating ) {
        $total_reviews = count( $all_reviews );
        
        // If we have 15 or fewer reviews, use all of them
        if ( $total_reviews <= 15 ) {
            return $all_reviews;
        }
        
        // For larger datasets, create a balanced sample
        $sample_size = min( 20, $total_reviews );
        
        // Calculate expected sentiment distribution based on overall rating
        $positive_ratio = max( 0.1, min( 0.9, ($overall_rating - 1) / 4 ) );
        
        // Separate reviews by estimated sentiment and recency
        $categorized_reviews = $this->categorize_reviews_by_sentiment( $all_reviews );
        
        // Sample strategically: ensure representation matches overall sentiment
        $positive_needed = round( $sample_size * $positive_ratio );
        $negative_needed = $sample_size - $positive_needed;
        
        $sampled_positive = $this->sample_reviews( $categorized_reviews['positive'], $positive_needed );
        $sampled_negative = $this->sample_reviews( $categorized_reviews['negative'], $negative_needed );
        
        // Shuffle to avoid temporal clustering
        $balanced_sample = array_merge( $sampled_positive, $sampled_negative );
        shuffle( $balanced_sample );
        
        return $balanced_sample;
    }

    /**
     * Categorize reviews by sentiment using improved keyword analysis
     */
    private function categorize_reviews_by_sentiment( $reviews ) {
        $positive_keywords = [
            // Strong positive words
            'great', 'excellent', 'amazing', 'wonderful', 'fantastic', 'love', 'perfect', 'awesome', 'best', 'outstanding', 'superb', 'brilliant', 'exceptional',
            // Positive emotions/experiences
            'recommend', 'satisfied', 'pleased', 'happy', 'enjoyed', 'delicious', 'tasty', 'fresh', 'quality', 'beautiful', 'clean', 'cozy', 'comfortable',
            // Positive service words
            'friendly', 'helpful', 'professional', 'polite', 'courteous', 'welcoming', 'attentive', 'efficient', 'quick', 'fast', 'prompt',
            // Positive value words
            'worth', 'value', 'reasonable', 'affordable', 'fair', 'generous', 'large', 'plenty',
            // Positive context words
            'thank', 'grateful', 'appreciate', 'impressed', 'surprised', 'exceeded', 'beyond'
        ];
        
        $negative_keywords = [
            // Strong negative words
            'terrible', 'awful', 'bad', 'worst', 'horrible', 'disappointing', 'poor', 'pathetic', 'disgusting', 'unacceptable', 'inadequate', 'unsatisfactory',
            // Negative service words
            'rude', 'unfriendly', 'unprofessional', 'ignored', 'dismissed', 'arrogant', 'condescending',
            // Negative quality words
            'dirty', 'filthy', 'stale', 'cold', 'burnt', 'overcooked', 'undercooked', 'tasteless', 'bland',
            // Negative experience words
            'slow', 'long wait', 'waited forever', 'delay', 'mistake', 'wrong', 'forgot', 'missing',
            // Negative value words
            'expensive', 'overpriced', 'rip off', 'waste', 'not worth', 'small portion', 'tiny',
            // Strong negative actions
            'avoid', 'never again', 'regret', 'hate', 'refuse'
        ];
        
        // Context modifiers that can flip sentiment
        $positive_modifiers = ['grateful', 'thankful', 'appreciate', 'love that', 'glad', 'nice to', 'good to', 'happy to'];
        $negative_modifiers = ['unfortunately', 'sadly', 'too bad', 'wish', 'should have', 'could be better'];
        
        $categorized = ['positive' => [], 'negative' => []];
        
        foreach ( $reviews as $review ) {
            $review_lower = strtolower( $review );
            $positive_score = 0;
            $negative_score = 0;
            
            // Check for positive modifiers that might override negative keywords
            $has_positive_modifier = false;
            foreach ( $positive_modifiers as $modifier ) {
                if ( strpos( $review_lower, $modifier ) !== false ) {
                    $has_positive_modifier = true;
                    $positive_score += 2; // Boost positive score
                    break;
                }
            }
            
            // Check for negative modifiers
            $has_negative_modifier = false;
            foreach ( $negative_modifiers as $modifier ) {
                if ( strpos( $review_lower, $modifier ) !== false ) {
                    $has_negative_modifier = true;
                    $negative_score += 1;
                    break;
                }
            }
            
            // Count keyword occurrences
            foreach ( $positive_keywords as $keyword ) {
                $positive_score += substr_count( $review_lower, $keyword );
            }
            
            foreach ( $negative_keywords as $keyword ) {
                $negative_score += substr_count( $review_lower, $keyword );
            }
            
            // Special case: If review has positive modifier, lean towards positive even with some negative words
            if ( $has_positive_modifier && $positive_score >= $negative_score ) {
                $categorized['positive'][] = $review;
            } elseif ( $negative_score > $positive_score + 1 ) { // Require clear negative dominance
                $categorized['negative'][] = $review;
            } else {
                $categorized['positive'][] = $review; // Default to positive for neutral/unclear
            }
        }
        
        return $categorized;
    }

    /**
     * Sample reviews with preference for more recent ones
     */
    private function sample_reviews( $reviews, $needed_count ) {
        if ( empty( $reviews ) || $needed_count <= 0 ) {
            return [];
        }
        
        if ( count( $reviews ) <= $needed_count ) {
            return $reviews;
        }
        
        // For Google reviews, recent ones are typically at the beginning
        // Take a mix of recent and older reviews
        $recent_count = min( $needed_count, ceil( $needed_count * 0.6 ) );
        $older_count = $needed_count - $recent_count;
        
        $recent_reviews = array_slice( $reviews, 0, $recent_count );
        
        if ( $older_count > 0 && count( $reviews ) > $recent_count ) {
            $older_reviews = array_slice( $reviews, $recent_count );
            $sampled_older = array_rand( $older_reviews, min( $older_count, count( $older_reviews ) ) );
            
            if ( is_array( $sampled_older ) ) {
                foreach ( $sampled_older as $index ) {
                    $recent_reviews[] = $older_reviews[$index];
                }
            } else {
                $recent_reviews[] = $older_reviews[$sampled_older];
            }
        }
        
        return $recent_reviews;
    }

/**
     * Gathers reviews from both WordPress comments and Google Places API transient.
     */
    public function get_reviews_for_listing( $post_id ) {
        $all_reviews = [];

        // 1. Get WordPress Comments & Reviews
        // Fetches both standard 'comment' and theme-specific 'review' types.
        $wp_comments = get_comments( ['post_id' => $post_id, 'status' => 'approve', 'type__in' => ['comment', 'review']] );
        foreach ( $wp_comments as $comment ) {
            if ( ! empty( $comment->comment_content ) ) {
                $all_reviews[] = "WordPress review from " . $comment->comment_author . ": " . strip_tags($comment->comment_content);
            }
        }

        // 2. Get Google Reviews from the correct Listeo transient
        // The correct transient name is 'listeo_reviews_{post_id}'
        $transient_name = 'listeo_reviews_' . $post_id;
        $google_data = get_transient( $transient_name );

        // Check if the data exists and has the expected nested structure
        if ( ! empty( $google_data ) && isset( $google_data['result']['reviews'] ) && is_array( $google_data['result']['reviews'] ) ) {
            
            $google_reviews = $google_data['result']['reviews']; // This is our array of reviews
            
            foreach ( $google_reviews as $review ) {
                // Use array notation ['key'] because the data is in arrays, not objects
                if ( ! empty( $review['text'] ) ) {
                    $author = isset($review['author_name']) ? $review['author_name'] : 'Google User';
                    $all_reviews[] = "Google review from " . $author . ": " . strip_tags($review['text']);
                }
            }
        }

        return $all_reviews;
    }

    /**
     * Generates the summary by calling the OpenAI API, but only if enough reviews exist.
     */
    public function generate_summary( $post_id ) {
        // --- The New "Fuse" Logic ---
        $options = get_option( 'airs_settings' );
        $min_reviews_required = isset( $options['min_reviews'] ) ? (int) $options['min_reviews'] : 3; // Default to 3

        $reviews = $this->get_reviews_for_listing( $post_id );
        $review_count = count( $reviews );

        // Check if the number of reviews meets the minimum requirement.
        if ( $review_count < $min_reviews_required ) {
            // Return a specific error that the AJAX handler will display.
            return new WP_Error(
                'not_enough_reviews',
                sprintf(
                    'Skipped: Not enough reviews to summarize (%d found, %d required).',
                    $review_count,
                    $min_reviews_required
                )
            );
        }
        // --- End of Fuse Logic ---

        // Get overall rating context for better AI decision making
        $overall_rating = $this->get_overall_rating( $post_id );
        
        // Always use balanced sampling to prevent misleading highlights
        $balanced_reviews = $this->get_balanced_review_sample( $post_id, $reviews, $overall_rating );
        $review_text = implode( "\n - ", $balanced_reviews );
        
        $prompt = $this->construct_prompt( $review_text, $overall_rating );
        $api_response = $this->call_openai_api( $prompt );

        if ( is_wp_error( $api_response ) ) {
            return $api_response;
        }

        // Clean the response: OpenAI sometimes wraps the JSON in ```json ... ```
        $json_response = preg_replace('/```json\s*([\s\S]*?)\s*```/', '$1', $api_response);
        $summary_data = json_decode( $json_response, true );

        if ( json_last_error() !== JSON_ERROR_NONE || !isset($summary_data['summaries']) || !is_array($summary_data['summaries']) ) {
            return new WP_Error( 'invalid_json', 'Failed to decode AI response or invalid format.', ['raw_response' => $api_response] );
        }
        
        $this->save_summary( $post_id, $summary_data );

        return $summary_data;
    }

/**
 * Constructs the detailed prompt for OpenAI based on admin settings.
 */
private function construct_prompt( $review_text, $overall_rating = null ) {
    $options = get_option( 'airs_settings' );

    // Get settings with robust defaults
    $total_points    = isset( $options['total_points'] ) ? (int) $options['total_points'] : 6;
    $summary_language = isset( $options['summary_language'] ) ? $options['summary_language'] : get_locale();
    
    // Convert language code to readable language name
    $language_name = $this->get_language_name($summary_language);

    // Build rating context
    $rating_context = '';
    if ( $overall_rating ) {
        $rating_stars = str_repeat( '⭐', floor( $overall_rating ) );
        $rating_context = "\n\nIMPORTANT CONTEXT: This business has an overall rating of {$overall_rating}/5.0 stars {$rating_stars}. Your summary should reflect this overall positive/negative trend. If the overall rating is high (4.0+), the majority of your highlights should be positive unless there are genuinely serious recurring issues.";
        
        if ( $overall_rating >= 4.0 ) {
            $rating_context .= "\n\nSince this business has a high overall rating ({$overall_rating}/5), focus primarily on what makes it successful. Only include negative points if they represent genuine, recurring issues that significantly impact the customer experience.";
        } elseif ( $overall_rating <= 2.5 ) {
            $rating_context .= "\n\nSince this business has a low overall rating ({$overall_rating}/5), you may include more negative points, but also highlight any positive aspects that customers appreciate.";
        }
    }

    return "You are an expert review analyst. Analyze the following customer reviews for a business.{$rating_context}
    
Your task is to generate a concise summary of the MOST IMPORTANT points in {$language_name}. Your response MUST be a valid JSON object and nothing else.

IMPORTANT: All text in your response must be in {$language_name}. Do not use English unless the target language is English.

The JSON object must have a single root key called 'summaries'.

CRITICAL RULES FOR POINT SELECTION:
1. Only include points that SIGNIFICANTLY impact the customer experience
2. Your highlights should align with the overall rating - don't contradict the general consensus
3. READ CAREFULLY for context - don't misinterpret positive statements as negative:
   - \"grateful for late hours\" = POSITIVE (customer appreciates extended hours)
   - \"wish they were open later\" = NEGATIVE (complaint about limited hours)
   - \"love that they stay open\" = POSITIVE (praise for hours)
   - \"hard to find parking but worth it\" = POSITIVE overall with minor inconvenience noted

4. For NEGATIVE points, only include issues that would genuinely affect someone's decision to visit/purchase:
   - Service failures (long waits, rude staff, mistakes)
   - Quality problems (bad food, broken items, poor workmanship)
   - Value issues (overpriced, hidden fees, poor portions)
   - Safety/cleanliness concerns
   - Accessibility problems
   - Consistent complaints mentioned by multiple reviewers

5. DO NOT include as negative points:
   - Minor preferences or suggestions
   - Missing features that aren't essential (like tip jars, specific payment methods)
   - One-off incidents unless severe
   - Things that are matters of personal taste
   - Neutral observations
   - Isolated complaints that contradict the overall rating trend
   - Positive statements that mention a minor inconvenience but overall praise the business
   - Statements about \"no problems\" or \"no issues\" - these are POSITIVE, not negative!
   - Absence of complaints (\"didn't have any issues\", \"no significant problems\", \"nothing to complain about\")
   - Comments that say there were \"no negatives\" or \"no downsides\"

6. For POSITIVE points, focus on what truly stands out:
   - Exceptional service experiences
   - Outstanding quality
   - Great value
   - Unique features that customers love (like extended hours, convenient location)
   - Consistently praised aspects
   - Statements confirming lack of problems or issues (these show reliability)

7. CONTEXT MATTERS: If a reviewer says they're \"grateful\" or \"appreciate\" something, it's POSITIVE even if they mention a limitation of competitors. Similarly, if someone says \"no problems\" or \"no complaints\", this indicates satisfaction and should be treated as positive feedback about reliability.

TARGET: Generate {$total_points} summary points when there is sufficient review content to support them:
- Aim for {$total_points} total points if the reviews contain enough distinct, significant points
- If there aren't enough substantial points to reach {$total_points}, generate fewer high-quality points (minimum 3)
- Include UP TO {$negative_points} negative points, but ONLY if there are genuine, significant issues
- If the overall rating is high, prioritize positive highlights and be very selective with negative ones
- If there are no significant negative points, include only positive points
- Never force or invent points just to meet the target number
- Quality over quantity - only include points that genuinely matter to potential customers
- NEVER turn positive statements into negative points (e.g., \"no problems\" is positive, not negative)
- If customers say they had \"no issues\" or \"no complaints\", this means they're satisfied - don't twist this into a negative

CRITICAL OUTPUT ORDER: Always list ALL positive points first, then ALL negative points. Do not mix positive and negative points together.

Each object in the array must have three keys:
1. 'title': A short, impactful headline in {$language_name}
2. 'description': A 1-2 sentence explanation with specific examples from reviews
3. 'sentiment': Either 'positive' or 'negative'

OUTPUT FORMAT: Structure your response as:
[
  {positive point 1},
  {positive point 2},
  {positive point 3},
  {negative point 1},
  {negative point 2}
]

Remember: A business with overwhelmingly positive reviews should have a summary that reflects that. It's better to have 5 positive points and 1 genuine negative than to invent trivial complaints.

Here are the reviews to analyze:
---
{$review_text}
---
";
}

    /**
     * Convert language code to readable language name for the prompt
     */
    private function get_language_name($language_code) {
        $language_names = array(
            'en_US' => 'English',
            'es_ES' => 'Spanish',
            'fr_FR' => 'French',
            'de_DE' => 'German',
            'it_IT' => 'Italian',
            'pt_BR' => 'Portuguese',
            'ru_RU' => 'Russian',
            'ja' => 'Japanese',
            'ko_KR' => 'Korean',
            'zh_CN' => 'Chinese (Simplified)',
            'zh_TW' => 'Chinese (Traditional)',
            'ar' => 'Arabic',
            'hi_IN' => 'Hindi',
            'tr_TR' => 'Turkish',
            'pl_PL' => 'Polish',
            'nl_NL' => 'Dutch',
            'sv_SE' => 'Swedish',
            'da_DK' => 'Danish',
            'no' => 'Norwegian',
            'fi' => 'Finnish',
            'cs_CZ' => 'Czech',
            'sk_SK' => 'Slovak',
            'hu_HU' => 'Hungarian',
            'ro_RO' => 'Romanian',
            'bg_BG' => 'Bulgarian',
            'hr' => 'Croatian',
            'sl_SI' => 'Slovenian',
            'et' => 'Estonian',
            'lv' => 'Latvian',
            'lt_LT' => 'Lithuanian',
            'el' => 'Greek',
            'he_IL' => 'Hebrew',
            'th' => 'Thai',
            'vi' => 'Vietnamese',
            'id_ID' => 'Indonesian',
            'ms_MY' => 'Malay',
            'uk' => 'Ukrainian',
            'be_BY' => 'Belarusian',
            'mk_MK' => 'Macedonian',
            'sr_RS' => 'Serbian',
            'bs_BA' => 'Bosnian',
            'sq' => 'Albanian',
            'is_IS' => 'Icelandic',
            'mt_MT' => 'Maltese',
            'ga' => 'Irish',
            'cy' => 'Welsh',
            'eu' => 'Basque',
            'ca' => 'Catalan',
            'gl_ES' => 'Galician',
            'af' => 'Afrikaans',
            'sw' => 'Swahili',
            'am' => 'Amharic',
            'bn_BD' => 'Bengali',
            'ur' => 'Urdu',
            'fa_IR' => 'Persian',
            'ta_IN' => 'Tamil',
            'te' => 'Telugu',
            'kn' => 'Kannada',
            'ml_IN' => 'Malayalam',
            'gu' => 'Gujarati',
            'pa_IN' => 'Punjabi',
            'or' => 'Odia',
            'as' => 'Assamese',
            'ne_NP' => 'Nepali',
            'si_LK' => 'Sinhala',
            'my_MM' => 'Burmese',
            'km' => 'Khmer',
            'lo' => 'Lao',
            'ka_GE' => 'Georgian',
            'hy' => 'Armenian',
            'az' => 'Azerbaijani',
            'kk' => 'Kazakh',
            'ky_KY' => 'Kyrgyz',
            'uz_UZ' => 'Uzbek',
            'tg' => 'Tajik',
            'mn' => 'Mongolian',
        );

        return isset($language_names[$language_code]) ? $language_names[$language_code] : 'English';
    }

    /**
     * Calls the OpenAI API.
     */
    private function call_openai_api( $prompt ) {
        $options = get_option( 'airs_settings' );
        $api_key = isset( $options['openai_api_key'] ) ? $options['openai_api_key'] : '';
        $model = isset( $options['openai_model'] ) ? $options['openai_model'] : 'gpt-4o';

        if ( empty( $api_key ) ) {
            return new WP_Error( 'no_api_key', 'OpenAI API key is not set.' );
        }

        $api_url = 'https://api.openai.com/v1/chat/completions';
        
        $body = [
            'model'    => $model,
            'messages' => [
                ['role' => 'system', 'content' => 'You are a helpful assistant that provides responses in valid JSON format.'],
                ['role' => 'user', 'content' => $prompt]
            ],
            'response_format' => ['type' => 'json_object'], // Use JSON mode for reliability
            'temperature' => 0.5,
        ];

        $response = wp_remote_post( $api_url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type'  => 'application/json',
            ],
            'body'    => json_encode( $body ),
            'timeout' => 60, // 60-second timeout
        ]);

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );

        if ( $response_code !== 200 ) {
            $error_data = json_decode( $response_body, true );
            $error_message = isset( $error_data['error']['message'] ) ? $error_data['error']['message'] : 'Unknown API Error';
            return new WP_Error( 'api_error', $error_message, ['status_code' => $response_code] );
        }

        $data = json_decode( $response_body, true );
        return $data['choices'][0]['message']['content'];
    }

    /**
     * Saves the generated summary to the custom database table.
     */
    public function save_summary( $post_id, $summary_data ) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ai_review_summaries';

        $data = [
            'listing_id'   => $post_id,
            'summary_data' => wp_json_encode( $summary_data, JSON_UNESCAPED_UNICODE ),
            'updated_at'   => current_time( 'mysql' ),
        ];
        
        $formats = [ '%d', '%s', '%s' ];

        // Check if a record already exists
        $exists = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM $table_name WHERE listing_id = %d", $post_id ) );

        if ($exists) {
            // Update existing record
            $wpdb->update( $table_name, $data, [ 'listing_id' => $post_id ], $formats, [ '%d' ] );
        } else {
            // Insert new record
            $data['created_at'] = current_time( 'mysql' );
            $formats[] = '%s';
            $wpdb->insert( $table_name, $data, $formats );
        }
    }
}