<?php
/**
 * AJAX Handlers for Admin Operations
 */

if (!defined('ABSPATH')) {
    exit;
}

class Listeo_iCal_Admin_Ajax_Handlers {

    public function __construct() {
        add_action('wp_ajax_listeo_ical_admin_force_update', array($this, 'handle_force_update'));
        add_action('wp_ajax_listeo_ical_admin_preview_import', array($this, 'handle_preview_import'));
        add_action('wp_ajax_listeo_ical_admin_remove_import', array($this, 'handle_remove_import'));
        add_action('wp_ajax_listeo_ical_admin_bulk_operation', array($this, 'handle_bulk_operation'));
        add_action('wp_ajax_listeo_ical_admin_get_status', array($this, 'handle_get_status'));
        add_action('wp_ajax_listeo_ical_admin_add_import', array($this, 'handle_add_import'));
        add_action('wp_ajax_listeo_ical_admin_copy_export_url', array($this, 'handle_copy_export_url'));
    }

    /**
     * Verify nonce and permissions
     */
    private function verify_request() {
        if (!wp_verify_nonce($_POST['nonce'] ?? '', 'listeo_ical_admin_nonce')) {
            wp_die(__('Security check failed', 'listeo-ical-admin'));
        }

        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'listeo-ical-admin'));
        }
    }

    /**
     * Log import operation
     */
    private function log_import_operation($listing_id, $ical_name, $ical_url, $type, $status, $result = array()) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'listeo_ical_import_history';

        $wpdb->insert(
            $table_name,
            array(
                'listing_id'      => $listing_id,
                'ical_name'       => $ical_name,
                'ical_url'        => $ical_url,
                'import_type'     => $type,
                'status'          => $status,
                'events_imported' => isset($result['imported']) ? $result['imported'] : 0,
                'events_skipped'  => isset($result['skipped_already_booked']) ? $result['skipped_already_booked'] + $result['skipped_missing_slot'] + $result['skipped_past'] : 0,
                'error_message'   => isset($result['error']) ? $result['error'] : null,
                'created_at'      => current_time('mysql')
            ),
            array('%d', '%s', '%s', '%s', '%s', '%d', '%d', '%s', '%s')
        );
    }

    /**
     * Handle force update request
     */
    public function handle_force_update() {
        $this->verify_request();

        $listing_id = absint($_POST['listing_id'] ?? 0);
        $import_index = isset($_POST['import_index']) ? absint($_POST['import_index']) : null;

        if (!$listing_id) {
            wp_send_json_error(__('Invalid listing ID', 'listeo-ical-admin'));
        }

        $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);
        if (!is_array($ical_imports) || empty($ical_imports)) {
            wp_send_json_error(__('No iCal imports found', 'listeo-ical-admin'));
        }

        $results = array();

        // If specific import index provided, update only that import
        if ($import_index !== null) {
            if (!isset($ical_imports[$import_index])) {
                wp_send_json_error(__('Import not found', 'listeo-ical-admin'));
            }

            $import = $ical_imports[$import_index];
            $import['force_update'] = 'true'; // Force the update

            try {
                if (class_exists('Listeo_Core_iCal')) {
                    $ical_core = Listeo_Core_iCal::instance();
                    $result = $ical_core->import_bookings_from_ical($import, $listing_id);

                    $status = isset($result['error']) ? 'error' : 'success';
                    $this->log_import_operation($listing_id, $import['name'], $import['url'], 'admin_force', $status, $result);

                    $results[] = array(
                        'import_name' => $import['name'],
                        'result' => $result
                    );
                } else {
                    wp_send_json_error(__('Listeo Core iCal class not found', 'listeo-ical-admin'));
                }
            } catch (Exception $e) {
                $error_result = array('error' => $e->getMessage());
                $this->log_import_operation($listing_id, $import['name'], $import['url'], 'admin_force', 'error', $error_result);
                wp_send_json_error($e->getMessage());
            }
        } else {
            // Update all imports for the listing
            foreach ($ical_imports as $index => $import) {
                $import['force_update'] = 'true';

                try {
                    if (class_exists('Listeo_Core_iCal')) {
                        $ical_core = Listeo_Core_iCal::instance();
                        $result = $ical_core->import_bookings_from_ical($import, $listing_id);

                        $status = isset($result['error']) ? 'error' : 'success';
                        $this->log_import_operation($listing_id, $import['name'], $import['url'], 'admin_bulk_force', $status, $result);

                        $results[] = array(
                            'import_name' => $import['name'],
                            'result' => $result
                        );
                    }
                } catch (Exception $e) {
                    $error_result = array('error' => $e->getMessage());
                    $this->log_import_operation($listing_id, $import['name'], $import['url'], 'admin_bulk_force', 'error', $error_result);

                    $results[] = array(
                        'import_name' => $import['name'],
                        'result' => $error_result
                    );
                }

                // Add small delay between imports
                usleep(500000); // 0.5 seconds
            }
        }

        wp_send_json_success(array(
            'message' => __('Import completed', 'listeo-ical-admin'),
            'results' => $results
        ));
    }

    /**
     * Handle preview import request
     */
    public function handle_preview_import() {
        $this->verify_request();

        $listing_id = absint($_POST['listing_id'] ?? 0);
        $import_index = isset($_POST['import_index']) ? absint($_POST['import_index']) : null;

        if (!$listing_id) {
            wp_send_json_error(__('Invalid listing ID', 'listeo-ical-admin'));
        }

        $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);
        if (!is_array($ical_imports) || empty($ical_imports)) {
            wp_send_json_error(__('No iCal imports found', 'listeo-ical-admin'));
        }

        $previews = array();

        // If specific import index provided, preview only that import
        $imports_to_preview = ($import_index !== null) ? array($import_index => $ical_imports[$import_index]) : $ical_imports;

        foreach ($imports_to_preview as $index => $import) {
            try {
                // Fetch and parse iCal data without importing
                $preview_data = $this->preview_ical_content($import['url']);

                $previews[] = array(
                    'import_name' => $import['name'],
                    'url' => $import['url'],
                    'preview' => $preview_data
                );

            } catch (Exception $e) {
                $previews[] = array(
                    'import_name' => $import['name'],
                    'url' => $import['url'],
                    'error' => $e->getMessage()
                );
            }
        }

        wp_send_json_success(array(
            'message' => __('Preview generated', 'listeo-ical-admin'),
            'previews' => $previews
        ));
    }

    /**
     * Preview iCal content without importing
     */
    private function preview_ical_content($url) {
        if (!class_exists('Listeo_Core_iCal_Reader')) {
            throw new Exception(__('iCal Reader class not found', 'listeo-ical-admin'));
        }

        $ical = new Listeo_Core_iCal_Reader(array($url));

        if (!$ical->has_events()) {
            return array(
                'events_count' => 0,
                'events' => array(),
                'message' => __('No events found in calendar', 'listeo-ical-admin')
            );
        }

        $events = array();
        $event_count = 0;
        $max_preview_events = 20; // Limit preview to first 20 events

        foreach ($ical->events() as $event) {
            if ($event_count >= $max_preview_events) {
                break;
            }

            $events[] = array(
                'summary' => isset($event->summary) ? $event->summary : __('No title', 'listeo-ical-admin'),
                'start' => isset($event->dtstart) ? $event->dtstart : '',
                'end' => isset($event->dtend) ? $event->dtend : '',
                'description' => isset($event->description) ? substr($event->description, 0, 100) . '...' : '',
                'all_day' => isset($event->all_day_event) ? $event->all_day_event : false
            );

            $event_count++;
        }

        return array(
            'events_count' => $ical->event_count,
            'events' => $events,
            'showing_count' => count($events),
            'message' => sprintf(__('Found %d events (showing first %d)', 'listeo-ical-admin'), $ical->event_count, count($events))
        );
    }

    /**
     * Handle remove import request
     */
    public function handle_remove_import() {
        $this->verify_request();

        $listing_id = absint($_POST['listing_id'] ?? 0);
        $import_index = absint($_POST['import_index'] ?? 0);

        if (!$listing_id) {
            wp_send_json_error(__('Invalid listing ID', 'listeo-ical-admin'));
        }

        $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);
        if (!is_array($ical_imports) || !isset($ical_imports[$import_index])) {
            wp_send_json_error(__('Import not found', 'listeo-ical-admin'));
        }

        $removed_import = $ical_imports[$import_index];

        // Remove the import from array
        unset($ical_imports[$import_index]);
        $ical_imports = array_values($ical_imports); // Reindex array

        // Update post meta
        update_post_meta($listing_id, 'listeo_ical_imports', $ical_imports);

        // Remove associated bookings
        try {
            if (class_exists('Listeo_Core_iCal')) {
                $ical_core = Listeo_Core_iCal::instance();
                $removed_count = $ical_core->remove_from_ical($removed_import, $listing_id);

                // Log the removal
                $this->log_import_operation(
                    $listing_id,
                    $removed_import['name'],
                    $removed_import['url'],
                    'admin_remove',
                    'success',
                    array('removed_bookings' => $removed_count)
                );

                wp_send_json_success(array(
                    'message' => sprintf(__('Import removed. %d associated bookings deleted.', 'listeo-ical-admin'), $removed_count)
                ));
            }
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }

    /**
     * Handle bulk operations
     */
    public function handle_bulk_operation() {
        $this->verify_request();

        $operation = sanitize_text_field($_POST['operation'] ?? '');
        $listing_ids = isset($_POST['listing_ids']) ? array_map('absint', $_POST['listing_ids']) : array();

        if (empty($listing_ids)) {
            wp_send_json_error(__('No listings selected', 'listeo-ical-admin'));
        }

        $results = array();

        switch ($operation) {
            case 'force_update_all':
                $results = $this->bulk_force_update_all($listing_ids);
                break;
            case 'preview_all':
                $results = $this->bulk_preview_all($listing_ids);
                break;
            default:
                wp_send_json_error(__('Invalid operation', 'listeo-ical-admin'));
        }

        wp_send_json_success($results);
    }

    /**
     * Bulk force update all imports for multiple listings
     */
    private function bulk_force_update_all($listing_ids) {
        $results = array();
        $total_imports = 0;
        $successful_imports = 0;

        foreach ($listing_ids as $listing_id) {
            $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);
            if (!is_array($ical_imports) || empty($ical_imports)) {
                continue;
            }

            $listing_results = array();

            foreach ($ical_imports as $import) {
                $total_imports++;
                $import['force_update'] = 'true';

                try {
                    if (class_exists('Listeo_Core_iCal')) {
                        $ical_core = Listeo_Core_iCal::instance();
                        $result = $ical_core->import_bookings_from_ical($import, $listing_id);

                        $status = isset($result['error']) ? 'error' : 'success';
                        if ($status === 'success') $successful_imports++;

                        $this->log_import_operation($listing_id, $import['name'], $import['url'], 'admin_bulk', $status, $result);

                        $listing_results[] = array(
                            'import_name' => $import['name'],
                            'result' => $result
                        );
                    }
                } catch (Exception $e) {
                    $error_result = array('error' => $e->getMessage());
                    $this->log_import_operation($listing_id, $import['name'], $import['url'], 'admin_bulk', 'error', $error_result);

                    $listing_results[] = array(
                        'import_name' => $import['name'],
                        'result' => $error_result
                    );
                }

                // Add delay between imports
                usleep(200000); // 0.2 seconds
            }

            $results[$listing_id] = $listing_results;
        }

        return array(
            'message' => sprintf(__('Bulk update completed. %d/%d imports successful.', 'listeo-ical-admin'), $successful_imports, $total_imports),
            'total_imports' => $total_imports,
            'successful_imports' => $successful_imports,
            'results' => $results
        );
    }

    /**
     * Bulk preview all imports for multiple listings
     */
    private function bulk_preview_all($listing_ids) {
        $results = array();

        foreach ($listing_ids as $listing_id) {
            $listing = get_post($listing_id);
            $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);

            if (!is_array($ical_imports) || empty($ical_imports)) {
                $results[$listing_id] = array(
                    'listing_title' => $listing->post_title,
                    'message' => __('No iCal imports found', 'listeo-ical-admin'),
                    'previews' => array()
                );
                continue;
            }

            $listing_previews = array();

            foreach ($ical_imports as $import) {
                try {
                    $preview_data = $this->preview_ical_content($import['url']);
                    $listing_previews[] = array(
                        'import_name' => $import['name'],
                        'url' => $import['url'],
                        'preview' => $preview_data
                    );
                } catch (Exception $e) {
                    $listing_previews[] = array(
                        'import_name' => $import['name'],
                        'url' => $import['url'],
                        'error' => $e->getMessage()
                    );
                }
            }

            $results[$listing_id] = array(
                'listing_title' => $listing->post_title,
                'previews' => $listing_previews
            );
        }

        return array(
            'message' => sprintf(__('Preview generated for %d listings', 'listeo-ical-admin'), count($listing_ids)),
            'results' => $results
        );
    }

    /**
     * Get import status for a listing
     */
    public function handle_get_status() {
        $this->verify_request();

        $listing_id = absint($_POST['listing_id'] ?? 0);

        if (!$listing_id) {
            wp_send_json_error(__('Invalid listing ID', 'listeo-ical-admin'));
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'listeo_ical_import_history';

        $recent_imports = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name WHERE listing_id = %d ORDER BY created_at DESC LIMIT 5",
            $listing_id
        ));

        wp_send_json_success(array(
            'recent_imports' => $recent_imports
        ));
    }

    /**
     * Handle add new import
     */
    public function handle_add_import() {
        $this->verify_request();

        $listing_id = absint($_POST['listing_id'] ?? 0);
        $import_name = sanitize_text_field($_POST['import_name'] ?? '');
        $import_url = esc_url_raw($_POST['import_url'] ?? '');
        $force_update = isset($_POST['force_update']) ? 'true' : 'false';

        if (!$listing_id || !$import_name || !$import_url) {
            wp_send_json_error(__('Missing required fields', 'listeo-ical-admin'));
        }

        // Validate URL
        if (!filter_var($import_url, FILTER_VALIDATE_URL)) {
            wp_send_json_error(__('Invalid URL format', 'listeo-ical-admin'));
        }

        // Check if booking is enabled for this listing
        $booking_enabled = get_post_meta($listing_id, '_booking_status', true);
        if ($booking_enabled !== 'on') {
            wp_send_json_error(__('Booking is not enabled for this listing', 'listeo-ical-admin'));
        }

        // Get existing imports
        $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);
        $ical_imports = is_array($ical_imports) ? $ical_imports : array();

        // Check for duplicate URLs
        foreach ($ical_imports as $existing_import) {
            if ($existing_import['url'] === $import_url) {
                wp_send_json_error(__('This iCal URL is already imported for this listing', 'listeo-ical-admin'));
            }
        }

        // Add new import
        $new_import = array(
            'name' => $import_name,
            'url' => $import_url,
            'force_update' => $force_update
        );

        $ical_imports[] = $new_import;

        // Update post meta
        update_post_meta($listing_id, 'listeo_ical_imports', $ical_imports);

        // Try to import immediately
        try {
            if (class_exists('Listeo_Core_iCal')) {
                $ical_core = Listeo_Core_iCal::instance();
                $result = $ical_core->import_bookings_from_ical($new_import, $listing_id);

                $status = isset($result['error']) ? 'error' : 'success';

                $this->log_import_operation(
                    $listing_id,
                    $import_name,
                    $import_url,
                    'admin_add',
                    $status,
                    $result
                );

                if (isset($result['error'])) {
                    wp_send_json_error($result['error']);
                } else {
                    wp_send_json_success(array(
                        'message' => sprintf(__('Import added successfully. %d events imported.', 'listeo-ical-admin'), $result['imported'] ?? 0),
                        'result' => $result
                    ));
                }
            } else {
                wp_send_json_error(__('Listeo Core iCal class not found', 'listeo-ical-admin'));
            }
        } catch (Exception $e) {
            $this->log_import_operation($listing_id, $import_name, $import_url, 'admin_add', 'error', array('error' => $e->getMessage()));
            wp_send_json_error($e->getMessage());
        }
    }

    /**
     * Handle copy export URL (for logging purposes)
     */
    public function handle_copy_export_url() {
        $this->verify_request();

        $listing_id = absint($_POST['listing_id'] ?? 0);
        $export_url = sanitize_url($_POST['export_url'] ?? '');

        if (!$listing_id || !$export_url) {
            wp_send_json_error(__('Invalid parameters', 'listeo-ical-admin'));
        }

        // Log the copy action
        global $wpdb;
        $table_name = $wpdb->prefix . 'listeo_ical_import_history';

        $wpdb->insert(
            $table_name,
            array(
                'listing_id'      => $listing_id,
                'ical_name'       => 'Export URL',
                'ical_url'        => $export_url,
                'import_type'     => 'admin_export_copy',
                'status'          => 'success',
                'events_imported' => 0,
                'events_skipped'  => 0,
                'error_message'   => null,
                'created_at'      => current_time('mysql')
            ),
            array('%d', '%s', '%s', '%s', '%s', '%d', '%d', '%s', '%s')
        );

        wp_send_json_success(array(
            'message' => __('Export URL copied to clipboard', 'listeo-ical-admin')
        ));
    }
}