<?php
/**
 * Import History Table
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('WP_List_Table')) {
    require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');
}

class Listeo_iCal_Admin_History_Table extends WP_List_Table {

    public function __construct() {
        parent::__construct(array(
            'singular' => 'import',
            'plural'   => 'imports',
            'ajax'     => false
        ));
    }

    /**
     * Get table columns
     */
    public function get_columns() {
        return array(
            'cb'             => '<input type="checkbox" />',
            'created_at'     => __('Date', 'listeo-ical-admin'),
            'listing_title'  => __('Listing', 'listeo-ical-admin'),
            'ical_name'      => __('Calendar Name', 'listeo-ical-admin'),
            'import_type'    => __('Type', 'listeo-ical-admin'),
            'status'         => __('Status', 'listeo-ical-admin'),
            'events'         => __('Events', 'listeo-ical-admin'),
            'details'        => __('Details', 'listeo-ical-admin')
        );
    }

    /**
     * Get sortable columns
     */
    public function get_sortable_columns() {
        return array(
            'created_at'    => array('created_at', true),
            'listing_title' => array('listing_title', false),
            'ical_name'     => array('ical_name', false),
            'status'        => array('status', false),
            'import_type'   => array('import_type', false)
        );
    }

    /**
     * Get bulk actions
     */
    public function get_bulk_actions() {
        return array(
            'delete' => __('Delete History Records', 'listeo-ical-admin'),
            'export' => __('Export to CSV', 'listeo-ical-admin')
        );
    }

    /**
     * Process bulk actions
     */
    public function process_bulk_action() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $action = $this->current_action();
        $history_ids = isset($_POST['import']) ? array_map('absint', $_POST['import']) : array();

        if (empty($history_ids)) {
            return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'listeo_ical_import_history';

        switch ($action) {
            case 'delete':
                $placeholders = implode(',', array_fill(0, count($history_ids), '%d'));
                $wpdb->query($wpdb->prepare(
                    "DELETE FROM $table_name WHERE id IN ($placeholders)",
                    $history_ids
                ));

                add_action('admin_notices', function() use ($history_ids) {
                    echo sprintf(
                        '<div class="notice notice-success"><p>%s</p></div>',
                        sprintf(__('Deleted %d history records.', 'listeo-ical-admin'), count($history_ids))
                    );
                });
                break;

            case 'export':
                $this->export_history_csv($history_ids);
                break;
        }
    }

    /**
     * Prepare table items
     */
    public function prepare_items() {
        global $wpdb;

        $this->process_bulk_action();

        $table_name = $wpdb->prefix . 'listeo_ical_import_history';
        $per_page = 50;
        $current_page = $this->get_pagenum();

        // Get search and filter parameters
        $search = isset($_REQUEST['s']) ? sanitize_text_field($_REQUEST['s']) : '';
        $status_filter = isset($_REQUEST['status_filter']) ? sanitize_text_field($_REQUEST['status_filter']) : '';
        $type_filter = isset($_REQUEST['type_filter']) ? sanitize_text_field($_REQUEST['type_filter']) : '';
        $listing_id_filter = isset($_REQUEST['listing_id']) ? absint($_REQUEST['listing_id']) : 0;

        // Get sort parameters
        $orderby = isset($_REQUEST['orderby']) ? sanitize_sql_orderby($_REQUEST['orderby']) : 'created_at';
        $order = isset($_REQUEST['order']) && $_REQUEST['order'] === 'asc' ? 'ASC' : 'DESC';

        // Build WHERE clause
        $where_conditions = array('1=1');
        $where_values = array();

        if (!empty($search)) {
            $where_conditions[] = "(h.ical_name LIKE %s OR h.ical_url LIKE %s OR p.post_title LIKE %s)";
            $search_term = '%' . $wpdb->esc_like($search) . '%';
            $where_values[] = $search_term;
            $where_values[] = $search_term;
            $where_values[] = $search_term;
        }

        if (!empty($status_filter)) {
            $where_conditions[] = "h.status = %s";
            $where_values[] = $status_filter;
        }

        if (!empty($type_filter)) {
            $where_conditions[] = "h.import_type = %s";
            $where_values[] = $type_filter;
        }

        if ($listing_id_filter) {
            $where_conditions[] = "h.listing_id = %d";
            $where_values[] = $listing_id_filter;
        }

        $where_clause = implode(' AND ', $where_conditions);

        // Count total items
        $count_sql = "SELECT COUNT(*) FROM $table_name h
                      LEFT JOIN {$wpdb->posts} p ON h.listing_id = p.ID
                      WHERE $where_clause";

        if (!empty($where_values)) {
            $total_items = $wpdb->get_var($wpdb->prepare($count_sql, $where_values));
        } else {
            $total_items = $wpdb->get_var($count_sql);
        }

        // Get items for current page
        $offset = ($current_page - 1) * $per_page;

        $items_sql = "SELECT h.*, p.post_title
                      FROM $table_name h
                      LEFT JOIN {$wpdb->posts} p ON h.listing_id = p.ID
                      WHERE $where_clause
                      ORDER BY h.$orderby $order
                      LIMIT %d OFFSET %d";

        $query_values = array_merge($where_values, array($per_page, $offset));

        if (!empty($where_values)) {
            $this->items = $wpdb->get_results($wpdb->prepare($items_sql, $query_values));
        } else {
            $this->items = $wpdb->get_results($wpdb->prepare(
                "SELECT h.*, p.post_title
                 FROM $table_name h
                 LEFT JOIN {$wpdb->posts} p ON h.listing_id = p.ID
                 ORDER BY h.$orderby $order
                 LIMIT %d OFFSET %d",
                $per_page, $offset
            ));
        }

        $this->set_pagination_args(array(
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil($total_items / $per_page)
        ));
    }

    /**
     * Default column rendering
     */
    public function column_default($item, $column_name) {
        switch ($column_name) {
            case 'created_at':
                return $this->render_date_column($item);
            case 'listing_title':
                return $this->render_listing_column($item);
            case 'ical_name':
                return esc_html($item->ical_name);
            case 'import_type':
                return $this->render_type_column($item);
            case 'status':
                return $this->render_status_column($item);
            case 'events':
                return $this->render_events_column($item);
            case 'details':
                return $this->render_details_column($item);
            default:
                return '';
        }
    }

    /**
     * Render checkbox column
     */
    public function column_cb($item) {
        return sprintf('<input type="checkbox" name="import[]" value="%s" />', $item->id);
    }

    /**
     * Render date column
     */
    private function render_date_column($item) {
        $date = mysql2date('Y-m-d H:i:s', $item->created_at);
        $time_diff = human_time_diff(strtotime($item->created_at), current_time('timestamp'));

        return sprintf(
            '<abbr title="%s">%s ago</abbr>',
            esc_attr($date),
            esc_html($time_diff)
        );
    }

    /**
     * Render listing column
     */
    private function render_listing_column($item) {
        if (!$item->post_title) {
            return sprintf('<em>%s (ID: %d)</em>', __('Deleted listing', 'listeo-ical-admin'), $item->listing_id);
        }

        $listing_link = admin_url('admin.php?page=listeo-ical-manager&action=view&listing_id=' . $item->listing_id);
        $edit_link = get_edit_post_link($item->listing_id);

        $title = sprintf('<strong><a href="%s">%s</a></strong>', esc_url($listing_link), esc_html($item->post_title));

        $actions = array(
            'manage' => sprintf('<a href="%s">%s</a>', esc_url($listing_link), __('Manage iCal', 'listeo-ical-admin')),
            'edit'   => sprintf('<a href="%s">%s</a>', esc_url($edit_link), __('Edit Listing', 'listeo-ical-admin'))
        );

        return $title . $this->row_actions($actions);
    }

    /**
     * Render type column
     */
    private function render_type_column($item) {
        $type_labels = array(
            'manual'       => __('Manual', 'listeo-ical-admin'),
            'cron'         => __('Automatic', 'listeo-ical-admin'),
            'admin_force'  => __('Admin Force', 'listeo-ical-admin'),
            'admin_bulk'   => __('Admin Bulk', 'listeo-ical-admin'),
            'admin_remove' => __('Admin Remove', 'listeo-ical-admin')
        );

        $label = isset($type_labels[$item->import_type]) ? $type_labels[$item->import_type] : ucfirst($item->import_type);

        $type_colors = array(
            'manual'       => 'blue',
            'cron'         => 'green',
            'admin_force'  => 'orange',
            'admin_bulk'   => 'purple',
            'admin_remove' => 'red'
        );

        $color = isset($type_colors[$item->import_type]) ? $type_colors[$item->import_type] : 'gray';

        return sprintf('<span class="import-type import-type-%s">%s</span>', esc_attr($color), esc_html($label));
    }

    /**
     * Render status column
     */
    private function render_status_column($item) {
        $status_colors = array(
            'success' => 'green',
            'error'   => 'red',
            'warning' => 'orange'
        );

        $color = isset($status_colors[$item->status]) ? $status_colors[$item->status] : 'gray';

        return sprintf(
            '<span class="status-badge status-%s">%s</span>',
            esc_attr($color),
            esc_html(ucfirst($item->status))
        );
    }

    /**
     * Render events column
     */
    private function render_events_column($item) {
        if ($item->import_type === 'admin_remove') {
            return sprintf('<span class="events-removed">%s</span>', __('Removed', 'listeo-ical-admin'));
        }

        $total = $item->events_imported + $item->events_skipped;
        $output = sprintf('<strong>%d</strong> %s', $item->events_imported, __('imported', 'listeo-ical-admin'));

        if ($item->events_skipped > 0) {
            $output .= sprintf('<br><small>%d %s</small>', $item->events_skipped, __('skipped', 'listeo-ical-admin'));
        }

        return $output;
    }

    /**
     * Render details column
     */
    private function render_details_column($item) {
        $details = array();

        // Add URL (truncated)
        $url = $this->truncate_url($item->ical_url, 30);
        $details[] = sprintf('<strong>%s:</strong> <a href="%s" target="_blank" title="%s">%s</a>',
            __('URL', 'listeo-ical-admin'),
            esc_url($item->ical_url),
            esc_attr($item->ical_url),
            esc_html($url)
        );

        // Add error message if present
        if (!empty($item->error_message)) {
            $details[] = sprintf('<strong>%s:</strong> <span class="error-message" title="%s">%s</span>',
                __('Error', 'listeo-ical-admin'),
                esc_attr($item->error_message),
                esc_html($this->truncate_text($item->error_message, 50))
            );
        }

        return implode('<br>', $details);
    }

    /**
     * Display extra table navigation
     */
    protected function extra_tablenav($which) {
        if ($which !== 'top') {
            return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'listeo_ical_import_history';

        // Get filter options
        $statuses = $wpdb->get_col("SELECT DISTINCT status FROM $table_name ORDER BY status");
        $types = $wpdb->get_col("SELECT DISTINCT import_type FROM $table_name ORDER BY import_type");

        $status_filter = isset($_REQUEST['status_filter']) ? $_REQUEST['status_filter'] : '';
        $type_filter = isset($_REQUEST['type_filter']) ? $_REQUEST['type_filter'] : '';

        echo '<div class="alignleft actions">';

        // Status filter
        echo '<select name="status_filter">';
        echo '<option value="">' . __('All Statuses', 'listeo-ical-admin') . '</option>';
        foreach ($statuses as $status) {
            printf(
                '<option value="%s"%s>%s</option>',
                esc_attr($status),
                selected($status_filter, $status, false),
                esc_html(ucfirst($status))
            );
        }
        echo '</select>';

        // Type filter
        echo '<select name="type_filter">';
        echo '<option value="">' . __('All Types', 'listeo-ical-admin') . '</option>';
        foreach ($types as $type) {
            printf(
                '<option value="%s"%s>%s</option>',
                esc_attr($type),
                selected($type_filter, $type, false),
                esc_html(ucfirst(str_replace('_', ' ', $type)))
            );
        }
        echo '</select>';

        submit_button(__('Filter', 'listeo-ical-admin'), '', 'filter_action', false, array('id' => 'post-query-submit'));

        echo '</div>';
    }

    /**
     * Export history to CSV
     */
    private function export_history_csv($history_ids) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'listeo_ical_import_history';
        $placeholders = implode(',', array_fill(0, count($history_ids), '%d'));

        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT h.*, p.post_title
             FROM $table_name h
             LEFT JOIN {$wpdb->posts} p ON h.listing_id = p.ID
             WHERE h.id IN ($placeholders)
             ORDER BY h.created_at DESC",
            $history_ids
        ));

        $filename = 'listeo-ical-history-' . date('Y-m-d-H-i-s') . '.csv';

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        $fp = fopen('php://output', 'w');

        // CSV headers
        fputcsv($fp, array(
            'Date',
            'Listing ID',
            'Listing Title',
            'Calendar Name',
            'Calendar URL',
            'Import Type',
            'Status',
            'Events Imported',
            'Events Skipped',
            'Error Message'
        ));

        foreach ($results as $row) {
            fputcsv($fp, array(
                $row->created_at,
                $row->listing_id,
                $row->post_title ?: 'Deleted Listing',
                $row->ical_name,
                $row->ical_url,
                $row->import_type,
                $row->status,
                $row->events_imported,
                $row->events_skipped,
                $row->error_message
            ));
        }

        fclose($fp);
        exit;
    }

    /**
     * Truncate URL for display
     */
    private function truncate_url($url, $length = 50) {
        if (strlen($url) <= $length) {
            return $url;
        }
        return substr($url, 0, $length) . '...';
    }

    /**
     * Truncate text for display
     */
    private function truncate_text($text, $length = 100) {
        if (strlen($text) <= $length) {
            return $text;
        }
        return substr($text, 0, $length) . '...';
    }
}