<?php
/**
 * Listings Table with iCal Data
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('WP_List_Table')) {
    require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');
}

class Listeo_iCal_Admin_Listings_Table extends WP_List_Table {

    private $total_items = 0;

    public function __construct() {
        parent::__construct(array(
            'singular' => 'listing',
            'plural'   => 'listings',
            'ajax'     => false
        ));
    }

    /**
     * Get table columns
     */
    public function get_columns() {
        return array(
            'cb'              => '<input type="checkbox" />',
            'title'           => __('Listing', 'listeo-ical-admin'),
            'type'            => __('Type', 'listeo-ical-admin'),
            'status'          => __('Status', 'listeo-ical-admin'),
            'ical_management' => __('iCal Management', 'listeo-ical-admin'),
            'ical_export'     => __('iCal Export', 'listeo-ical-admin'),
            'last_import'     => __('Last Import', 'listeo-ical-admin')
        );
    }

    /**
     * Get sortable columns
     */
    public function get_sortable_columns() {
        return array(
            'title'       => array('post_title', false),
            'type'        => array('_listing_type', false),
            'status'      => array('post_status', false)
        );
    }

    /**
     * Get bulk actions
     */
    public function get_bulk_actions() {
        return array(
            'force_update' => __('Force Update iCal Imports', 'listeo-ical-admin'),
            'preview_all'  => __('Preview All Imports', 'listeo-ical-admin'),
            'export_data'  => __('Export Import Data', 'listeo-ical-admin')
        );
    }

    /**
     * Handle bulk actions
     */
    public function process_bulk_action() {
        if (!current_user_can('manage_options')) {
            return;
        }

        $action = $this->current_action();
        $listing_ids = isset($_POST['listing']) ? array_map('absint', $_POST['listing']) : array();

        if (empty($listing_ids)) {
            return;
        }

        switch ($action) {
            case 'force_update':
                $this->bulk_force_update($listing_ids);
                break;
            case 'preview_all':
                $this->bulk_preview($listing_ids);
                break;
            case 'export_data':
                $this->bulk_export_data($listing_ids);
                break;
        }
    }

    /**
     * Prepare table items
     */
    public function prepare_items() {
        $this->process_bulk_action();

        $per_page = 20;
        $current_page = $this->get_pagenum();

        // Get search term
        $search = isset($_REQUEST['s']) ? sanitize_text_field($_REQUEST['s']) : '';

        // Get sort parameters
        $orderby = isset($_REQUEST['orderby']) ? sanitize_text_field($_REQUEST['orderby']) : 'post_date';
        $order = isset($_REQUEST['order']) && $_REQUEST['order'] === 'asc' ? 'ASC' : 'DESC';

        // Build query - show all listings
        $args = array(
            'post_type'      => 'listing',
            'post_status'    => array('publish', 'pending', 'draft', 'private'),
            'posts_per_page' => $per_page,
            'paged'          => $current_page,
            'orderby'        => $orderby,
            'order'          => $order
        );

        // Handle meta field sorting
        if ($orderby === '_listing_type') {
            $args['meta_key'] = '_listing_type';
            $args['orderby'] = 'meta_value';
        }

        if (!empty($search)) {
            $args['s'] = $search;
        }

        $query = new WP_Query($args);
        $this->items = $query->posts;
        $this->total_items = $query->found_posts;

        // Debug information (remove in production)
        error_log('Listeo iCal Admin: Found ' . $this->total_items . ' listings');
        if (empty($this->items)) {
            error_log('Listeo iCal Admin: No items returned from query');
            error_log('Query args: ' . print_r($args, true));
        } else {
            error_log('Listeo iCal Admin: Items array: ' . print_r(array_map(function($item) {
                return $item->ID . ' - ' . $item->post_title;
            }, $this->items), true));
        }

        $this->set_pagination_args(array(
            'total_items' => $this->total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil($this->total_items / $per_page)
        ));
    }

    /**
     * Default column rendering
     */
    public function column_default($item, $column_name) {
        switch ($column_name) {
            case 'type':
                return 'Service'; // Simple hardcoded for now
            case 'status':
                return ucfirst($item->post_status);
            case 'ical_management':
                return 'iCal Management Placeholder';
            case 'ical_export':
                return 'Export Placeholder';
            case 'last_import':
                return 'Never';
            default:
                return '&mdash;';
        }
    }

    /**
     * Render checkbox column
     */
    public function column_cb($item) {
        return sprintf('<input type="checkbox" name="listing[]" value="%s" />', $item->ID);
    }

    /**
     * Render title column
     */
    public function column_title($item) {
        error_log('Listeo iCal Admin: Rendering title for item ID: ' . $item->ID . ' - ' . $item->post_title);
        $title = '<strong>' . esc_html($item->post_title) . '</strong>';
        return $title;
    }

    /**
     * Render type column
     */
    private function render_type_column($item) {
        $listing_type = get_post_meta($item->ID, '_listing_type', true);
        if (empty($listing_type)) {
            $listing_type = 'service'; // Default type
        }

        $type_labels = array(
            'service' => __('Service', 'listeo-ical-admin'),
            'rental' => __('Rental', 'listeo-ical-admin'),
            'event' => __('Event', 'listeo-ical-admin'),
            'classifieds' => __('Classified', 'listeo-ical-admin')
        );

        $label = isset($type_labels[$listing_type]) ? $type_labels[$listing_type] : ucfirst($listing_type);
        return '<span class="listing-type listing-type-' . esc_attr($listing_type) . '">' . esc_html($label) . '</span>';
    }

    /**
     * Render status column
     */
    private function render_status_column($item) {
        $status_colors = array(
            'publish' => 'green',
            'pending' => 'orange',
            'draft'   => 'gray',
            'private' => 'blue'
        );

        $color = isset($status_colors[$item->post_status]) ? $status_colors[$item->post_status] : 'gray';

        return sprintf(
            '<span class="status-badge status-%s">%s</span>',
            esc_attr($color),
            esc_html(ucfirst($item->post_status))
        );
    }

    /**
     * Render iCal management column
     */
    private function render_ical_management_column($item) {
        try {
            $ical_imports = get_post_meta($item->ID, 'listeo_ical_imports', true);
            $ical_imports = is_array($ical_imports) ? $ical_imports : array();
            $booking_enabled = get_post_meta($item->ID, '_booking_status', true);

            $output = '<div class="ical-management-wrapper" data-listing-id="' . esc_attr($item->ID) . '">';

            // Show existing imports
            if (!empty($ical_imports)) {
                $output .= '<div class="existing-imports">';
                $output .= '<h4>' . sprintf(__('%d iCal Import(s)', 'listeo-ical-admin'), count($ical_imports)) . '</h4>';

                foreach ($ical_imports as $index => $import) {
                    if (!isset($import['name']) || !isset($import['url'])) {
                        continue; // Skip malformed imports
                    }

                    $output .= '<div class="import-item" data-import-index="' . esc_attr($index) . '">';
                    $output .= '<div class="import-details">';
                    $output .= '<strong>' . esc_html($import['name']) . '</strong><br>';
                    $output .= '<small><a href="' . esc_url($import['url']) . '" target="_blank" title="' . esc_attr($import['url']) . '">';
                    $output .= esc_html(strlen($import['url']) > 40 ? substr($import['url'], 0, 40) . '...' : $import['url']);
                    $output .= '</a></small>';
                    $output .= '</div>';
                    $output .= '<div class="import-actions">';
                    $output .= '<button type="button" class="button button-small ical-force-update" data-listing-id="' . esc_attr($item->ID) . '" data-import-index="' . esc_attr($index) . '">' . __('Update', 'listeo-ical-admin') . '</button>';
                    $output .= '<button type="button" class="button button-small ical-preview" data-listing-id="' . esc_attr($item->ID) . '" data-import-index="' . esc_attr($index) . '">' . __('Preview', 'listeo-ical-admin') . '</button>';
                    $output .= '<button type="button" class="button button-small button-link-delete ical-remove" data-listing-id="' . esc_attr($item->ID) . '" data-import-index="' . esc_attr($index) . '">' . __('Remove', 'listeo-ical-admin') . '</button>';
                    $output .= '</div>';
                    $output .= '<div id="ical-status-' . esc_attr($item->ID) . '-' . esc_attr($index) . '" class="ical-import-status"></div>';
                    $output .= '</div>';
                }
                $output .= '</div>';
            }

            // Add new import form
            if ($booking_enabled === 'on') {
                $output .= '<div class="add-import-form" style="' . (!empty($ical_imports) ? 'margin-top: 15px;' : '') . '">';
                $output .= '<h4>' . __('Add New iCal Import', 'listeo-ical-admin') . '</h4>';
                $output .= '<form class="inline-import-form" data-listing-id="' . esc_attr($item->ID) . '">';
                $output .= '<input type="text" placeholder="' . esc_attr__('Import Name', 'listeo-ical-admin') . '" name="import_name" class="small-text" required>';
                $output .= '<input type="url" placeholder="' . esc_attr__('iCal URL', 'listeo-ical-admin') . '" name="import_url" class="regular-text" required>';
                $output .= '<label><input type="checkbox" name="force_update"> ' . __('Force Update', 'listeo-ical-admin') . '</label>';
                $output .= '<button type="submit" class="button button-small">' . __('Add Import', 'listeo-ical-admin') . '</button>';
                $output .= '</form>';
                $output .= '</div>';
            } else {
                if (empty($ical_imports)) {
                    $output .= '<p class="description">' . __('Booking not enabled for this listing', 'listeo-ical-admin') . '</p>';
                }
            }

            $output .= '</div>';
            return $output;
        } catch (Exception $e) {
            return '<span class="error">Error: ' . esc_html($e->getMessage()) . '</span>';
        }
    }

    /**
     * Render iCal export column
     */
    private function render_ical_export_column($item) {
        try {
            $booking_enabled = get_post_meta($item->ID, '_booking_status', true);

            if ($booking_enabled !== 'on') {
                return '<span class="status-inactive">' . __('Booking disabled', 'listeo-ical-admin') . '</span>';
            }

            // Check if listeo_ical_export_url function exists (from Listeo Core)
            if (function_exists('listeo_ical_export_url')) {
                $export_url = listeo_ical_export_url($item->ID);

                $output = '<div class="ical-export-wrapper">';
                $output .= '<input type="text" value="' . esc_attr($export_url) . '" class="regular-text ical-export-url" readonly>';
                $output .= '<button type="button" class="button button-small copy-export-url" data-url="' . esc_attr($export_url) . '">' . __('Copy', 'listeo-ical-admin') . '</button>';
                $output .= '</div>';

                return $output;
            }

            return '<span class="status-inactive">' . __('Export not available', 'listeo-ical-admin') . '</span>';
        } catch (Exception $e) {
            return '<span class="error">Error: ' . esc_html($e->getMessage()) . '</span>';
        }
    }

    /**
     * Render last import column
     */
    private function render_last_import_column($item) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'listeo_ical_import_history';
        $last_import = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE listing_id = %d ORDER BY created_at DESC LIMIT 1",
            $item->ID
        ));

        if (!$last_import) {
            return '<span class="status-inactive">' . __('Never', 'listeo-ical-admin') . '</span>';
        }

        $time_diff = human_time_diff(strtotime($last_import->created_at), current_time('timestamp'));
        $status_class = $last_import->status === 'success' ? 'status-success' : 'status-error';

        $output = sprintf('<div class="last-import %s">', $status_class);
        $output .= sprintf('<strong>%s ago</strong><br>', $time_diff);
        $output .= sprintf('<small>%s: %s</small>', __('Status', 'listeo-ical-admin'), ucfirst($last_import->status));
        if ($last_import->events_imported > 0) {
            $output .= sprintf('<br><small>%d %s</small>', $last_import->events_imported, __('events', 'listeo-ical-admin'));
        }
        $output .= '</div>';

        return $output;
    }


    /**
     * Handle bulk force update
     */
    private function bulk_force_update($listing_ids) {
        $updated = 0;
        foreach ($listing_ids as $listing_id) {
            // Trigger force update via AJAX endpoint
            $updated++;
        }

        add_action('admin_notices', function() use ($updated) {
            echo sprintf(
                '<div class="notice notice-success"><p>%s</p></div>',
                sprintf(__('Triggered force update for %d listings.', 'listeo-ical-admin'), $updated)
            );
        });
    }

    /**
     * Handle bulk preview
     */
    private function bulk_preview($listing_ids) {
        // Store listing IDs in transient for AJAX processing
        set_transient('listeo_ical_bulk_preview_' . get_current_user_id(), $listing_ids, 300);

        add_action('admin_notices', function() use ($listing_ids) {
            echo sprintf(
                '<div class="notice notice-info"><p>%s</p></div>',
                sprintf(__('Preview prepared for %d listings. Click "Process Preview" to continue.', 'listeo-ical-admin'), count($listing_ids))
            );
        });
    }

    /**
     * Handle bulk export data
     */
    private function bulk_export_data($listing_ids) {
        // Generate CSV export
        $filename = 'listeo-ical-data-' . date('Y-m-d-H-i-s') . '.csv';
        $filepath = wp_upload_dir()['basedir'] . '/' . $filename;

        $fp = fopen($filepath, 'w');

        // CSV headers
        fputcsv($fp, array(
            'Listing ID',
            'Title',
            'Status',
            'Booking Type',
            'iCal Import Name',
            'iCal URL',
            'Force Update',
            'Last Import',
            'Import Status'
        ));

        foreach ($listing_ids as $listing_id) {
            $listing = get_post($listing_id);
            $ical_imports = get_post_meta($listing_id, 'listeo_ical_imports', true);

            if (empty($ical_imports)) {
                fputcsv($fp, array(
                    $listing_id,
                    $listing->post_title,
                    $listing->post_status,
                    listeo_get_booking_type($listing_id),
                    'No imports',
                    '',
                    '',
                    '',
                    ''
                ));
            } else {
                foreach ($ical_imports as $import) {
                    fputcsv($fp, array(
                        $listing_id,
                        $listing->post_title,
                        $listing->post_status,
                        listeo_get_booking_type($listing_id),
                        $import['name'],
                        $import['url'],
                        isset($import['force_update']) ? $import['force_update'] : 'false',
                        '', // Will be filled by last import data
                        ''  // Will be filled by import status
                    ));
                }
            }
        }

        fclose($fp);

        $upload_url = wp_upload_dir()['baseurl'] . '/' . $filename;

        add_action('admin_notices', function() use ($upload_url, $filename) {
            echo sprintf(
                '<div class="notice notice-success"><p>%s <a href="%s" download>%s</a></p></div>',
                __('Export completed:', 'listeo-ical-admin'),
                esc_url($upload_url),
                esc_html($filename)
            );
        });
    }

    /**
     * Display table with custom wrapper
     */
    public function display() {
        echo '<div class="listeo-ical-admin-table-wrapper">';
        error_log('Listeo iCal Admin: Displaying table with ' . count($this->items) . ' items');
        parent::display();
        echo '</div>';
    }
}