<?php

/**
 * Merchandiser Class
 */

define( 'MERCHANDISER_THEME_WEBSITE',               'https://merchandiser.getbowtied.com' );
define( 'MERCHANDISER_DOCUMENTATION_LINK',          'https://www.getbowtied.com/docs/merchandiser/' );
define( 'MERCHANDISER_GOOGLE_FONTS_WEBSITE',        'https://fonts.google.com' );
define( 'MERCHANDISER_SAFE_FONTS_WEBSITE',          'https://www.w3schools.com/cssref/css_websafe_fonts.asp' );

define( 'MERCHANDISER_WOOCOMMERCE_IS_ACTIVE',       class_exists( 'WooCommerce' ) );
define( 'MERCHANDISER_VISUAL_COMPOSER_IS_ACTIVE',   defined( 'WPB_VC_VERSION' ) );
define( 'MERCHANDISER_WISHLIST_IS_ACTIVE',          class_exists( 'YITH_WCWL' ) );

if ( !class_exists( 'Merchandiser' ) ) {
	class Merchandiser {

		/**
		 * Convert string to slug.
		 *
		 * @param [string] $str String to be converted.
		 *
		 * @return [string] Processed string.
		 */
		public static function convert_string_to_slug( $str ) {
			$str = strtolower(trim($str));
			$str = preg_replace('/[^a-z0-9-]/', '_', $str);
			$str = preg_replace('/-+/', "_", $str);

			return $str;
		}

		/**
		 * Get theme's name.
		 *
		 * @return [string] Theme's name.
		 */
		public static function get_theme_name() {
			$theme = wp_get_theme(get_template());
			$theme_name = $theme->get('Name');

			return $theme_name;
		}

		/**
		 * Get theme's slug.
		 *
		 * @return [string] Theme's slug.
		 */
		public static function get_theme_slug() {
			$getbowtied_theme = wp_get_theme(get_template());

			return Merchandiser::convert_string_to_slug( $getbowtied_theme->get('Name') );
		}

		/**
		 * Get theme's author.
		 *
		 * @return [string] Theme's author.
		 */
		public static function get_theme_author() {
			$getbowtied_theme = wp_get_theme(get_template());

			return $getbowtied_theme->get('Author');
		}

		/**
		 * Get theme's description.
		 *
		 * @return [string] Theme's description.
		 */
		public static function get_theme_description() {
			$getbowtied_theme = wp_get_theme(get_template());

			return $getbowtied_theme->get('Description');
		}

		/**
		 * Get theme's version.
		 *
		 * @return [string] Theme's version.
		 */
		public static function get_theme_version() {
			$getbowtied_theme = wp_get_theme(get_template());

			return $getbowtied_theme->get('Version');
		}

		/**
		 * Convert HEX to RGB.
		 *
		 * @param [string] $hex String to be converted.
		 *
		 * @return [string] Processed string.
		 */
		public static function convert_hex_to_rgb( $hex ) {
			$hex = str_replace("#", "", $hex);

			if(strlen($hex) == 3) {
				$r = hexdec(substr($hex,0,1).substr($hex,0,1));
				$g = hexdec(substr($hex,1,1).substr($hex,1,1));
				$b = hexdec(substr($hex,2,1).substr($hex,2,1));
			} else {
				$r = hexdec(substr($hex,0,2));
				$g = hexdec(substr($hex,2,2));
				$b = hexdec(substr($hex,4,2));
			}
			$rgb = array($r, $g, $b);

			return implode(",", $rgb); // returns the rgb values separated by commas
		}

		/**
		 * Get page ID.
		 *
		 * @return [string] Page ID.
		 */
		public static function get_page_id() {
			$page_id = "";
			if ( is_single() || is_page() ) {
			    $page_id = get_the_ID();
			} else if ( is_home() ) {
			    $page_id = get_option('page_for_posts');
			} else if ( MERCHANDISER_WOOCOMMERCE_IS_ACTIVE && is_shop() ) {
				$page_id = wc_get_page_id( 'shop' );
			}

			return $page_id;
		}

		/**
		 * Get file contents.
		 *
		 * @param [string] $file_path File path.
		 *
		 * @return [string] File contents.
		 */
		public static function get_local_file_contents( $file_path ) {

			$url_get_contents_data = false;

			if (function_exists('ob_start') && function_exists('ob_get_clean') && ($url_get_contents_data == false))
			{
			 ob_start();
			    include $file_path;
			    $url_get_contents_data = ob_get_clean();
			}

			return $url_get_contents_data;
		}

		/**
		 * Get Ajax URL.
		 *
		 * @return [string] Ajax URL.
		 */
		public static function get_ajax_url() {

			$ajax_url = admin_url( 'admin-ajax.php' );

			if ( class_exists('SitePress') ) {
				$my_current_lang = apply_filters( 'wpml_current_language', NULL );
				if ( $my_current_lang ) {
				    $ajax_url = add_query_arg( 'wpml_lang', $my_current_lang, $ajax_url );
			}}

			return $ajax_url;
		}

		/**
		 * Sanitizes select controls.
		 *
		 * @param string $input [the input].
		 * @param string $setting [the settings].
		 *
		 * @return string
		 */
		public static function sanitize_select( $input, $setting ) {
			$input   = sanitize_key( $input );
			$choices = isset($setting->manager->get_control( $setting->id )->choices) ? $setting->manager->get_control( $setting->id )->choices : '';

			return ( $choices && array_key_exists( $input, $choices ) ) ? $input : $setting->default;
		}

		/**
		 * Sanitizes image upload.
		 *
		 * @param string $input potentially dangerous data.
		 */
		public static function sanitize_image( $input ) {
			$filetype = wp_check_filetype( $input );
			if ( $filetype['ext'] && ( wp_ext2type( $filetype['ext'] ) === 'image' || $filetype['ext'] === 'svg' ) ) {
				return esc_url( $input );
			}

			return '';
		}

		/**
		 * Converts string to bool.
		 *
		 * @param string $string [the input].
		 *
		 * @return boolean
		 */
		public static function convert_string_to_bool( $string ) {

		    return is_bool( $string ) ? $string : ( 'yes' === $string || 1 === $string || 'true' === $string || '1' === $string );
		}

		/**
		 * Sanitizes checkbox controls.
		 * Returns true if checkbox is checked.
		 *
		 * @param string $input [the input].
		 *
		 * @return boolean
		 */
		public static function sanitize_checkbox( $input ){

			return Merchandiser::convert_string_to_bool($input);
		}
	}
}
