<?php

namespace Elementor;

use Thim_EL_Kit\Elementor;
use Thim_EL_Kit\GroupControlTrait;
use Thim_EL_Kit\Elementor\Controls\Controls_Manager as Thim_Control_Manager;
use WP_Query;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Thim_Projects_El extends Widget_Base {

	use GroupControlTrait;

	public function __construct( $data = array(), $args = null ) {
		parent::__construct( $data, $args );
	}

	public function get_name() {
		return 'thim-projects';
	}

	public function get_title() {
		return esc_html__( 'Thim: Projects', 'coaching' );
	}

	protected function get_html_wrapper_class() {
		return 'thim-projects';
	}

	public function get_icon() {
		return 'thim-eicon eicon-slider-push';
	}
	public function get_style_depends(): array {
		return [ 'e-swiper' ];
	}
	public function get_categories() {
		return [ 'thim_coaching' ];
	}

	public function get_keywords() {
		return [
			'thim',
			'project',
			'slide',
		];
	}

	protected function register_controls() {
		$this->start_controls_section(
			'content',
			[
				'label' => esc_html__( 'Project', 'coaching' )
			]
		);

		$this->add_control(
			'layout',
			array(
				'label'   => esc_html__( 'Choose layout', 'coaching' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'slider',
				'options' => array(
					'slider' => esc_html__( 'Slide', 'coaching' ),
					'grid'   => esc_html__( 'Grid', 'coaching' ),
				),
			)
		);

		$this->add_control(
			'cat_ids',
			[
				'label'       => esc_html__( 'Select Project Category', 'coaching' ),
				'type'        => Controls_Manager::SELECT2,
				'label_block' => true,
				'multiple'    => true,
				'options'     => Elementor::get_cat_taxonomy( 'portfolio_category' ),
			]
		);

		$this->add_responsive_control(
			'columns',
			[
				'label'     => esc_html__( 'Columns', 'coaching' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => '3',
				'options'   => array(
					'1' => '1',
					'2' => '2',
					'3' => '3',
					'4' => '4',
				),
				'selectors' => array(
					'{{WRAPPER}}' => '--thim-project-columns: repeat({{VALUE}}, 1fr)',
				),
				'condition' => array(
					'layout' => 'grid'
				)
			]
		);

		$this->add_control(
			'data_project',
			array(
				'label'       => esc_html__( 'Data Project', 'coaching' ),
				'label_block' => true,
				'type'        => Thim_Control_Manager::SELECT2,
				'default'     => array( 'img', 'logo', 'description', 'readmore' ),
				'multiple'    => true,
				'sortable'    => true,
				'options'     => [
					'img'         => esc_html__( 'Thumbnail', 'coaching' ),
					'title'       => esc_html__( 'Title', 'coaching' ),
					'logo'        => esc_html__( 'Logo', 'coaching' ),
					'description' => esc_html__( 'Description', 'coaching' ),
					'readmore'    => esc_html__( 'Read More', 'coaching' ),
				],
			)
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			array(
				'name'      => 'thumbnail_size',
				'default'   => 'full',
				'condition' => array(
					'data_project' => 'img'
				)
			)
		);

		$this->add_control(
			'project_link_text',
			[
				'label'     => esc_html__( 'Link Text', 'coaching' ),
				'type'      => Controls_Manager::TEXT,
				'condition' => array(
					'data_project' => 'readmore'
				)
			]
		);

		$this->add_control(
			'number_project',
			array(
				'label'   => esc_html__( 'Number Project', 'coaching' ),
				'default' => '4',
				'min'     => 0,
				'step'    => 1,
				'type'    => Controls_Manager::NUMBER,
			)
		);

		$this->add_control(
			'orderby',
			[
				'label'   => esc_html__( 'Order by', 'coaching' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'date'       => esc_html__( 'Date', 'coaching' ),
					'rand'       => esc_html__( 'Random', 'coaching' ),
					'title'      => esc_html__( 'Title', 'coaching' ),
					'menu_order' => esc_html__( 'Sort custom', 'coaching' ),
				],
				'default' => 'date',
			]
		);
		$this->add_control(
			'order',
			[
				'label'   => esc_html__( 'Order', 'coaching' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'desc' => esc_html__( 'DESC', 'coaching' ),
					'asc'  => esc_html__( 'ASC', 'coaching' ),
				],
				'default' => 'desc',
			]
		);

		$this->end_controls_section();

		$this->_register_settings_slider(
			array(
				'layout' => 'slider',
			)
		);

		$this->register_item_project_style();
		$this->register_thumbnail_project_style();
		$this->register_thumbnail_project_title();
		$this->register_thumbnail_project_description();
		$this->register_thumbnail_project_readmore();
		$this->_register_setting_slider_dot_style();
		$this->_register_setting_slider_nav_style();
	}

	protected function register_style_typo_color_margin( string $prefix_name, string $selector ) {
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => $prefix_name . '_typography',
				'selector' => "{{WRAPPER}} $selector",
			)
		);

		$this->add_responsive_control(
			$prefix_name . '_margin',
			array(
				'label'      => esc_html__( 'Margin', 'coaching' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					"{{WRAPPER}} $selector" => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			$prefix_name . '_color',
			array(
				'label'     => esc_html__( 'Color', 'coaching' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					"{{WRAPPER}} $selector" => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			$prefix_name . '_color_hover',
			array(
				'label'     => esc_html__( 'Color Hover', 'coaching' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					"{{WRAPPER}} $selector:hover" => 'color: {{VALUE}};',
				),
			)
		);
	}

	protected function register_item_project_style() {
		$this->start_controls_section(
			'section_item_project',
			array(
				'label' => esc_html__( 'Item', 'coaching' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'item_align',
			array(
				'label'     => esc_html__( 'Alignment', 'coaching' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'left'   => array(
						'title' => esc_html__( 'Left', 'coaching' ),
						'icon'  => 'eicon-text-align-left',
					),
					'center' => array(
						'title' => esc_html__( 'Center', 'coaching' ),
						'icon'  => 'eicon-text-align-center',
					),
					'right'  => array(
						'title' => esc_html__( 'Right', 'coaching' ),
						'icon'  => 'eicon-text-align-right',
					),
				),
				'default'   => 'center',
				'toggle'    => true,
				'selectors' => array(
					'{{WRAPPER}} .thim-project__article' => 'text-align: {{VALUE}};',
				),
			)
		);

		$this->add_responsive_control(
			'project_gap',
			array(
				'label'     => esc_html__( 'Gap', 'coaching' ),
				'type'      => Controls_Manager::SLIDER,
				'default'   => array(
					'size' => 30,
				),
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 100,
					),
				),
				'selectors' => array(
					'{{WRAPPER}}' => '--thim-ekits-project-gap: {{SIZE}}{{UNIT}}',
				),
				'separator' => 'before',
				'condition' => array(
					'layout' => 'grid',
				),
			)
		);

		$this->add_responsive_control(
			'item_padding',
			[
				'label'      => esc_html__( 'Padding', 'coaching' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'default'    => array(
					'top'    => 0,
					'right'  => 30,
					'bottom' => 24,
					'left'   => 30,
					'unit'   => 'px',
				),
				'selectors'  => [
					'{{WRAPPER}} .thim-project__article' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'item_bg_color',
			[
				'label'     => esc_html__( 'Background Color', 'coaching' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#fff',
				'selectors' => [
					'{{WRAPPER}} .thim-project__article' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();
	}

	protected function register_thumbnail_project_style() {
		$this->start_controls_section(
			'section_thumb_project',
			array(
				'label' => esc_html__( 'Thumbnail', 'coaching' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'thumb_margin',
			[
				'label'      => esc_html__( 'Margin', 'coaching' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'default'    => array(
					'top'    => 0,
					'right'  => - 30,
					'bottom' => 24,
					'left'   => - 30,
					'unit'   => 'px',
				),
				'selectors'  => [
					'{{WRAPPER}} .thim-project__image' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition'  => array(
					'data_project' => 'img'
				)
			]
		);

		$this->add_control(
			'logo_project',
			array(
				'label'     => esc_html__( 'Logo', 'thim-elementor-kit' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => array(
					'data_project' => 'logo'
				)
			)
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			array(
				'name'      => 'logo_size',
				'default'   => 'full',
				'condition' => array(
					'data_project' => 'logo'
				)
			)
		);

		$this->add_responsive_control(
			'logo_margin',
			[
				'label'      => esc_html__( 'Margin', 'coaching' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'default'    => array(
					'top'    => 0,
					'right'  => 0,
					'bottom' => 12,
					'left'   => 0,
					'unit'   => 'px',
				),
				'selectors'  => [
					'{{WRAPPER}} .thim-project__logo' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition'  => array(
					'data_project' => 'img'
				)
			]
		);

		$this->end_controls_section();
	}

	protected function register_thumbnail_project_title() {
		$this->start_controls_section(
			'section_title_project',
			array(
				'label'     => esc_html__( 'Title', 'coaching' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'data_project' => 'title'
				)
			)
		);

		$this->register_style_typo_color_margin( 'title_project', '.thim-project__title' );

		$this->end_controls_section();
	}

	protected function register_thumbnail_project_description() {
		$this->start_controls_section(
			'section_description_project',
			array(
				'label'     => esc_html__( 'Description', 'coaching' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'data_project' => 'description'
				)
			)
		);

		$this->register_style_typo_color_margin( 'description_project', '.thim-excerpt' );

		$this->add_control(
			'excerpt_max_line',
			array(
				'label'       => esc_html__( 'Max Line', 'coaching' ),
				'type'        => Controls_Manager::NUMBER,
				'label_block' => false,
				'min'         => 0,
				'step'        => 1,
				'selectors'   => array(
					'{{WRAPPER}} .thim-excerpt' => 'display: -webkit-box; text-overflow: ellipsis; -webkit-line-clamp: {{VALUE}};-webkit-box-orient:vertical; overflow: hidden;',
				),
			)
		);

		$this->end_controls_section();
	}

	protected function register_thumbnail_project_readmore() {
		$this->start_controls_section(
			'section_readmore_project',
			array(
				'label'     => esc_html__( 'Read More', 'coaching' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'data_project' => 'readmore'
				)
			)
		);

		$this->add_control(
			'icon_readmore',
			array(
				'label'       => esc_html__( 'Icon Read More', 'coaching' ),
				'type'        => Controls_Manager::ICONS,
				'skin'        => 'inline',
				'label_block' => false,
			)
		);

		$this->add_control(
			'icon_spacing',
			array(
				'label'     => esc_html__( 'Icon Spacing (px)', 'coaching' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'max' => 50,
					),
				),
				'default'   => array(
					'size' => 5,
					'unit' => 'px',
				),
				'selectors' => array(
					'{{WRAPPER}} .thim-project__readmore i' => 'margin-left: {{SIZE}}{{UNIT}};',
				),
				'condition' => array( 'icon_readmore[value]!' => '' ),
			)
		);

		$this->register_button_style( 'readmore_project', '.thim-project__readmore' );

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		$query_args = array(
			'post_type'           => 'portfolio',
			'posts_per_page'      => absint( $settings['number_project'] ),
			'order'               => ( 'asc' == $settings['order'] ) ? 'asc' : 'desc',
			'ignore_sticky_posts' => true,
		);

		if ( $settings['cat_ids'] && $settings['cat_ids'] != 'all' ) {
			$tax_query               = array(
				array(
					'taxonomy' => 'portfolio_category',
					'field'    => 'slug',
					'terms'    => $settings['cat_ids'],
				),
			);
			$query_args['tax_query'] = $tax_query;
		}

		switch ( $settings['orderby'] ) {
			case 'title':
				$query_args['orderby'] = 'post_title';
				break;
			case 'menu_order':
				$query_args['orderby'] = 'menu_order';
				break;
			case 'rand':
				$query_args['orderby'] = 'rand';
				break;
			default:
				$query_args['orderby'] = 'date';
		}

		$query_vars = new WP_Query( $query_args );

		$class       = 'thim-project';
		$class_inner = 'thim-project__inner';
		$class_item  = 'thim-project__article';

		if ( $query_vars->have_posts() ) {
			if ( isset( $settings['layout'] ) && $settings['layout'] == 'slider' ) {
				$swiper_class = Plugin::$instance->experiments->is_feature_active( 'e_swiper_latest' ) ? 'swiper' : 'swiper-container';
				$class        .= ' thim-ekits-sliders ' . $swiper_class;
				$class_inner  = 'swiper-wrapper';
				$class_item   .= ' swiper-slide';

				$this->render_nav_pagination_slider( $settings );
			}
			?>
            <div class="<?php
			echo esc_attr( $class ); ?>">
                <div class="<?php
				echo esc_attr( $class_inner ); ?>">
					<?php
					while ( $query_vars->have_posts() ) {
						$query_vars->the_post();
						$data_project = $settings['data_project'];

						echo '<div class="' . esc_attr( $class_item ) . '">';
						if ( ! empty( $data_project ) ) {
							foreach ( $data_project as $data ) {
								switch ( $data ) {
									case 'img':
										$this->render_thumbnail( $settings );
										break;
									case 'logo':
										$this->render_logo( $settings );
										break;
									case 'description':
										echo thim_excerpt( 15 );
										break;
									case 'title':
										$this->render_title();
										break;
									case 'readmore':
										$this->render_readmore( $settings );
										break;
								}
							}
						}
						echo '</div>';
					}
					?>
                </div>
            </div>

			<?php
		} else {
			echo '<div class="message-info">' . __( 'No data were found matching your selection, you need to create Project or select Category of Widget.',
					'coaching' ) . '</div>';
		}

		wp_reset_postdata();
	}

	protected function render_thumbnail( $settings ) {
		$settings['thumbnail_size'] = array(
			'id' => get_post_thumbnail_id( get_the_ID() ),
		);
		$thumbnail_html             = Group_Control_Image_Size::get_attachment_image_html( $settings,
			'thumbnail_size' );
		echo '<div class="thim-project__image">';
		echo '<a href="' . esc_url( get_permalink( get_the_ID() ) ) . '" title="' . esc_attr( get_the_title( get_the_ID() ) ) . '" >' . wp_kses_post( $thumbnail_html ) . '</a>';
		echo '</div>';
	}

	protected function render_logo( $settings ) {
		$logo = get_post_meta( get_the_ID(), 'thim_portfolio_project_logo', false );

		if ( isset( $logo ) && ! empty( $logo ) ) {
			$settings['logo_size'] = array(
				'id' => $logo[0],
			);
			$logo_html             = Group_Control_Image_Size::get_attachment_image_html( $settings, 'logo_size' );

			echo '<div class="thim-project__logo">' . wp_kses_post( $logo_html ) . '</div>';
		}
	}

	protected function render_title() {
		echo '<h5 class="thim-project__title"><a href="' . esc_url( get_permalink( get_the_ID() ) ) . '" title="' . esc_attr( get_the_title( get_the_ID() ) ) . '" >' . get_the_title( get_the_ID() ) . '</a></h5>';
	}

	protected function render_readmore( $settings ) {
		$text = ! empty( $settings['project_link_text'] ) ? $settings['project_link_text'] : esc_html__( 'View Detail',
			'coaching' );

		echo '<a class="thim-project__readmore" href="' . esc_url( get_permalink( get_the_ID() ) ) . '" title="' . esc_attr( get_the_title( get_the_ID() ) ) . '" >';
		echo '<span>' . $text . '</span>';
		if ( isset( $settings['icon_readmore'] ) && ! empty( $settings['icon_readmore']['value'] ) ) {
			echo '<i class="' . $settings['icon_readmore']['value'] . '"></i> ';
		}
		echo '</a>';
	}
}
