<?php

/**
 * Animation
 *
 * @param $css_animation
 *
 * @return string
 */

use Thim_EL_Kit\Functions;

if ( ! function_exists( 'thim_getCSSAnimation' ) ) {
	function thim_getCSSAnimation( $css_animation ) {
		$output = '';
		if ( $css_animation != '' ) {
			wp_enqueue_script( 'thim-waypoints' );
			$output = ' wpb_animate_when_almost_visible wpb_' . $css_animation;
		}

		return $output;
	}
}


/**
 * Custom excerpt
 *
 * @param $limit
 *
 * @return array|mixed|string|void
 */
function thim_excerpt( $limit ) {
	$excerpt = explode( ' ', get_the_excerpt(), $limit );
	if ( count( $excerpt ) >= $limit ) {
		array_pop( $excerpt );
		$excerpt = implode( " ", $excerpt ) . '...';
	} else {
		$excerpt = implode( " ", $excerpt );
	}
	$excerpt = preg_replace( '`\[[^\]]*\]`', '', $excerpt );
	$excerpt = strip_tags( $excerpt );

	return '<div class="thim-excerpt">' . $excerpt . '</div>';
}


/**
 * Get related posts
 *
 * @param     $post_id
 * @param int $number_posts
 *
 * @return WP_Query
 */
if ( ! function_exists( 'thim_get_related_posts' ) ) {
	function thim_get_related_posts( $post_id, $number_posts = - 1 ) {
		$query = new WP_Query();
		$args  = '';
		if ( $number_posts == 0 ) {
			return $query;
		}
		$args = wp_parse_args(
			$args, array(
				'posts_per_page'      => $number_posts,
				'post__not_in'        => array( $post_id ),
				'ignore_sticky_posts' => 0,
				'meta_key'            => '_thumbnail_id',
				'category__in'        => wp_get_post_categories( $post_id )
			)
		);

		$query = new WP_Query( $args );

		return $query;
	}
}


// bbPress
function thim_use_bbpress() {
	if ( function_exists( 'is_bbpress' ) ) {
		return is_bbpress();
	} else {
		return false;
	}
}

/************ List Comment ***************/
if ( ! function_exists( 'thim_comment' ) ) {
	function thim_comment( $comment, $args, $depth ) {
		$GLOBALS['comment'] = $comment;
		//extract( $args, EXTR_SKIP );
		if ( 'div' == $args['style'] ) {
			$tag       = 'div';
			$add_below = 'comment';
		} else {
			$tag       = 'li';
			$add_below = 'div-comment';
		}
		?>
		<<?php echo ent2ncr( $tag . ' ' ) ?><?php comment_class( 'description_comment' ) ?> id="comment-<?php comment_ID() ?>">
		<div class="wrapper-comment">
			<?php
			if ( $args['avatar_size'] != 0 ) {
				echo '<div class="avatar">';
				echo get_avatar( $comment, $args['avatar_size'] );
				echo '</div>';
			}
			?>
			<div class="comment-right">
				<?php if ( $comment->comment_approved == '0' ) : ?>
					<em
						class="comment-awaiting-moderation"><?php esc_html_e( 'Your comment is awaiting moderation.', 'coaching' ) ?></em>
				<?php endif; ?>

				<div class="comment-extra-info">
					<div class="author"><span class="author-name"><?php echo get_comment_author_link(); ?></span></div>
					<div class="date" itemprop="commentTime">
						<?php printf( get_comment_date(), get_comment_time() ) ?></div>

					<div class="content-comment">
						<?php comment_text() ?>
					</div>
					<?php edit_comment_link( esc_html__( 'Edit', 'coaching' ), '', '' ); ?>
					<?php comment_reply_link(
						array_merge(
							$args, array(
								'add_below' => $add_below,
								'depth'     => $depth,
								'max_depth' => $args['max_depth']
							)
						)
					) ?>
				</div>


			</div>
		</div>
		<?php
	}
}

// dislay setting layout
require THIM_DIR . 'inc/wrapper-before-after.php';
require THIM_DIR . 'inc/templates/heading-top.php';

/**
 * @param $mtb_setting
 *
 * @return mixed
 */
function thim_mtb_setting_after_created( $mtb_setting ) {
	$mtb_setting->removeOption( array( 11 ) );
	$option_name_space = $mtb_setting->owner->optionNamespace;

	$settings   = array(
		'name'      => esc_html__( 'Color Sub Title', 'coaching' ),
		'id'        => 'mtb_color_sub_title',
		'type'      => 'color-opacity',
		'desc'      => ' ',
		'row_class' => 'child_of_' . $option_name_space . '_mtb_using_custom_heading thim_sub_option',
	);
	$settings_1 = array(
		'name' => esc_html__( 'No Padding Content', 'coaching' ),
		'id'   => 'mtb_no_padding',
		'type' => 'checkbox',
		'desc' => ' ',
	);

	$mtb_setting->insertOptionBefore( $settings, 11 );
	$mtb_setting->insertOptionBefore( $settings_1, 16 );

	return $mtb_setting;
}

add_filter( 'thim_mtb_setting_after_created', 'thim_mtb_setting_after_created', 10, 2 );


/**
 * @param $tabs
 *
 * @return array
 */
function thim_widget_group( $tabs ) {
	$tabs[] = array(
		'title'  => esc_html__( 'Thim Widget', 'coaching' ),
		'filter' => array(
			'groups' => array( 'thim_widget_group' )
		)
	);

	return $tabs;
}

add_filter( 'siteorigin_panels_widget_dialog_tabs', 'thim_widget_group', 19 );


/**
 * @return string
 */
function thim_excerpt_length() {
	$theme_options_data = get_theme_mods();
	if ( isset( $theme_options_data['thim_archive_excerpt_length'] ) ) {
		$length = $theme_options_data['thim_archive_excerpt_length'];
	} else {
		$length = '50';
	}

	return $length;
}

add_filter( 'excerpt_length', 'thim_excerpt_length', 999 );

/**
 * Social sharing
 */
if ( ! function_exists( 'thim_social_share' ) ) {
	function thim_social_share() {
		$sharings = get_theme_mod( 'group_sharing' );

		if ( isset( $sharings ) && $sharings ) {
			echo '<ul class="thim-social-share">';
			do_action( 'thim_before_social_list' );
			echo '<li class="heading">' . esc_html__( 'Share:', 'coaching' ) . '</li>';
			foreach ( $sharings as $sharing ) {
				switch ( $sharing ) {
					case 'facebook':
						echo '<li><div class="facebook-social"><a target="_blank" class="facebook"  href="https://www.facebook.com/sharer.php?u=' . urlencode( get_permalink() ) . '" title="' . esc_attr__( 'Facebook', 'coaching' ) . '"><i class="fa fa-facebook"></i></a></div></li>';
						break;
					case 'google':
						echo '<li><div class="googleplus-social"><a target="_blank" class="googleplus" href="https://plus.google.com/share?url=' . urlencode( get_permalink() ) . '&amp;title=' . rawurlencode( esc_attr( get_the_title() ) ) . '" title="' . esc_attr__( 'Google Plus', 'coaching' ) . '><i class="fa fa-google"></i></a></div></li>';
						break;
					case 'twitter':
						echo '<li><div class="twitter-social"><a target="_blank" class="twitter" href="https://twitter.com/share?url=' . urlencode( get_permalink() ) . '&amp;text=' . rawurlencode( esc_attr( get_the_title() ) ) . '" title="' . esc_attr__( 'Twitter', 'coaching' ) . '"><i class="fa fa-x-twitter"></i></a></div></li>';
						break;

					case 'pinterest':
						echo '<li><div class="pinterest-social"><a target="_blank" class="pinterest"  href="http://pinterest.com/pin/create/button/?url=' . urlencode( get_permalink() ) . '&amp;description=' . rawurlencode( esc_attr( get_the_excerpt() ) ) . '&amp;media=' . urlencode( wp_get_attachment_url( get_post_thumbnail_id() ) ) . '" onclick="window.open(this.href); return false;" title="' . esc_attr__( 'Pinterest', 'coaching' ) . '"><i class="fa fa-pinterest-p"></i></a></div></li>';
						break;
					case 'linkedin':
						echo '<li><div class="linkedin-social"><a target="_blank" class="linkedin" href="https://www.linkedin.com/shareArticle?mini=true&url=' . urlencode( get_permalink() ) . '&title=' . rawurlencode( esc_attr( get_the_title() ) ) . '&summary=&source=' . rawurlencode( esc_attr( get_the_excerpt() ) ) . '"><i class="fa fa-linkedin-square"></i></a></div></li>';
						break;
				}
			}
			do_action( 'thim_after_social_list' );

			echo '</ul>';
		}
	}
}
add_action( 'thim_social_share', 'thim_social_share' );


/**
 * Display favicon
 */
function thim_favicon() {
	if ( function_exists( 'wp_site_icon' ) ) {
		if ( function_exists( 'has_site_icon' ) ) {
			if ( ! has_site_icon() ) {
				// Icon default
				$thim_favicon_src = get_template_directory_uri() . "/images/favicon.png";
				echo '<link rel="shortcut icon" href="' . esc_url( $thim_favicon_src ) . '" type="image/x-icon" />';

				return;
			}

			return;
		}
	}

	/**
	 * Support WordPress < 4.3
	 */
	$theme_options_data = get_theme_mods();
	$thim_favicon_src   = '';
	if ( isset( $theme_options_data['thim_favicon'] ) ) {
		$thim_favicon       = $theme_options_data['thim_favicon'];
		$favicon_attachment = wp_get_attachment_image_src( $thim_favicon, 'full' );
		$thim_favicon_src   = $favicon_attachment[0];
	}
	if ( ! $thim_favicon_src ) {
		$thim_favicon_src = get_template_directory_uri() . "/images/favicon.png";
	}
	echo '<link rel="shortcut icon" href="' . esc_url( $thim_favicon_src ) . '" type="image/x-icon" />';
}

add_action( 'wp_head', 'thim_favicon' );

/**
 * Redirect to custom login page
 */
if ( ! function_exists( 'thim_login_failed' ) ) {
	function thim_login_failed() {
		if ( ! isset( $_REQUEST['is_theme_thimpress'] ) ) {
			return;
		}

		if ( ( isset( $_REQUEST['action'] ) && $_REQUEST['action'] == 'thim_login_ajax' ) || ( isset( $_REQUEST['lp-ajax'] ) && $_REQUEST['lp-ajax'] == 'login' ) || ( is_admin() && defined( 'DOING_AJAX' ) && DOING_AJAX ) ) {
			return;
		}

		if ( is_user_logged_in() ) {
			return;
		}

		$url = add_query_arg( 'result', 'failed', thim_get_login_page_url() );

		if ( isset( $_POST['g-recaptcha-response'] ) ) {
			if ( ! $_POST['g-recaptcha-response'] ) {
				$url = add_query_arg( 'gglcptch_error', '1', $url );
			}
		}

		wp_redirect( $url );
		exit;
	}

	add_action( 'wp_login_failed', 'thim_login_failed', 1000 );
}

/**
 * Filter register link
 *
 * @param $register_url
 *
 * @return string|void
 */
if ( ! function_exists( 'thim_get_register_url' ) ) {
	function thim_get_register_url() {
		$url = add_query_arg( 'action', 'register', thim_get_login_page_url() );

		return $url;
	}
}


/**
 * Register failed
 *
 * @param $sanitized_user_login
 * @param $user_email
 * @param $errors
 */
if ( ! function_exists( 'thim_register_failed' ) ) {
	function thim_register_failed( $errors ) {
		if ( ! $errors || is_wp_error( $errors ) ) {
			if ( $errors && $errors->get_error_code() ) {

				//setup your custom URL for redirection
				$url = add_query_arg( 'action', 'register', thim_get_login_page_url() );

				foreach ( $errors->errors as $e => $m ) {
					$url = add_query_arg( $e, '1', $url );
				}
				wp_redirect( $url );
				exit;
			}
		}

		return $errors;
	}

	add_action( 'registration_errors', 'thim_register_failed', 99, 1 );
}

function thim_check_extra_register_fields( $login, $email, $errors ) {
	$theme_options_data = get_theme_mods();
	if ( empty( $theme_options_data['thim_auto_login'] ) || ( isset( $theme_options_data['thim_auto_login'] ) && $theme_options_data['thim_auto_login'] == '0' ) ) {
		if ( isset( $_POST['password'] ) && $_POST['repeat_password'] && $_POST['password'] !== $_POST['repeat_password'] ) {
			$errors->add( 'passwords_not_matched', "<strong>" . esc_html__( 'ERROR:', 'coaching' ) . "</strong>" . esc_html__( ' Passwords must match', 'coaching' ) );
		}
	}
}

add_action( 'register_post', 'thim_check_extra_register_fields', 10, 3 );

/**
 * Update password
 *
 * @param $user_id
 */
function thim_register_extra_fields( $user_id ) {
	$theme_options_data = get_theme_mods();
	if ( empty( $theme_options_data['thim_auto_login'] ) || ( isset( $theme_options_data['thim_auto_login'] ) && $theme_options_data['thim_auto_login'] == '0' ) ) {
		if ( ! isset( $_POST['password'] ) || ! isset( $_POST['repeat_password'] ) ) {
			return false;
		}

		$pw         = sanitize_text_field( $_POST['password'] );
		$confirm_pw = sanitize_text_field( $_POST['repeat_password'] );

		if ( $pw !== $confirm_pw ) {
			return false;
		}

		$user_data              = array();
		$user_data['ID']        = $user_id;
		$user_data['user_pass'] = $pw;

		add_filter( 'send_password_change_email', '__return_false' );

		$new_user_id = wp_update_user( $user_data );

		// Login after registered
		if ( ! is_admin() ) {
			wp_set_current_user( $user_id );
			wp_set_auth_cookie( $user_id );
			wp_new_user_notification( $user_id, null, 'both' );
			if ( ( isset( $_POST['billing_email'] ) && ! empty( $_POST['billing_email'] ) ) || ( isset( $_POST['bconfirmemail'] ) && ! empty( $_POST['bconfirmemail'] ) ) ) {
				return;
			} elseif ( isset( $_POST['gateway'] ) && $_POST['gateway'] == 'paypalexpress' && isset( $_POST['token'] ) && isset( $_POST['confirm'] ) && isset( $_POST['confirm'] ) && isset( $_POST['level'] ) && isset( $_POST['checkjavascript'] ) ) {
				return;
			} else {
				if ( ! empty( $_REQUEST['redirect_to'] ) ) {
					wp_redirect( $_REQUEST['redirect_to'] );
				} else {
					$theme_options_data = get_theme_mods();
					if ( ! empty( $_REQUEST['option'] ) && $_REQUEST['option'] == 'moopenid' ) {
						if ( isset( $_SERVER['HTTPS'] ) && ! empty( $_SERVER['HTTPS'] ) && $_SERVER['HTTPS'] != 'off' ) {
							$http = "https://";
						} else {
							$http = "http://";
						}
						$redirect_url = urldecode( html_entity_decode( esc_url( $http . $_SERVER["HTTP_HOST"] . str_replace( '?option=moopenid', '', $_SERVER['REQUEST_URI'] ) ) ) );
						if ( html_entity_decode( esc_url( remove_query_arg( 'ss_message', $redirect_url ) ) ) == wp_login_url() || strpos( $_SERVER['REQUEST_URI'], 'wp-login.php' ) !== false || strpos( $_SERVER['REQUEST_URI'], 'wp-admin' ) !== false ) {
							$redirect_url = site_url() . '/';
						}

						wp_redirect( $redirect_url );

						return;
					}
					if ( ! empty( $theme_options_data['thim_register_redirect'] ) ) {
						wp_redirect( $theme_options_data['thim_register_redirect'] );
					} else {
						wp_redirect( home_url() );
					}
				}
				exit;
			}
		}
	}
}

add_action( 'user_register', 'thim_register_extra_fields', 1000 );

/**
 * Redirect to custom register page in case multi sites
 *
 * @param $url
 *
 * @return mixed
 */
if ( ! function_exists( 'thim_multisite_register_redirect' ) ) {
	function thim_multisite_register_redirect( $url ) {

		if ( ! is_user_logged_in() ) {
			if ( is_multisite() ) {
				$url = add_query_arg( 'action', 'register', thim_get_login_page_url() );
			}

			$user_login = isset( $_POST['user_login'] ) ? $_POST['user_login'] : '';
			$user_email = isset( $_POST['user_email'] ) ? $_POST['user_email'] : '';

			$errors = register_new_user( $user_login, $user_email );

			if ( ! is_wp_error( $errors ) ) {
				$redirect_to = ! empty( $_POST['redirect_to'] ) ? $_POST['redirect_to'] : 'wp-login.php?checkemail=registered';
				wp_safe_redirect( $redirect_to );
				exit();
			}
		}

		return $url;
	}
}

add_filter( 'wp_signup_location', 'thim_multisite_register_redirect' );

if ( ! function_exists( 'thim_multisite_signup_redirect' ) ) {
	function thim_multisite_signup_redirect() {
		if ( is_multisite() ) {
			wp_redirect( wp_registration_url() );
			die();
		}
	}
}

add_action( 'signup_header', 'thim_multisite_signup_redirect' );


/**
 * Filter lost password link
 *
 * @param $url
 *
 * @return string
 */
if ( ! function_exists( 'thim_get_lost_password_url' ) ) {
	function thim_get_lost_password_url() {
		$url = add_query_arg( 'action', 'lostpassword', thim_get_login_page_url() );

		return $url;
	}
}


/**
 * Add lost password link into login form
 *
 * @param $content
 * @param $args
 *
 * @return string
 */
if ( ! function_exists( 'thim_add_lost_password_link' ) ) {
	function thim_add_lost_password_link() {
		$content = '<a class="lost-pass-link" href="' . thim_get_lost_password_url() . '" title="' . esc_attr__( 'Lost Password', 'coaching' ) . '">' . esc_html__( 'Lost your password?', 'coaching' ) . '</a>';

		return $content;
	}
}

//add_filter( 'login_form_middle', 'thim_add_lost_password_link', 999 );

/**
 * Register failed
 */
if ( ! function_exists( 'thim_reset_password_failed' ) ) {
	function thim_reset_password_failed() {

		//setup your custom URL for redirection
		$url = add_query_arg( 'action', 'lostpassword', thim_get_login_page_url() );

		if ( empty( $_POST['user_login'] ) ) {
			$url = add_query_arg( 'empty', '1', $url );
			wp_redirect( $url );
			exit;
		} elseif ( strpos( $_POST['user_login'], '@' ) ) {
			$user_data = get_user_by( 'email', trim( $_POST['user_login'] ) );
			if ( empty( $user_data ) ) {
				$url = add_query_arg( 'user_not_exist', '1', $url );
				wp_redirect( $url );
				exit;
			}
		} elseif ( ! username_exists( $_POST['user_login'] ) ) {
			$url = add_query_arg( 'user_not_exist', '1', $url );
			wp_redirect( $url );
			exit;
		}


	}
}
add_action( 'lostpassword_post', 'thim_reset_password_failed', 999 );

/**
 * Get login page url
 *
 * @return false|string
 */
if ( ! function_exists( 'thim_get_login_page_url' ) ) {
	function thim_get_login_page_url() {

		//		if ( ! thim_plugin_active( 'siteorigin-panels/siteorigin-panels.php' ) && ! thim_plugin_active( 'js_composer/js_composer.php' ) ) {
		//			return wp_login_url();
		//		}

		if ( $page = get_option( 'thim_login_page' ) ) {
			return get_permalink( $page );
		} else {
			global $wpdb;
			$page = $wpdb->get_col(
				$wpdb->prepare(
					"SELECT p.ID FROM $wpdb->posts AS p INNER JOIN $wpdb->postmeta AS pm ON p.ID = pm.post_id
			WHERE 	pm.meta_key = %s
			AND 	pm.meta_value = %s
			AND		p.post_type = %s
			AND		p.post_status = %s",
					'thim_login_page',
					'1',
					'page',
					'publish'
				)
			);
			if ( ! empty( $page[0] ) ) {
				return get_permalink( $page[0] );
			}
		}

		return wp_login_url();

	}
}

/**
 * Display feature image
 *
 * @param $attachment_id
 * @param $size_type
 * @param $width
 * @param $height
 * @param $alt
 * @param $title
 *
 * @return string
 */

if ( ! function_exists( 'thim_get_feature_image' ) ) {
	function thim_get_feature_image( $attachment_id, $size_type = null, $width = null, $height = null, $alt = null, $title = null ) {


		if ( ! $size_type ) {
			$size_type = 'full';
		}
		$style = '';
		if ( $width && $height ) {
			$src   = wp_get_attachment_image_src( $attachment_id, array( $width, $height ) );
			$style = ' width="' . $width . '" height="' . $height . '"';
		} else {
			$src = wp_get_attachment_image_src( $attachment_id, $size_type );
			if ( ! empty( $src[1] ) && ! empty( $src[2] ) ) {
				$style = ' width="' . $src[1] . '" height="' . $src[2] . '"';
			}
		}

		if ( ! $src ) {
			$query_args    = array(
				'post_type'   => 'attachment',
				'post_status' => 'inherit',
				'meta_query'  => array(
					array(
						'key'     => '_wp_attached_file',
						'compare' => 'LIKE',
						'value'   => 'demo_image.jpg'
					)
				)
			);
			$attachment_id = get_posts( $query_args );
			if ( ! empty( $attachment_id ) && $attachment_id[0] ) {
				$attachment_id = $attachment_id[0]->ID;
				$src           = wp_get_attachment_image_src( $attachment_id, 'full' );
			}
		}

		if ( ! $alt ) {
			$alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true ) ? get_post_meta( $attachment_id, '_wp_attachment_image_alt', true ) : get_the_title( $attachment_id );
		}
		//		if ( $no_lazyload = 1 ) {
		//			$style .= ' data-skip-lazy';
		//		}
		if ( ! $title ) {
			$title = get_the_title( $attachment_id );
		}
		if ( empty( $src ) ) {
			$src    = [];
			$src[0] = THIM_URI . 'images/demo_images/demo_image.jpg';
		}

		return '<img src="' . esc_url( $src[0] ) . '" alt="' . esc_attr( $alt ) . '" title="' . esc_attr( $title ) . '" ' . $style . '>';

	}
}

function thim_get_src_demo_image() {

	// Get demo image
	global $wpdb;
	$attachment_id = $wpdb->get_col(
		$wpdb->prepare(
			"SELECT p.ID FROM $wpdb->posts AS p INNER JOIN $wpdb->postmeta AS pm ON p.ID = pm.post_id
				WHERE 	pm.meta_key = %s
				AND 	pm.meta_value LIKE %s",
			'_wp_attached_file',
			'%demo_image.jpg'
		)
	);

	if ( empty( $attachment_id[0] ) ) {
		return;
	}

	$attachment_id = $attachment_id[0];
	$src           = wp_get_attachment_image_src( $attachment_id, 'full' );

	return $src;

}


/**
 * Adds a box to the main column on the Post and Page edit screens.
 */
function thim_event_add_meta_boxes() {

	if ( ! post_type_exists( 'tp_event' ) || ! post_type_exists( 'our_team' ) ) {
		return;
	}
	add_meta_box(
		'thim_organizers',
		esc_html__( 'Organizers', 'coaching' ),
		'thim_event_meta_boxes_callback',
		'tp_event'
	);
}

add_action( 'add_meta_boxes', 'thim_event_add_meta_boxes' );

/**
 * Prints the box content.
 *
 * @param WP_Post $post The object for the current post/page.
 */
function thim_event_meta_boxes_callback( $post ) {

	// Add a nonce field so we can check for it later.
	wp_nonce_field( 'thim_event_save_meta_boxes', 'thim_event_meta_boxes_nonce' );

	// Get all team
	$team = new WP_Query(
		array(
			'post_type'           => 'our_team',
			'post_status'         => 'publish',
			'ignore_sticky_posts' => true,
			'posts_per_page'      => - 1
		)
	);
	if ( empty( $team->post_count ) ) {
		echo '<p>' . esc_html__( 'No members exists. You can create a member data from', 'coaching' ) . ' <a target="_blank" href="' . esc_url( admin_url( 'post-new.php?post_type=our_team' ) ) . '">' . esc_html__( 'Our Team', 'coaching' ) . '</a></p>';

		return;
	}

	echo '<label for="thim_event_members">';
	esc_html_e( 'Get Members', 'coaching' );
	echo '</label> ';
	echo '<select id="thim_event_members" name="thim_event_members[]" multiple>';
	echo '<option value="">Remove all member</option>';
	if ( isset( $team->posts ) ) {
		$team = $team->posts;
		foreach ( $team as $member ) {
			echo '<option value="' . esc_attr( $member->ID ) . '">' . get_the_title( $member->ID ) . '</option>';
		}
	}
	echo '</select>';
	echo '<span>';
	esc_html_e( 'Hold down the Ctrl (Windows) / Command (Mac) button to select multiple options.', 'coaching' );
	echo '</span><br>';
	wp_reset_postdata();

	/*
	 * Use get_post_meta() to retrieve an existing value
	 * from the database and use the value for the form.
	 */
	$members = get_post_meta( $post->ID, 'thim_event_members', true );
	echo '<p>' . esc_html__( 'Current Members: ', 'coaching' );

	if ( empty( array_filter( $members ) ) ) {
		echo esc_html__( 'None', 'coaching' ) . '</p>';
	} else {

		foreach ( $members as $key => $id ) {
			echo '<strong><a target="_blank" href="' . get_edit_post_link( $id ) . '">' . get_the_title( $id ) . '</a></strong>';
			if ( $key != count( $members ) ) {
				echo ', ';
			}
		}
	}
}

/**
 * When the post is saved, saves our custom data.
 *
 * @param int $post_id The ID of the post being saved.
 */
function thim_event_save_meta_boxes( $post_id ) {

	/*
	 * We need to verify this came from our screen and with proper authorization,
	 * because the save_post action can be triggered at other times.
	 */

	// Check if our nonce is set.
	if ( ! isset( $_POST['thim_event_meta_boxes_nonce'] ) ) {
		return;
	}

	// Verify that the nonce is valid.
	if ( ! wp_verify_nonce( $_POST['thim_event_meta_boxes_nonce'], 'thim_event_save_meta_boxes' ) ) {
		return;
	}

	// If this is an autosave, our form has not been submitted, so we don't want to do anything.
	if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
		return;
	}

	// Check the user's permissions.
	if ( isset( $_POST['post_type'] ) && 'tp_event' == $_POST['post_type'] ) {

		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

	}

	/* OK, it's safe for us to save the data now. */

	// Make sure that it is set.
	if ( ! isset( $_POST['thim_event_members'] ) ) {
		return;
	}

	// Update the meta field in the database.
	update_post_meta( $post_id, 'thim_event_members', $_POST['thim_event_members'] );
}

add_action( 'save_post', 'thim_event_save_meta_boxes' );

if ( ! function_exists( 'thim_child_add_extra_event' ) ) {
	function thim_child_add_extra_event( $post ) {
		$on_map = get_post_meta( $post->ID, 'tp_event_view_map', true ) ? get_post_meta( $post->ID, 'tp_event_view_map', true ) : '';
		?>
		<tr>
			<th>
				<label><?php esc_html_e( 'Map Link', 'coaching' ) ?></label>
			</th>
			<td>
				<p id="tp_event_datetime_end">
					<input type="text" name="tp_event_view_map" value="<?php echo esc_attr( $on_map ); ?>">
				</p>
			</td>
		</tr>
		<?php
	}
}

add_action( 'event_after_meta_box', 'thim_child_add_extra_event' );


/**
 * Change default comment fields
 *
 * @param $field
 *
 * @return string
 */
function thim_new_comment_fields( $fields ) {
	$commenter = wp_get_current_commenter();
	$req       = get_option( 'require_name_email', true );
	$aria_req  = ( $req ? 'aria-required=true' : '' );

	$fields = array(
		'author' => '<p class="comment-form-author">' . '<input placeholder="' . esc_attr__( 'Name', 'coaching' ) . ( $req ? ' *' : '' ) . '" id="author" name="author" type="text" value="' . esc_attr( $commenter['comment_author'] ) . '" size="30" ' . $aria_req . ' /></p>',
		'email'  => '<p class="comment-form-email">' . '<input placeholder="' . esc_attr__( 'Email', 'coaching' ) . ( $req ? ' *' : '' ) . '" id="email" name="email" type="text" value="' . esc_attr( $commenter['comment_author_email'] ) . '" size="30" ' . $aria_req . ' /></p>',
		'url'    => '<p class="comment-form-url">' . '<input placeholder="' . esc_attr__( 'Website', 'coaching' ) . '" id="url" name="url" type="text" value="' . esc_attr( $commenter['comment_author_url'] ) . '" size="30" /></p>',
	);

	return $fields;
}

add_filter( 'comment_form_default_fields', 'thim_new_comment_fields', 1 );


/**
 * Remove Emoji scripts
 */
remove_action( 'wp_head', 'print_emoji_detection_script', 7 );
remove_action( 'wp_print_styles', 'print_emoji_styles' );


/**
 * Optimize script files
 */
if ( ! function_exists( 'thim_optimize_scripts' ) ) {
	function thim_optimize_scripts() {
		global $wp_scripts;
		if ( ! is_a( $wp_scripts, 'WP_Scripts' ) ) {
			return;
		}
		foreach ( $wp_scripts->registered as $handle => $script ) {
			$wp_scripts->registered[ $handle ]->ver = null;
		}
	}
}


/**
 * Optimize style files
 */
if ( ! function_exists( 'thim_optimize_styles' ) ) {
	function thim_optimize_styles() {
		global $wp_styles;
		if ( ! is_a( $wp_styles, 'WP_Styles' ) ) {
			return;
		}
		foreach ( $wp_styles->registered as $handle => $style ) {
			if ( $handle !== 'thim-rtl' ) {
				$wp_styles->registered[ $handle ]->ver = null;
			}
		}
	}
}

/**
 * Remove query string in css files & js files
 */
$theme_remove_string = get_theme_mod( 'thim_remove_query_string', false );
if ( $theme_remove_string ) {
	add_action( 'wp_print_scripts', 'thim_optimize_scripts', 999 );
	add_action( 'wp_print_footer_scripts', 'thim_optimize_scripts', 999 );
	add_action( 'admin_print_styles', 'thim_optimize_styles', 999 );
	add_action( 'wp_print_styles', 'thim_optimize_styles', 999 );
}

/**
 * Get our wp_nav_menu() fallback, wp_page_menu(), to show a home link.
 *
 * @param array $args Configuration arguments.
 *
 * @return array
 */
function thim_page_menu_args( $args ) {
	$args['show_home'] = true;

	return $args;
}

add_filter( 'wp_page_menu_args', 'thim_page_menu_args' );

/**
 * Sets the authordata global when viewing an author archive.
 *
 * @return void
 * @global WP_Query $wp_query WordPress Query object.
 */
function thim_setup_author() {
	global $wp_query;

	if ( $wp_query->is_author() && isset( $wp_query->post ) ) {
		$GLOBALS['authordata'] = get_userdata( $wp_query->post->post_author );
	}
}

add_action( 'wp', 'thim_setup_author' );

function thim_add_event_admin_styles() {
	?>
	<style type="text/css">
		#thim_event_members {
			min-height: 200px;
		}
	</style>
	<?php
}

add_action( 'admin_print_styles', 'thim_add_event_admin_styles' );

/**
 * Check a plugin activate
 *
 * @param $plugin
 *
 * @return bool
 */
function thim_plugin_active( $plugin ) {
	include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
	if ( is_plugin_active( $plugin ) ) {
		return true;
	}

	return false;
}


/**
 * Custom WooCommerce breadcrumbs
 *
 * @return array
 */
function thim_woocommerce_breadcrumbs() {
	return array(
		'delimiter'   => '',
		'wrap_before' => '<ul class="breadcrumbs" id="breadcrumbs" itemtype="http://schema.org/BreadcrumbList" itemscope="" itemprop="breadcrumb">',
		'wrap_after'  => '</ul>',
		'before'      => '<li itemtype="http://schema.org/ListItem" itemscope="" itemprop="itemListElement">',
		'after'       => '</li>',
		'home'        => esc_html__( 'Home', 'coaching' ),
	);
}

add_filter( 'woocommerce_breadcrumb_defaults', 'thim_woocommerce_breadcrumbs' );

/**
 * Display post thumbnail by default
 *
 * @param $size
 */
function thim_default_get_post_thumbnail( $size ) {

	if ( thim_plugin_active( 'thim-framework/tp-framework.php' ) ) {
		return;
	}

	if ( get_the_post_thumbnail( get_the_ID(), $size ) ) {
		?>
		<div class='post-formats-wrapper'>
			<a class="post-image" href="<?php echo esc_url( get_permalink() ); ?>">
				<?php echo get_the_post_thumbnail( get_the_ID(), $size ); ?>
			</a>
		</div>
		<?php
	}
}

//add_action( 'thim_entry_top', 'thim_default_get_post_thumbnail', 20 );


/**
 * Set unlimit events in archive
 *
 * @param $query
 */
function thim_event_post_filter( $query ) {
	global $wp_query;

	if ( is_post_type_archive( 'tp_event' ) && 'tp_event' == $query->get( 'post_type' ) ) {
		$query->set( 'posts_per_page', - 1 );

		return;
	}
}

add_action( 'pre_get_posts', 'thim_event_post_filter' );


function thim_start_widget_element_content( $content, $panels_data, $post_id ) {
	global $siteorigin_panels_inline_css;

	if ( ! empty( $siteorigin_panels_inline_css[ $post_id ] ) ) {
		$content = '<style scoped>' . ( $siteorigin_panels_inline_css[ $post_id ] ) . '</style>' . $content;
	}

	return $content;
}

remove_action( 'wp_footer', 'siteorigin_panels_print_inline_css' );
add_filter( 'siteorigin_panels_before_content', 'thim_start_widget_element_content', 10, 3 );

//Override ajax-loader contact form
function thim_wpcf7_ajax_loader() {
	return THIM_URI . 'images/ajax-loader.gif';
}

add_filter( 'wpcf7_ajax_loader', 'thim_wpcf7_ajax_loader' );


/**
 * Testing with CF7 scripts
 */
if ( ! function_exists( 'thim_disable_cf7_cache' ) ) {
	function thim_disable_cf7_cache() {
		global $wp_scripts;
		if ( ! empty( $wp_scripts->registered['contact-form-7'] ) ) {
			if ( ! empty( $wp_scripts->registered['contact-form-7']->extra['data'] ) ) {
				$localize                                                = $wp_scripts->registered['contact-form-7']->extra['data'];
				$localize                                                = str_replace( '"cached":"1"', '"cached":0', $localize );
				$wp_scripts->registered['contact-form-7']->extra['data'] = $localize;
			}
		}
	}
}

add_action( 'wpcf7_enqueue_scripts', 'thim_disable_cf7_cache' );


//Function thim_related_our_team
if ( ! function_exists( 'thim_related_our_team' ) ) {
	function thim_related_our_team( $post_id, $number_posts = - 1 ) {
		$query = new WP_Query();
		$args  = '';
		if ( $number_posts == 0 ) {
			return $query;
		}
		$args  = wp_parse_args(
			$args, array(
				'posts_per_page'      => $number_posts,
				'post_type'           => 'our_team',
				'post__not_in'        => array( $post_id ),
				'ignore_sticky_posts' => true,
				'tax_query'           => array(
					array(
						'taxonomy' => 'our_team_category',
						// taxonomy name
						'field'    => 'term_id',
						// term_id, slug or name
						'operator' => 'IN',
						'terms'    => wp_get_post_terms( $post_id, 'our_team_category', array( "fields" => "ids" ) ),
						// term id, term slug or term name
					)
				),
			)
		);
		$query = new WP_Query( $args );

		return $query;
	}
}

/**
 * Process events order
 */

add_filter( 'posts_fields', 'thim_event_posts_fields' );
add_filter( 'posts_join_paged', 'thim_event_posts_join_paged' );
add_filter( 'posts_where_paged', 'thim_event_posts_where_paged' );
add_filter( 'posts_orderby', 'thim_event_posts_orderby' );

function thim_is_events_archive() {
	global $pagenow, $post_type;
	if ( ! is_post_type_archive( 'tp_event' ) || ! is_main_query() ) {
		return false;
	}

	return true;
}

function thim_event_posts_fields( $fields ) {
	if ( ! thim_is_events_archive() ) {
		return $fields;
	}

	$fields = " DISTINCT " . $fields;
	$fields .= ', concat( str_to_date( pm1.meta_value, \'%m/%d/%Y\' ), \' \', str_to_date(pm2.meta_value, \'%h:%i %p\' ) ) as start_date_time ';

	return $fields;
}

function thim_event_posts_join_paged( $join ) {
	if ( ! thim_is_events_archive() ) {
		return $join;
	}

	global $wpdb;
	$join .= " LEFT JOIN {$wpdb->postmeta} pm1 ON pm1.post_id = {$wpdb->posts}.ID AND pm1.meta_key = 'tp_event_date_start'";
	$join .= " LEFT JOIN {$wpdb->postmeta} pm2 ON pm2.post_id = {$wpdb->posts}.ID AND pm2.meta_key = 'tp_event_time_start'";


	return $join;
}

function thim_event_posts_where_paged( $where ) {
	if ( ! thim_is_events_archive() ) {
		return $where;
	}

	return $where;
}

function thim_event_posts_orderby( $order_by_statement ) {

	if ( ! thim_is_events_archive() ) {
		return $order_by_statement;
	}
	$order_by_statement = "start_date_time ASC"; // ASC

	return $order_by_statement;
}

if ( ! function_exists( 'thim_replace_retrieve_password_message' ) ) {
	function thim_replace_retrieve_password_message( $message, $key, $user_login, $user_data ) {

		$reset_link = add_query_arg(
			array(
				'action' => 'rp',
				'key'    => $key,
				'login'  => rawurlencode( $user_login )
			), thim_get_login_page_url()
		);

		// Create new message
		$message = esc_html__( 'Someone has requested a password reset for the following account:', 'coaching' ) . "\r\n\r\n";
		$message .= network_home_url( '/' ) . "\r\n\r\n";
		$message .= sprintf( esc_html__( 'Username: %s', 'coaching' ), $user_login ) . "\r\n\r\n";
		$message .= esc_html__( 'If this was a mistake, just ignore this email and nothing will happen.', 'coaching' ) . "\r\n\r\n";
		$message .= esc_html__( 'To reset your password, visit the following address:', 'coaching' ) . "\r\n\r\n";
		$message .= $reset_link . "\r\n";

		return $message;
	}
}
/**
 * Add filter if without using wpengine
 */
if ( ! function_exists( 'is_wpe' ) && ! function_exists( 'is_wpe_snapshot' ) ) {
	add_filter( 'retrieve_password_message', 'thim_replace_retrieve_password_message', 10, 4 );
}


function thim_do_password_reset() {

	$login_page = thim_get_login_page_url();
	if ( 'POST' == $_SERVER['REQUEST_METHOD'] ) {

		if ( ! isset( $_REQUEST['key'] ) || ! isset( $_REQUEST['login'] ) ) {
			return;
		}

		$key   = $_REQUEST['key'];
		$login = $_REQUEST['login'];

		$user = check_password_reset_key( $key, $login );

		if ( ! $user || is_wp_error( $user ) ) {
			if ( $user && $user->get_error_code() === 'expired_key' ) {
				wp_redirect( add_query_arg(
					array(
						'action'      => 'rp',
						'expired_key' => '1',
					), $login_page
				) );
			} else {
				wp_redirect( add_query_arg(
					array(
						'action'      => 'rp',
						'invalid_key' => '1',
					), $login_page
				) );
			}
			exit;
		}

		if ( isset( $_POST['password'] ) ) {

			if ( empty( $_POST['password'] ) ) {
				// Password is empty
				wp_redirect( add_query_arg(
					array(
						'action'           => 'rp',
						'key'              => $_REQUEST['key'],
						'login'            => $_REQUEST['login'],
						'invalid_password' => '1',
					), $login_page
				) );
				exit;
			}

			// Parameter checks OK, reset password
			reset_password( $user, $_POST['password'] );
			wp_redirect( add_query_arg(
				array(
					'result' => 'changed',
				), $login_page
			) );
		} else {
			esc_html_e( 'Invalid request.', 'coaching' );
		}

		exit;
	}
}

add_action( 'login_form_rp', 'thim_do_password_reset', 1000 );
add_action( 'login_form_resetpass', 'thim_do_password_reset', 1000 );

if ( ! function_exists( 'thim_related_portfolio' ) ) {
	function thim_related_portfolio( $post_id ) {
		$hover_effect = ! empty( get_theme_mod( 'thim_portfolio_cate_style_chosen', false ) ) ? get_theme_mod( 'thim_portfolio_cate_style_chosen', false ) : 'style01';
		?>
		<div class="related-portfolio col-md-12 <?php echo esc_attr( $hover_effect ); ?>">
			<div class="module_title">
				<h4 class="widget-title"><?php esc_html_e( 'Related Items', 'coaching' ); ?></h4>
			</div>

			<?php //Get Related posts by category	-->
			$args      = array(
				'posts_per_page' => 3,
				'post_type'      => 'portfolio',
				'post_status'    => 'publish',
				'post__not_in'   => array( $post_id )
			);
			$port_post = get_posts( $args );
			?>

			<ul class="row">
				<?php
				foreach ( $port_post as $post ) : setup_postdata( $post ); ?>
					<?php
					$bk_ef = get_post_meta( $post->ID, 'thim_portfolio_bg_color_ef', true );
					if ( $bk_ef == '' ) {
						$bk_ef = get_post_meta( $post->ID, 'thim_portfolio_bg_color_ef', true );
						$bg    = '';
					} else {
						$bk_ef = get_post_meta( $post->ID, 'thim_portfolio_bg_color_ef', true );
						$bg    = 'style="background-color:' . $bk_ef . ';"';
					}
					$logo = get_post_meta( $post->ID, 'thim_portfolio_project_logo', false );
					if ( isset( $logo ) && ! empty( $logo ) ) {
						$logo_attachment = wp_get_attachment_image_src( $logo[0], 'full' );
						if ( $logo_attachment ) {
							$thim_logo_src = $logo_attachment[0];
							$logo_size     = ' width="' . $logo_attachment[1] . '" height="' . $logo_attachment[2] . '"';
						}
						$html_title = '<img class="project-logo" src="' . $thim_logo_src . '" alt="' . esc_attr( get_the_title( $post->ID ) ) . '" ' . $logo_size . ' />';
					} else {
						$html_title = get_the_title( $post->ID );
					}
					?>
					<li class="col-sm-4">
						<?php

						$imImage = get_permalink( $post->ID );

						$image_url = thim_get_feature_image( get_post_thumbnail_id( $post->ID ), 'full', '480', '320' );
						echo '<div data-color="' . $bk_ef . '" ' . $bg . '>';
						echo '<div class="portfolio-image" ' . $bg . '>' . $image_url . '
					<div class="portfolio_hover"><div class="thumb-bg"><div class="mask-content">';
						echo '<h3><a href="' . esc_url( get_permalink( $post->ID ) ) . '" title="' . esc_attr( get_the_title( $post->ID ) ) . '" >' . $html_title . '</a></h3>';
						echo '<span class="p_line"></span>';
						$terms    = get_the_terms( $post->ID, 'portfolio_category' );
						$cat_name = "";
						if ( $terms && ! is_wp_error( $terms ) ) :
							foreach ( $terms as $term ) {
								if ( $cat_name ) {
									$cat_name .= ', ';
								}
								$cat_name .= '<a href="' . esc_url( get_term_link( $term ) ) . '">' . $term->name . "</a>";
							}
							echo '<div class="cat_portfolio">' . $cat_name . '</div>';
						endif;
						if ( $hover_effect === 'style09' || $hover_effect === 'style10' || $hover_effect === 'style11' ) {
							echo thim_excerpt( 15 );
						}
						echo '<a href="' . esc_url( $imImage ) . '" title="' . esc_attr( get_the_title( $post->ID ) ) . '" class="btn_zoom ">' . esc_html__( 'View More', 'coaching' ) . '</a>';
						echo '</div></div></div></div></div>';
						?>
					</li>
				<?php endforeach; ?>
			</ul>
			<?php wp_reset_postdata(); ?>
		</div>
		<!--#portfolio_related-->
		<?php
	}
}


//Function ajax widget gallery-posts
add_action( 'wp_ajax_thim_gallery_popup', 'thim_gallery_popup' );
add_action( 'wp_ajax_nopriv_thim_gallery_popup', 'thim_gallery_popup' );
/** widget gallery posts ajax output **/
if ( ! function_exists( 'thim_gallery_popup' ) ) {
	function thim_gallery_popup() {
		global $post;
		$post_id = $_POST["post_id"];
		$post    = get_post( $post_id );

		$format = get_post_format( $post_id->ID );

		$error = true;
		$link  = get_edit_post_link( $post_id );
		ob_start();

		if ( $format == 'video' ) {
			$url_video = get_post_meta( $post_id, 'thim_video', true );
			if ( empty( $url_video ) ) {
				echo '<div class="thim-gallery-message"><a class="link" href="' . $link . '">' . esc_html__( 'This post doesn\'t have config video, please add the video!', 'coaching' ) . '</a></div>';
			}
			// If URL: show oEmbed HTML
			if ( filter_var( $url_video, FILTER_VALIDATE_URL ) ) {
				if ( $oembed = @wp_oembed_get( $url_video ) ) {
					echo '<div class="video">' . $oembed . '</div>';
				}
			} else {
				echo '<div class="video">' . $url_video . '</div>';
			}

		} else {
			$images = thim_meta( 'thim_gallery', "type=image&single=false&size=full" );
			// Get category permalink


			if ( ! empty( $images ) ) {
				foreach ( $images as $k => $value ) {
					$url_image = $value['url'];
					if ( $url_image && $url_image != '' ) {
						echo '<a href="' . $url_image . '">';
						echo '<img src="' . $url_image . '" />';
						echo '</a>';
						$error = false;
					}
				}
			}
			if ( $error ) {
				if ( is_user_logged_in() ) {
					echo '<div class="thim-gallery-message"><a class="link" href="' . $link . '">' . esc_html__( 'This post doesn\'t have any gallery images, please add some!', 'coaching' ) . '</a></div>';
				} else {
					echo '<div class="thim-gallery-message">' . esc_html__( 'This post doesn\'t have any gallery images, please add some!', 'coaching' ) . '</div>';
				}

			}
		}
		$output = ob_get_contents();
		ob_end_clean();
		echo ent2ncr( $output );
		die();
	}
}


//Function ajax widget gallery-videos
add_action( 'wp_ajax_thim_gallery_video', 'thim_gallery_video' );
add_action( 'wp_ajax_nopriv_thim_gallery_video', 'thim_gallery_video' );
/** widget gallery videos ajax output **/
if ( ! function_exists( 'thim_gallery_video' ) ) {
	function thim_gallery_video() {
		global $post;
		$post_id = $_POST["post_id"];
		$post    = get_post( $post_id );
		$video   = thim_meta_theme( 'thim_video' );
		// Get category permalink
		ob_start();
		echo '<div class="thim-gallery-video-content">';
		if ( filter_var( $video, FILTER_VALIDATE_URL ) ) {
			if ( $video_temp = @wp_oembed_get( $video ) ) {
				echo ent2ncr( $video_temp );
			}
		} // If embed code: just display
		else {
			echo ent2ncr( $video );
		}
		echo '</div>';
		//		?>

		<?php
		$output = ob_get_contents();
		ob_end_clean();
		echo ent2ncr( $output );
		die();
	}
}

//Remove action single event
remove_action( 'tp_event_after_loop_event_item', 'event_auth_register' );
if ( class_exists( 'WPEMS' ) ) {
	remove_action( 'tp_event_after_single_event', 'wpems_single_event_register' );
} else {
	remove_action( 'tp_event_after_single_event', 'event_auth_register' );
}

add_filter( 'event_auth_create_pages', 'thim_remove_create_page_action_event_auth' );
if ( ! function_exists( 'thim_remove_create_page_action_event_auth' ) ) {
	function thim_remove_create_page_action_event_auth( $return ) {
		return false;
	}
}

add_filter( 'event_auth_login_url', 'thim_get_login_page_url' );


add_action( 'wp_head', 'thim_define_ajaxurl', 1000 );
if ( ! function_exists( 'thim_define_ajaxurl' ) ) {
	function thim_define_ajaxurl() {
		?>
		<script type="text/javascript">
			if (typeof ajaxurl === 'undefined') {
				/* <![CDATA[ */
				var ajaxurl = "<?php echo esc_js( admin_url( 'admin-ajax.php' ) ); ?>";
				/* ]]> */
			}
		</script>
		<?php
	}
}


//Ajax widget login-popup
add_action( 'wp_ajax_nopriv_thim_login_ajax', 'thim_login_ajax_callback' );
add_action( 'wp_ajax_thim_login_ajax', 'thim_login_ajax_callback' );
if ( ! function_exists( 'thim_login_ajax_callback' ) ) {
	function thim_login_ajax_callback() {
		//ob_start();
		if ( empty( $_REQUEST['data'] ) ) {
			$response_data = array(
				'code'    => - 1,
				'message' => '<p class="message message-error">' . esc_html__( 'Something wrong. Please try again.', 'coaching' ) . '</p>'
			);
		} else {

			parse_str( $_REQUEST['data'], $login_data );

			foreach ( $login_data as $k => $v ) {
				$_POST[ $k ] = $v;
			}

			$user_verify = wp_signon( array(), is_ssl() );

			$code    = 1;
			$message = '';

			if ( is_wp_error( $user_verify ) ) {
				if ( ! empty( $user_verify->errors ) ) {
					$errors = $user_verify->errors;

					if ( ! empty( $errors['gglcptch_error'] ) ) {
						$message = '<p class="message message-error">' . esc_html__( 'You have entered an incorrect reCAPTCHA value.', 'coaching' ) . '</p>';
					} elseif ( ! empty( $errors['invalid_username'] ) || ! empty( $errors['incorrect_password'] ) ) {
						$message = '<p class="message message-error">' . esc_html__( 'Invalid username or password. Please try again!', 'coaching' ) . '</p>';
					} else if ( ! empty( $errors['cptch_error'] ) && is_array( $errors['cptch_error'] ) ) {
						foreach ( $errors['cptch_error'] as $key => $value ) {
							$message .= '<p class="message message-error">' . $value . '</p>';
						}
					} else {
						$message = '<p class="message message-error">' . esc_html__( 'Something wrong. Please try again.', 'coaching' ) . '</p>';
					}
				} else {
					$message = '<p class="message message-error">' . esc_html__( 'Something wrong. Please try again.', 'coaching' ) . '</p>';
				}
				$code = - 1;
			} else {
				$message = '<p class="message message-success">' . esc_html__( 'Login successful, redirecting...', 'coaching' ) . '</p>';
			}

			$response_data = array(
				'code'    => $code,
				'message' => $message
			);

			if ( ! empty( $login_data['redirect_to'] ) ) {
				$response_data['redirect'] = $login_data['redirect_to'];
			}
		}
		echo json_encode( $response_data );
		die(); // this is required to return a proper result
	}
}

/**
 * tunn added 04 Apr 2016
 */

if ( class_exists( 'TP_Event_Authentication' ) ) {
	$auth = TP_Event_Authentication::getInstance()->auth;

	remove_action( 'login_form_login', array( $auth, 'redirect_to_login_page' ) );
	remove_action( 'login_form_register', array( $auth, 'login_form_register' ) );
	remove_action( 'login_form_lostpassword', array( $auth, 'redirect_to_lostpassword' ) );
	remove_action( 'login_form_rp', array( $auth, 'resetpass' ) );
	remove_action( 'login_form_resetpass', array( $auth, 'resetpass' ) );

	remove_action( 'wp_logout', array( $auth, 'wp_logout' ) );
	remove_filter( 'login_url', array( $auth, 'login_url' ) );

}

function thim_redirect_rp_url() {
	if (
		! empty( $_REQUEST['action'] ) && $_REQUEST['action'] == 'rp'
		&& ! empty( $_REQUEST['key'] ) && ! empty( $_REQUEST['login'] )
	) {
		$reset_link = add_query_arg(
			array(
				'action' => 'rp',
				'key'    => $_REQUEST['key'],
				'login'  => rawurlencode( $_REQUEST['login'] )
			), thim_get_login_page_url()
		);

		if ( ! thim_is_current_url( $reset_link ) ) {
			wp_redirect( $reset_link );
			exit();
		}
	}
}

/**
 * Add action if without using wpengine
 */
if ( ! function_exists( 'is_wpe' ) && ! function_exists( 'is_wpe_snapshot' ) ) {
	add_action( 'init', 'thim_redirect_rp_url' );
}

if ( ! function_exists( 'thim_get_current_url' ) ) {
	function thim_get_current_url() {
		static $current_url;
		if ( ! $current_url ) {
			if ( ! empty( $_REQUEST['login'] ) ) {
				$url = add_query_arg( array( 'login' => rawurlencode( $_REQUEST['login'] ) ) );
			} else {
				$url = add_query_arg();
			}

			if ( ! preg_match( '!^https?!', $url ) ) {
				$segs1 = explode( '/', get_site_url() );
				$segs2 = explode( '/', $url );
				if ( $removed = array_intersect( $segs1, $segs2 ) ) {
					$segs2 = array_diff( $segs2, $removed );
					$url   = get_site_url() . '/' . join( '/', $segs2 );
				}
			}

			$current_url = $url;

		}

		return $current_url;
	}
}
if ( ! function_exists( 'thim_is_current_url' ) ) {
	function thim_is_current_url( $url ) {
		return strcmp( thim_get_current_url(), $url ) == 0;
	}
}

//Filter post_status tp_event
if ( ! function_exists( 'thim_get_upcoming_events' ) ) {
	function thim_get_upcoming_events( $args = array() ) {
		if ( is_tax( 'tp_event_category' ) ) {
			if ( version_compare( WPEMS_VER, '2.1.5', '>=' ) ) {
				$args = wp_parse_args(
					$args,
					array(
						'post_type'  => 'tp_event',
						'meta_query' => array(
							array(
								'key'     => 'tp_event_status',
								'value'   => 'upcoming',
								'compare' => '=',
							),
						),
						'tax_query'  => array(
							array(
								'taxonomy' => 'tp_event_category',
								'field'    => 'slug',
								'terms'    => get_query_var( 'term' ),
							)
						),
					)
				);
			} else {
				$args = wp_parse_args(
					$args,
					array(
						'post_type'   => 'tp_event',
						'post_status' => 'tp-event-upcoming',
						'tax_query'   => array(
							array(
								'taxonomy' => 'tp_event_category',
								'field'    => 'slug',
								'terms'    => get_query_var( 'term' ),
							)
						),
					)
				);
			}
		} else {
			if ( version_compare( WPEMS_VER, '2.1.5', '>=' ) ) {
				$args = wp_parse_args(
					$args,
					array(
						'post_type'  => 'tp_event',
						'meta_query' => array(
							array(
								'key'     => 'tp_event_status',
								'value'   => 'upcoming',
								'compare' => '=',
							),
						),
					)
				);
			} else {
				$args = wp_parse_args(
					$args,
					array(
						'post_type'   => 'tp_event',
						'post_status' => 'tp-event-upcoming',
					)
				);
			}
		}


		return new WP_Query( $args );
	}
}

if ( ! function_exists( 'thim_get_expired_events' ) ) {
	function thim_get_expired_events( $args = array() ) {
		if ( is_tax( 'tp_event_category' ) ) {
			if ( version_compare( WPEMS_VER, '2.1.5', '>=' ) ) {
				$args = wp_parse_args(
					$args,
					array(
						'post_type'  => 'tp_event',
						'meta_query' => array(
							array(
								'key'     => 'tp_event_status',
								'value'   => 'expired',
								'compare' => '=',
							),
						),
						'tax_query'  => array(
							array(
								'taxonomy' => 'tp_event_category',
								'field'    => 'slug',
								'terms'    => get_query_var( 'term' ),
							)
						),
					)
				);
			} else {
				$args = wp_parse_args(
					$args,
					array(
						'post_type'   => 'tp_event',
						'post_status' => 'tp-event-expired',
						'tax_query'   => array(
							array(
								'taxonomy' => 'tp_event_category',
								'field'    => 'slug',
								'terms'    => get_query_var( 'term' ),
							)
						),
					)
				);
			}
		} else {
			if ( version_compare( WPEMS_VER, '2.1.5', '>=' ) ) {
				$args = wp_parse_args(
					$args,
					array(
						'post_type'  => 'tp_event',
						'meta_query' => array(
							array(
								'key'     => 'tp_event_status',
								'value'   => 'expired',
								'compare' => '=',
							),
						),
					)
				);
			} else {
				$args = wp_parse_args(
					$args,
					array(
						'post_type'   => 'tp_event',
						'post_status' => 'tp-event-expired',
					)
				);
			}
		}

		return new WP_Query( $args );
	}
}

if ( ! function_exists( 'thim_get_happening_events' ) ) {
	function thim_get_happening_events( $args = array() ) {
		if ( is_tax( 'tp_event_category' ) ) {
			if ( version_compare( WPEMS_VER, '2.1.5', '>=' ) ) {
				$args = wp_parse_args(
					$args,
					array(
						'post_type'  => 'tp_event',
						'meta_query' => array(
							array(
								'key'     => 'tp_event_status',
								'value'   => 'happening',
								'compare' => '=',
							),
						),
						'tax_query'  => array(
							array(
								'taxonomy' => 'tp_event_category',
								'field'    => 'slug',
								'terms'    => get_query_var( 'term' ),
							)
						),
					)
				);
			} else {
				$args = wp_parse_args(
					$args,
					array(
						'post_type'   => 'tp_event',
						'post_status' => 'tp-event-happenning',
						'tax_query'   => array(
							array(
								'taxonomy' => 'tp_event_category',
								'field'    => 'slug',
								'terms'    => get_query_var( 'term' ),
							)
						),
					)
				);
			}
		} else {
			if ( version_compare( WPEMS_VER, '2.1.5', '>=' ) ) {
				$args = wp_parse_args(
					$args,
					array(
						'post_type'  => 'tp_event',
						'meta_query' => array(
							array(
								'key'     => 'tp_event_status',
								'value'   => 'happening',
								'compare' => '=',
							),
						),
					)
				);
			} else {
				$args = wp_parse_args(
					$args,
					array(
						'post_type'   => 'tp_event',
						'post_status' => 'tp-event-happenning',
					)
				);
			}
		}

		return new WP_Query( $args );
	}
}

if ( ! function_exists( 'thim_archive_event_template' ) ) {
	function thim_archive_event_template( $template ) {
		if ( get_post_type() == 'tp_event' && is_post_type_archive( 'tp_event' ) || is_tax( 'tp_event_category' ) ) {
			$GLOBALS['thim_happening_events'] = thim_get_happening_events();
			$GLOBALS['thim_upcoming_events']  = thim_get_upcoming_events();
			$GLOBALS['thim_expired_events']   = thim_get_expired_events();
		}

		return $template;
	}
}
add_action( 'template_include', 'thim_archive_event_template' );


function thim_get_nav_single_event( $event_id = null ) {

	if ( ! $event_id ) {
		$event_id = get_the_ID();
	}

	$post_status = get_post_status( $event_id );

	global $wpdb;
	$query = $wpdb->get_col( $wpdb->prepare(
		"
				  SELECT      t1.id
				  FROM        $wpdb->posts AS t1
				  WHERE t1.post_type = %s
				  ORDER BY post_date ASC
				  ",
		'tp_event'
	) );

	$key = array_search( $event_id, $query );

	$navigation = array();

	if ( $key != '0' ) {
		$navigation['prev_event'] = $query[ $key - 1 ];
	}

	if ( $key < count( $query ) - 1 ) {
		$navigation['next_event'] = $query[ $key + 1 ];
	}

	return $navigation;
}

if ( ! function_exists( 'thim_add_testimonial' ) ) {
	function thim_add_testimonial() {
		$meta_boxes[] = array(
			'id'     => 'testimonials_settings',
			'title'  => 'Testimonials Settings',
			'pages'  => array( 'testimonials' ),
			'fields' => array(

				array(
					'name' => esc_html__( 'Regency', 'coaching' ),
					'id'   => 'regency',
					'type' => 'textfield',
					'desc' => ''
				),
				array(
					'name' => esc_html__( 'Author', 'coaching' ),
					'id'   => 'author',
					'type' => 'textfield',
					'desc' => ''
				),

				array(
					'name' => esc_html__( 'Website URL', 'coaching' ),
					'id'   => 'website_url',
					'type' => 'textfield',
					'desc' => ''
				),
			)
		);

		return $meta_boxes;
	}
}
add_filter( 'testimonials_meta_boxes', 'thim_add_testimonial', 21 );


/**
 * @param $settings
 *
 * @return array
 */
if ( ! function_exists( 'thim_update_metabox_settings' ) ) {
	function thim_update_metabox_settings( $settings ) {
		$settings[] = 'lp_course';
		$settings[] = 'tp_event';

		return $settings;
	}
}

add_filter( 'thim_framework_metabox_settings', 'thim_update_metabox_settings' );


// Custom JS option
if ( ! function_exists( 'thim_add_custom_js' ) ) {
	function thim_add_custom_js() {
		$theme_options_data = get_theme_mods();
		if ( ! empty( $theme_options_data['thim_custom_js'] ) ) {
			?>
			<script data-cfasync="false" type="text/javascript">
				<?php echo ent2ncr( $theme_options_data['thim_custom_js'] ); ?>
			</script>
			<?php
		}
		//Add code js to open login-popup if not logged in.
		$enable_single_popup = get_theme_mod( 'thim_learnpress_single_popup', true );
		global $post;
		$redirect_url = '';
		if ( ! empty( $post->ID ) && get_option( 'permalink_structure' ) ) {
			$redirect_url = add_query_arg( array(
				'enroll-course' => $post->ID,
			), get_the_permalink( $post->ID ) );
		}
		if ( $enable_single_popup && is_singular( 'lp_course' ) ) {
			?>
			<script data-cfasync="true" type="text/javascript">
				(function ($) {
					'use strict';
					$(document).on('click',
						'body:not(".logged-in") .enroll-course .button-enroll-course, body:not(".logged-in") .purchase-course:not(".guest_checkout,.learn-press-pmpro-buy-membership ") .button',
						function (e) {
							if ($(window).width() > 767) {
								if ($('.thim-login-popup .login').length > 0) {
									e.preventDefault();
									$('#thim-popup-login #loginform [name="redirect_to"]').val('<?php echo $redirect_url; ?>');
									$('.thim-login-popup .login').trigger('click');
								}
							} else {
								e.preventDefault();
								$(this).parent().find('[name="redirect_to"]').val(
									'<?php echo thim_get_login_page_url(); ?>?redirect_to=<?php echo urlencode( $redirect_url ); ?>'
								);
								var redirect = $(this).parent().find('[name="redirect_to"]').val();
								window.location = redirect;
							}
							if ($('#thim-popup-login .register').length > 0) {
								$('#thim-popup-login .register').each(function () {
									var link = $(this).attr('href'),
										new_link = link +
											'<?php echo ! empty( $redirect_url ) ? '&redirect_to=' . $redirect_url : ''; ?>';
									$(this).prop('href', new_link);
								});
							}
						});
				})(jQuery);
			</script>
			<?php
		} else {
			?>
			<script data-cfasync="true" type="text/javascript">
				(function ($) {
					'use strict';
					$(document).on('click',
						'body:not(".logged-in") .enroll-course .button-enroll-course, body:not(".logged-in") .purchase-course:not(".guest_checkout") .button, body:not(".logged-in") .purchase-course:not(".guest_checkout,.learn-press-pmpro-buy-membership") .button',
						function (e) {
							e.preventDefault();
							$(this).parent().find('[name="redirect_to"]').val('<?php echo thim_get_login_page_url(); ?>?redirect_to=<?php echo $redirect_url; ?>');
							var redirect = $(this).parent().find('[name="redirect_to"]').val();
							window.location = redirect;
						});
				})(jQuery);
			</script>
			<?php
		}

	}

}
add_action( 'wp_footer', 'thim_add_custom_js' );

if ( ! function_exists( 'thim_oembed_filter' ) ) {
	function thim_oembed_filter( $return, $data, $url ) {
		$return = str_replace( 'frameborder="0"', 'style="border: none"', $return );

		return $return;
	}
}
add_filter( 'oembed_dataparse', 'thim_oembed_filter', 90, 3 );

if ( ! function_exists( 'thim_search_form' ) ) {
	function thim_search_form( $form ) {
		$form = '
    <div class="layout-overlay">
		<div class="search-popup-bg"></div>
	    <form role="search" method="get" class="search-form" action="' . home_url( '/' ) . '" >
		    <label>
				<span class="screen-reader-text">' . esc_html__( 'Search for:', 'coaching' ) . '</span>
				<input type="search" class="search-field" placeholder="' . esc_html__( 'Search ...', 'coaching' ) . '" value="" name="s" />
				<input type="hidden" name="post_type" value="post" />
			</label>
		    <button type="submit" class="search-submit" value="' . esc_html__( 'Search', 'coaching' ) . '">
		    	<i class="fa fa-search"></i>
		    </button>
	    </form>
	</div>';

		return $form;
	}

}
add_filter( 'get_search_form', 'thim_search_form', 100 );

if ( ! function_exists( 'thim_meta_theme' ) ) {
	function thim_meta_theme( $key, $args = array(), $post_id = null ) {
		$post_id = empty( $post_id ) ? get_the_ID() : $post_id;

		$args = wp_parse_args( $args, array(
			'type' => 'text',
		) );

		// Image
		if ( in_array( $args['type'], array( 'image' ) ) ) {
			if ( isset( $args['single'] ) && $args['single'] == "false" ) {
				// Gallery
				$temp = array();
				$data = array();

				$attachment_id = get_post_meta( $post_id, $key, true );
				if ( ! $attachment_id ) {
					return $data;
				}
				$value = explode( ',', $attachment_id );

				if ( empty( $value ) ) {
					return $data;
				}
				foreach ( $value as $k => $v ) {
					$image_attributes = wp_get_attachment_image_src( $v, $args['size'] );
					$temp['url']      = $image_attributes[0];
					$data[]           = $temp;
				}

				return $data;
			} else {
				// Single Image
				$attachment_id    = get_post_meta( $post_id, $key, true );
				$image_attributes = wp_get_attachment_image_src( $attachment_id, $args['size'] );

				return $image_attributes;
			}
		}

		return get_post_meta( $post_id, $key, $args );
	}
}


//add wishlist link to shop detail
if ( thim_plugin_active( 'yith-woocommerce-wishlist/init.php' ) ) {
	add_action( 'woocommerce_single_product_summary', 'thim_add_wishlist', 100 );
	function thim_add_wishlist() {
		$url = substr( YITH_WCWL()->get_wishlist_url(), 0, - 5 );
		if ( function_exists( 'YITH_WCWL' ) && YITH_WCWL()->count_products() ): ?>
			<a class="wishlist-link"
			   href="<?php echo esc_url( $url ) ?>"><?php esc_html_e( 'View wishlist', 'coaching' ) ?></a>
		<?php endif;

	}
}
//Filter wordpress search widget
if ( ! function_exists( 'thim_search_widget_filter' ) ) {
	function thim_search_widget_filter( $query ) {
		if ( ! is_admin() && $query->is_main_query() ) {
			if ( $query->is_search ) {
				wp_reset_query();
				if ( empty( $_REQUEST['post_type'] ) ) {
					$query->set( 'post_type', array( 'post', 'page' ) );
				}
				//$query->set( 'post_type', array( 'post', 'page' ) );
				$query->set( 'post_status', array( 'publish' ) );
			}
		}
	}
}
remove_filter( 'pre_get_posts', 'learn_press_query_taxonomy' );
add_action( 'pre_get_posts', 'thim_search_widget_filter', 1000 );


/**
 * LearnPress section
 */

if ( class_exists( 'LearnPress' ) ) {
	require_once THIM_DIR . 'inc/class-lp-layout.php';
	// require_once THIM_DIR . 'inc/learnpress-functions.php';
	require_once THIM_DIR . 'inc/learnpress-functions.php';
	function thim_new_learnpress_template_path( $slash ) {
		return 'learnpress-v4';
	}

	add_filter( 'learn_press_template_path', 'thim_new_learnpress_template_path', 999 );
}

/**
 * Check new version of LearnPress
 *
 * @return mixed
 */
function thim_is_new_learnpress( $version ) {
	if ( defined( 'LEARNPRESS_VERSION' ) ) {
		return version_compare( LEARNPRESS_VERSION, $version, '>=' );
	} else {
		return version_compare( get_option( 'learnpress_version' ), $version, '>=' );
	}
}

/**
 * Check new version of addons LearnPress woo
 *
 * @return mixed
 */
function thim_is_version_addons_woo( $version ) {
	if ( defined( 'LP_ADDON_WOOCOMMERCE_PAYMENT_VER' ) ) {
		return ( version_compare( LP_ADDON_WOOCOMMERCE_PAYMENT_VER, $version, '>=' ) && version_compare( LP_ADDON_WOOCOMMERCE_PAYMENT_VER, (int) $version + 1, '<' ) );
	}

	return false;
}

/**
 * Check new version of addons LearnPress course review
 *
 * @return mixed
 */
function thim_is_version_addons_review( $version ) {
	if ( defined( 'LP_ADDON_COURSE_REVIEW_VER' ) ) {
		return ( version_compare( LP_ADDON_COURSE_REVIEW_VER, $version, '>=' ) );
	}

	return false;
}

/**
 * Check new version of addons LearnPress bbpress
 *
 * @return mixed
 */
function thim_is_version_addons_bbpress( $version ) {
	if ( defined( 'LP_ADDON_BBPRESS_VER' ) ) {
		return ( version_compare( LP_ADDON_BBPRESS_VER, $version, '>=' ) && version_compare( LP_ADDON_BBPRESS_VER, (int) $version + 1, '<' ) );
	}

	return false;
}

/**
 * Check new version of addons LearnPress certificate
 *
 * @return mixed
 */
function thim_is_version_addons_certificates( $version ) {
	if ( defined( 'LP_ADDON_CERTIFICATES_VER' ) ) {
		return ( version_compare( LP_ADDON_CERTIFICATES_VER, $version, '>=' ) );
	}

	return false;
}

/**
 * Check new version of addons LearnPress wishlist
 *
 * @return mixed
 */
function thim_is_version_addons_wishlist( $version ) {

	if ( defined( 'LP_ADDON_WISHLIST_VER' ) ) {
		return ( version_compare( LP_ADDON_WISHLIST_VER, $version, '>=' ) );
	}

	return false;
}

/**
 * Check new version of addons LearnPress Co-instructor
 *
 * @return mixed
 */
function thim_is_version_addons_instructor( $version ) {
	if ( defined( 'LP_ADDON_CO_INSTRUCTOR_VER' ) ) {
		return ( version_compare( LP_ADDON_CO_INSTRUCTOR_VER, $version, '>=' ) && version_compare( LP_ADDON_CO_INSTRUCTOR_VER, (int) $version + 1, '<' ) );
	}

	return false;
}


if ( ! function_exists( 'thim_check_is_course' ) ) {
	function thim_check_is_course() {

		if ( function_exists( 'learn_press_is_courses' ) && learn_press_is_courses() ) {
			return true;
		} else {
			return false;
		}
	}
}

if ( ! function_exists( 'thim_check_is_course_taxonomy' ) ) {
	function thim_check_is_course_taxonomy() {

		if ( function_exists( 'learn_press_is_course_taxonomy' ) && learn_press_is_course_taxonomy() ) {
			return true;
		} else {
			return false;
		}
	}
}


function thim_allowAuthorEditing() {
	add_post_type_support( 'lp_course', 'author' );
}

add_action( 'init', 'thim_allowAuthorEditing' );

if ( ! function_exists( 'thim_time_ago' ) ) {
	function thim_time_ago( $time ) {
		$periods = array(
			esc_html__( 'second', 'coaching' ),
			esc_html__( 'minute', 'coaching' ),
			esc_html__( 'hour', 'coaching' ),
			esc_html__( 'day', 'coaching' ),
			esc_html__( 'week', 'coaching' ),
			esc_html__( 'month', 'coaching' ),
			esc_html__( 'year', 'coaching' ),
			esc_html__( 'decade', 'coaching' ),
		);
		$lengths = array(
			'60',
			'60',
			'24',
			'7',
			'4.35',
			'12',
			'10'
		);


		$now = time();

		$difference = $now - $time;
		$tense      = esc_html__( 'ago', 'coaching' );

		for ( $j = 0; $difference >= $lengths[ $j ] && $j < count( $lengths ) - 1; $j ++ ) {
			$difference /= $lengths[ $j ];
		}

		$difference = round( $difference );

		if ( $difference != 1 ) {
			$periods[ $j ] .= "s";
		}

		return "$difference $periods[$j] $tense";
	}
}

//Filter image all-demo tp-chameleon
if ( ! function_exists( 'thim_override_demo_image_tp_chameleon' ) ) {
	function thim_override_demo_image_tp_chameleon() {
		return THIM_URI . 'images/chameleon-coaching.jpg';
	}
}
add_filter( 'tp_chameleon_get_image_sprite_demos', 'thim_override_demo_image_tp_chameleon' );

/**
 * Remove redirect register url buddypress
 */
remove_filter( 'register_url', 'bp_get_signup_page' );
remove_action( 'bp_init', 'bp_core_wpsignup_redirect' );

if ( ! function_exists( 'thim_ssl_secure_url' ) ) {
	function thim_ssl_secure_url( $sources ) {
		$scheme = parse_url( site_url(), PHP_URL_SCHEME );
		if ( 'https' == $scheme ) {
			if ( stripos( $sources, 'http://' ) === 0 ) {
				$sources = 'https' . substr( $sources, 4 );
			}

			return $sources;
		}

		return $sources;
	}
}

if ( ! function_exists( 'thim_ssl_secure_image_srcset' ) ) {
	function thim_ssl_secure_image_srcset( $sources ) {
		$scheme = parse_url( site_url(), PHP_URL_SCHEME );
		if ( 'https' == $scheme ) {
			foreach ( $sources as &$source ) {
				if ( stripos( $source['url'], 'http://' ) === 0 ) {
					$source['url'] = 'https' . substr( $source['url'], 4 );
				}
			}

			return $sources;
		}

		return $sources;
	}
}

add_filter( 'wp_calculate_image_srcset', 'thim_ssl_secure_image_srcset' );
add_filter( 'wp_get_attachment_url', 'thim_ssl_secure_url', 1000 );
add_filter( 'image_widget_image_url', 'thim_ssl_secure_url' );


/**
 * Footer Bottom
 */
if ( ! function_exists( 'thim_footer_bottom' ) ) {
	function thim_footer_bottom() {
		if ( is_active_sidebar( 'footer_bottom' ) ) {
			$fb_style         = '';
			$footer_bottom_bg = get_theme_mod( 'thim_footer_bottom_bg_img', false );
			if ( ! empty( $footer_bottom_bg ) ) {
				if ( is_numeric( $footer_bottom_bg ) ) {
					$url_bg   = wp_get_attachment_image_src( $footer_bottom_bg, 'full' );
					$fb_style = ! empty( $url_bg[0] ) ? 'style="background-image: url(' . $url_bg[0] . ');"' : '';
				} else {
					$fb_style = 'style="background-image: url(' . $footer_bottom_bg . ');"';
				}
			}
			?>
			<div class="footer-bottom">
				<?php if ( ! empty( $fb_style ) ) : ?>
					<div class="thim-bg-overlay-color-half" <?php echo ent2ncr( $fb_style ); ?>>
						<div class="container">
							<?php dynamic_sidebar( 'footer_bottom' ); ?>
						</div>
					</div>
				<?php else : ?>
					<div class="container">
						<?php dynamic_sidebar( 'footer_bottom' ); ?>
					</div>
				<?php endif; ?>
			</div>
		<?php }
	}
}
add_action( 'thim_end_content_pusher', 'thim_footer_bottom' );

/**
 * Footer Class
 */
if ( ! function_exists( 'thim_footer_class' ) ) {
	function thim_footer_class() {
		$footer_class    = is_active_sidebar( 'footer_bottom' ) ? ' has-footer-bottom ' : '';
		$custom_class    = get_theme_mod( 'thim_footer_custom_class', '' );
		$footer_bg_image = get_theme_mod( 'thim_footer_background_img', '' );
		$custom_class    .= ! empty( $footer_bg_image ) ? ' footer-bg-image' : '';

		$footer_class .= $custom_class;

		echo esc_attr( $footer_class );
	}
}

/**
 * Check and update term meta for tax meta class
 */
if ( ! get_option( 'thim_update_tax_meta', false ) ) {
	global $wpdb;
	$querystr      = "
	    SELECT option_name, option_value
	    FROM $wpdb->options
	    WHERE $wpdb->options.option_name LIKE 'tax_meta_%'
	 ";
	$list_tax_meta = $wpdb->get_results( $querystr );

	if ( ! empty( $list_tax_meta ) ) {
		foreach ( $list_tax_meta as $tax_meta ) {
			$term_id   = str_replace( 'tax_meta_', '', $tax_meta->option_name );
			$term_meta = unserialize( $tax_meta->option_value );
			if ( is_array( $term_meta ) && ! empty( $term_meta ) ) {
				foreach ( $term_meta as $key => $value ) {
					if ( is_array( $value ) ) {
						if ( ! empty( $value['src'] ) ) {
							$value['url'] = $value['src'];
							unset( $value['src'] );
						}
					}
					update_term_meta( $term_id, $key, $value );
				}
			}
		}
	}
	update_option( 'thim_update_tax_meta', '1' );
}

/**
 * @param $settings
 *
 * @return array
 */
if ( ! function_exists( 'thim_import_add_basic_settings' ) ) {
	function thim_import_add_basic_settings( $settings ) {
		$settings[] = 'siteorigin_panels_settings';
		$settings[] = 'learn_press_archive_course_limit';
		$settings[] = 'learn_press_course_thumbnail_dimensions';
		$settings[] = 'thim_ekits_widget_settings';
		$settings[] = 'elementor_css_print_method';
		$settings[] = 'elementor_experiment-e_font_icon_svg';
		$settings[] = 'elementor_experiment-nested-elements';
		$settings[] = 'elementor_disable_typography_schemes';
		$settings[] = 'elementor_cpt_support';

		return $settings;
	}
}
add_filter( 'thim_importer_basic_settings', 'thim_import_add_basic_settings' );

/**
 * @param $settings
 *
 * @return array
 */
if ( ! function_exists( 'thim_import_add_page_id_settings' ) ) {
	function thim_import_add_page_id_settings( $settings ) {
		$settings[] = 'learn_press_courses_page_id';
		$settings[] = 'learn_press_profile_page_id';
		$settings[] = 'elementor_active_kit';

		return $settings;
	}
}
add_filter( 'thim_importer_page_id_settings', 'thim_import_add_page_id_settings' );

//Add info for Dashboard Admin
if ( ! function_exists( 'thim_coaching_links_guide_user' ) ) {
	function thim_coaching_links_guide_user() {
		return array(
			'docs'      => 'https://docs.thimpress.com/coachingwp/',
			'support'   => 'https://help.thimpress.com/',
			'knowledge' => 'https://thimpress.com/knowledge-base/',
		);
	}
}
add_filter( 'thim_theme_links_guide_user', 'thim_coaching_links_guide_user' );

/**
 * Link purchase theme.
 */
if ( ! function_exists( 'thim_coaching_link_purchase' ) ) {
	function thim_coaching_link_purchase() {
		return 'https://themeforest.net/item/speaker-and-life-coach-wordpress-theme-coaching-wp/17097658?ref=thimpress&utm_source=thimcoredashboard&utm_medium=dashboard';
	}
}
add_filter( 'thim_envato_link_purchase', 'thim_coaching_link_purchase' );

/**
 * Envato id.
 */
if ( ! function_exists( 'thim_coaching_envato_item_id' ) ) {
	function thim_coaching_envato_item_id() {
		return '17097658';
	}
}
add_filter( 'thim_envato_item_id', 'thim_coaching_envato_item_id' );

/**
 * Arguments form subscribe.
 */
if ( ! function_exists( 'thim_coaching_args_form_subscribe' ) ) {
	function thim_coaching_args_form_subscribe() {
		return array(
			'user' => 'e514ab4788b7083cb36eed163',
			'form' => '1beedf87e5',
		);
	}
}
add_filter( 'thim_core_args_form_subscribe', 'thim_coaching_args_form_subscribe' );


/**
 * Field name custom css theme mods.
 */
if ( ! function_exists( 'thim_coaching_field_name_custom_css_theme' ) ) {
	function thim_coaching_field_name_custom_css_theme() {
		return 'thim_custom_css';
	}
}
add_filter( 'thim_core_field_name_custom_css_theme', 'thim_coaching_field_name_custom_css_theme' );


function thim_coaching_register_meta_boxes_portfolio( $meta_boxes ) {
	$prefix = 'thim_';
	// $meta_boxes[] = array(
	// 	'id'         => 'portfolio_bg_color',
	// 	'title'      => __( 'Portfolio Meta', 'coaching' ),
	// 	'post_types' => 'portfolio',
	// 	'fields'     => array(
	// 		array(
	// 			'name' => __( 'Background Color', 'coaching' ),
	// 			'id'   => $prefix . 'portfolio_bg_color_ef',
	// 			'type' => 'color',
	// 		),
	// 		array(
	// 			'name' => __( 'Project Logo', 'coaching' ),
	// 			'id'   => $prefix . 'portfolio_project_logo',
	// 			'type' => 'image_advanced',
	// 			'max_file_uploads' => 1,
	// 		),
	// 	)
	// );
	$meta_boxes[] = array(
		'id'         => 'post_sticky_post',
		'title'      => __( 'Post Meta', 'coaching' ),
		'post_types' => 'post',
		'fields'     => array(
			array(
				'name' => __( 'Sticky Post', 'coaching' ),
				'id'   => $prefix . 'sticky_post',
				'type' => 'checkbox',
			),
			array(
				'name'    => __( 'Number related', 'coaching' ),
				'id'      => $prefix . 'number_related',
				'type'    => 'number_related',
				'default' => 3,
				'min'     => '1',
				'max'     => '10'
			),
		)
	);
	$meta_boxes[] = array(
		'id'         => 'testimonials_image_after',
		'title'      => __( 'Testimonials Meta', 'coaching' ),
		'post_types' => 'testimonials',
		'fields'     => array(
			array(
				'name'             => __( 'Image Before', 'coaching' ),
				'id'               => $prefix . 'image_before',
				'type'             => 'image_advanced',
				'max_file_uploads' => 1,

			),
			array(
				'name'             => __( 'Image After', 'coaching' ),
				'id'               => $prefix . 'image_after',
				'type'             => 'image_advanced',
				'max_file_uploads' => 1,
			),
		)
	);

	return $meta_boxes;
}

add_filter( 'rwmb_meta_boxes', 'thim_coaching_register_meta_boxes_portfolio' );

function coaching_metabox_audio( $metabox_audio ) {
	$prefix = 'thim_';

	$metabox_audio = array(
		'id'         => $prefix . 'metabox_audio',
		'context'    => 'normal',
		'priority'   => 'high',
		'title'      => __( 'Post format: Audio', 'coaching' ),
		'post_types' => array( 'post' ),
		'show'       => array(
			'post_format' => array( 'audio' ),
		),
		'fields'     => array(
			array(
				'id'    => $prefix . 'audio_mp3',
				'label' => __( 'Use MP3 file', 'coaching' ),
				'desc'  => '',
				'type'  => 'checkbox',
				'std'   => false,
			),
			array(
				'label'  => __( 'Audio Url MP3', 'coaching' ),
				'id'     => $prefix . 'audio_url',
				'type'   => 'text',
				'hidden' => array( $prefix . 'audio_mp3', '=', false ),
			),
			array(
				'label'  => __( 'Embeded Code', 'coaching' ),
				'id'     => $prefix . 'audio',
				'type'   => 'textarea',
				'hidden' => array( $prefix . 'audio_mp3', '=', true ),
			),
		),
	);

	return $metabox_audio;
}

add_filter( 'thim_metabox_audio', 'coaching_metabox_audio' );

function thim_coaching_after_switch_theme() {
	update_option( 'thim_coaching_version', THIM_THEME_VERSION );
}

add_action( 'after_switch_theme', 'thim_coaching_after_switch_theme' );

/**
 * Function print preload
 */
if ( ! function_exists( 'thim_print_preload' ) ) {
	function thim_print_preload() {
		$enable_preload     = get_theme_mod( 'thim_preload', 'default' );
		$thim_preload_image = get_theme_mod( 'thim_preload_image', false );
		$item_only          = ! empty( $_REQUEST['content-item-only'] ) ? $_REQUEST['content-item-only'] : false;
		if ( ! empty( $enable_preload ) && empty( $item_only ) ) { ?>
			<div id="preload">
				<?php
				if ( $thim_preload_image && $enable_preload == 'image' ) {
					if ( is_numeric( $thim_preload_image ) ) {
						echo wp_get_attachment_image( $thim_preload_image, 'full' );
					} else {
						echo '<img src="' . $thim_preload_image . '" alt="' . esc_html__( 'Preaload Image', 'coaching' ) . '"/>';
					}
				} else {
					switch ( $enable_preload ) {
						case 'style_1':
							$output_preload = '<div class="cssload-loader-style-1">
													<div class="cssload-inner cssload-one"></div>
													<div class="cssload-inner cssload-two"></div>
													<div class="cssload-inner cssload-three"></div>
												</div>';
							break;
						case 'style_2':
							$output_preload = '<div class="cssload-loader-style-2">
												<div class="cssload-loader-inner"></div>
											</div>';
							break;
						case 'style_3':
							$output_preload = '<div class="sk-folding-cube">
												<div class="sk-cube1 sk-cube"></div>
												<div class="sk-cube2 sk-cube"></div>
												<div class="sk-cube4 sk-cube"></div>
												<div class="sk-cube3 sk-cube"></div>
											</div>';
							break;
						case 'wave':
							$output_preload = '<div class="sk-wave">
										        <div class="sk-rect sk-rect1"></div>
										        <div class="sk-rect sk-rect2"></div>
										        <div class="sk-rect sk-rect3"></div>
										        <div class="sk-rect sk-rect4"></div>
										        <div class="sk-rect sk-rect5"></div>
										      </div>';
							break;
						case 'rotating-plane':
							$output_preload = '<div class="sk-rotating-plane"></div>';
							break;
						case 'double-bounce':
							$output_preload = '<div class="sk-double-bounce">
										        <div class="sk-child sk-double-bounce1"></div>
										        <div class="sk-child sk-double-bounce2"></div>
										      </div>';
							break;
						case 'wandering-cubes':
							$output_preload = '<div class="sk-wandering-cubes">
										        <div class="sk-cube sk-cube1"></div>
										        <div class="sk-cube sk-cube2"></div>
										      </div>';
							break;
						case 'spinner-pulse':
							$output_preload = '<div class="sk-spinner sk-spinner-pulse"></div>';
							break;
						case 'chasing-dots':
							$output_preload = '<div class="sk-chasing-dots">
										        <div class="sk-child sk-dot1"></div>
										        <div class="sk-child sk-dot2"></div>
										      </div>';
							break;
						case 'three-bounce':
							$output_preload = '<div class="sk-three-bounce">
										        <div class="sk-child sk-bounce1"></div>
										        <div class="sk-child sk-bounce2"></div>
										        <div class="sk-child sk-bounce3"></div>
										      </div>';
							break;
						case 'cube-grid':
							$output_preload = '<div class="sk-cube-grid">
										        <div class="sk-cube sk-cube1"></div>
										        <div class="sk-cube sk-cube2"></div>
										        <div class="sk-cube sk-cube3"></div>
										        <div class="sk-cube sk-cube4"></div>
										        <div class="sk-cube sk-cube5"></div>
										        <div class="sk-cube sk-cube6"></div>
										        <div class="sk-cube sk-cube7"></div>
										        <div class="sk-cube sk-cube8"></div>
										        <div class="sk-cube sk-cube9"></div>
										      </div>';
							break;
						default:
							$output_preload = '<div class="sk-folding-cube">
												<div class="sk-cube1 sk-cube"></div>
												<div class="sk-cube2 sk-cube"></div>
												<div class="sk-cube4 sk-cube"></div>
												<div class="sk-cube3 sk-cube"></div>
											</div>';
					}
					echo ent2ncr( $output_preload );
				}
				?>
			</div>
		<?php }
	}
}
add_action( 'thim_before_body', 'thim_print_preload' );

/**
 * Add google analytics & facebook pixel code
 */
if ( ! function_exists( 'thim_add_marketing_code' ) ) {
	function thim_add_marketing_code() {
		$theme_options_data = get_theme_mods();
		if ( ! empty( $theme_options_data['thim_google_analytics'] ) ) {
			?>
			<script>
				(function (i, s, o, g, r, a, m) {
					i['GoogleAnalyticsObject'] = r;
					i[r] = i[r] || function () {
						(i[r].q = i[r].q || []).push(arguments);
					}, i[r].l = 1 * new Date();
					a = s.createElement(o),
						m = s.getElementsByTagName(o)[0];
					a.async = 1;
					a.src = g;
					m.parentNode.insertBefore(a, m);
				})(window, document, 'script', 'https://www.google-analytics.com/analytics.js', 'ga');

				ga('create', '<?php echo esc_html( $theme_options_data['thim_google_analytics'] ); ?>', 'auto');
				ga('send', 'pageview');
			</script>
			<?php
		}
		if ( ! empty( $theme_options_data['thim_facebook_pixel'] ) ) {
			?>
			<script>
				!function (f, b, e, v, n, t, s) {
					if (f.fbq) return;
					n = f.fbq = function () {
						n.callMethod ?
							n.callMethod.apply(n, arguments) : n.queue.push(arguments);
					};
					if (!f._fbq) f._fbq = n;
					n.push = n;
					n.loaded = !0;
					n.version = '2.0';
					n.queue = [];
					t = b.createElement(e);
					t.async = !0;
					t.src = v;
					s = b.getElementsByTagName(e)[0];
					s.parentNode.insertBefore(t, s);
				}(window, document, 'script',
					'https://connect.facebook.net/en_US/fbevents.js');
				fbq('init', '<?php echo esc_html( $theme_options_data['thim_facebook_pixel'] ); ?>');
				fbq('track', 'PageView');
			</script>
			<noscript>
				<img height="1" width="1" style="display:none"
					 src="https://www.facebook.com/tr?id=<?php echo $theme_options_data['thim_facebook_pixel']; ?>&ev=PageView&noscript=1"/>
			</noscript>
			<?php
		}
	}
}
add_action( 'wp_footer', 'thim_add_marketing_code' );

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 *
 * @return array
 */
if ( ! function_exists( 'thim_body_classes' ) ) {
	function thim_body_classes( $classes ) {
		$item_only = ! empty( $_REQUEST['content-item-only'] ) ? $_REQUEST['content-item-only'] : false;
		// Adds a class of group-blog to blogs with more than 1 published author.
		if ( is_multi_author() ) {
			$classes[] = 'group-blog';
		}

		if ( get_theme_mod( 'thim_body_custom_class', false ) ) {
			$classes[] = get_theme_mod( 'thim_body_custom_class', false );
		}

		if ( get_theme_mod( 'thim_rtl_support', false ) || is_rtl() ) {
			$classes[] = 'rtl';
		}

		if ( get_theme_mod( 'thim_preload', true ) && empty( $item_only ) ) {
			$classes[] = 'thim-body-preload';
		} else {
			$classes[] = 'thim-body-load-overlay';
		}

		if ( get_theme_mod( 'thim_box_layout', 'wide' ) == 'boxed' ) {
			$classes[] = 'boxed-area';
		}

		if ( get_theme_mod( 'thim_size_wrap', '' ) == 'wide' ) {
			$classes[] = 'size_wide';
		} elseif ( get_theme_mod( 'thim_size_wrap', '' ) == 'elementor' ) {
			$classes[] = 'size_elementor';
		}

		if ( get_theme_mod( 'thim_bg_boxed_type', 'image' ) == 'image' ) {
			$classes[] = 'bg-boxed-image';
		} else {
			$classes[] = 'bg-boxed-pattern';
		}

		if ( class_exists( 'LearnPress' ) ) {
			$classes[] = 'learnpress-v4';
		}
		$classes[] = 'nofixcss';

		return $classes;
	}
}
add_filter( 'body_class', 'thim_body_classes' );

/**
 * Remove hook tp-event-auth
 */
if ( class_exists( 'TP_Event_Authentication' ) ) {
	if ( ! version_compare( get_option( 'event_auth_version' ), '1.0.3', '>=' ) ) {
		$auth = TP_Event_Authentication::getInstance()->auth;

		remove_action( 'login_form_login', array( $auth, 'redirect_to_login_page' ) );
		remove_action( 'login_form_register', array( $auth, 'login_form_register' ) );
		remove_action( 'login_form_lostpassword', array( $auth, 'redirect_to_lostpassword' ) );
		remove_action( 'login_form_rp', array( $auth, 'resetpass' ) );
		remove_action( 'login_form_resetpass', array( $auth, 'resetpass' ) );

		remove_action( 'wp_logout', array( $auth, 'wp_logout' ) );
		remove_filter( 'login_url', array( $auth, 'login_url' ) );
		remove_filter( 'login_redirect', array( $auth, 'login_redirect' ) );
	}
}
/**
 * Filter event login url
 */
add_filter( 'tp_event_login_url', 'thim_get_login_page_url' );
add_filter( 'event_auth_login_url', 'thim_get_login_page_url' );

/**
 * Add filter login redirect
 */
add_filter( 'login_redirect', 'thim_login_redirect', 1000 );
if ( ! function_exists( 'thim_login_redirect' ) ) {
	function thim_login_redirect() {
		if ( empty( $_REQUEST['redirect_to'] ) ) {
			$redirect_url = get_theme_mod( 'thim_login_redirect' );
			if ( ! empty( $redirect_url ) ) {
				return $redirect_url;
			} else {
				return home_url();
			}
		} else {
			return $_REQUEST['redirect_to'];
		}
	}
}

/**
 * Filters Paid Membership pro login redirect & register redirect
 */
remove_filter( 'login_redirect', 'pmpro_login_redirect', 10 );
add_filter( 'pmpro_register_redirect', '__return_false' );

/**
 * Set login cookie
 *
 * @param $logged_in_cookie
 * @param $expire
 * @param $expiration
 * @param $user_id
 * @param $logged_in
 */
function thim_set_logged_in_cookie( $logged_in_cookie, $expire, $expiration, $user_id, $logged_in ) {
	if ( $logged_in == 'logged_in' ) {
		// Hack for wp checking empty($_COOKIE[LOGGED_IN_COOKIE]) after user logged in
		// in "private mode". $_COOKIE is not set after calling setcookie util the response
		// is sent back to client (do not know why in "private mode").
		// @see wp-login.php line #789
		$_COOKIE[ LOGGED_IN_COOKIE ] = $logged_in_cookie;
	}
}

add_action( 'set_logged_in_cookie', 'thim_set_logged_in_cookie', 100, 5 );

/**
 * Filter map single event 2.0
 */
if ( ! function_exists( 'thim_filter_event_map' ) ) {
	function thim_filter_event_map( $map_data ) {
		$map_data['height']                  = '210px';
		$map_data['map_data']['scroll-zoom'] = false;
		$map_data['map_data']['marker-icon'] = get_template_directory_uri() . '/images/map_icon.png';

		return $map_data;
	}
}
add_filter( 'tp_event_filter_event_location_map', 'thim_filter_event_map' );

/**
 * Get prefix for page title
 */
if ( ! function_exists( 'thim_get_prefix_page_title' ) ) {
	function thim_get_prefix_page_title() {
		if ( get_post_type() == "product" ) {
			$prefix = 'thim_woo';
		} elseif ( get_post_type() == "lpr_course" || get_post_type() == "lpr_quiz" || get_post_type() == "lp_course" || get_post_type() == "lp_quiz" || thim_check_is_course() || thim_check_is_course_taxonomy() ) {
			$prefix = 'thim_learnpress';
		} elseif ( get_post_type() == "lp_collection" ) {
			$prefix = 'thim_collection';
		} elseif ( get_post_type() == "tp_event" ) {
			$prefix = 'thim_event';
		} elseif ( get_post_type() == "our_team" ) {
			$prefix = 'thim_team';
		} elseif ( get_post_type() == "testimonials" ) {
			$prefix = 'thim_testimonials';
		} elseif ( get_post_type() == "portfolio" ) {
			$prefix = 'thim_portfolio';
		} elseif ( get_post_type() == "forum" ) {
			$prefix = 'thim_forum';
		} elseif ( is_front_page() || is_home() ) {
			$prefix = 'thim';
		} else {
			$prefix = 'thim_archive';
		}

		return $prefix;
	}
}

/**
 * Get prefix inner for page title
 */
if ( ! function_exists( 'thim_get_prefix_inner_page_title' ) ) {
	function thim_get_prefix_inner_page_title() {
		if ( ( is_page() || is_single() ) && ! is_front_page() && ! is_home() ) {
			$prefix_inner = '_single_';
			if ( is_page() && get_post_type() == "portfolio" ) {
				$prefix_inner = '_cate_';
			}
		} else {
			if ( is_front_page() || is_home() ) {
				$prefix_inner = '_front_page_';
			} else {
				$prefix_inner = '_cate_';
				if ( get_post_type() == "lp_collection" ) {
					$prefix_inner = '_single_';
				}
			}
		}

		return $prefix_inner;
	}
}

/**
 * Print breadcrumbs
 */
if ( ! function_exists( 'thim_print_breadcrumbs' ) ) {
	function thim_print_breadcrumbs() {
		?>
		<div class="breadcrumbs-wrapper">
			<div class="container">
				<?php
				//Check seo by yoast breadcrumbs
				$wpseo = get_option( 'wpseo_internallinks' );
				if ( is_plugin_active( 'wordpress-seo/wp-seo.php' ) && ( isset( $wpseo['breadcrumbs-enable'] ) && $wpseo['breadcrumbs-enable'] ) ) {
					if ( function_exists( 'yoast_breadcrumb' ) ) {
						yoast_breadcrumb( '<ul id="breadcrumbs">', '</ul>' );
					}
				} else {
					if ( thim_use_bbpress() ) {
						bbp_breadcrumb();
					} elseif ( get_post_type() == "lp_collection" && function_exists( 'learn_press_breadcrumb' ) ) {
						learn_press_breadcrumb();
					} else {
						do_action( 'thim_breadcrumbs' );
					}
				}
				?>
			</div>
		</div>
		<?php
	}
}

/**
 * Get page title
 */
if ( ! function_exists( 'thim_get_page_title' ) ) {
	function thim_get_page_title( $custom_title, $front_title ) {
		$heading_title = esc_html__( 'Page title', 'coaching' );
		if ( get_post_type() == 'product' ) {
			$heading_title = woocommerce_page_title( false );
		} elseif ( get_post_type() == 'lp_course' || get_post_type() == 'lp_quiz' || thim_check_is_course() || thim_check_is_course_taxonomy() ) {
			if ( is_single() ) {
				if ( ! empty( $custom_title ) ) {
					$heading_title = $custom_title;
				} else {
					$course_cat = get_the_terms( get_the_ID(), 'course_category' );
					if ( ! empty( $course_cat ) ) {
						$heading_title = $course_cat[0]->name;
					} else {
						$heading_title = __( 'Course', 'coaching' );
					}
				}
			} else {
				$heading_title = learn_press_page_title();
			}
		} elseif ( get_post_type() == 'lp_collection' ) {
			$heading_title = learn_press_collections_page_title( false );
		} elseif ( ( is_category() || is_archive() || is_search() || is_404() ) && ! thim_use_bbpress() ) {
			if ( get_post_type() == 'tp_event' ) {
				$heading_title = esc_html__( 'Events', 'coaching' );
			} elseif ( get_post_type() == 'testimonials' ) {
				$heading_title = esc_html__( 'Testimonials', 'coaching' );
			} else {
				$heading_title = thim_archive_title();
			}
		} elseif ( thim_use_bbpress() ) {
			$heading_title = thim_forum_title();
		} elseif ( is_page() || is_single() ) {
			if ( is_single() ) {
				if ( $custom_title ) {
					$heading_title = $custom_title;
				} else {
					if ( get_post_type() == 'post' ) {
						$category = get_the_category();
						if ( $category ) {
							$category_id   = get_cat_ID( $category[0]->cat_name );
							$heading_title = get_category_parents( $category_id, false, " " );
						}
					}
					if ( get_post_type() == 'tp_event' ) {
						$heading_title = esc_html__( 'Events', 'coaching' );
					}
					if ( get_post_type() == 'portfolio' ) {
						$heading_title = esc_html__( 'Portfolio', 'coaching' );
					}
					if ( get_post_type() == 'our_team' ) {
						$heading_title = esc_html__( 'Our Team', 'coaching' );
					}
					if ( get_post_type() == 'testimonials' ) {
						$heading_title = esc_html__( 'Testimonials', 'coaching' );
					}
				}

			} else {
				$heading_title = ! empty( $custom_title ) ? $custom_title : get_the_title();
			}

		} elseif ( ! is_front_page() && is_home() ) {
			$heading_title = ! empty( $front_title ) ? $front_title : esc_html__( 'Blog', 'coaching' );
		}

		return $heading_title;
	}
}

/**
 * Format price
 */
add_filter( 'pmpro_format_price', 'thim_pmpro_formatPrice', 10, 4 );
if ( ! function_exists( 'thim_pmpro_formatPrice' ) ) {
	function thim_pmpro_formatPrice( $formatted, $price, $pmpro_currency, $pmpro_currency_symbol ) {
		if ( is_numeric( $price ) && ( intval( $price ) == floatval( $price ) ) ) {
			return $pmpro_currency_symbol . number_format( $price );
		} else {
			return $pmpro_currency_symbol . number_format( $price, 2 );
		}

	}
}


/**
 * Filter level cost text paid membership pro
 */
add_filter( 'pmpro_level_cost_text', 'thim_pmpro_getLevelCost', 10, 4 );
if ( ! function_exists( 'thim_pmpro_getLevelCost' ) ) {
	function thim_pmpro_getLevelCost( $r, $level, $tags, $short ) {
		//initial payment
		if ( ! $short ) {
			$r = sprintf( __( 'The price for membership is <p class="price">%s</p>', 'coaching' ), pmpro_formatPrice( $level->initial_payment ) );
		} else {
			$r = sprintf( __( '%s', 'coaching' ), pmpro_formatPrice( $level->initial_payment ) );
		}

		//recurring part
		if ( $level->billing_amount != '0.00' ) {
			if ( $level->billing_limit > 1 ) {
				if ( $level->cycle_number == '1' ) {
					$r .= sprintf( __( '<p class="expired">then %s per %s for %d more %s</p>', 'coaching' ), pmpro_formatPrice( $level->billing_amount ), pmpro_translate_billing_period( $level->cycle_period ), $level->billing_limit, pmpro_translate_billing_period( $level->cycle_period, $level->billing_limit ) );
				} else {
					$r .= sprintf( __( '<p class="expired">then %s every %d %s for %d more payments</p>', 'coaching' ), pmpro_formatPrice( $level->billing_amount ), $level->cycle_number, pmpro_translate_billing_period( $level->cycle_period, $level->cycle_number ), $level->billing_limit );
				}
			} elseif ( $level->billing_limit == 1 ) {
				$r .= sprintf( __( '<p class="expired">then %s after %d %s</p>', 'coaching' ), pmpro_formatPrice( $level->billing_amount ), $level->cycle_number, pmpro_translate_billing_period( $level->cycle_period, $level->cycle_number ) );
			} else {
				if ( $level->billing_amount === $level->initial_payment ) {
					if ( $level->cycle_number == '1' ) {
						if ( ! $short ) {
							$r = sprintf( __( 'The price for membership is <strong>%s per %s</strong>', 'coaching' ), pmpro_formatPrice( $level->initial_payment ), pmpro_translate_billing_period( $level->cycle_period ) );
						} else {
							$r = sprintf( __( '<p class="expired">%s per %s</p>', 'coaching' ), pmpro_formatPrice( $level->initial_payment ), pmpro_translate_billing_period( $level->cycle_period ) );
						}
					} else {
						if ( ! $short ) {
							$r = sprintf( __( 'The price for membership is <strong>%s every %d %s</strong>', 'coaching' ), pmpro_formatPrice( $level->initial_payment ), $level->cycle_number, pmpro_translate_billing_period( $level->cycle_period, $level->cycle_number ) );
						} else {
							$r = sprintf( __( '<p class="expired">%s every %d %s</p>', 'coaching' ), pmpro_formatPrice( $level->initial_payment ), $level->cycle_number, pmpro_translate_billing_period( $level->cycle_period, $level->cycle_number ) );
						}
					}
				} else {
					if ( $level->cycle_number == '1' ) {
						$r .= sprintf( __( '<p class="expired">then %s per %s</p>', 'coaching' ), pmpro_formatPrice( $level->billing_amount ), pmpro_translate_billing_period( $level->cycle_period ) );
					} else {
						$r .= sprintf( __( '<p class="expired">and then %s every %d %s</p>', 'coaching' ), pmpro_formatPrice( $level->billing_amount ), $level->cycle_number, pmpro_translate_billing_period( $level->cycle_period, $level->cycle_number ) );
					}
				}
			}
		}

		//trial part
		if ( $level->trial_limit ) {
			if ( $level->trial_amount == '0.00' ) {
				if ( $level->trial_limit == '1' ) {
					$r .= ' ' . __( 'After your initial payment, your first payment is Free.', 'coaching' );
				} else {
					$r .= ' ' . sprintf( __( 'After your initial payment, your first %d payments are Free.', 'coaching' ), $level->trial_limit );
				}
			} else {
				if ( $level->trial_limit == '1' ) {
					$r .= ' ' . sprintf( __( 'After your initial payment, your first payment will cost %s.', 'coaching' ), pmpro_formatPrice( $level->trial_amount ) );
				} else {
					$r .= ' ' . sprintf( __( 'After your initial payment, your first %d payments will cost %s.', 'coaching' ), $level->trial_limit, pmpro_formatPrice( $level->trial_amount ) );
				}
			}
		}

		//taxes part
		$tax_state = pmpro_getOption( "tax_state" );
		$tax_rate  = pmpro_getOption( "tax_rate" );

		if ( $tax_state && $tax_rate && ! pmpro_isLevelFree( $level ) ) {
			$r .= sprintf( __( 'Customers in %s will be charged %s%% tax.', 'coaching' ), $tax_state, round( $tax_rate * 100, 2 ) );
		}

		if ( ! $tags ) {
			$r = strip_tags( $r );
		}

		return $r;
	}
}

/**
 * Filter price paid membership pro
 */
add_filter( 'pmpro_format_price', 'thim_pmpro_format_price', 10, 4 );
if ( ! function_exists( 'thim_pmpro_format_price' ) ) {
	function thim_pmpro_format_price( $formatted, $price, $pmpro_currency, $pmpro_currency_symbol ) {
		global $pmpro_currency, $pmpro_currency_symbol, $pmpro_currencies;

		//start with the price formatted with two decimals
		$formatted = number_format( (double) $price, 0 );

		//settings stored in array?
		if ( ! empty( $pmpro_currencies[ $pmpro_currency ] ) && is_array( $pmpro_currencies[ $pmpro_currency ] ) ) {
			//format number do decimals, with decimal_separator and thousands_separator
			$formatted = number_format( $price,
				( isset( $pmpro_currencies[ $pmpro_currency ]['decimals'] ) ? (int) $pmpro_currencies[ $pmpro_currency ]['decimals'] : 2 ),
				( isset( $pmpro_currencies[ $pmpro_currency ]['decimal_separator'] ) ? $pmpro_currencies[ $pmpro_currency ]['decimal_separator'] : '.' ),
				( isset( $pmpro_currencies[ $pmpro_currency ]['thousands_separator'] ) ? $pmpro_currencies[ $pmpro_currency ]['thousands_separator'] : ',' )
			);

			//which side is the symbol on?
			if ( ! empty( $pmpro_currencies[ $pmpro_currency ]['position'] ) && $pmpro_currencies[ $pmpro_currency ]['position'] == 'left' ) {
				$formatted = '<span class="currency">' . $pmpro_currency_symbol . '</span>' . $formatted;
			} else {
				$formatted = $formatted . '<span class="currency">' . $pmpro_currency_symbol . '</span>';
			}
		} else {
			$formatted = '<span class="currency">' . $pmpro_currency_symbol . '</span>' . $formatted;
		}    //default to symbol on the left

		return $formatted;
	}
}

/*
 * Handle conflict betweeen Google captcha plugin vs Revolution Slider plugin
 */
if ( thim_plugin_active( 'google-captcha/google-captcha.php' ) ) {
	remove_filter( 'widget_text', 'do_shortcode' );
}

/*
 * Add google captcha register check to register form ( multisite case )
 */
if ( is_multisite() && function_exists( 'gglcptch_register_check' ) ) {
	global $gglcptch_ip_in_whitelist;

	if ( ! $gglcptch_ip_in_whitelist ) {
		add_action( 'registration_errors', 'gglcptch_register_check', 10, 1 );
	}
}

if ( ! function_exists( "thim_get_cat_taxonomy" ) ) {
	function thim_get_cat_taxonomy( $term = 'category', $cats = false, $vc = false ) {
		if ( ! $cats ) {
			$cats = array();
		}
		if ( is_admin() ) {

			$terms = new WP_Term_Query(
				array(
					'taxonomy'   => $term,
					'orderby'    => 'name',
					'order'      => 'DESC',
					'child_of'   => 0,
					'parent'     => 0,
					'fields'     => 'all',
					'hide_empty' => false,
				)
			);

			if ( is_wp_error( $terms ) ) {
			} else {
				if ( empty( $terms->terms ) ) {
				} else {
					$prefix = '';
					foreach ( $terms->terms as $term ) {
						if ( $term->parent > 0 ) {
							$prefix = "--";
						}
						if ( $vc == true ) {
							$cats[ $prefix . $term->name ] = $term->term_id;
						} else {
							$cats[ $term->term_id ] = $prefix . $term->name;
						}
					}
				}
			}
		}

		return $cats;
	}
}
/*
 * Get List image size
 */
if ( ! function_exists( "thim_sc_get_list_image_size" ) ) {
	function thim_sc_get_list_image_size() {
		global $_wp_additional_image_sizes;

		$sizes                        = array();
		$get_intermediate_image_sizes = get_intermediate_image_sizes();

		// Create the full array with sizes and crop info
		foreach ( $get_intermediate_image_sizes as $_size ) {

			if ( in_array( $_size, array( 'thumbnail', 'medium', 'large' ) ) ) {

				$sizes[ $_size ]['width']  = get_option( $_size . '_size_w' );
				$sizes[ $_size ]['height'] = get_option( $_size . '_size_h' );
				$sizes[ $_size ]['crop']   = (bool) get_option( $_size . '_crop' );

			} elseif ( isset( $_wp_additional_image_sizes[ $_size ] ) ) {

				$sizes[ $_size ] = array(
					'width'  => $_wp_additional_image_sizes[ $_size ]['width'],
					'height' => $_wp_additional_image_sizes[ $_size ]['height'],
					'crop'   => $_wp_additional_image_sizes[ $_size ]['crop']
				);

			}

		}


		$image_size                                             = array();
		$image_size[ esc_html__( "No Image", 'coaching' ) ]     = 'none';
		$image_size[ esc_html__( "Custom Image", 'coaching' ) ] = 'custom_image';
		$image_size[ esc_html__( "Custom Size", 'coaching' ) ]  = 'custom_size';
		if ( ! empty( $sizes ) ) {
			foreach ( $sizes as $key => $value ) {
				if ( $value['width'] && $value['height'] ) {
					$image_size[ $value['width'] . 'x' . $value['height'] ] = $key;
				} else {
					$image_size[ $key ] = $key;
				}
			}
		}

		return $image_size;
	}
}
/**
 * Get popular list courses
 *
 * Count all user enroll, buy course (No discrimination order status)
 *
 * @param int $limit
 *
 * @return array|false|mixed
 * @since 4.2.9.7
 * @note  should write on LP | function is temporary | see same get_popular_courses function of LP
 * @author tungnx
 *
 */
function coaching_lp_get_popular_courses( $limit = 10 ) {
	global $wpdb;

	$result = wp_cache_get( 'lp-popular-course', '', true );

	if ( ! $result ) {
		$query = $wpdb->prepare( "
        SELECT ID, a+IF(b IS NULL, 0, b) AS students FROM(
		SELECT p.ID as ID, IF(pm.meta_value, pm.meta_value, 0) as a, (
	       SELECT COUNT(*)
          FROM (SELECT COUNT(item_id), item_id, user_id FROM {$wpdb->prefix}learnpress_user_items GROUP BY item_id, user_id) AS Y
          GROUP BY item_id
          HAVING item_id = p.ID
        ) AS b
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} AS pm ON p.ID = pm.post_id  AND pm.meta_key = %s
        WHERE p.post_type = %s AND p.post_status = %s
        GROUP BY ID
        ) AS Z
        ORDER BY students DESC
        	  LIMIT 0, $limit
         ", '_lp_students', 'lp_course', 'publish' );

		$result = $wpdb->get_col( $query );
	}

	$time_cache = apply_filters( 'lp/time-cache/popular-courses', 60 * 60 * 60 );

	wp_cache_set( 'lp-popular-courses', $result, '', current_time( 'timestamp' ) + $time_cache );

	return $result;
}

/**
 * Get options from customizer
 *
 *
 * @author tuanta
 *
 */
if ( ! function_exists( 'thim_header_class' ) ) {
	function thim_header_class() {
		$theme_options_data = get_theme_mods();

		$header_class = '';
		if ( isset( $theme_options_data['thim_config_att_sticky'] ) && $theme_options_data['thim_config_att_sticky'] == 'sticky_custom' ) {
			$header_class .= ' bg-custom-sticky';
		}
		if ( isset( $theme_options_data['thim_header_sticky'] ) && $theme_options_data['thim_header_sticky'] == 1 && ! is_singular( 'lp_course' ) ) {
			$header_class .= ' sticky-header';
		}
		if ( isset( $theme_options_data['thim_header_position'] ) ) {
			$header_class .= ' ' . $theme_options_data['thim_header_position'];
		}

		if ( isset( $theme_options_data['thim_header_style'] ) ) {
			$header_class .= ' ' . $theme_options_data['thim_header_style'];
		} else {
			$header_class .= ' header_v1';
		}

		//Custom Header Class
		if ( isset( $theme_options_data['thim_header_custom_class'] ) ) {
			$header_class .= ' ' . $theme_options_data['thim_header_custom_class'];
		}

		// mobile menu custom class
		if ( isset( $theme_options_data['thim_config_logo_mobile'] ) && $theme_options_data['thim_config_logo_mobile'] == 'custom_logo' ) {
			if ( wp_is_mobile() && ( ( isset( $theme_options_data['thim_logo_mobile'] ) && $theme_options_data['thim_logo_mobile'] <> '' ) || ( isset( $theme_options_data['thim_sticky_logo_mobile'] ) && $theme_options_data['thim_sticky_logo_mobile'] <> '' )
				)
			) {
				$header_class .= ' mobile-logo-custom';
			}
		}

		echo esc_attr( $header_class );
	}
}

if ( ! function_exists( "thim_message_before_importer" ) ) {
	function thim_message_before_importer() {
		$title = 'Import data demo with Elementor Page Builder';
		if ( get_theme_mod( 'thim_page_builder_chosen' ) == 'visual_composer' ) {
			$title = 'You has import data demo with WPBakery Page Builder';
		} elseif ( get_theme_mod( 'thim_page_builder_chosen' ) == 'site_origin' ) {
			$title = 'You has import data demo with SiteOrigin Page Builder';
		}

		$desc = '<b>WPBakery</b> or <b>SiteOrigin</b>';
		if ( apply_filters( 'thim-importer-demo-vc', false ) ) {
			$title = 'You has enabled import data demo with WPBakery Page Builder';
			$desc  = '<b>Elementor</b> or <b>SiteOrigin</b>';
		} elseif ( apply_filters( 'thim-importer-demo-so', false ) ) {
			$title = 'You has enabled import data demo with SiteOrigin Page Builder';
			$desc  = '<b>WPBakery</b> or <b>Elementor</b>';
		}
		echo '<div class="thim-message-import"><h3>' . esc_html__( $title, 'coaching' ) . '</h3>';
		echo '<p><i>If you want to import data with ' . $desc . ' Page Builder <a href="https://thimpress.com/knowledge-base/how-to-import-data-with-wpbakery-or-siteorigin/" target="_blank">Please read the guide here.</a></i></p></div>';
	}
}
add_filter( 'thim-message-before-importer', 'thim_message_before_importer' );

if ( ! function_exists( 'thim_enable_upload_svg' ) ) {
	function thim_enable_upload_svg( $mimes ) {
		$mimes['svg'] = 'image/svg+xml';

		return $mimes;
	}

	add_filter( 'upload_mimes', 'thim_enable_upload_svg' );
}

// hook custom post audio
if ( ! function_exists( 'thim_podcast_mp3' ) ) {
	function thim_podcast_mp3() {
		if ( ! has_post_format( 'audio' ) ) {
			return;
		}
		wp_enqueue_script( 'thim-amplitude' );
		$icon_duration    = '<svg xmlns="http://www.w3.org/2000/svg" width="21" height="20" viewBox="0 0 21 20" fill="none"><path d="M10.5 0C4.97581 0 0.5 4.47581 0.5 10C0.5 15.5242 4.97581 20 10.5 20C16.0242 20 20.5 15.5242 20.5 10C20.5 4.47581 16.0242 0 10.5 0ZM10.5 18.0645C6.04435 18.0645 2.43548 14.4556 2.43548 10C2.43548 5.54435 6.04435 1.93548 10.5 1.93548C14.9556 1.93548 18.5645 5.54435 18.5645 10C18.5645 14.4556 14.9556 18.0645 10.5 18.0645ZM12.9919 13.8548L9.56855 11.3669C9.44355 11.2742 9.37097 11.129 9.37097 10.9758V4.35484C9.37097 4.08871 9.58871 3.87097 9.85484 3.87097H11.1452C11.4113 3.87097 11.629 4.08871 11.629 4.35484V10.0685L14.3226 12.0282C14.5403 12.1855 14.5847 12.4879 14.4274 12.7056L13.6694 13.75C13.5121 13.9637 13.2097 14.0121 12.9919 13.8548Z" fill="#FC5739"/></svg>';
		$icon_share       = '<svg width="26" height="27" viewBox="0 0 26 27" fill="none" xmlns="http://www.w3.org/2000/svg"><path id="Vector" d="M15.132 19.7676L9.60697 16.7544C8.88563 17.5256 7.94903 18.0619 6.91887 18.2938C5.88871 18.5256 4.81263 18.4423 3.83047 18.0545C2.84832 17.6667 2.0055 16.9925 1.41156 16.1195C0.817611 15.2464 0.5 14.2149 0.5 13.159C0.5 12.1031 0.817611 11.0715 1.41156 10.1985C2.0055 9.32545 2.84832 8.65124 3.83047 8.26348C4.81263 7.87573 5.88871 7.79236 6.91887 8.02421C7.94903 8.25606 8.88563 8.79241 9.60697 9.56355L15.1334 6.55035C14.8201 5.30819 14.9706 3.9944 15.5567 2.85527C16.1428 1.71613 17.1242 0.829868 18.317 0.362602C19.5098 -0.104664 20.8321 -0.120847 22.036 0.317085C23.2399 0.755017 24.2427 1.617 24.8565 2.74144C25.4703 3.86589 25.6529 5.1756 25.3702 6.42506C25.0874 7.67453 24.3586 8.77796 23.3204 9.52851C22.2822 10.2791 21.006 10.6252 19.7308 10.5021C18.4557 10.3789 17.2693 9.79495 16.3939 8.85959L10.8675 11.8728C11.0796 12.7167 11.0796 13.6 10.8675 14.4439L16.3939 17.4571C17.2697 16.5222 18.4564 15.9388 19.7316 15.8163C21.0067 15.6937 22.2828 16.0405 23.3206 16.7915C24.3584 17.5426 25.0867 18.6463 25.3689 19.8959C25.6511 21.1454 25.4679 22.455 24.8536 23.5792C24.2393 24.7033 23.2362 25.5649 22.0321 26.0023C20.8281 26.4397 19.5058 26.4229 18.3132 25.9552C17.1206 25.4875 16.1396 24.6008 15.5539 23.4615C14.9683 22.3222 14.8183 21.0083 15.132 19.7663V19.7676ZM5.7635 15.7899C6.46144 15.7899 7.1308 15.5127 7.62433 15.0192C8.11785 14.5256 8.39511 13.8563 8.39511 13.1583C8.39511 12.4604 8.11785 11.791 7.62433 11.2975C7.1308 10.804 6.46144 10.5267 5.7635 10.5267C5.06555 10.5267 4.39619 10.804 3.90266 11.2975C3.40914 11.791 3.13188 12.4604 3.13188 13.1583C3.13188 13.8563 3.40914 14.5256 3.90266 15.0192C4.39619 15.5127 5.06555 15.7899 5.7635 15.7899ZM20.2374 7.89511C20.9353 7.89511 21.6047 7.61785 22.0982 7.12433C22.5917 6.6308 22.869 5.96144 22.869 5.2635C22.869 4.56555 22.5917 3.89619 22.0982 3.40267C21.6047 2.90914 20.9353 2.63188 20.2374 2.63188C19.5394 2.63188 18.8701 2.90914 18.3765 3.40267C17.883 3.89619 17.6058 4.56555 17.6058 5.2635C17.6058 5.96144 17.883 6.6308 18.3765 7.12433C18.8701 7.61785 19.5394 7.89511 20.2374 7.89511ZM20.2374 23.6848C20.9353 23.6848 21.6047 23.4075 22.0982 22.914C22.5917 22.4205 22.869 21.7511 22.869 21.0532C22.869 20.3552 22.5917 19.6859 22.0982 19.1923C21.6047 18.6988 20.9353 18.4216 20.2374 18.4216C19.5394 18.4216 18.8701 18.6988 18.3765 19.1923C17.883 19.6859 17.6058 20.3552 17.6058 21.0532C17.6058 21.7511 17.883 22.4205 18.3765 22.914C18.8701 23.4075 19.5394 23.6848 20.2374 23.6848Z" fill="#FC5739"/></svg>';
		$icon_save_audio  = '<svg width="26" height="24" viewBox="0 0 26 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M25 7C25 3.68629 22.2018 1 18.75 1C16.1692 1 13.9537 2.5017 13 4.64456C12.0463 2.5017 9.83082 1 7.25 1C3.79822 1 1 3.68629 1 7C1 14.6072 8.49219 20.1822 11.6365 22.187C12.4766 22.7226 13.5234 22.7226 14.3635 22.187C17.5078 20.1822 25 14.6072 25 7Z" stroke="#94A3B8" stroke-width="2" stroke-linejoin="round"/></svg>';
		$icon_volume_down = '<svg xmlns="http://www.w3.org/2000/svg" width="15" height="20" viewBox="0 0 15 20" fill="none" class="amplitude-volume-down"><path d="M13.5354 0.115441C13.3324 0.0276326 13.1098 -0.00474977 12.8902 0.0215983C12.6706 0.0479465 12.462 0.132081 12.2856 0.265416L6.31155 4.98964H1.7498C1.41833 4.98964 1.10044 5.12132 0.866057 5.3557C0.631674 5.59008 0.5 5.90797 0.5 6.23944V13.7382C0.5 14.0697 0.631674 14.3876 0.866057 14.622C1.10044 14.8563 1.41833 14.988 1.7498 14.988H6.31155L12.2231 19.7122C12.443 19.8887 12.7161 19.9856 12.998 19.9872C13.1846 19.9903 13.3692 19.9474 13.5354 19.8622C13.7481 19.7609 13.9278 19.6016 14.0538 19.4026C14.1799 19.2036 14.2471 18.973 14.2478 18.7374V1.24026C14.2471 1.00469 14.1799 0.774103 14.0538 0.575089C13.9278 0.376075 13.7481 0.216735 13.5354 0.115441ZM11.7482 16.1378L7.52385 12.7634C7.30396 12.587 7.03089 12.4901 6.74898 12.4884H2.99959V7.48924H6.74898C7.03089 7.4876 7.30396 7.3907 7.52385 7.21428L11.7482 3.83983V16.1378Z" fill="#FC5739"/></svg>';
		$icon_volume_up   = '<svg xmlns="http://www.w3.org/2000/svg" width="26" height="20" viewBox="0 0 26 20" fill="none" class="amplitude-volume-up"><path d="M13.2834 0.115441C13.0805 0.0276326 12.8578 -0.00474977 12.6383 0.0215983C12.4187 0.0479465 12.21 0.132081 12.0336 0.265416L6.0596 4.98964H1.49784C1.16638 4.98964 0.848486 5.12132 0.614103 5.3557C0.379721 5.59008 0.248047 5.90797 0.248047 6.23944V13.7382C0.248047 14.0697 0.379721 14.3876 0.614103 14.622C0.848486 14.8563 1.16638 14.988 1.49784 14.988H6.0596L11.9711 19.7122C12.191 19.8887 12.4641 19.9856 12.746 19.9872C12.9327 19.9903 13.1173 19.9474 13.2834 19.8622C13.4961 19.7609 13.6758 19.6016 13.8019 19.4026C13.9279 19.2036 13.9951 18.973 13.9958 18.7374V1.24026C13.9951 1.00469 13.9279 0.774103 13.8019 0.575089C13.6758 0.376075 13.4961 0.216735 13.2834 0.115441ZM11.4962 16.1378L7.2719 12.7634C7.05201 12.587 6.77893 12.4901 6.49703 12.4884H2.74764V7.48924H6.49703C6.77893 7.4876 7.05201 7.3907 7.2719 7.21428L11.4962 3.83983V16.1378ZM22.3194 2.91498C22.0841 2.67964 21.7649 2.54743 21.4321 2.54743C21.0993 2.54743 20.7801 2.67964 20.5447 2.91498C20.3094 3.15032 20.1772 3.46952 20.1772 3.80234C20.1772 4.13516 20.3094 4.45435 20.5447 4.68969C21.2807 5.4245 21.8557 6.30435 22.2334 7.27334C22.611 8.24233 22.7829 9.27928 22.7383 10.3183C22.6936 11.3573 22.4333 12.3757 21.9739 13.3087C21.5145 14.2417 20.8661 15.069 20.0698 15.7379C19.8764 15.9032 19.7381 16.1237 19.6734 16.3697C19.6086 16.6158 19.6205 16.8758 19.7075 17.1149C19.7944 17.354 19.9523 17.5609 20.16 17.7079C20.3676 17.8549 20.6152 17.935 20.8697 17.9375C21.1617 17.9381 21.4447 17.8364 21.6695 17.6501C22.733 16.7593 23.5995 15.657 24.214 14.4133C24.8284 13.1695 25.1772 11.8115 25.2383 10.4256C25.2995 9.0397 25.0715 7.65625 24.569 6.36323C24.0665 5.0702 23.3004 3.89589 22.3194 2.91498ZM18.7825 6.45191C18.666 6.33538 18.5276 6.24294 18.3754 6.17987C18.2231 6.11681 18.06 6.08435 17.8952 6.08435C17.7304 6.08435 17.5672 6.11681 17.4149 6.17987C17.2627 6.24294 17.1243 6.33538 17.0078 6.45191C16.8913 6.56843 16.7988 6.70677 16.7358 6.85903C16.6727 7.01128 16.6403 7.17446 16.6403 7.33926C16.6403 7.50406 16.6727 7.66724 16.7358 7.81949C16.7988 7.97175 16.8913 8.11009 17.0078 8.22662C17.4772 8.69323 17.7424 9.32696 17.7452 9.98883C17.7455 10.353 17.6662 10.7127 17.5129 11.043C17.3597 11.3733 17.1361 11.6662 16.8578 11.901C16.7313 12.0059 16.6267 12.1348 16.55 12.2802C16.4733 12.4256 16.4261 12.5847 16.411 12.7484C16.3959 12.9121 16.4132 13.0772 16.462 13.2341C16.5108 13.3911 16.5901 13.5369 16.6954 13.6632C16.8012 13.7889 16.9307 13.8924 17.0765 13.968C17.2224 14.0435 17.3817 14.0896 17.5454 14.1035C17.709 14.1174 17.8738 14.099 18.0304 14.0492C18.1869 13.9993 18.3321 13.9192 18.4576 13.8132C19.0164 13.3447 19.4659 12.7596 19.7746 12.0989C20.0834 11.4383 20.2438 10.7181 20.2448 9.98883C20.2377 8.66418 19.7129 7.39481 18.7825 6.45191Z" fill="#FC5739"/></svg>';
		$icon_prev        = '<svg width="32" height="32" viewBox="0 0 32 32" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M26 7C26 5.76393 24.5889 5.05836 23.6 5.8L11.6 14.8C10.8 15.4 10.8 16.6 11.6 17.2L23.6 26.2C24.5889 26.9416 26 26.2361 26 25V7Z" fill="#94A3B8" stroke="#94A3B8" stroke-width="2" stroke-linejoin="round"/><path d="M6 5L6 27" stroke="#94A3B8" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>';
		$icon_next        = '<svg width="32" height="32" viewBox="0 0 32 32" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M6 7C6 5.76393 7.41115 5.05836 8.4 5.8L20.4 14.8C21.2 15.4 21.2 16.6 20.4 17.2L8.4 26.2C7.41115 26.9416 6 26.2361 6 25V7Z" fill="#94A3B8" stroke="#94A3B8" stroke-width="2" stroke-linejoin="round"/><path d="M26 5L26 27" stroke="#94A3B8" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>';
		$icon_play        = '<svg id="play-icon" class="ml-[10px]" width="31" height="37" viewBox="0 0 31 37" fill="none" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M29.6901 16.6608L4.00209 0.747111C2.12875 -0.476923 0.599998 0.421814 0.599998 2.75545V33.643C0.599998 35.9728 2.12747 36.8805 4.00209 35.6514L29.6901 19.7402C29.6901 19.7402 30.6043 19.0973 30.6043 18.2012C30.6043 17.3024 29.6901 16.6608 29.6901 16.6608Z" class="fill-slate-500 dark:fill-slate-400"/></svg>';
		$icon_pause       = '<svg id="pause-icon" width="24" height="36" viewBox="0 0 24 36" fill="none" xmlns="http://www.w3.org/2000/svg"><rect width="6" height="36" rx="3" class="fill-slate-500 dark:fill-slate-400"/><rect x="18" width="6" height="36" rx="3" class="fill-slate-500 dark:fill-slate-400" /></svg>';
		$icon_repeat      = '<svg width="26" height="24" viewBox="0 0 26 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M17.7071 15.7071C18.0976 15.3166 18.0976 14.6834 17.7071 14.2929C17.3166 13.9024 16.6834 13.9024 16.2929 14.2929L17.7071 15.7071ZM13 19L12.2929 18.2929C11.9024 18.6834 11.9024 19.3166 12.2929 19.7071L13 19ZM16.2929 23.7071C16.6834 24.0976 17.3166 24.0976 17.7071 23.7071C18.0976 23.3166 18.0976 22.6834 17.7071 22.2929L16.2929 23.7071ZM19.9359 18.7035L19.8503 17.7072L19.9359 18.7035ZM8.95082 19.9005C9.50243 19.9277 9.97163 19.5025 9.99879 18.9509C10.026 18.3993 9.6008 17.9301 9.04918 17.9029L8.95082 19.9005ZM6.06408 18.7035L5.97851 19.6998L6.06408 18.7035ZM1.07501 13.4958L0.075929 13.5387L1.07501 13.4958ZM1.07501 6.50423L0.0759292 6.46127L1.07501 6.50423ZM6.06409 1.29649L6.14965 2.29282L6.06409 1.29649ZM19.9359 1.29649L19.8503 2.29283L19.9359 1.29649ZM24.925 6.50423L23.9259 6.54718L24.925 6.50423ZM24.925 13.4958L25.9241 13.5387V13.5387L24.925 13.4958ZM16.2929 14.2929L12.2929 18.2929L13.7071 19.7071L17.7071 15.7071L16.2929 14.2929ZM12.2929 19.7071L16.2929 23.7071L17.7071 22.2929L13.7071 18.2929L12.2929 19.7071ZM19.8503 17.7072C17.5929 17.901 15.3081 18 13 18V20C15.3653 20 17.7072 19.8986 20.0215 19.6998L19.8503 17.7072ZM9.04918 17.9029C8.07792 17.8551 7.1113 17.7898 6.14964 17.7072L5.97851 19.6998C6.96438 19.7845 7.95525 19.8515 8.95082 19.9005L9.04918 17.9029ZM2.07408 13.4528C2.02486 12.3081 2 11.157 2 10H0C0 11.1856 0.0254804 12.3654 0.075929 13.5387L2.07408 13.4528ZM2 10C2 8.84302 2.02486 7.69192 2.07408 6.54718L0.0759292 6.46127C0.0254806 7.63461 0 8.81436 0 10H2ZM6.14965 2.29282C8.4071 2.09896 10.6919 2 13 2V0C10.6347 0 8.29281 0.101411 5.97853 0.30016L6.14965 2.29282ZM13 2C15.3081 2 17.5929 2.09896 19.8503 2.29283L20.0215 0.30016C17.7072 0.101411 15.3653 0 13 0V2ZM23.9259 6.54718C23.9751 7.69192 24 8.84302 24 10H26C26 8.81436 25.9745 7.63461 25.9241 6.46127L23.9259 6.54718ZM24 10C24 11.157 23.9751 12.3081 23.9259 13.4528L25.9241 13.5387C25.9745 12.3654 26 11.1856 26 10H24ZM19.8503 2.29283C22.092 2.48534 23.8293 4.29889 23.9259 6.54718L25.9241 6.46127C25.7842 3.20897 23.2653 0.578736 20.0215 0.30016L19.8503 2.29283ZM6.14964 17.7072C3.90797 17.5147 2.17075 15.7011 2.07408 13.4528L0.075929 13.5387C0.215764 16.791 2.7347 19.4213 5.97851 19.6998L6.14964 17.7072ZM2.07408 6.54718C2.17075 4.29889 3.90798 2.48534 6.14965 2.29282L5.97853 0.30016C2.73471 0.578735 0.215764 3.20897 0.0759292 6.46127L2.07408 6.54718ZM20.0215 19.6998C23.2653 19.4213 25.7842 16.791 25.9241 13.5387L23.9259 13.4528C23.8292 15.7011 22.092 17.5147 19.8503 17.7072L20.0215 19.6998Z" fill="#94A3B8"/></svg>';
		?>
		<div id="audio-player" class="audio-player"
			 data-audio="<?php echo get_post_meta( get_the_ID(), 'thim_audio_url', true ); ?>">
			<div class="audio-player__box">
				<div class="audio-player__box__header">
					<div class="audio-player__box__header__image">
						<?php the_post_thumbnail(); ?>
					</div>
					<div class="audio-player__box__header__content">
						<?php the_title( '<h2 class="entry-title">', '</h2>' ); ?>
						<div class="audio-player__box__header__content__duration">
							<?php echo $icon_duration; ?>
							<span class="amplitude-duration-time"></span>
							<span class="duration-text"><?php echo esc_html__( 'Minutes', 'coaching' ) ?></span>
						</div>
						<div class="description"><?php the_excerpt(); ?></div>
					</div>
					<div class="audio-player__box__header__meta">
						<div id="audio-social-share" class="audio-player__box__header__meta__share">
							<?php echo $icon_share; ?>
							<?php do_action( 'thim_social_share' ); ?>
						</div>
						<div class="audio-player__box__header__meta__save" id="song-saved">
							<?php echo $icon_save_audio; ?>
						</div>
					</div>
				</div>
				<div class="audio-player__box__player">
					<input type="range" id="song-percentage-played"
						   class="amplitude-song-slider audio-player__box__player__range" step=".1"/>
					<div class="audio-player__box__player__duration">
						<span class="amplitude-current-time"></span>
						<span class="amplitude-duration-time"></span>
					</div>
				</div>
				<div class="audio-player__box__bottom">
					<div class="audio-player__box__bottom__left">
						<div class="audio-player__box__bottom__volume">
							<?php echo $icon_volume_down; ?>
							<input type="range" class="amplitude-volume-slider"/>
							<?php echo $icon_volume_up; ?>
						</div>
						<div class="amplitude-playback-speed">
							<span class="speed10x"><?php echo esc_html__( 'Speed 1X', 'coaching' ) ?></span>
							<span class="speed15x"><?php echo esc_html__( 'Speed 1.5X', 'coaching' ) ?></span>
							<span class="speed20x"><?php echo esc_html__( 'Speed 2X', 'coaching' ) ?></span>
						</div>
					</div>
					<div class="audio-player__box__bottom__center">
						<div class="amplitude-prev">
							<?php echo $icon_prev; ?>
						</div>
						<div class="amplitude-play-pause">
							<?php echo $icon_play; ?>
							<?php echo $icon_pause; ?>
						</div>
						<div class="amplitude-next">
							<?php echo $icon_next; ?>
						</div>
					</div>
					<div class="audio-player__box__bottom__right">
						<div class="amplitude-repeat-song">
							<?php echo $icon_repeat; ?>
						</div>
						<h4 class="audio-player__box__bottom__author">
							<?php printf( '<a href="%1$s">%2$s</a>',
								esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
								esc_html( get_the_author() )
							); ?>
						</h4>
					</div>
				</div>
			</div>
		</div>
		<?php
	}
}
add_action( 'thim_podcast_mp3', 'thim_podcast_mp3' );

// custom date post audio
if ( ! function_exists( "thim_check_duration_audio" ) ) {
	function thim_check_duration_audio() {
		$thim_audio_mp3 = get_post_meta( get_the_ID(), 'thim_audio_mp3', true );
		$audio          = ! empty( get_post_meta( get_the_ID(), 'thim_audio_url', true ) ) ? get_post_meta( get_the_ID(), 'thim_audio_url', true ) : '';
		if ( has_post_format( 'audio' ) && $thim_audio_mp3 && $audio != '' ) {
			echo '<span class="podcast" data-audio="' . $audio . '">';
			echo '<span class="duration-audio"></span>';
			echo '<span class="duration-text">' . esc_html__( 'Minutes', 'coaching' ) . '</span>';
			echo '</span>';
		} else {
			return get_the_date( get_option( 'date_format' ) );
		}
	}

	add_filter( 'the_date', 'thim_check_duration_audio' );
}

// custom thim ekit widget thumbnail
if ( ! function_exists( "thim_ekit_custom_single_widget_thumbnail" ) ) {
	function thim_ekit_custom_single_widget_thumbnail( $thumbnail_html ) {
		wp_enqueue_script( 'thim-flexslider' );
		switch ( get_post_format() ) {
			case 'image':
			case 'gallery':
			case 'video':
				do_action( 'thim_entry_top', 'full' );
				break;
			case 'audio':
				$thim_audio_mp3 = get_post_meta( get_the_ID(), 'thim_audio_mp3', true );
				if ( $thim_audio_mp3 ) {
					do_action( 'thim_podcast_mp3' );
				} else {
					do_action( 'thim_entry_top', 'full' );
				}
				break;
			default:
				return $thumbnail_html;
		}
	}

	add_filter( 'thim-ekit-single-post-thumbnail', 'thim_ekit_custom_single_widget_thumbnail' );
}

if ( defined( 'THIM_CORE_VERSION' ) && version_compare( THIM_CORE_VERSION, '2.1.3', '<' ) ) {
	// Breadcrumb
	require_once THIM_DIR . 'inc/libs/thim-breadcrumb.php';
}

// add internal css in header
add_filter( 'thim_custom_internal_css', 'thim_custom_internal_css' );
function thim_custom_internal_css() {
	ob_start();
	$body_class = get_theme_mod( 'thim_body_custom_class', false );

	if ( ! empty( $body_class ) ) {
		if ( strpos( $body_class, 'demo-healthy-coaching' ) != false ) {
			echo "\r\n/** CSS Demo Healthy Coaching */\r\n";
			include_once THIM_DIR . "assets/css/libs/demo-healthy-coaching.css";
		}

		if ( $body_class == 'demo-life-coaching' ) {
			echo "\r\n/** CSS Demo Life Coaching */\r\n";
			include_once THIM_DIR . "assets/css/libs/demo-life-coaching.css";
		}

		if ( $body_class == 'business-consulting' ) {
			echo "\r\n/** CSS Demo Business Consulting */\r\n";
			include_once THIM_DIR . "assets/css/libs/business-consulting.css";
		}
		if ( $body_class == 'demo-business-v2' ) {
			echo "\r\n/** CSS Demo Business v2 */\r\n";
			include_once THIM_DIR . "assets/css/libs/demo-business-v2.css";
		}
	}

	$css = ob_get_contents();
	ob_end_clean();

	return $css;
}

if ( ! function_exists( 'thim_customizer_extral_class' ) ) {
	function thim_customizer_extral_class( $type, $rules = array( 'all' ) ) {
		if ( $type == 'archive-post' && count( $rules ) == 1 ) {
			$rules = array( 'all', 'post_categories', 'post_tags', 'post_tags', 'post_search', 'post_term', 'select_post_author' );
		}
		if ( class_exists( '\Thim_EL_Kit\Functions' ) ) {
			$build_el = Functions::instance()->get_conditions_by_type( $type );
			foreach ( $rules as $rule ) {
				if ( ! empty( $build_el[ $rule ] ) ) {
					return ' hidden';
				}
			}
		}

		return;
	}
}
// disable mega menu in thim-core
if ( class_exists( 'Thim_EL_Kit' ) ) {
	$ekits_module_settings = get_option( 'thim_ekits_module_settings' );
	if ( ( ! $ekits_module_settings || ! empty( $ekits_module_settings['megamenu'] ) ) ) {
		add_filter( 'thim-support-mega-menu', '__return_false' );
	}
}
