<?php

namespace Togo_Framework\Payment\Woo;

if (! defined('ABSPATH')) {
	exit;
}

/**
 * Class Metabox
 */
class Checkout
{

	/**
	 * Instance
	 *
	 * @var $instance
	 */
	private static $instance;

	/**
	 * Initiator
	 *
	 * @since 1.0.0
	 * @return object
	 */
	public static function instance()
	{
		if (! isset(self::$instance)) {
			self::$instance = new self();
		}

		return self::$instance;
	}
	/**
	 * Constructor SP_Loader
	 * *******************************************************
	 */
	public function __construct()
	{
		// Include files
		$this->include_files();

		// Override default WooCommerce hooks
		$this->override_hooks();
	}

	function include_files()
	{
		include_once TOGO_FRAMEWORK_PATH . 'inc/payment/woo/classes/class-wc-product.php';
		include_once TOGO_FRAMEWORK_PATH . 'inc/payment/woo/classes/class-wc-data-store.php';
		include_once TOGO_FRAMEWORK_PATH . 'inc/payment/woo/classes/class-wc-order-item.php';
		include_once TOGO_FRAMEWORK_PATH . 'inc/payment/woo/classes/class-wc-order-item-store.php';
	}

	function override_hooks()
	{
		add_filter('woocommerce_product_class', array($this, 'product_class'), 10, 4);
		add_filter('woocommerce_cart_item_class', array($this, 'checkout_item_classes'), 10, 4);
		add_filter('woocommerce_order_item_class', array($this, 'order_item_classes'), 10, 2);
		add_filter('woocommerce_data_stores', array($this, 'data_store_integration'), 10, 1);
		add_filter('woocommerce_product_type_query', array($this, 'data_store_post_type_override'), 10, 2);
		add_filter('woocommerce_order_type_to_group', array($this, 'data_store_post_type_to_order_type_group'), 10, 1);
		add_filter('woocommerce_checkout_create_order_line_item_object', array($this, 'order_line_item_object_override'), 10, 4);
		add_filter('woocommerce_get_order_item_classname', array($this, 'order_item_classname_override'), 10, 3);
		add_filter('woocommerce_get_items_key', array($this, 'order_item_override'), 10, 2);
		add_filter('woocommerce_order_get_items', array($this, 'order_item_types_global_extend'), 10, 3);
		add_filter('woocommerce_order_get_items', array($this, 'convert_order_item_trip_to_product_for_non_default_payment'), 11, 3);
		add_action('woocommerce_admin_order_item_headers', array($this, 'add_order_item_headers'));
		add_action('woocommerce_admin_order_item_values', array($this, 'add_order_item_values'), 11, 3);
		
		// Hiển thị uploaded files trong admin
		add_action('woocommerce_admin_order_data_after_billing_address', array($this, 'display_uploaded_files_in_admin'));
		
		// Hiển thị guest info trong email
		add_action('woocommerce_email_after_order_table', array($this, 'display_guest_info_in_email'), 20, 4);
		
		// Thêm file upload fields
		add_action('woocommerce_checkout_after_customer_details', array($this, 'add_guest_file_upload_fields'));
		add_action('woocommerce_checkout_process', array($this, 'validate_guest_file_uploads'));
		add_action('woocommerce_checkout_update_order_meta', array($this, 'save_guest_file_uploads'), 10, 1);
		add_action('woocommerce_new_order', array($this, 'save_guest_file_uploads_fallback'), 10, 1);

		
		// AJAX file upload handlers
		add_action('wp_ajax_upload_guest_files', array($this, 'ajax_upload_guest_files'));
		add_action('wp_ajax_nopriv_upload_guest_files', array($this, 'ajax_upload_guest_files'));
		add_action('wp_enqueue_scripts', array($this, 'enqueue_checkout_scripts'));
		
		// Cleanup handlers
		add_action('woocommerce_cart_emptied', array($this, 'cleanup_session_files'));
		add_action('wp_logout', array($this, 'cleanup_session_files'));
		add_action('woocommerce_cart_item_removed', array($this, 'cleanup_trip_files'), 10, 2);
		
		// Schedule cleanup for expired sessions
		add_action('init', array($this, 'schedule_file_cleanup'));
		add_action('togo_cleanup_expired_files', array($this, 'cleanup_expired_files'));
		
		// Cleanup files when order status changes
		add_action('woocommerce_order_status_cancelled', array($this, 'cleanup_order_files'));
		add_action('woocommerce_order_status_failed', array($this, 'cleanup_order_files'));
		
		// Plugin deactivation cleanup
		register_deactivation_hook(TOGO_FRAMEWORK_PATH . 'togo-framework.php', array($this, 'plugin_deactivation_cleanup'));

		add_action('customize_controls_enqueue_scripts', array($this, 'enqueue_customizer_scripts'));
	}

	public function supported_post_types()
	{
		return apply_filters('uxper_filter_checkout_integration_post_types', array());
	}

	public function product_class($classname, $product_type, $post_type, $product_id)
	{
		$post_types = $this->supported_post_types();

		if (in_array($post_type, $post_types)) {
			$classname = 'WC_Product_' . $this->transform_class_name($post_type);
		}

		return $classname;
	}

	public function data_store_integration($data_stores)
	{
		$custom_data_stores = array();
		$post_types         = $this->supported_post_types();

		foreach ($post_types as $post_type) {
			$custom_data_stores[$post_type]                 = 'WC_' . $this->transform_class_name($post_type) . '_Data_Store';
			$custom_data_stores['order-item-' . $post_type] = 'Uxper_Woo_Order_Item_' . $this->transform_class_name($post_type) . '_Data_Store';
		}

		return array_merge($data_stores, $custom_data_stores);
	}

	public function checkout_item_classes($classes, $cart_item, $cart_item_key)
	{
		$classes .= ' uxper-product-type-' . get_post_type($cart_item['product_id']);

		return $classes;
	}

	public function order_item_classes($classes, $item)
	{
		$classes .= ' uxper-product-type-' . get_post_type($item['product_id']);

		return $classes;
	}

	public function data_store_post_type_override($classname, $product_id)
	{
		$supported_types = $this->supported_post_types();
		foreach ($supported_types as $supported_type) {
			if ($supported_type == get_post_type($product_id)) {
				return $supported_type;
			}
		}

		return false;
	}

	public function data_store_post_type_to_order_type_group($order_groups)
	{
		$supported_types = $this->supported_post_types();

		foreach ($supported_types as $supported_type) {
			$type_edited                     = str_replace('-', '_', $supported_type);
			$order_groups[$supported_type] = $type_edited . '_lines';
		}

		return $order_groups;
	}

	public function order_line_item_object_override($order_item, $cart_item_key, $values, $order)
	{
		$supported_types = $this->supported_post_types();
		foreach ($supported_types as $supported_type) {
			if (get_post_type($values['product_id']) == $supported_type) {
				$classname  = 'Uxper_Woo_Order_Item_' . $this->transform_class_name($supported_type);
				$order_item = new $classname;
				break;
			}
		}

		return $order_item;
	}

	public function order_item_classname_override($classname, $item_type, $id)
	{
		$supported_types = $this->supported_post_types();

		foreach ($supported_types as $supported_type) {
			if ($item_type == $supported_type) {
				$classname = 'Uxper_Woo_Order_Item_' . $this->transform_class_name($supported_type);
				break;
			}
		}

		return $classname;
	}

	public function order_item_override($line_item, $item)
	{
		$supported_types = $this->supported_post_types();

		foreach ($supported_types as $supported_type) {
			$classname = 'Uxper_Woo_Order_Item_' . $this->transform_class_name($supported_type);

			if (is_a($item, $classname)) {
				$type      = str_replace('-', '_', $supported_type);
				$line_item = $type . '_lines';
				break;
			}
		}

		return $line_item;
	}

	public function order_item_types_global_extend($items, $order, $types)
	{
		if (is_array($types) && sizeof($types) == 1 && in_array('line_item', $types)) {
			$supported_types   = $this->supported_post_types();
			$supported_types[] = 'line_item';
			$items             = $order->get_items($supported_types, true);
		}

		return $items;
	}

	private function transform_class_name($name)
	{
		$name = ucfirst(str_replace('-', '_', $name));
		$name = implode('_', array_map('ucwords', explode('_', $name)));

		return $name;
	}

	/**
	 * Convert Uxper_WC_Order_Item_Room to WC_Order_Item_Prouct to by pass type check Fatal Errof
	 * when using Woocommerce Paypal Payment
	 *
	 * @param WC_Order_Item[] $items Array of WC_Order_Item
	 * @param WC_Order $order Array of WC_Order
	 * @param array $types WC Item types
	 * @return WC_Order_Item[]
	 **/
	public function convert_order_item_trip_to_product_for_non_default_payment($items, $order, $types)
	{
		$is_checkout = is_checkout();
		$is_thankyou = is_wc_endpoint_url('order-received');
		if ($is_checkout && $is_thankyou) {
			return $items;
		}

		if (!isset($_POST['payment_method'])) {
			return $items;
		}

		$is_default_gateway = in_array($_POST['payment_method'], ['bacs', 'cheque', 'cod']);
		if ($is_default_gateway) {
			return $items;
		}

		foreach ($items as $key => $item) {
			if ($item instanceof Uxper_Woo_Order_Item_Togo_Trip) {
				$converted = new WC_Order_Item_Product($item);
				$reflect = new ReflectionClass($converted);
				$set_prop_ref = $reflect->getMethod('set_prop');
				$set_prop_ref->setAccessible(TRUE);
				$set_prop_ref->invokeArgs($converted, ['product_id', $item->get_product_id()]);
				$items[$key] = $converted;
			}
		}

		return $items;
	}

	public function add_order_item_headers()
	{
		echo '<th class="custom-text">' . esc_html__('Details') .  '</th>';
	}

	public function add_order_item_values($product, $item, $item_id)
	{
		$order_id = isset($_GET['id']) ? intval($_GET['id']) : 0;
		$product_id = $product instanceof WC_Product ? $product->get_id() : 0;
		$reservation_data = get_post_meta($order_id, 'trip_order_reservation_data_' . $product_id, true);
		if (!empty($reservation_data)) {
			$booking_date = $reservation_data['booking_date'];
			$trip_id = $reservation_data['trip_id'];
			$pricing_type = $reservation_data['pricing_type'];
			$guests = $reservation_data['guests'];
			$time_type = $reservation_data['time_type'];
			$time = $reservation_data['time'];
			$opening_hours = $reservation_data['opening_hours'];
			$many_days_start_time = $reservation_data['many_days_start_time'];
			$services_without_price = $reservation_data['services_without_price'];
			$date_format = get_option('date_format');
			$pricing_categories = wp_get_post_terms($trip_id, 'togo_trip_pricing_categories');
			if (!empty($reservation_data)) {
				echo '<td class="reservation-detail">';

				if ($time_type == 'start_times') {
					echo '<div class="item">';
					echo \Togo\Icon::get_svg('calendar-check');
					echo '<span class="value">' . date($date_format, strtotime($booking_date)) . '</span>';
					echo '</div>';
					echo '<div class="item">';
					echo \Togo\Icon::get_svg('clock-circle');
					echo '<span class="value">' . \Togo_Framework\Helper::convert24To12($time) . '</span>';
					echo '</div>';
				} elseif ($time_type == 'opening_hours') {
					echo '<div class="item">';
					echo \Togo\Icon::get_svg('calendar-check');
					echo '<span class="value">' . date($date_format, strtotime($booking_date)) . '</span>';
					echo '</div>';
					echo '<div class="item">';
					echo \Togo\Icon::get_svg('clock-circle');
					echo '<span class="value">' . esc_html__('Open at', 'togo-framework') . ' ' . $opening_hours . '</span>';
					echo '</div>';
				} elseif ($time_type == 'many_days') {
					echo '<div class="item">';
					echo \Togo\Icon::get_svg('calendar-check');
					echo '<span class="value">' . date($date_format, strtotime($booking_date)) . '</span>';
					echo '</div>';
					echo '<div class="item">';
					echo \Togo\Icon::get_svg('clock-circle');
					echo '<span class="value">' . esc_html__('Departure at', 'togo-framework') . ' ' . $many_days_start_time . '</span>';
					echo '</div>';
				}

				if (!empty($pricing_categories) && $pricing_type == 'per_person') {
					echo '<div class="item">';
					echo \Togo\Icon::get_svg('users-group');
					echo '<div class="values">';
					foreach ($pricing_categories as $key => $category) {
						echo '<span class="value">' . $guests[$key] . ' ' . esc_html($category->name) . '</span>';
					}
					echo '</div>';
					echo '</div>';
				} elseif (!empty($pricing_categories) && $pricing_type == 'per_group') {
					echo '<div class="item">';
					echo \Togo\Icon::get_svg('users-group');
					echo '<div class="values">';
					echo '<span class="value">' . sprintf(_n('%d guest', '%d guests', $guests[0], 'togo-framework'), $guests[0]) . '</span>';
					echo '</div>';
					echo '</div>';
				}

				if (!empty($services_without_price)) {
					echo '<div class="item">';
					echo \Togo\Icon::get_svg('room-service');
					echo '<div class="values">';
					foreach ($services_without_price as $service) {
						echo '<span class="value">' . esc_html($service) . '</span>';
					}
					echo '</div>';
					echo '</div>';
				}

				echo '</td>';
			}
		}
	}

	/**
	 * Add file upload fields based on the number of guests
	 */
	public function add_guest_file_upload_fields()
	{
		// get guest information from cart
		$cart_items = WC()->cart->get_cart();
		if (empty($cart_items)) {
			return;
		}

		foreach ($cart_items as $cart_item) {
			$product_id = $cart_item['product_id'];
			
			// only process for trip products
			if (get_post_type($product_id) !== 'togo_trip') {
				continue;
			}
			//only process if there is only 1 product in the cart
			if (count($cart_items) > 1) {
				continue;
			}

			//get trip enable customer fields
			$trip_enable_customer_fields = get_post_meta($product_id, 'trip_enable_customer_fields', true);
			if ($trip_enable_customer_fields == '0') {
				continue;
			}

			//get customer fields
			$customer_fields = get_post_meta($product_id, 'trip_customer_fields', true);
			if (empty($customer_fields) || !is_array($customer_fields)) {
				continue;
			}
			
			$get_customer_options = \Togo\Helper::setting('single_trip_customer_fields');

			//get customer fields from get_customer_options if in
			$customer_fields_options = array();
			$customer_fields_options = array_filter($get_customer_options, function($field) use ($customer_fields) {
				return in_array($field['name'], $customer_fields);
			});

			// get reservation data from transient
			$reservation_data = \Togo_Framework\Helper::get_transient_woo_booking($product_id);
			
			if (empty($reservation_data) || empty($reservation_data['guests'])) {
				continue;
			}

			$guests = $reservation_data['guests'];
			$trip_id = $reservation_data['trip_id'];
			$pricing_type = $reservation_data['pricing_type'];
			$pricing_categories = wp_get_post_terms($trip_id, 'togo_trip_pricing_categories');
			
			// calculate total number of guests
			$total_guests = 0;
			if (is_array($guests)) {
				if ($pricing_type == 'per_person' && !empty($pricing_categories)) {
					foreach ($guests as $guest_count) {
						$total_guests += intval($guest_count);
					}
				} elseif ($pricing_type == 'per_group') {
					$total_guests = intval($guests[0]);
				}
			}

			if ($total_guests > 0 && !empty($customer_fields_options)) {
				echo '<div class="guest-information woocommerce-billing-fields">';
				echo '<h3>' . sprintf(__('Guest Information (%d Guests)', 'togo-framework'), $total_guests) . '</h3>';
				echo '<p>' . __('Please provide information each guest.', 'togo-framework') . '</p>';
				echo '<div class="woocommerce-billing-fields__field-wrapper">';
				
				for ($i = 1; $i <= $total_guests; $i++) {
					echo '<div class="guest-section" data-guest="' . $i . '">';
					echo '<h4>' . sprintf(__('Guest %d', 'togo-framework'), $i) . '</h4>';
					
					echo '<div class="woocommerce-billing-fields__field-wrapper">';
					
					// Generate fields dynamically from $customer_fields_options
					$field_count = 0;
					foreach ($customer_fields_options as $field) {
						$field_name = $field['name'];
						$field_label = $field['label'];
						$field_type = isset($field['type']) ? $field['type'] : 'text';
						$required = isset($field['required']) && $field['required'] == 'yes' ? true : false;
						$placeholder = isset($field['placeholder']) ? $field['placeholder'] : '';

						// Determine field width class
						$field_count++;
						$field_class = 'customer-form-row-wide';
						
						$field_id = 'guest_' . $field_name . '_' . $i;
						$field_name_attr = 'guest_' . $field_name . '_' . $i;
						
						echo '<div class="customer-form-row ' . $field_class . '">';
						echo '<label for="' . $field_id . '">' . esc_html($field_label);
						if ($required) {
							echo ' <span class="required">*</span>';
						}
						echo '</label>';
						
						// Generate input based on field type
						if ($field_type === 'text') {
							echo '<input type="text" name="' . $field_name_attr . '" id="' . $field_id . '" class="input-text" placeholder="' . esc_attr($placeholder) . '"';
							if ($required) echo ' required';
							echo ' />';
						}
						elseif ($field_type === 'number') {
							echo '<input type="number" name="' . $field_name_attr . '" id="' . $field_id . '" class="input-text" placeholder="' . esc_attr($placeholder) . '" min="1" max="120"';
							if ($required) echo ' required';
							echo ' />';
						}
						elseif ($field_type === 'email') {
							echo '<input type="email" name="' . $field_name_attr . '" id="' . $field_id . '" class="input-text" placeholder="' . esc_attr($placeholder) . '"';
							if ($required) echo ' required';
							echo ' />';
						}
						elseif ($field_type === 'tel') {
							echo '<input type="tel" name="' . $field_name_attr . '" id="' . $field_id . '" class="input-text" placeholder="' . esc_attr($placeholder) . '"';
							if ($required) echo ' required';
							echo ' />';
						}
						elseif ($field_type === 'select' && isset($field['options'])) {
							echo '<select name="' . $field_name_attr . '" id="' . $field_id . '" class="input-text"';
							if ($required) echo ' required';
							echo '>';
							echo '<option value="">' . __('Select...', 'togo-framework') . '</option>';
							foreach ($field['options'] as $option_value => $option_label) {
								echo '<option value="' . esc_attr($option_value) . '">' . esc_html($option_label) . '</option>';
							}
							echo '</select>';
						}
						elseif ($field_type === 'textarea') {
							echo '<textarea name="' . $field_name_attr . '" id="' . $field_id . '" class="input-text" placeholder="' . esc_attr($placeholder) . '" rows="3"';
							if ($required) echo ' required';
							echo '></textarea>';
						}
						elseif ($field_type === 'file') {
							echo '<input type="file" name="' . $field_name_attr . '" id="' . $field_id . '" accept=".pdf,.jpg,.jpeg,.png" data-required="' . ($required ? 'true' : 'false') . '"';
							if ($required) echo ' required';
							echo ' />';
							echo '<small>' . __('Accepted formats: PDF, JPG, PNG (Max 5MB)', 'togo-framework') . '</small>';
						}
						else {
							// Default to text input
							echo '<input type="text" name="' . $field_name_attr . '" id="' . $field_id . '" class="input-text" placeholder="' . esc_attr($placeholder) . '"';
							if ($required) echo ' required';
							echo ' />';
						}
						
						echo '</div>';
					}
					
					echo '</div>';
					echo '</div>'; // End guest-section
				}
				
				echo '</div>';
				echo '</div>';
			}
			break; // only process the first item
		}
	}

	/**
	 * Validate guest information and file uploads
	 */
	public function validate_guest_file_uploads()
	{
		// Get field options for validation
		$customer_fields_options = $this->get_customer_fields_options();
		
		if (empty($customer_fields_options)) {
			return;
		}

		// validate guest information fields
		$guest_fields = array();
		foreach ($_POST as $key => $value) {
			if (preg_match('/^guest_(.+)_(\d+)$/', $key, $matches)) {
				$field_name = $matches[1];
				$guest_number = $matches[2];
				$guest_fields[$guest_number][$field_name] = $value;
			}
		}

		// validate guest information based on dynamic fields
		foreach ($guest_fields as $guest_number => $guest_data) {
			foreach ($customer_fields_options as $field) {
				$field_name = $field['name'];
				$field_label = $field['label'];
				$field_type = isset($field['type']) ? $field['type'] : 'text';
				$required = isset($field['required']) && $field['required'] == 'yes' ? true : false;
				$field_value = isset($guest_data[$field_name]) ? $guest_data[$field_name] : '';

				// Skip validation if field is not required and empty
				if (!$required && empty($field_value)) {
					continue;
				}

				// Special handling for file fields
				if ($field_type === 'file') {
					// For file fields, check session instead of POST data
					if ($required) {
						$session_key = 'guest_files_' . WC()->session->get_customer_unique_id();
						$session_files = WC()->session->get($session_key, array());
						$file_field_name = 'guest_' . $field_name . '_' . $guest_number;
						
						if (empty($session_files[$file_field_name])) {
							wc_add_notice(sprintf(__('Guest %d: %s is required.', 'togo-framework'), $guest_number, $field_label), 'error');
						}
					}
					continue; // Skip other validation for file fields
				}

				// Required field validation for non-file fields
				if ($required && empty($field_value)) {
					wc_add_notice(sprintf(__('Guest %d: %s is required.', 'togo-framework'), $guest_number, $field_label), 'error');
					continue;
				}

				// Type-specific validation for non-file fields
				if (!empty($field_value)) {
					switch ($field_type) {
						case 'text':
							if (strlen(trim($field_value)) < 2) {
								wc_add_notice(sprintf(__('Guest %d: %s must be at least 2 characters.', 'togo-framework'), $guest_number, $field_label), 'error');
							}
							break;

						case 'number':
							if (!is_numeric($field_value) || $field_value < 1 || $field_value > 120) {
								wc_add_notice(sprintf(__('Guest %d: %s must be a valid number (1-120).', 'togo-framework'), $guest_number, $field_label), 'error');
							}
							break;

						case 'email':
							if (!is_email($field_value)) {
								wc_add_notice(sprintf(__('Guest %d: %s must be a valid email address.', 'togo-framework'), $guest_number, $field_label), 'error');
							}
							break;

						case 'tel':
							// Basic phone validation - can be customized
							if (strlen(preg_replace('/[^0-9+\-\s\(\)]/', '', $field_value)) < 8) {
								wc_add_notice(sprintf(__('Guest %d: %s must be a valid phone number.', 'togo-framework'), $guest_number, $field_label), 'error');
							}
							break;

						case 'select':
							if (isset($field['options']) && !array_key_exists($field_value, $field['options'])) {
								wc_add_notice(sprintf(__('Guest %d: Please select a valid %s.', 'togo-framework'), $guest_number, $field_label), 'error');
							}
							break;

						case 'textarea':
							if (strlen(trim($field_value)) < 10) {
								wc_add_notice(sprintf(__('Guest %d: %s must be at least 10 characters.', 'togo-framework'), $guest_number, $field_label), 'error');
							}
							break;
					}
				}
			}
		}

		// validate file uploads for dynamic file fields
		foreach ($_FILES as $key => $file) {
			if (preg_match('/^guest_(.+)_(\d+)$/', $key, $matches) && !empty($file['name'])) {
				$field_name = $matches[1];
				$guest_number = $matches[2];

				// Find field configuration
				$field_config = null;
				foreach ($customer_fields_options as $field) {
					if ($field['name'] === $field_name && isset($field['type']) && $field['type'] === 'file') {
						$field_config = $field;
						break;
					}
				}

				if (!$field_config) {
					continue; // Skip if not a file field
				}

				$field_label = $field_config['label'];

				// check upload error
				if ($file['error'] !== UPLOAD_ERR_OK) {
					wc_add_notice(sprintf(__('Guest %d: Error uploading %s.', 'togo-framework'), $guest_number, $field_label), 'error');
					continue;
				}

				// check file size (5MB)
				if ($file['size'] > 5 * 1024 * 1024) {
					wc_add_notice(sprintf(__('Guest %d: %s is too large. Maximum size is 5MB.', 'togo-framework'), $guest_number, $field_label), 'error');
				}

				// check file type
				$allowed_types = array('application/pdf', 'image/jpeg', 'image/jpg', 'image/png');
				$file_info = wp_check_filetype($file['name']);
				$file_type = wp_get_image_mime($file['tmp_name']);
				
				if (!$file_type) {
					$file_type = $file['type'];
				}

				if (!in_array($file_type, $allowed_types)) {
					wc_add_notice(sprintf(__('Guest %d: Invalid %s format. Only PDF, JPG, PNG are allowed.', 'togo-framework'), $guest_number, $field_label), 'error');
				}
			}
		}
	}

	/**
	 * Save guest information and uploaded files to order meta
	 */
	public function save_guest_file_uploads($order_id)
	{
		// save guest information first
		$this->save_guest_information($order_id);
		
		// get files from WooCommerce session  
		$session_key = 'guest_files_' . WC()->session->get_customer_unique_id();
		$session_files = WC()->session->get($session_key, array());
		
		if (!empty($session_files)) {
			// there are files in session, use these files
			$uploaded_files = array();
			foreach ($session_files as $key => $file_data) {
				// rename file to include order ID
				$old_path = $file_data['file_path'];
				$file_info = pathinfo($old_path);
				$new_filename = 'order_' . $order_id . '_' . $key . '_' . time() . '.' . $file_info['extension'];
				$new_path = $file_info['dirname'] . '/' . $new_filename;
				
				if (file_exists($old_path) && rename($old_path, $new_path)) {
					$uploaded_files[$key] = $file_data;
					$uploaded_files[$key]['file_path'] = $new_path;
					$uploaded_files[$key]['file_url'] = str_replace(basename($old_path), $new_filename, $file_data['file_url']);
				}
			}
			
			if (!empty($uploaded_files)) {
				update_post_meta($order_id, '_guest_uploaded_files', $uploaded_files);
				
				// Clear session
				WC()->session->__unset($session_key);
			}
			
			return;
		}

		// fallback: process $_FILES (for normal form submission)
		if (!function_exists('wp_handle_upload')) {
			require_once(ABSPATH . 'wp-admin/includes/file.php');
		}

		// check if there is any verification file
		$has_verification_files = false;
		foreach ($_FILES as $key => $file) {
			if (strpos($key, 'guest_verification_') === 0 && !empty($file['name'])) {
				$has_verification_files = true;
				break;
			}
		}

		if (!$has_verification_files) {
			return;
		}

		$uploaded_files = array();

		foreach ($_FILES as $key => $file) {
			if (strpos($key, 'guest_verification_') === 0 && !empty($file['name'])) {
				// check upload error
				if ($file['error'] !== UPLOAD_ERR_OK) {
					continue;
				}

				$upload_overrides = array(
					'test_form' => false,
					'unique_filename_callback' => function($dir, $name, $ext) use ($order_id, $key) {
						return 'order_' . $order_id . '_' . $key . '_' . time() . $ext;
					}
				);

				$uploaded_file = wp_handle_upload($file, $upload_overrides);

				if (!isset($uploaded_file['error'])) {
					$uploaded_files[$key] = array(
						'file_path' => $uploaded_file['file'],
						'file_url' => $uploaded_file['url'],
						'file_name' => $file['name'],
						'file_type' => $file['type'],
						'file_size' => $file['size'],
						'upload_date' => current_time('mysql')
					);
				}
			}
		}

		if (!empty($uploaded_files)) {
			update_post_meta($order_id, '_guest_uploaded_files', $uploaded_files);
		}
	}

	/**
	 * Fallback method to save files if the main method doesn't work
	 */
	public function save_guest_file_uploads_fallback($order_id)
	{
		// only run if there are no files saved
		$existing_files = get_post_meta($order_id, '_guest_uploaded_files', true);
		if (empty($existing_files)) {
			$this->save_guest_file_uploads($order_id);
		}
	}

	/**
	 * Display guest information and files in admin order details
	 */
	public function display_uploaded_files_in_admin($order)
	{
		$guest_information = get_post_meta($order->get_id(), '_guest_information', true);
		$uploaded_files = get_post_meta($order->get_id(), '_guest_uploaded_files', true);
		
		if (!empty($guest_information) || !empty($uploaded_files)) {
			echo '<div class="address">';
			echo '<p><strong>' . __('Guest Information & Documents:', 'togo-framework') . '</strong></p>';
			
			// display guest information
			if (!empty($guest_information)) {
				// Get field options for proper labeling
				$order_id = $order->get_id();
				$customer_fields_options = $this->get_customer_fields_options($order_id);
				$field_labels = array();
				
				foreach ($customer_fields_options as $field) {
					$field_labels[$field['name']] = $field['label'];
				}
				
				foreach ($guest_information as $guest_number => $guest_data) {
					echo '<div style="margin-bottom: 15px; padding: 10px; border: 1px solid #ddd; border-radius: 5px;">';
					echo '<h4>' . sprintf(__('Guest %s', 'togo-framework'), $guest_number) . '</h4>';
					
					// Display all guest data dynamically
					foreach ($guest_data as $field_name => $field_value) {
						if (!empty($field_value)) {
							$field_label = isset($field_labels[$field_name]) ? $field_labels[$field_name] : ucfirst($field_name);
							echo '<p><strong>' . esc_html($field_label) . ':</strong> ' . esc_html($field_value) . '</p>';
						}
					}
					
					// Display uploaded files for this guest
					foreach ($uploaded_files as $file_key => $file_data) {
						if (preg_match('/^guest_(.+)_' . $guest_number . '$/', $file_key, $matches)) {
							$field_name = $matches[1];
							$field_label = isset($field_labels[$field_name]) ? $field_labels[$field_name] : ucfirst($field_name);
							
							echo '<p><strong>' . esc_html($field_label) . ':</strong><br>';
							echo '<a href="' . esc_url($file_data['file_url']) . '" target="_blank">' . esc_html($file_data['file_name']) . '</a><br>';
							echo '<small>' . sprintf(__('Size: %s', 'togo-framework'), size_format($file_data['file_size'])) . '</small></p>';
						}
					}
					
					echo '</div>';
				}
			}
			
			echo '</div>';
		}
	}

	/**
	 * Enqueue scripts cho checkout
	 */
	public function enqueue_checkout_scripts()
	{
		if (is_checkout()) {
			wp_enqueue_script('togo-checkout-upload', TOGO_FRAMEWORK_DIR . 'assets/js/checkout-upload.js', array('jquery'), '1.0.0', true);
			wp_localize_script('togo-checkout-upload', 'togo_checkout', array(
				'ajax_url' => admin_url('admin-ajax.php'),
				'nonce' => wp_create_nonce('togo_upload_nonce'),
				'max_file_size' => 5 * 1024 * 1024, // 5MB
				'allowed_types' => array('application/pdf', 'image/jpeg', 'image/jpg', 'image/png')
			));
		}
	}

	/**
	 * AJAX handler cho upload files
	 */
	public function ajax_upload_guest_files()
	{
		// Verify nonce
		if (!wp_verify_nonce($_POST['nonce'], 'togo_upload_nonce')) {
			wp_die('Security check failed');
		}

		if (!function_exists('wp_handle_upload')) {
			require_once(ABSPATH . 'wp-admin/includes/file.php');
		}

		$response = array('success' => false, 'data' => array());
		$session_key = 'guest_files_' . WC()->session->get_customer_unique_id();

		// Get existing files from session
		$session_files = WC()->session->get($session_key, array());

		foreach ($_FILES as $key => $file) {
			if (preg_match('/^guest_(.+)_(\d+)$/', $key) && !empty($file['name'])) {
				// Validate file
				if ($file['error'] !== UPLOAD_ERR_OK) {
					$response['data'][$key] = array('error' => 'Upload error: ' . $file['error']);
					continue;
				}

				if ($file['size'] > 5 * 1024 * 1024) {
					$response['data'][$key] = array('error' => 'File too large');
					continue;
				}

				$allowed_types = array('application/pdf', 'image/jpeg', 'image/jpg', 'image/png');
				if (!in_array($file['type'], $allowed_types)) {
					$response['data'][$key] = array('error' => 'Invalid file type');
					continue;
				}

				// Upload file
				$upload_overrides = array(
					'test_form' => false,
					'unique_filename_callback' => function($dir, $name, $ext) use ($key) {
						return 'temp_' . str_replace(['guest_', '_'], ['', '_'], $key) . '_' . time() . $ext;
					}
				);
				$uploaded_file = wp_handle_upload($file, $upload_overrides);

				if (!isset($uploaded_file['error'])) {
					$session_files[$key] = array(
						'file_path' => $uploaded_file['file'],
						'file_url' => $uploaded_file['url'],
						'file_name' => $file['name'],
						'file_type' => $file['type'],
						'file_size' => $file['size'],
						'upload_date' => current_time('mysql'),
						'is_temporary' => true
					);
					
					$response['data'][$key] = array('success' => true, 'file_name' => $file['name']);
				} else {
					$response['data'][$key] = array('error' => $uploaded_file['error']);
				}
			}
		}

		// Save to session
		WC()->session->set($session_key, $session_files);
		
		$response['success'] = true;
		wp_send_json($response);
	}

	/**
	 * Save guest information to order meta
	 */
	private function save_guest_information($order_id)
	{
		$guest_information = array();
		
		// Collect guest data from POST
		foreach ($_POST as $key => $value) {
			if (preg_match('/^guest_(.+)_(\d+)$/', $key, $matches)) {
				$field_name = $matches[1];
				$guest_number = $matches[2];
				$guest_information[$guest_number][$field_name] = sanitize_text_field($value);
			}
		}
		
		if (!empty($guest_information)) {
			update_post_meta($order_id, '_guest_information', $guest_information);
		}
	}

	/**
	 * Get customer fields options from cart item or order
	 */
	private function get_customer_fields_options($order_id = null)
	{
		// If we have an order ID (admin context), get from order items
		if ($order_id) {
			$order = wc_get_order($order_id);
			if ($order) {
				foreach ($order->get_items() as $item) {
					$product_id = $item->get_product_id();
					if (get_post_type($product_id) === 'togo_trip') {
						return $this->get_trip_customer_fields_options($product_id);
					}
				}
			}
			return array();
		}

		// Frontend context - get from cart
		if (!function_exists('WC') || !WC() || !WC()->cart) {
			return array();
		}

		$cart_items = WC()->cart->get_cart();
		if (empty($cart_items)) {
			return array();
		}

		foreach ($cart_items as $cart_item) {
			$product_id = $cart_item['product_id'];
			
			if (get_post_type($product_id) === 'togo_trip') {
				return $this->get_trip_customer_fields_options($product_id);
			}
		}

		return array();
	}

	/**
	 * Get customer fields options for a specific trip product
	 */
	private function get_trip_customer_fields_options($product_id)
	{
		// Check if customer fields are enabled
		$trip_enable_customer_fields = get_post_meta($product_id, 'trip_enable_customer_fields', true);
		if ($trip_enable_customer_fields == '0') {
			return array();
		}

		// Get customer fields
		$customer_fields = get_post_meta($product_id, 'trip_customer_fields', true);
		if (empty($customer_fields) || !is_array($customer_fields)) {
			return array();
		}
		
		$get_customer_options = \Togo\Helper::setting('single_trip_customer_fields');

		// Get customer fields from options
		$customer_fields_options = array();
		$customer_fields_options = array_filter($get_customer_options, function($field) use ($customer_fields) {
			return in_array($field['name'], $customer_fields);
		});

		return $customer_fields_options;
	}

	/**
	 * Cleanup session files when cart is emptied or user logs out
	 */
	public function cleanup_session_files()
	{
		if (!WC()->session) {
			return;
		}

		$customer_id = WC()->session->get_customer_unique_id();
		if (!$customer_id) {
			return;
		}

		$session_key = 'guest_files_' . $customer_id;
		$session_files = WC()->session->get($session_key, array());

		if (!empty($session_files)) {
			// Delete physical files
			foreach ($session_files as $file_data) {
				if (isset($file_data['file_path']) && file_exists($file_data['file_path'])) {
					wp_delete_file($file_data['file_path']);
				}
			}

			// Clear session data
			WC()->session->__unset($session_key);
		}
	}

	/**
	 * Cleanup files when specific trip item is removed from cart
	 */
	public function cleanup_trip_files($cart_item_key, $cart)
	{
		$cart_item = $cart->removed_cart_contents[$cart_item_key];
		
		if (isset($cart_item['product_id']) && get_post_type($cart_item['product_id']) === 'togo_trip') {
			// If no more trip items in cart, cleanup all session files
			$has_trip_items = false;
			foreach (WC()->cart->get_cart() as $item) {
				if (get_post_type($item['product_id']) === 'togo_trip') {
					$has_trip_items = true;
					break;
				}
			}

			if (!$has_trip_items) {
				$this->cleanup_session_files();
			}
		}
	}

	/**
	 * Schedule daily cleanup for expired files
	 */
	public function schedule_file_cleanup()
	{
		if (!wp_next_scheduled('togo_cleanup_expired_files')) {
			wp_schedule_event(time(), 'daily', 'togo_cleanup_expired_files');
		}
	}

	/**
	 * Cleanup expired temporary files (older than 24 hours)
	 */
	public function cleanup_expired_files()
	{
		$upload_dir = wp_upload_dir();
		$upload_path = $upload_dir['basedir'];
		
		if (!is_dir($upload_path)) {
			return;
		}

		$files = array_merge(
			glob($upload_path . '/order_*_guest_verification_*'),
			glob($upload_path . '/temp_guest_verification_*')
		);
		$expire_time = 24 * 60 * 60; // 24 hours in seconds

		foreach ($files as $file) {
			if (is_file($file)) {
				$file_time = filemtime($file);
				$current_time = current_time('timestamp');
				
				// Check if file is older than 24 hours and doesn't belong to any order
				if (($current_time - $file_time) > $expire_time) {
					$filename = basename($file);
					
					// Handle temporary files (delete if older than 24 hours)
					if (strpos($filename, 'temp_') === 0) {
						wp_delete_file($file);
					}
					// Handle order files
					elseif (preg_match('/order_(\d+)_/', $filename, $matches)) {
						$order_id = $matches[1];
						
						// Check if order exists and is not completed
						$order = wc_get_order($order_id);
						if (!$order || in_array($order->get_status(), ['failed', 'cancelled', 'pending'])) {
							wp_delete_file($file);
						}
					} else {
						// Unknown file pattern - delete if old
						wp_delete_file($file);
					}
				}
			}
		}

		// Also cleanup session files for inactive sessions
		$this->cleanup_inactive_session_files();
	}

	/**
	 * Cleanup session files for inactive sessions
	 */
	private function cleanup_inactive_session_files()
	{
		global $wpdb;

		// Get all WC session data
		$session_table = $wpdb->prefix . 'woocommerce_sessions';
		
		if ($wpdb->get_var("SHOW TABLES LIKE '$session_table'") != $session_table) {
			return;
		}

		// Find sessions with guest verification data that are expired
		$expired_sessions = $wpdb->get_results($wpdb->prepare("
			SELECT session_key, session_value 
			FROM $session_table 
			WHERE session_value LIKE %s 
			AND session_expiry < %s
		", '%guest_verification_%', current_time('timestamp')));

		foreach ($expired_sessions as $session) {
			$session_data = maybe_unserialize($session->session_value);
			
			if (is_array($session_data)) {
				foreach ($session_data as $key => $value) {
					if (strpos($key, 'guest_verification_') === 0 && is_array($value)) {
						// Delete files from this expired session
						foreach ($value as $file_data) {
							if (isset($file_data['file_path']) && file_exists($file_data['file_path'])) {
								wp_delete_file($file_data['file_path']);
							}
						}
					}
				}
			}
		}

		// Clean up the expired session records
		$wpdb->query($wpdb->prepare("
			DELETE FROM $session_table 
			WHERE session_expiry < %s
		", current_time('timestamp')));
	}

	/**
	 * Cleanup files when order is cancelled or failed
	 */
	public function cleanup_order_files($order_id)
	{
		$uploaded_files = get_post_meta($order_id, '_guest_uploaded_files', true);
		
		if (!empty($uploaded_files)) {
			foreach ($uploaded_files as $file_data) {
				if (isset($file_data['file_path']) && file_exists($file_data['file_path'])) {
					wp_delete_file($file_data['file_path']);
				}
			}
			
			// Remove the meta data as well
			delete_post_meta($order_id, '_guest_uploaded_files');
		}
	}

	/**
	 * Cleanup all temporary files when plugin is deactivated
	 */
	public function plugin_deactivation_cleanup()
	{
		// Clear scheduled event
		wp_clear_scheduled_hook('togo_cleanup_expired_files');
		
		// Clean up all temporary guest verification files
		$upload_dir = wp_upload_dir();
		$upload_path = $upload_dir['basedir'];
		
		if (is_dir($upload_path)) {
			$temp_files = glob($upload_path . '/temp_guest_verification_*');
			foreach ($temp_files as $file) {
				if (is_file($file)) {
					wp_delete_file($file);
				}
			}
		}
		
		// Clean up all session data
		global $wpdb;
		$session_table = $wpdb->prefix . 'woocommerce_sessions';
		
		if ($wpdb->get_var("SHOW TABLES LIKE '$session_table'") == $session_table) {
			$wpdb->query("
				UPDATE $session_table 
				SET session_value = REPLACE(session_value, 'guest_verification_', 'removed_guest_verification_')
				WHERE session_value LIKE '%guest_verification_%'
			");
		}
	}

	public function enqueue_customizer_scripts()
	{
		$inline_js = "
			jQuery(document).ready(function($) {
				jQuery('#customize-control-single_trip_customer_fields .repeater-field-name input').attr('disabled', true);
			});
		";
		wp_add_inline_script('customize-controls', $inline_js);
	}

	/**
	 * Display guest information in order confirmation and admin emails
	 */
	public function display_guest_info_in_email($order, $sent_to_admin, $plain_text, $email)
	{
		// Only show in order confirmation and admin new order emails  
		if (!in_array($email->id, array('customer_processing_order', 'new_order'))) {
			return;
		}

		$order_id = $order->get_id();
		$guest_information = get_post_meta($order_id, '_guest_information', true);
		$guest_files = get_post_meta($order_id, '_guest_uploaded_files', true);

		if (empty($guest_information) && empty($guest_files)) {
			return;
		}

		// Get field options để có label
		$customer_fields_options = $this->get_customer_fields_options($order_id);
		
		if (empty($customer_fields_options)) {
			return;
		}

		if (!$plain_text) {
			// HTML email format
			echo '<div style="margin: 20px 0; border-top: 1px solid #ddd; padding-top: 20px;">';
			echo '<h3 style="margin-bottom: 15px; color: #333;">' . __('Guest Information:', 'togo-framework') . '</h3>';
			
			$guests_data = array();
			foreach ($customer_fields_options as $field) {
				$guests_data[$field['name']] = $field['label'];
			}

			foreach ($guest_information as $guest_number => $guest_data) {
				echo '<div style="margin-bottom: 20px;">';
				echo '<h4 style="margin-bottom: 10px; color: #666;">' . sprintf(__('Guest %d:', 'togo-framework'), $guest_number) . '</h4>';

				echo '<table style="width: 100%; border-collapse: collapse;">';
				// Display all guest data dynamically
				foreach ($guest_data as $field_name => $field_value) {
					if (!empty($field_value)) {
						$field_label = isset($field_labels[$field_name]) ? $field_labels[$field_name] : ucfirst($field_name);
						echo '<tr style="border-bottom: 1px solid #eee;">';
						echo '<td style="padding: 8px 12px; font-weight: bold; width: 30%;">' . esc_html($field_label) . ':</td>';
						echo '<td style="padding: 8px 12px;">' . esc_html($field_value) . '</td>';
						echo '</tr>';
					}
				}
				
				// Display uploaded files for this guest
				foreach ($guest_files as $file_key => $file_data) {
					if (preg_match('/^guest_(.+)_' . $guest_number . '$/', $file_key, $matches)) {
						$field_name = $matches[1];
						$field_label = isset($field_labels[$field_name]) ? $field_labels[$field_name] : ucfirst($field_name);
						echo '<tr style="border-bottom: 1px solid #eee;">';
						echo '<td style="padding: 8px 12px; font-weight: bold; width: 30%;">' . esc_html($field_label) . ':</td>';
						echo '<td style="padding: 8px 12px;">';
						echo '<a href="' . esc_url($file_data['file_url']) . '" target="_blank">' . esc_html($file_data['file_name']) . '</a><br>';
						echo '<small>' . sprintf(__('Size: %s', 'togo-framework'), size_format($file_data['file_size'])) . '</small></p>';
						echo '</td>';
						echo '</tr>';
					}
				}

				echo '</table>';
				
				echo '</div>';
			}
			echo '</div>';
		}
	}
}
