<?php

namespace Togo_Framework;

defined('ABSPATH') || exit;

class Currency_Manager
{
    private static $instance = null;
    private $base_currency = 'USD';
    private $exchange_rates = [];
    private $cache_duration = 3600; // 1 hour

    public static function get_instance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        add_action('init', [$this, 'init']);
        add_action('wp_ajax_togo_switch_currency', [$this, 'ajax_switch_currency']);
        add_action('wp_ajax_nopriv_togo_switch_currency', [$this, 'ajax_switch_currency']);
        
        // Hook to update number formatting based on selected currency
        add_filter('wc_price_args', [$this, 'modify_wc_price_args'], 10, 1);
        
        // Hook to update currency symbol based on selected currency
        add_filter('woocommerce_currency_symbol', [$this, 'modify_woocommerce_currency_symbol'], 10, 2);
    }

    public function init()
    {
        // Set base currency from WooCommerce or theme settings
        if (class_exists('WooCommerce')) {
            $this->base_currency = get_woocommerce_currency();
        } else {
            $this->base_currency = \Togo\Helper::setting('currency', 'USD');
        }
        
        // Load exchange rates
        $this->load_exchange_rates();
    }

    /**
     * Get current selected currency
     */
    public function get_current_currency()
    {
        if (isset($_COOKIE['togo_selected_currency'])) {
            return sanitize_text_field($_COOKIE['togo_selected_currency']);
        }
        return $this->base_currency;
    }

    /**
     * Set current currency
     */
    public function set_current_currency($currency_code)
    {
        $currency_code = strtoupper($currency_code);
        if ($this->is_valid_currency($currency_code)) {
            setcookie('togo_selected_currency', $currency_code, time() + (30 * 24 * 60 * 60), '/');
            return true;
        }
        return false;
    }

    /**
     * Check if currency is valid
     */
    public function is_valid_currency($currency_code)
    {
        // Check WooCommerce currencies first
        if (class_exists('WooCommerce')) {
            $currencies = get_woocommerce_currencies();
            if (isset($currencies[$currency_code])) {
                return true;
            }
        }
        
        // Fallback to Helper currencies
        $currencies = Helper::get_all_currency();
        return isset($currencies[$currency_code]);
    }

    /**
     * Get exchange rate for a currency
     */
    public function get_exchange_rate($from_currency, $to_currency, $manual_rate = null)
    {
        if ($from_currency === $to_currency) {
            return 1;
        }

        // Use manual rate if provided
        if ($manual_rate !== null && $manual_rate > 0) {
            return floatval($manual_rate);
        }

        // Check for stored manual rate
        if ($to_currency === $this->base_currency) {
            $rate_to_base = $this->get_manual_exchange_rate_from_option($from_currency);
            if ($rate_to_base !== null && $rate_to_base > 0) {
                $rate = 1 / $rate_to_base;
                $rate_to_bases = number_format($rate, 5, '.', '');
                return $rate_to_bases;
            }
        }

        $stored_manual_rate = $this->get_manual_exchange_rate_from_option($to_currency);
        if ($stored_manual_rate !== null && $stored_manual_rate > 0) {
            return $stored_manual_rate;
        }

        // Use fallback rates only (no external APIs)
        return $this->get_fallback_rate($from_currency, $to_currency);
    }

    /**
     * Convert price from one currency to another
     */
    public function convert_price($price, $from_currency = null, $to_currency = null, $manual_rate = null)
    {
        if ($from_currency === null) {
            $from_currency = $this->base_currency;
        }
        
        if ($to_currency === null) {
            $to_currency = $this->get_current_currency();
        }

        if ($from_currency === $to_currency) {
            return $price;
        }

        // Ensure price is numeric
        $price = floatval($price);
        if ($price <= 0) {
            return 0;
        }

        $rate = $this->get_exchange_rate($from_currency, $to_currency, $manual_rate);
        $price_converted = round($price * $rate, 5);
        $price_converted = number_format($price_converted, 4, '.', '');

        return $price_converted;
    }

    /**
     * Format price with current currency
     */
    public function format_price($price, $currency_code = null)
    {
        if ($currency_code === null) {
            $currency_code = $this->get_current_currency();
        }

        // Convert price if needed
        $converted_price = $this->convert_price($price, null, $currency_code);

        // Format price directly to avoid infinite loop
        return $this->format_price_direct($converted_price, $currency_code);
    }

    /**
     * Format price directly without calling Helper::togo_format_price
     */
    private function format_price_direct($price, $currency_code)
    {
        if ($price === null || $price === '' || !is_numeric($price)) {
            return '';
        }

        // Ensure price is numeric
        $price = floatval($price);

        // Get currency formatting from widget settings first
        $formatting = $this->get_currency_formatting($currency_code);
        
        // Get currency symbol and position
        if (class_exists('WooCommerce')) {
            // Temporarily change WooCommerce currency for symbol
            $original_currency = get_woocommerce_currency();
            update_option('woocommerce_currency', $currency_code);
            
            $symbol = get_woocommerce_currency_symbol($currency_code);
            $currency_position = get_option('woocommerce_currency_pos', 'left');
            
            // Restore original currency
            update_option('woocommerce_currency', $original_currency);
        } else {
            $symbol = Helper::get_currency_symbol($currency_code);
            $currency_position = \Togo\Helper::setting('currency_position', 'left');
        }

        // Use widget formatting if available, otherwise fallback to WooCommerce/theme settings
        if (!empty($formatting['thousand_separator'])) {
            $currency_thousand_separator = $formatting['thousand_separator'];
        } else {
            if (class_exists('WooCommerce')) {
                $currency_thousand_separator = get_option('woocommerce_thousand_sep', ',');
            } else {
                $currency_thousand_separator = \Togo\Helper::setting('currency_thousand_separator', ',');
            }
        }

        if (!empty($formatting['decimal_separator'])) {
            $currency_decimal_separator = $formatting['decimal_separator'];
        } else {
            if (class_exists('WooCommerce')) {
                $currency_decimal_separator = get_option('woocommerce_decimal_sep', '.');
            } else {
                $currency_decimal_separator = \Togo\Helper::setting('currency_decimal_separator', '.');
            }
        }

        if (isset($formatting['number_of_decimals']) && $formatting['number_of_decimals'] !== '') {
            $currency_number_of_decimals = intval($formatting['number_of_decimals']);
        } else {
            if (class_exists('WooCommerce')) {
                $currency_number_of_decimals = get_option('woocommerce_price_num_decimals', 0);
            } else {
                $currency_number_of_decimals = \Togo\Helper::setting('currency_number_of_decimals') ? \Togo\Helper::setting('currency_number_of_decimals') : 0;
            }
        }

        // Format the price
        $formatted_number = number_format($price, $currency_number_of_decimals, $currency_decimal_separator, $currency_thousand_separator);

        if ($currency_position == 'right') {
            return $formatted_number . $symbol;
        } elseif ($currency_position == 'right_space') {
            return $formatted_number . ' ' . $symbol;
        } elseif ($currency_position == 'left') {
            return $symbol . $formatted_number;
        } elseif ($currency_position == 'left_space') {
            return $symbol . ' ' . $formatted_number;
        }

        return $symbol . $formatted_number; // Default fallback
    }

    /**
     * Load exchange rates from cache or API
     */
    private function load_exchange_rates()
    {
        $cache_key = 'togo_exchange_rates';
        $cached_rates = get_transient($cache_key);

        if ($cached_rates !== false) {
            $this->exchange_rates = $cached_rates;
            return;
        }

        $this->exchange_rates = $this->fetch_all_exchange_rates();
        
        if (!empty($this->exchange_rates)) {
            set_transient($cache_key, $this->exchange_rates, $this->cache_duration);
        }
    }

    /**
     * Fetch exchange rate from fallback rates only
     */
    private function fetch_exchange_rate($from_currency, $to_currency)
    {
        // Only use fallback rates, no external APIs
        return $this->get_fallback_rate($from_currency, $to_currency);
    }

    /**
     * Fetch all exchange rates (fallback rates only)
     */
    private function fetch_all_exchange_rates()
    {
        $rates = [];
        
        // Get currencies from WooCommerce if available
        if (class_exists('WooCommerce')) {
            $currencies = array_keys(get_woocommerce_currencies());
        } else {
            // Fallback to common currencies
            $currencies = ['USD', 'EUR', 'GBP', 'JPY', 'AUD', 'CAD', 'CHF', 'CNY', 'VND'];
        }

        foreach ($currencies as $currency) {
            if ($currency !== $this->base_currency) {
                // Only use fallback rates, no external APIs
                $rate = $this->get_fallback_rate($this->base_currency, $currency);
                if ($rate > 0) {
                    $rates[$currency] = $rate;
                }
            }
        }

        return $rates;
    }


    /**
     * Get fallback exchange rates (manual rates)
     */
    private function get_fallback_rate($from_currency, $to_currency)
    {
        $fallback_rates = [
            'USD' => [
                'EUR' => 0.92,
                'GBP' => 0.79,
                'JPY' => 150.0,
                'AUD' => 1.52,
                'CAD' => 1.35,
                'CHF' => 0.88,
                'CNY' => 7.20,
                'VND' => 25000.0,
                'INR' => 83.0,
                'KRW' => 1300.0,
                'SGD' => 1.35,
                'THB' => 35.0,
                'MYR' => 4.70,
                'IDR' => 15500.0,
                'PHP' => 55.0,
                'BRL' => 5.0,
                'MXN' => 17.0,
                'RUB' => 90.0,
                'ZAR' => 18.0,
                'TRY' => 30.0,
            ],
            'EUR' => [
                'USD' => 1.09,
                'GBP' => 0.86,
                'JPY' => 163.0,
                'AUD' => 1.65,
                'CAD' => 1.47,
                'CHF' => 0.96,
                'CNY' => 7.85,
                'VND' => 26200.0,
                'INR' => 90.5,
                'KRW' => 1417.0,
                'SGD' => 1.47,
                'THB' => 38.2,
                'MYR' => 5.13,
                'IDR' => 16900.0,
                'PHP' => 60.0,
                'BRL' => 5.45,
                'MXN' => 18.5,
                'RUB' => 98.0,
                'ZAR' => 19.6,
                'TRY' => 32.7,
            ],
            'GBP' => [
                'USD' => 1.27,
                'EUR' => 1.16,
                'JPY' => 189.0,
                'AUD' => 1.92,
                'CAD' => 1.71,
                'CHF' => 1.12,
                'CNY' => 9.14,
                'VND' => 30400.0,
                'INR' => 105.0,
                'KRW' => 1647.0,
                'SGD' => 1.71,
                'THB' => 44.4,
                'MYR' => 5.96,
                'IDR' => 19600.0,
                'PHP' => 69.7,
                'BRL' => 6.35,
                'MXN' => 21.6,
                'RUB' => 114.0,
                'ZAR' => 22.8,
                'TRY' => 38.0,
            ],
        ];

        if (isset($fallback_rates[$from_currency][$to_currency])) {
            return $fallback_rates[$from_currency][$to_currency];
        }

        // Try reverse rate
        if (isset($fallback_rates[$to_currency][$from_currency])) {
            return 1 / $fallback_rates[$to_currency][$from_currency];
        }

        return 1;
    }

    /**
     * AJAX handler for currency switching
     */
    public function ajax_switch_currency()
    {
        check_ajax_referer('togo_currency_nonce', 'nonce');

        $currency_code = sanitize_text_field($_POST['currency']);
        $exchange_rate = floatval($_POST['exchange_rate'] ?? 1);
        $thousand_separator = sanitize_text_field($_POST['thousand_separator'] ?? '');
        $decimal_separator = sanitize_text_field($_POST['decimal_separator'] ?? '');
        $number_of_decimals = isset($_POST['number_of_decimals']) && $_POST['number_of_decimals'] !== '' ? intval($_POST['number_of_decimals']) : 2;
        
        if ($this->set_current_currency($currency_code)) {
            // Store manual rate if provided
            if ($exchange_rate > 0) {
                $this->set_manual_exchange_rate($currency_code, $exchange_rate);
            }
            
            // Store currency formatting settings
            $this->set_currency_formatting($currency_code, [
                'thousand_separator' => $thousand_separator,
                'decimal_separator' => $decimal_separator,
                'number_of_decimals' => $number_of_decimals
            ]);
            
            wp_send_json_success([
                'message' => __('Currency switched successfully', 'togo-framework'),
                'currency' => $currency_code,
                'exchange_rate' => $exchange_rate
            ]);
        } else {
            wp_send_json_error([
                'message' => __('Invalid currency code', 'togo-framework')
            ]);
        }
    }


    /**
     * Get all available currencies with exchange rates
     */
    public function get_currencies_with_rates()
    {
        // Get currencies from WooCommerce if available
        if (class_exists('WooCommerce')) {
            $currencies = get_woocommerce_currencies();
        } else {
            $currencies = Helper::get_all_currency();
        }
        
        $current_currency = $this->get_current_currency();
        $result = [];

        foreach ($currencies as $code => $name) {
            $rate = 1;
            if ($code !== $this->base_currency) {
                $rate = $this->get_exchange_rate($this->base_currency, $code);
            }

            $result[$code] = [
                'name' => $name,
                'rate' => $rate,
                'is_current' => $code === $current_currency
            ];
        }

        return $result;
    }

    /**
     * Set currency formatting settings
     */
    public function set_currency_formatting($currency_code, $formatting_settings)
    {
        $currency_code = strtoupper($currency_code);
        $formatting_key = 'togo_currency_formatting_' . $currency_code;
        
        // Store in option instead of transient for better persistence
        update_option($formatting_key, $formatting_settings);
    }

    /**
     * Get currency formatting settings
     */
    public function get_currency_formatting($currency_code)
    {
        $currency_code = strtoupper($currency_code);
        $formatting_key = 'togo_currency_formatting_' . $currency_code;
        
        $formatting = get_option($formatting_key, false);
        
        if ($formatting === false) {
            // Return default formatting if not found
            return [
                'thousand_separator' => '',
                'decimal_separator' => '',
                'number_of_decimals' => 2
            ];
        }
        
        return $formatting;
    }

    /**
     * Get manual exchange rate from widget settings
     */
    public function get_manual_exchange_rate($currency_code, $widget_settings = [])
    {
        if (empty($widget_settings['currency_list'])) {
            return null;
        }

        foreach ($widget_settings['currency_list'] as $currency) {
            if (strtoupper($currency['currency_code']) === strtoupper($currency_code)) {
                if (!empty($currency['exchange_rate'])) {
                    return floatval($currency['exchange_rate']);
                }
            }
        }

        return null;
    }

    /**
     * Convert price with manual rates from widget
     */
    public function convert_price_with_widget($price, $from_currency, $to_currency, $widget_settings = [])
    {
        if ($from_currency === $to_currency) {
            return $price;
        }

        // Try to get manual rate from widget settings
        $manual_rate = $this->get_manual_exchange_rate($to_currency, $widget_settings);
        
        if ($manual_rate !== null) {
            return $price * $manual_rate;
        }

        // Fallback to auto rate
        return $this->convert_price($price, $from_currency, $to_currency);
    }

    /**
     * Set manual exchange rate for a currency
     */
    public function set_manual_exchange_rate($currency_code, $rate)
    {
        $currency_code = strtoupper($currency_code);
        $manual_rates = get_option('togo_manual_exchange_rates', []);
        $manual_rates[$currency_code] = floatval($rate);
        update_option('togo_manual_exchange_rates', $manual_rates);
    }

    /**
     * Get manual exchange rate for a currency
     */
    public function get_manual_exchange_rate_from_option($currency_code)
    {
        $currency_code = strtoupper($currency_code);
        $manual_rates = get_option('togo_manual_exchange_rates', []);
        return isset($manual_rates[$currency_code]) ? floatval($manual_rates[$currency_code]) : null;
    }

    /**
     * Get all available currencies for widget
     */
    public function get_available_currencies()
    {
        if (class_exists('WooCommerce')) {
            return get_woocommerce_currencies();
        } else {
            return Helper::get_all_currency();
        }
    }

    /**
     * Get currency symbol for a specific currency
     */
    public function get_currency_symbol_for_currency($currency_code)
    {
        if (class_exists('WooCommerce')) {
            return get_woocommerce_currency_symbol($currency_code);
        } else {
            return Helper::get_currency_symbol($currency_code);
        }
    }

    /**
     * Get base currency
     */
    public function get_base_currency()
    {
        return $this->base_currency;
    }

    /**
     * Modify WooCommerce price arguments based on selected currency
     */
    public function modify_wc_price_args($args)
    {
        // Only apply when selected currency is different from default currency
        $current_currency = $this->get_current_currency();
        if ($current_currency === $this->base_currency) {
            return $args;
        }

        // Get formatting from currency formatting settings
        $formatting = $this->get_currency_formatting($current_currency);
        
        // Update thousand separator
        if (!empty($formatting['thousand_separator'])) {
            $args['thousand_separator'] = $formatting['thousand_separator'];
        }
        
        // Update decimal separator
        if (!empty($formatting['decimal_separator'])) {
            $args['decimal_separator'] = $formatting['decimal_separator'];
        }
        
        // Update number of decimals
        if (isset($formatting['number_of_decimals']) && $formatting['number_of_decimals'] !== '') {
            $args['decimals'] = intval($formatting['number_of_decimals']);
        }

        return $args;
    }

    /**
     * Modify WooCommerce currency symbol based on selected currency
     */
    public function modify_woocommerce_currency_symbol($currency_symbol, $currency_code)
    {
        // Only apply when selected currency is different from default currency
        $current_currency = $this->get_current_currency();
        if ($current_currency === $this->base_currency) {
            return $currency_symbol;
        }

        // Prevent infinite loop by checking if we're already processing this currency
        static $processing = false;
        if ($processing) {
            return $currency_symbol;
        }

        // Only modify if the requested currency matches our selected currency
        if ($currency_code !== $current_currency) {
            $processing = true;
            
            // Get currency symbol from WooCommerce or Helper
            if (class_exists('WooCommerce')) {
                $new_symbol = get_woocommerce_currency_symbol($current_currency);
            } else {
                $new_symbol = Helper::get_currency_symbol($current_currency);
            }
            
            $processing = false;
            return $new_symbol;
        }

        return $currency_symbol;
    }

}
