<?php
/**
 * Theme Functions
 *
 * @package Torfa
 * @author NeuronThemes
 * @link http://neuronthemes.com
 */

/**
 * Global Variables
 * 
 * Defining global variables to make
 * usage easier.
 */
define('TORFA_THEME_DIR', get_template_directory());
define('TORFA_THEME_URI', get_template_directory_uri());
define('TORFA_THEME_STYLESHEET', get_stylesheet_uri());
define('TORFA_THEME_PLACEHOLDER', get_template_directory_uri() . '/assets/images/placeholder.png');
define('TORFA_THEME_NAME', 'torfa');
define('TORFA_THEME_VERSION', wp_get_theme()->get('Version'));

/**
 * Content Width
 * 
 * Maximum content width is set at 1920,
 * larger images or videos will be cropped
 * to that resolution.
 */
!isset($content_width) ? $content_width = 1920 : '';

/**
 * Text Domain
 * 
 * Makes theme available for translation,
 * translations can be found in the /languages/ directory.
 */
load_theme_textdomain('torfa', TORFA_THEME_DIR . '/languages');

// Action to call init
add_action('after_setup_theme', 'torfa_init');

/**
 * Init
 * 
 * Global function which adds theme support,
 * register nav menus and call actions for
 * different php, js and css files.
 */
function torfa_init() {
    // Theme Support
	add_theme_support('post-thumbnails');
	add_theme_support('automatic-feed-links');
	add_theme_support('title-tag');

	/**
	 * WooCommerce Theme Support
	 * 
	 * Theme fully supports plugin WooCommerce
	 * also it's features in single product
	 * as zoom, lightbox and slider.
	 */
	if (class_exists('WooCommerce')) {
		add_theme_support('woocommerce');
		add_theme_support('wc-product-gallery-zoom');
		add_theme_support('wc-product-gallery-lightbox');
		add_theme_support('wc-product-gallery-slider');
	}

	// Image Sizes
	$torfa_general_image_sizes = get_theme_mod('general_image_sizes');
	if ($torfa_general_image_sizes) {
		$index = 1;
		foreach ($torfa_general_image_sizes as $image_size) {
			add_image_size('torfa_image_size_' . $index, isset($image_size['image_size_width']) ? $image_size['image_size_width'] : '', isset($image_size['image_size_height']) ? $image_size['image_size_height'] : 9999, true);
			$index++;
		}
	}
    
	// Include custom files
	include(TORFA_THEME_DIR . '/includes/functions/neuron-functions.php');
    include(TORFA_THEME_DIR . '/includes/functions/style-functions.php');
	include(TORFA_THEME_DIR . '/includes/admin/extra.php');
	include_once(TORFA_THEME_DIR . '/includes/tgm/class-tgm-plugin-activation.php');
	include_once(TORFA_THEME_DIR . '/includes/admin/acf/fields.php');
	get_theme_mod('custom_fields_panel', '2') == '2' ? define('ACF_LITE' , true) : '';

    // Theme actions within init function
    add_action('tgmpa_register', 'torfa_plugins');
    add_action('wp_enqueue_scripts', 'torfa_external_css');
    add_action('wp_enqueue_scripts', 'torfa_external_js');
    add_action('admin_enqueue_scripts', 'torfa_add_extra_scripts');
	add_action('widgets_init', 'torfa_widgets_init');

	// Theme Filters
	if (class_exists('Kirki')){
		add_filter('kirki_telemetry', '__return_false');
	}
    
    // Register Menus
	register_nav_menus(
		array(
			'main-menu' => esc_html__('Main Menu', 'torfa')
		)
	);
}

/**
 * TGMPA
 * 
 * An addon which helps theme to install
 * and activate different plugins.
 */
function torfa_plugins() {
    $plugins = array(
        array(
            'name'      => esc_html__('Advanced Custom Fields', 'torfa'),
            'slug'      => 'advanced-custom-fields',
            'required'  => true
        ),
        array(
			'name'      => esc_html__('Elementor', 'torfa'),
            'slug'      => 'elementor',
            'required'  => true
        ),
        array(
			'name'        => esc_html__('Neuron Core', 'torfa'),
            'slug'        => 'neuron-core-torfa',
			'source'      => 'https://cdn.neuronthemes.com/plugins/neuron-core-torfa.zip',
		    'required'    => true
		),
        array(
			'name'        => esc_html__('Slider Revolution', 'torfa'),
            'slug'        => 'revslider',
			'source'      => 'https://cdn.neuronthemes.com/plugins/revslider.zip',
		    'required'    => false
		),
        array(
            'name'      => esc_html__('WooCommerce', 'torfa'),
            'slug'      => 'woocommerce',
            'required'  => false
        ),
        array(
            'name'       => esc_html__('One Click Demo Import', 'torfa'),
            'slug'       => 'one-click-demo-import',
            'required'   => false
		),
        array(
            'name'       => esc_html__('Contact Form 7', 'torfa'),
            'slug'       => 'contact-form-7',
            'required'   => false
        )
    );
    $config = array(
        'id'           => 'tgmpa',
        'default_path' => '',
        'menu'         => 'tgmpa-install-plugins',
        'parent_slug'  => 'themes.php',
        'capability'   => 'edit_theme_options',
        'has_notices'  => true,
        'dismissable'  => true,
        'dismiss_msg'  => '',
        'is_automatic' => false,
        'message'      => ''
    );
    tgmpa($plugins, $config);
}

// External CSS
function torfa_external_css() {
    wp_enqueue_style('torfa-main-style', TORFA_THEME_URI . '/assets/styles/torfa.css', false, TORFA_THEME_VERSION, null);
    wp_enqueue_style('magnific-popup', TORFA_THEME_URI . '/assets/styles/magnific-popup.css', false, TORFA_THEME_VERSION, null);
    wp_enqueue_style('owl-carousel', TORFA_THEME_URI . '/assets/styles/owl.carousel.min.css', false, TORFA_THEME_VERSION, null);
    wp_enqueue_style('perfect-scrollbar', TORFA_THEME_URI . '/assets/styles/perfect-scrollbar.css', false, TORFA_THEME_VERSION, null);
	wp_enqueue_style('torfa-wp-style', TORFA_THEME_STYLESHEET);
	wp_enqueue_style('torfa-fonts', torfa_fonts_url(), array(), TORFA_THEME_VERSION);
	
	// Custom Style and Fonts
	if (function_exists('neuron_color_lightness') && function_exists('neuron_hexToRgb')) {
		wp_add_inline_style('torfa-wp-style', torfa_custom_style());
	}
	wp_add_inline_style('torfa-wp-style', torfa_body_offset());
}

// External Javascript
function torfa_external_js() {
	if (!is_admin()) {
		wp_enqueue_script('isotope', TORFA_THEME_URI . '/assets/scripts/isotope.pkgd.min.js', array('jquery'), TORFA_THEME_VERSION, TRUE);
		wp_enqueue_script('packery-mode', TORFA_THEME_URI . '/assets/scripts/packery-mode.pkgd.min.js', array('jquery'), TORFA_THEME_VERSION, TRUE);
		wp_enqueue_script('magnific-popup', TORFA_THEME_URI . '/assets/scripts/jquery.magnific-popup.min.js', array('jquery'), TORFA_THEME_VERSION, TRUE);
		wp_enqueue_script('owl-carousel', TORFA_THEME_URI . '/assets/scripts/owl.carousel.min.js', array('jquery'), TORFA_THEME_VERSION, TRUE);
		wp_enqueue_script('typed', TORFA_THEME_URI . '/assets/scripts/typed.min.js', array('jquery'), TORFA_THEME_VERSION, TRUE);
		wp_enqueue_script('wow', TORFA_THEME_URI . '/assets/scripts/wow.min.js', array('jquery'), TORFA_THEME_VERSION, TRUE);
		wp_enqueue_script('theia-sticky-sidebar', TORFA_THEME_URI . '/assets/scripts/theia-sticky-sidebar.js', array('jquery'), TORFA_THEME_VERSION, TRUE);
		wp_enqueue_script('headroom', TORFA_THEME_URI . '/assets/scripts/headroom.js', array('jquery'), TORFA_THEME_VERSION, TRUE);
		wp_enqueue_script('headroom-zepto', TORFA_THEME_URI . '/assets/scripts/jQuery.headroom.js', array('jquery'), TORFA_THEME_VERSION, TRUE);
		wp_enqueue_script('perfect-scrollbar', TORFA_THEME_URI . '/assets/scripts/perfect-scrollbar.min.js', array('jquery'), TORFA_THEME_VERSION, TRUE);
		wp_enqueue_script('torfa-scripts', TORFA_THEME_URI . '/assets/scripts/torfa.js', array('jquery'), TORFA_THEME_VERSION, TRUE);

        is_singular() ? wp_enqueue_script('comment-reply') : '';
	}
}

// Enqueue Extra Scripts
function torfa_add_extra_scripts() {
	wp_enqueue_style('torfa-admin-style', TORFA_THEME_URI . '/includes/admin/style.css', false, TORFA_THEME_VERSION, null);
	wp_enqueue_script('torfa-admin-script', TORFA_THEME_URI . '/includes/admin/script.js', array('jquery'), TORFA_THEME_VERSION, TRUE);
}

// Init Widgets
function torfa_widgets_init() {
	$torfa_sidebars = [
		[
			'name' => __('Main Sidebar', 'torfa'),
			'description' => __('Widgets on this sidebar are displayed in Blog Page.', 'torfa'),
			'id' => 'main-sidebar'
		],
		[
			'name' => __('Shop Sidebar', 'torfa'),
			'description' => __('Widgets on this sidebar are displayed in Shop Pages.', 'torfa'),
			'id' => 'shop-sidebar',
			'condition' => class_exists('WooCommerce')
		],
		[
			'name' => __('Footer Sidebar 1', 'torfa'),
			'description' => __('Widgets on this sidebar are placed on the first column of footer.', 'torfa'),
			'id' => 'sidebar-footer-1'
		],
		[
			'name' => __('Footer Sidebar 2', 'torfa'),
			'description' => __('Widgets on this sidebar are placed on the second column of footer.', 'torfa'),
			'id' => 'sidebar-footer-2'
		],
		[
			'name' => __('Footer Sidebar 3', 'torfa'),
			'description' => __('Widgets on this sidebar are placed on the third column of footer.', 'torfa'),
			'id' => 'sidebar-footer-3'
		],
		[
			'name' => __('Footer Sidebar 4', 'torfa'),
			'description' => __('Widgets on this sidebar are placed on the fourth column of footer.', 'torfa'),
			'id' => 'sidebar-footer-4'
		],
		[
			'name' => __('Footer Sidebar 5', 'torfa'),
			'description' => __('Widgets on this sidebar are placed on the fifth column of footer.', 'torfa'),
			'id' => 'sidebar-footer-5'
		],
		[
			'name' => __('Footer Sidebar 6', 'torfa'),
			'description' => __('Widgets on this sidebar are placed on the sixth column of footer.', 'torfa'),
			'id' => 'sidebar-footer-6'
		],
		[
			'name' => __('Sliding Bar Sidebar', 'torfa'),
			'description' => __('Widgets on this sidebar are placed on the sliding bar of header.', 'torfa'),
			'id' => 'sliding-bar',
		],
	];

	if (get_theme_mod('general_sidebars')) {
		foreach (get_theme_mod('general_sidebars') as $sidebar) {
			$torfa_sidebars[] = [
				'name' => esc_html__($sidebar['sidebar_title'], 'torfa'),
				'description' => esc_html__($sidebar['sidebar_description'], 'torfa'),
				'id' => strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $sidebar['sidebar_title']))),
			];
		}
	}

	foreach ($torfa_sidebars as $sidebar) {
		$sidebar['condition'] = isset($sidebar['condition']) ? $sidebar['condition'] : true;

		if ($sidebar['condition'] == false) {
			continue;
		}

		register_sidebar(
			[
				'name' => esc_html($sidebar['name']),
				'description' => esc_html($sidebar['description']),
				'id' => esc_attr($sidebar['id']),
				'before_widget' => '<div id="%1$s" class="widget %2$s">',
				'after_widget'  => '</div>',
				'before_title'  => '<div class="widgettitle-wrapper"><h4 class="widgettitle">',
				'after_title'   => '</h4></div>'
			]
		);
	}
}

/**
 * Mega Menu Classes
 * 
 * Add classes to the menu item when
 * mega menu option is clicked.
 */
add_filter('wp_nav_menu_objects', 'torfa_mega_menu_class', 10, 2);
function torfa_mega_menu_class($items, $args) {
	foreach ($items as $item) {
		// Activate
		if (get_field('mega_menu', $item)) {
			$item->classes[] = 'm-mega-menu';
		}

		// Columns
		switch (get_field('mega_menu_columns', $item)) {
			case '1':
				$item->classes[] = 'm-mega-menu--two';
				break;
			case '2':
				$item->classes[] = 'm-mega-menu--three';
				break;
			case '3':
				$item->classes[] = 'm-mega-menu--four';
				break;
			case '4':
				$item->classes[] = 'm-mega-menu--five';
				break;
		}

		// Unclickable
		if (get_field('menu_unclickable', $item)) {
			$item->classes[] = 'disabled';
		}

		// Label
		if (get_field('menu_label', $item) == '2') {
			$item->classes[] = 'a-menu-badge a-menu-badge--new';
		} elseif (get_field('menu_label', $item) == '3') {
			$item->classes[] = 'a-menu-badge a-menu-badge--hot';
		}
	}
	return $items;
}

/**
 * Remove Mega Menu Classes
 * 
 * Remove clases from the menu
 * items, useful for builder.
 */
function torfa_remove_mega_menu_class($items, $args) {
	foreach ($items as $item) {
		foreach($item->classes as $key => $class) {
			if(strpos($class, 'm-mega-menu') !== false) {
				unset($item->classes[$key]);
			}
		}
	}
	return $items;
}

/**
 * Rewrite the ACF functions incase ACF fails to activate
 */
if (!function_exists('get_field') && !is_admin() && !function_exists('get_sub_field')) {
	function get_field($field_id, $post_id = null) {
		return null;
	}

	function get_sub_field($field_id, $post_id = null){
		return null;
	}
}

/**
 * Portfolio Arguments
 * 
 * Rewrite the url of portfolio
 * and remove the archive page.
 */
add_filter('portfolioposttype_args', 'torfa_change_portfolio_labels');
function torfa_change_portfolio_labels(array $args) {

	if (get_theme_mod('portfolio_prefix')) {
		$args['rewrite'] = array('slug' => get_theme_mod('portfolio_prefix'));
	}

    $args['has_archive'] = false;

	return $args;
}

/**
 * WooCommerce Placeholder
 */
add_filter('woocommerce_placeholder_img_src', 'torfa_woocommerce_placeholder_img_src');
function torfa_woocommerce_placeholder_img_src($src) {
	$src = TORFA_THEME_PLACEHOLDER;
	 
	return $src;
}

/**
 * Register Fonts
 */
function torfa_fonts_url() {
	$font_url = '';
	if ('off' !== _x('on', 'Google font: on or off', 'torfa')) {
		$font_url = add_query_arg('family', urlencode('Poppins:300,500,600,700'), '//fonts.googleapis.com/css');
	}
	return $font_url;
}

/**
 * Custom Template
 */
function torfa_get_custom_template($id) {
	if (!class_exists('Elementor\Plugin')) {
		return;
	}

	if (empty($id)) {
		return;
	}

	$content = \Elementor\Plugin::instance()->frontend->get_builder_content_for_display($id, true);

	return $content;
}

/**
 * Body Offset
 */
function torfa_body_offset() {
	if (torfa_inherit_option('general_body_offset', 'body_offset', '2') == '2' || is_search())  {
		return '';
	}

	$torfa_offset_output = [];
	$torfa_body_offset_padding = [
		'theme-options' => get_theme_mod('body_offset_padding'),
		'acf' => [
			'padding-left' => get_field('general_body_offset_padding_left', get_queried_object()),
			'padding-right' => get_field('general_body_offset_padding_right', get_queried_object())
		]
	];

	if (get_field('general_body_offset', get_queried_object()) == '2') {
		$torfa_body_offset_values = $torfa_body_offset_padding['acf'];
	} else {
		$torfa_body_offset_values = $torfa_body_offset_padding['theme-options'];
	}

	$torfa_offset_output[] = isset($torfa_body_offset_values['padding-left']) && $torfa_body_offset_values['padding-left'] != 0 ? 'padding-left:' . $torfa_body_offset_values['padding-left'] : '';
	$torfa_offset_output[] = isset($torfa_body_offset_values['padding-right']) && $torfa_body_offset_values['padding-right'] != 0 ? 'padding-right:' . $torfa_body_offset_values['padding-right'] : ''; 

	// Offset Breakpoint
	if (torfa_inherit_option('general_body_offset_breakpoint', 'body_offset_breakpoint', '1') == '1') {
		$torfa_offset_media_query = '1039px';
	} else {
		$torfa_offset_media_query = '745px';
	}

	return $torfa_offset_output ? '@media (min-width: '. $torfa_offset_media_query .'){ body, .l-primary-header--sticky .l-primary-header, .l-primary-footer--parallax {' . implode('; ', $torfa_offset_output) . '}}' : '';
}

/**
 * Demo Importer
 * 
 * Import the content, widgets and
 * the customizer settings via the
 * plugin one click demo importer.
 */
add_filter('pt-ocdi/import_files', 'torfa_ocdi_import_files');
function torfa_ocdi_import_files() {
	return array(
		array(
			'import_file_name'           => esc_html__('Main Demo', 'torfa'),
			'categories'                 => array('Demo'),
			'import_file_url'            => 'https://neuronthemes.com/torfa/demo-importer/content.xml',
			'import_widget_file_url'     => 'https://neuronthemes.com/torfa/demo-importer/widgets.json',
			'import_customizer_file_url' => 'https://neuronthemes.com/torfa/demo-importer/customizer.dat',
			'import_notice'              => esc_html__('Everything that is listed in our demo will be imported.', 'torfa'),
		),
		array(
			'import_file_name'           => esc_html__('Header Templates', 'torfa'),
			'categories'                 => array('Templates'),
			'import_file_url'            => 'https://neuronthemes.com/torfa/demo-importer/header-templates.xml',
			'import_preview_image_url'   => TORFA_THEME_URI . '/assets/images/screenshot-headers.png',
			'import_notice'              => esc_html__('Only the Header Templates will be imported.', 'torfa'),
		),
	);
}

/**
 * After Import Setup
 * 
 * Set the Classic Home Page as front
 * page and assign the menu to 
 * the main menu location.
 */
add_action('pt-ocdi/after_import', 'torfa_ocdi_after_import_setup');
function torfa_ocdi_after_import_setup() {
	$main_menu = get_term_by('name', 'Main Menu', 'nav_menu');

	if ($main_menu) {
		set_theme_mod('nav_menu_locations', array('main-menu' => $main_menu->term_id));
	}

	$front_page_id = get_page_by_title('Corporate Agency');
	if ($front_page_id) {
		update_option('page_on_front', $front_page_id->ID);
		update_option('show_on_front', 'page');
	}	
	$blog_page_id = get_page_by_title('Blog');
	if ($blog_page_id) {
		update_option('page_for_posts', $blog_page_id->ID);
	}
}

/**
 * Custom Body Classes
 */
function torfa_custom_body_classes($classes) {
    /**
     * Theme Borders
     */
    if (torfa_inherit_option('theme_borders', 'theme_borders', '2') == '1' && apply_filters('torfa_display_theme_borders', true)) {
        $classes[] = 'h-theme-borders';
    }

    /**
     * Parallax Footer
     */
    if (torfa_inherit_option('footer_parallax', 'footer_parallax', '2') == '1') {
        $classes[] = 'h-parallax-footer';
    }

    return $classes;
}
add_filter('body_class', 'torfa_custom_body_classes');

/**
 * WooCommerce Product Gallery
 * 
 * Changes the image sizes from
 * thumbnail to medium for sharper
 * resolution.
 */
add_filter('woocommerce_gallery_thumbnail_size', 'torfa_woocommerce_gallery_thumbnail_size');
function torfa_woocommerce_gallery_thumbnail_size() {
	return 'medium';
}
