/**
 * Wp Alpha Core FrameWork
 * Alpha Animated Text Library
 * 
 * @package Wp Alpha Core FrameWork
 * @version 1.0
 */
'use strict';

/**
 * http://animejs.com
 * JavaScript animation engine
 * @version v2.2.0
 * @author Julian Garnier
 * @copyright ©2017 Julian Garnier
 * Released under the MIT license
**/

(function (root, factory) {
    if (typeof define === 'function' && define.amd) {
        // AMD. Register as an anonymous module.
        define([], factory);
    } else if (typeof module === 'object' && module.exports) {
        // Node. Does not work with strict CommonJS, but
        // only CommonJS-like environments that support module.exports,
        // like Node.
        module.exports = factory();
    } else {
        // Browser globals (root is window)
        root.anime = factory();
    }
}(this, () => {

    // Defaults

    const defaultInstanceSettings = {
        update: undefined,
        begin: undefined,
        run: undefined,
        complete: undefined,
        loop: 1,
        direction: 'normal',
        autoplay: true,
        offset: 0
    }

    const defaultTweenSettings = {
        duration: 1000,
        delay: 0,
        easing: 'easeOutElastic',
        elasticity: 500,
        round: 0
    }

    const validTransforms = ['translateX', 'translateY', 'translateZ', 'rotate', 'rotateX', 'rotateY', 'rotateZ', 'scale', 'scaleX', 'scaleY', 'scaleZ', 'skewX', 'skewY', 'perspective'];
    let transformString;

    // Utils

    function stringContains(str, text) {
        return str.indexOf(text) > -1;
    }

    const is = {
        arr: a => Array.isArray(a),
        obj: a => stringContains(Object.prototype.toString.call(a), 'Object'),
        pth: a => is.obj(a) && a.hasOwnProperty('totalLength'),
        svg: a => a instanceof SVGElement,
        dom: a => a.nodeType || is.svg(a),
        str: a => typeof a === 'string',
        fnc: a => typeof a === 'function',
        und: a => typeof a === 'undefined',
        hex: a => /(^#[0-9A-F]{6}$)|(^#[0-9A-F]{3}$)/i.test(a),
        rgb: a => /^rgb/.test(a),
        hsl: a => /^hsl/.test(a),
        col: a => (is.hex(a) || is.rgb(a) || is.hsl(a))
    }

    // BezierEasing https://github.com/gre/bezier-easing

    const bezier = (() => {

        const kSplineTableSize = 11;
        const kSampleStepSize = 1.0 / (kSplineTableSize - 1.0);

        function A(aA1, aA2) { return 1.0 - 3.0 * aA2 + 3.0 * aA1 };
        function B(aA1, aA2) { return 3.0 * aA2 - 6.0 * aA1 };
        function C(aA1) { return 3.0 * aA1 };

        function calcBezier(aT, aA1, aA2) { return ((A(aA1, aA2) * aT + B(aA1, aA2)) * aT + C(aA1)) * aT };
        function getSlope(aT, aA1, aA2) { return 3.0 * A(aA1, aA2) * aT * aT + 2.0 * B(aA1, aA2) * aT + C(aA1) };

        function binarySubdivide(aX, aA, aB, mX1, mX2) {
            let currentX, currentT, i = 0;
            do {
                currentT = aA + (aB - aA) / 2.0;
                currentX = calcBezier(currentT, mX1, mX2) - aX;
                if (currentX > 0.0) { aB = currentT } else { aA = currentT };
            } while (Math.abs(currentX) > 0.0000001 && ++i < 10);
            return currentT;
        }

        function newtonRaphsonIterate(aX, aGuessT, mX1, mX2) {
            for (let i = 0; i < 4; ++i) {
                const currentSlope = getSlope(aGuessT, mX1, mX2);
                if (currentSlope === 0.0) return aGuessT;
                const currentX = calcBezier(aGuessT, mX1, mX2) - aX;
                aGuessT -= currentX / currentSlope;
            }
            return aGuessT;
        }

        function bezier(mX1, mY1, mX2, mY2) {

            if (!(0 <= mX1 && mX1 <= 1 && 0 <= mX2 && mX2 <= 1)) return;
            let sampleValues = new Float32Array(kSplineTableSize);

            if (mX1 !== mY1 || mX2 !== mY2) {
                for (let i = 0; i < kSplineTableSize; ++i) {
                    sampleValues[i] = calcBezier(i * kSampleStepSize, mX1, mX2);
                }
            }

            function getTForX(aX) {

                let intervalStart = 0.0;
                let currentSample = 1;
                const lastSample = kSplineTableSize - 1;

                for (; currentSample !== lastSample && sampleValues[currentSample] <= aX; ++currentSample) {
                    intervalStart += kSampleStepSize;
                }

                --currentSample;

                const dist = (aX - sampleValues[currentSample]) / (sampleValues[currentSample + 1] - sampleValues[currentSample]);
                const guessForT = intervalStart + dist * kSampleStepSize;
                const initialSlope = getSlope(guessForT, mX1, mX2);

                if (initialSlope >= 0.001) {
                    return newtonRaphsonIterate(aX, guessForT, mX1, mX2);
                } else if (initialSlope === 0.0) {
                    return guessForT;
                } else {
                    return binarySubdivide(aX, intervalStart, intervalStart + kSampleStepSize, mX1, mX2);
                }

            }

            return x => {
                if (mX1 === mY1 && mX2 === mY2) return x;
                if (x === 0) return 0;
                if (x === 1) return 1;
                return calcBezier(getTForX(x), mY1, mY2);
            }

        }

        return bezier;

    })();

    const easings = (() => {

        const names = ['Quad', 'Cubic', 'Quart', 'Quint', 'Sine', 'Expo', 'Circ', 'Back', 'Elastic'];

        // Elastic easing adapted from jQueryUI http://api.jqueryui.com/easings/

        function elastic(t, p) {
            return t === 0 || t === 1 ? t :
                -Math.pow(2, 10 * (t - 1)) * Math.sin((((t - 1) - (p / (Math.PI * 2.0) * Math.asin(1))) * (Math.PI * 2)) / p);
        }

        // Approximated Penner equations http://matthewlein.com/ceaser/

        const equations = {
            In: [
                [0.550, 0.085, 0.680, 0.530], /* InQuad */
                [0.550, 0.055, 0.675, 0.190], /* InCubic */
                [0.895, 0.030, 0.685, 0.220], /* InQuart */
                [0.755, 0.050, 0.855, 0.060], /* InQuint */
                [0.470, 0.000, 0.745, 0.715], /* InSine */
                [0.950, 0.050, 0.795, 0.035], /* InExpo */
                [0.600, 0.040, 0.980, 0.335], /* InCirc */
                [0.600, -0.280, 0.735, 0.045], /* InBack */
                elastic /* InElastic */
            ], Out: [
                [0.250, 0.460, 0.450, 0.940], /* OutQuad */
                [0.215, 0.610, 0.355, 1.000], /* OutCubic */
                [0.165, 0.840, 0.440, 1.000], /* OutQuart */
                [0.230, 1.000, 0.320, 1.000], /* OutQuint */
                [0.390, 0.575, 0.565, 1.000], /* OutSine */
                [0.190, 1.000, 0.220, 1.000], /* OutExpo */
                [0.075, 0.820, 0.165, 1.000], /* OutCirc */
                [0.175, 0.885, 0.320, 1.275], /* OutBack */
                (t, f) => 1 - elastic(1 - t, f) /* OutElastic */
            ], InOut: [
                [0.455, 0.030, 0.515, 0.955], /* InOutQuad */
                [0.645, 0.045, 0.355, 1.000], /* InOutCubic */
                [0.770, 0.000, 0.175, 1.000], /* InOutQuart */
                [0.860, 0.000, 0.070, 1.000], /* InOutQuint */
                [0.445, 0.050, 0.550, 0.950], /* InOutSine */
                [1.000, 0.000, 0.000, 1.000], /* InOutExpo */
                [0.785, 0.135, 0.150, 0.860], /* InOutCirc */
                [0.680, -0.550, 0.265, 1.550], /* InOutBack */
                (t, f) => t < .5 ? elastic(t * 2, f) / 2 : 1 - elastic(t * -2 + 2, f) / 2 /* InOutElastic */
            ]
        }

        let functions = {
            linear: bezier(0.250, 0.250, 0.750, 0.750)
        }

        for (let type in equations) {
            equations[type].forEach((f, i) => {
                functions['ease' + type + names[i]] = is.fnc(f) ? f : bezier.apply(this, f);
            });
        }

        return functions;

    })();

    // Strings

    function stringToHyphens(str) {
        return str.replace(/([a-z])([A-Z])/g, '$1-$2').toLowerCase();
    }

    function selectString(str) {
        if (is.col(str)) return;
        try {
            let nodes = document.querySelectorAll(str);
            return nodes;
        } catch (e) {
            return;
        }
    }

    // Arrays

    function filterArray(arr, callback) {
        const len = arr.length;
        const thisArg = arguments.length >= 2 ? arguments[1] : void 0;
        let result = [];
        for (let i = 0; i < len; i++) {
            if (i in arr) {
                const val = arr[i];
                if (callback.call(thisArg, val, i, arr)) {
                    result.push(val);
                }
            }
        }
        return result;
    }

    function flattenArray(arr) {
        return arr.reduce((a, b) => a.concat(is.arr(b) ? flattenArray(b) : b), []);
    }

    function toArray(o) {
        if (is.arr(o)) return o;
        if (is.str(o)) o = selectString(o) || o;
        if (o instanceof NodeList || o instanceof HTMLCollection) return [].slice.call(o);
        return [o];
    }

    function arrayContains(arr, val) {
        return arr.some(a => a === val);
    }

    // Objects

    function cloneObject(o) {
        let clone = {};
        for (let p in o) clone[p] = o[p];
        return clone;
    }

    function replaceObjectProps(o1, o2) {
        let o = cloneObject(o1);
        for (let p in o1) o[p] = o2.hasOwnProperty(p) ? o2[p] : o1[p];
        return o;
    }

    function mergeObjects(o1, o2) {
        let o = cloneObject(o1);
        for (let p in o2) o[p] = is.und(o1[p]) ? o2[p] : o1[p];
        return o;
    }

    // Colors

    function rgbToRgba(rgbValue) {
        const rgb = /rgb\((\d+,\s*[\d]+,\s*[\d]+)\)/g.exec(rgbValue);
        return rgb ? `rgba(${rgb[1]},1)` : rgbValue;
    }

    function hexToRgba(hexValue) {
        const rgx = /^#?([a-f\d])([a-f\d])([a-f\d])$/i;
        const hex = hexValue.replace(rgx, (m, r, g, b) => r + r + g + g + b + b);
        const rgb = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
        const r = parseInt(rgb[1], 16);
        const g = parseInt(rgb[2], 16);
        const b = parseInt(rgb[3], 16);
        return `rgba(${r},${g},${b},1)`;
    }

    function hslToRgba(hslValue) {
        const hsl = /hsl\((\d+),\s*([\d.]+)%,\s*([\d.]+)%\)/g.exec(hslValue) || /hsla\((\d+),\s*([\d.]+)%,\s*([\d.]+)%,\s*([\d.]+)\)/g.exec(hslValue);
        const h = parseInt(hsl[1]) / 360;
        const s = parseInt(hsl[2]) / 100;
        const l = parseInt(hsl[3]) / 100;
        const a = hsl[4] || 1;
        function hue2rgb(p, q, t) {
            if (t < 0) t += 1;
            if (t > 1) t -= 1;
            if (t < 1 / 6) return p + (q - p) * 6 * t;
            if (t < 1 / 2) return q;
            if (t < 2 / 3) return p + (q - p) * (2 / 3 - t) * 6;
            return p;
        }
        let r, g, b;
        if (s == 0) {
            r = g = b = l;
        } else {
            const q = l < 0.5 ? l * (1 + s) : l + s - l * s;
            const p = 2 * l - q;
            r = hue2rgb(p, q, h + 1 / 3);
            g = hue2rgb(p, q, h);
            b = hue2rgb(p, q, h - 1 / 3);
        }
        return `rgba(${r * 255},${g * 255},${b * 255},${a})`;
    }

    function colorToRgb(val) {
        if (is.rgb(val)) return rgbToRgba(val);
        if (is.hex(val)) return hexToRgba(val);
        if (is.hsl(val)) return hslToRgba(val);
    }

    // Units

    function getUnit(val) {
        const split = /([\+\-]?[0-9#\.]+)(%|px|pt|em|rem|in|cm|mm|ex|ch|pc|vw|vh|vmin|vmax|deg|rad|turn)?$/.exec(val);
        if (split) return split[2];
    }

    function getTransformUnit(propName) {
        if (stringContains(propName, 'translate') || propName === 'perspective') return 'px';
        if (stringContains(propName, 'rotate') || stringContains(propName, 'skew')) return 'deg';
    }

    // Values

    function minMaxValue(val, min, max) {
        return Math.min(Math.max(val, min), max);
    }

    function getFunctionValue(val, animatable) {
        if (!is.fnc(val)) return val;
        return val(animatable.target, animatable.id, animatable.total);
    }

    function getCSSValue(el, prop) {
        if (prop in el.style) {
            return getComputedStyle(el).getPropertyValue(stringToHyphens(prop)) || '0';
        }
    }

    function getAnimationType(el, prop) {
        if (is.dom(el) && arrayContains(validTransforms, prop)) return 'transform';
        if (is.dom(el) && (el.getAttribute(prop) || (is.svg(el) && el[prop]))) return 'attribute';
        if (is.dom(el) && (prop !== 'transform' && getCSSValue(el, prop))) return 'css';
        if (el[prop] != null) return 'object';
    }

    function getTransformValue(el, propName) {
        const defaultUnit = getTransformUnit(propName);
        const defaultVal = stringContains(propName, 'scale') ? 1 : 0 + defaultUnit;
        const str = el.style.transform;
        if (!str) return defaultVal;
        let match = [];
        let props = [];
        let values = [];
        const rgx = /(\w+)\((.+?)\)/g;
        while (match = rgx.exec(str)) {
            props.push(match[1]);
            values.push(match[2]);
        }
        const value = filterArray(values, (val, i) => props[i] === propName);
        return value.length ? value[0] : defaultVal;
    }

    function getOriginalTargetValue(target, propName) {
        switch (getAnimationType(target, propName)) {
            case 'transform': return getTransformValue(target, propName);
            case 'css': return getCSSValue(target, propName);
            case 'attribute': return target.getAttribute(propName);
        }
        return target[propName] || 0;
    }

    function getRelativeValue(to, from) {
        const operator = /^(\*=|\+=|-=)/.exec(to);
        if (!operator) return to;
        const u = getUnit(to) || 0;
        const x = parseFloat(from);
        const y = parseFloat(to.replace(operator[0], ''));
        switch (operator[0][0]) {
            case '+': return x + y + u;
            case '-': return x - y + u;
            case '*': return x * y + u;
        }
    }

    function validateValue(val, unit) {
        if (is.col(val)) return colorToRgb(val);
        const originalUnit = getUnit(val);
        const unitLess = originalUnit ? val.substr(0, val.length - originalUnit.length) : val;
        return unit && !/\s/g.test(val) ? unitLess + unit : unitLess;
    }

    // getTotalLength() equivalent for circle, rect, polyline, polygon and line shapes. 
    // adapted from https://gist.github.com/SebLambla/3e0550c496c236709744

    function getDistance(p1, p2) {
        return Math.sqrt(Math.pow(p2.x - p1.x, 2) + Math.pow(p2.y - p1.y, 2));
    }

    function getCircleLength(el) {
        return 2 * Math.PI * el.getAttribute('r');
    }

    function getRectLength(el) {
        return (el.getAttribute('width') * 2) + (el.getAttribute('height') * 2);
    }

    function getLineLength(el) {
        return getDistance(
            { x: el.getAttribute('x1'), y: el.getAttribute('y1') },
            { x: el.getAttribute('x2'), y: el.getAttribute('y2') }
        );
    }

    function getPolylineLength(el) {
        const points = el.points;
        let totalLength = 0;
        let previousPos;
        for (let i = 0; i < points.numberOfItems; i++) {
            const currentPos = points.getItem(i);
            if (i > 0) totalLength += getDistance(previousPos, currentPos);
            previousPos = currentPos;
        }
        return totalLength;
    }

    function getPolygonLength(el) {
        const points = el.points;
        return getPolylineLength(el) + getDistance(points.getItem(points.numberOfItems - 1), points.getItem(0));
    }

    // Path animation

    function getTotalLength(el) {
        if (el.getTotalLength) return el.getTotalLength();
        switch (el.tagName.toLowerCase()) {
            case 'circle': return getCircleLength(el);
            case 'rect': return getRectLength(el);
            case 'line': return getLineLength(el);
            case 'polyline': return getPolylineLength(el);
            case 'polygon': return getPolygonLength(el);
        }
    }

    function setDashoffset(el) {
        const pathLength = getTotalLength(el);
        el.setAttribute('stroke-dasharray', pathLength);
        return pathLength;
    }

    // Motion path

    function getPath(path, percent) {
        const el = is.str(path) ? selectString(path)[0] : path;
        const p = percent || 100;
        return function (prop) {
            return {
                el: el,
                property: prop,
                totalLength: getTotalLength(el) * (p / 100)
            }
        }
    }

    function getPathProgress(path, progress) {
        function point(offset = 0) {
            const l = progress + offset >= 1 ? progress + offset : 0;
            return path.el.getPointAtLength(l);
        }
        const p = point();
        const p0 = point(-1);
        const p1 = point(+1);
        switch (path.property) {
            case 'x': return p.x;
            case 'y': return p.y;
            case 'angle': return Math.atan2(p1.y - p0.y, p1.x - p0.x) * 180 / Math.PI;
        }
    }

    // Decompose value

    function decomposeValue(val, unit) {
        const rgx = /-?\d*\.?\d+/g;
        const value = validateValue((is.pth(val) ? val.totalLength : val), unit) + '';
        return {
            original: value,
            numbers: value.match(rgx) ? value.match(rgx).map(Number) : [0],
            strings: (is.str(val) || unit) ? value.split(rgx) : []
        }
    }

    // Animatables

    function parseTargets(targets) {
        const targetsArray = targets ? (flattenArray(is.arr(targets) ? targets.map(toArray) : toArray(targets))) : [];
        return filterArray(targetsArray, (item, pos, self) => self.indexOf(item) === pos);
    }

    function getAnimatables(targets) {
        const parsed = parseTargets(targets);
        return parsed.map((t, i) => {
            return { target: t, id: i, total: parsed.length };
        });
    }

    // Properties

    function normalizePropertyTweens(prop, tweenSettings) {
        let settings = cloneObject(tweenSettings);
        if (is.arr(prop)) {
            const l = prop.length;
            const isFromTo = (l === 2 && !is.obj(prop[0]));
            if (!isFromTo) {
                // Duration divided by the number of tweens
                if (!is.fnc(tweenSettings.duration)) settings.duration = tweenSettings.duration / l;
            } else {
                // Transform [from, to] values shorthand to a valid tween value
                prop = { value: prop };
            }
        }
        return toArray(prop).map((v, i) => {
            // Default delay value should be applied only on the first tween
            const delay = !i ? tweenSettings.delay : 0;
            // Use path object as a tween value
            let obj = is.obj(v) && !is.pth(v) ? v : { value: v };
            // Set default delay value
            if (is.und(obj.delay)) obj.delay = delay;
            return obj;
        }).map(k => mergeObjects(k, settings));
    }

    function getProperties(instanceSettings, tweenSettings, params) {
        let properties = [];
        const settings = mergeObjects(instanceSettings, tweenSettings);
        for (let p in params) {
            if (!settings.hasOwnProperty(p) && p !== 'targets') {
                properties.push({
                    name: p,
                    offset: settings['offset'],
                    tweens: normalizePropertyTweens(params[p], tweenSettings)
                });
            }
        }
        return properties;
    }

    // Tweens

    function normalizeTweenValues(tween, animatable) {
        let t = {};
        for (let p in tween) {
            let value = getFunctionValue(tween[p], animatable);
            if (is.arr(value)) {
                value = value.map(v => getFunctionValue(v, animatable));
                if (value.length === 1) value = value[0];
            }
            t[p] = value;
        }
        t.duration = parseFloat(t.duration);
        t.delay = parseFloat(t.delay);
        return t;
    }

    function normalizeEasing(val) {
        return is.arr(val) ? bezier.apply(this, val) : easings[val];
    }

    function normalizeTweens(prop, animatable) {
        let previousTween;
        return prop.tweens.map(t => {
            let tween = normalizeTweenValues(t, animatable);
            const tweenValue = tween.value;
            const originalValue = getOriginalTargetValue(animatable.target, prop.name);
            const previousValue = previousTween ? previousTween.to.original : originalValue;
            const from = is.arr(tweenValue) ? tweenValue[0] : previousValue;
            const to = getRelativeValue(is.arr(tweenValue) ? tweenValue[1] : tweenValue, from);
            const unit = getUnit(to) || getUnit(from) || getUnit(originalValue);
            tween.from = decomposeValue(from, unit);
            tween.to = decomposeValue(to, unit);
            tween.start = previousTween ? previousTween.end : prop.offset;
            tween.end = tween.start + tween.delay + tween.duration;
            tween.easing = normalizeEasing(tween.easing);
            tween.elasticity = (1000 - minMaxValue(tween.elasticity, 1, 999)) / 1000;
            tween.isPath = is.pth(tweenValue);
            tween.isColor = is.col(tween.from.original);
            if (tween.isColor) tween.round = 1;
            previousTween = tween;
            return tween;
        });
    }

    // Tween progress

    const setTweenProgress = {
        css: (t, p, v) => t.style[p] = v,
        attribute: (t, p, v) => t.setAttribute(p, v),
        object: (t, p, v) => t[p] = v,
        transform: (t, p, v, transforms, id) => {
            if (!transforms[id]) transforms[id] = [];
            transforms[id].push(`${p}(${v})`);
        }
    }

    // Animations

    function createAnimation(animatable, prop) {
        const animType = getAnimationType(animatable.target, prop.name);
        if (animType) {
            const tweens = normalizeTweens(prop, animatable);
            return {
                type: animType,
                property: prop.name,
                animatable: animatable,
                tweens: tweens,
                duration: tweens[tweens.length - 1].end,
                delay: tweens[0].delay
            }
        }
    }

    function getAnimations(animatables, properties) {
        return filterArray(flattenArray(animatables.map(animatable => {
            return properties.map(prop => {
                return createAnimation(animatable, prop);
            });
        })), a => !is.und(a));
    }

    // Create Instance

    function getInstanceTimings(type, animations, instanceSettings, tweenSettings) {
        const isDelay = (type === 'delay');
        if (animations.length) {
            return (isDelay ? Math.min : Math.max).apply(Math, animations.map(anim => anim[type]));
        } else {
            return isDelay ? tweenSettings.delay : instanceSettings.offset + tweenSettings.delay + tweenSettings.duration;
        }
    }

    function createNewInstance(params) {
        const instanceSettings = replaceObjectProps(defaultInstanceSettings, params);
        const tweenSettings = replaceObjectProps(defaultTweenSettings, params);
        const animatables = getAnimatables(params.targets);
        const properties = getProperties(instanceSettings, tweenSettings, params);
        const animations = getAnimations(animatables, properties);
        return mergeObjects(instanceSettings, {
            children: [],
            animatables: animatables,
            animations: animations,
            duration: getInstanceTimings('duration', animations, instanceSettings, tweenSettings),
            delay: getInstanceTimings('delay', animations, instanceSettings, tweenSettings)
        });
    }

    // Core

    let activeInstances = [];
    let raf = 0;

    const engine = (() => {
        function play() { raf = requestAnimationFrame(step); };
        function step(t) {
            const activeLength = activeInstances.length;
            if (activeLength) {
                let i = 0;
                while (i < activeLength) {
                    if (activeInstances[i]) activeInstances[i].tick(t);
                    i++;
                }
                play();
            } else {
                cancelAnimationFrame(raf);
                raf = 0;
            }
        }
        return play;
    })();


    // Public Instance

    function anime(params = {}) {

        let now, startTime, lastTime = 0;

        let resolve = null;

        function makePromise() {
            return window.Promise && new Promise(_resolve => resolve = _resolve);
        }

        let promise = makePromise();

        let instance = createNewInstance(params);

        function toggleInstanceDirection() {
            instance.reversed = !instance.reversed;
        }

        function adjustTime(time) {
            return instance.reversed ? instance.duration - time : time;
        }

        function syncInstanceChildren(time) {
            const children = instance.children;
            const childrenLength = children.length;
            if (time >= instance.currentTime) {
                for (let i = 0; i < childrenLength; i++) children[i].seek(time);
            } else {
                for (let i = childrenLength; i--;) children[i].seek(time);
            }
        }

        function setAnimationsProgress(insTime) {
            let i = 0;
            let transforms = {};
            const animations = instance.animations;
            const animationsLength = animations.length;
            while (i < animationsLength) {
                const anim = animations[i];
                const animatable = anim.animatable;
                const tweens = anim.tweens;
                const tweenLength = tweens.length - 1;
                let tween = tweens[tweenLength];
                // Only check for keyframes if there is more than one tween
                if (tweenLength) tween = filterArray(tweens, t => (insTime < t.end))[0] || tween;
                const elapsed = minMaxValue(insTime - tween.start - tween.delay, 0, tween.duration) / tween.duration;
                const eased = isNaN(elapsed) ? 1 : tween.easing(elapsed, tween.elasticity);
                const strings = tween.to.strings;
                const round = tween.round;
                let numbers = [];
                let progress;
                const toNumbersLength = tween.to.numbers.length;
                for (let n = 0; n < toNumbersLength; n++) {
                    let value;
                    const toNumber = tween.to.numbers[n];
                    const fromNumber = tween.from.numbers[n];
                    if (!tween.isPath) {
                        value = fromNumber + (eased * (toNumber - fromNumber));
                    } else {
                        value = getPathProgress(tween.value, eased * toNumber);
                    }
                    if (round) {
                        if (!(tween.isColor && n > 2)) {
                            value = Math.round(value * round) / round;
                        }
                    }
                    numbers.push(value);
                }
                // Manual Array.reduce for better performances
                const stringsLength = strings.length;
                if (!stringsLength) {
                    progress = numbers[0];
                } else {
                    progress = strings[0];
                    for (let s = 0; s < stringsLength; s++) {
                        const a = strings[s];
                        const b = strings[s + 1];
                        const n = numbers[s];
                        if (!isNaN(n)) {
                            if (!b) {
                                progress += n + ' ';
                            } else {
                                progress += n + b;
                            }
                        }
                    }
                }
                if ((animatable.target.closest('.animating-text-rise') && anim.property == 'translateY' && progress != '50%') ||
                    (animatable.target.closest('.animating-text-fall') && anim.property == 'translateY' && progress != '-50%')) {
                    if (animatable.target.closest('.animating-text-rise') && !animatable.target.classList.contains('text-rise')) {
                        animatable.target.classList.add('text-rise');
                        animatable.target.style.transform = '';
                        animatable.target.style.animationDuration = tween.duration + 'ms';
                    } if (animatable.target.closest('.animating-text-fall') && !animatable.target.classList.contains('text-fall')) {
                        animatable.target.classList.add('text-fall');
                        animatable.target.style.transform = '';
                        animatable.target.style.animationDuration = tween.duration + 'ms';
                    }
                } else {
                    setTweenProgress[anim.type](animatable.target, anim.property, progress, transforms, animatable.id);
                    anim.currentValue = progress;
                }
                i++;
            }
            const transformsLength = Object.keys(transforms).length;
            if (transformsLength) {
                for (let id = 0; id < transformsLength; id++) {
                    if (typeof transforms[id] != 'undefined') {
                        if (!transformString) {
                            const t = 'transform';
                            transformString = (getCSSValue(document.body, t) ? t : `-webkit-${t}`);
                        }
                        instance.animatables[id].target.style[transformString] = transforms[id].join(' ');
                    }
                }
            }
            instance.currentTime = insTime;
            instance.progress = (insTime / instance.duration) * 100;
        }

        function setCallback(cb) {
            if (instance[cb]) instance[cb](instance);
        }

        function countIteration() {
            if (instance.remaining && instance.remaining !== true) {
                instance.remaining--;
            }
        }

        function setInstanceProgress(engineTime) {
            const insDuration = instance.duration;
            const insOffset = instance.offset;
            const insStart = insOffset + instance.delay;
            const insCurrentTime = instance.currentTime;
            const insReversed = instance.reversed;
            const insTime = adjustTime(engineTime);
            if (instance.children.length) syncInstanceChildren(insTime);
            if (insTime >= insStart || !insDuration) {
                if (!instance.began) {
                    instance.began = true;
                    setCallback('begin');
                }
                setCallback('run');
            }
            if (insTime > insOffset && insTime < insDuration) {
                setAnimationsProgress(insTime);
            } else {
                if (insTime <= insOffset && insCurrentTime !== 0) {
                    setAnimationsProgress(0);
                    if (insReversed) countIteration();
                }
                if ((insTime >= insDuration && insCurrentTime !== insDuration) || !insDuration) {
                    setAnimationsProgress(insDuration);
                    if (!insReversed) countIteration();
                }
            }
            setCallback('update');
            if (engineTime >= insDuration) {
                if (instance.remaining) {
                    startTime = now;
                    if (instance.direction === 'alternate') toggleInstanceDirection();
                } else {
                    instance.pause();
                    if (!instance.completed) {
                        instance.completed = true;
                        setCallback('complete');
                        if ('Promise' in window) {
                            resolve();
                            promise = makePromise();
                        }
                    }
                }
                lastTime = 0;
            }
        }

        instance.reset = function () {
            const direction = instance.direction;
            const loops = instance.loop;
            instance.currentTime = 0;
            instance.progress = 0;
            instance.paused = true;
            instance.began = false;
            instance.completed = false;
            instance.reversed = direction === 'reverse';
            instance.remaining = direction === 'alternate' && loops === 1 ? 2 : loops;
            setAnimationsProgress(0);
            for (let i = instance.children.length; i--;) {
                instance.children[i].reset();
            }
            for (let i = 0; i < instance.animatables.length; i++) {
                instance.animatables[i].target.classList.remove('text-rise', 'text-fall');
            }
        }

        instance.tick = function (t) {
            now = t;
            if (!startTime) startTime = now;
            const engineTime = (lastTime + now - startTime) * anime.speed;
            setInstanceProgress(engineTime);
        }

        instance.seek = function (time) {
            setInstanceProgress(adjustTime(time));
        }

        instance.pause = function () {
            const i = activeInstances.indexOf(instance);
            if (i > -1) activeInstances.splice(i, 1);
            instance.paused = true;
        }

        instance.play = function () {
            if (!instance.paused) return;
            instance.paused = false;
            startTime = 0;
            lastTime = adjustTime(instance.currentTime);
            activeInstances.push(instance);
            if (!raf) engine();
        }

        instance.reverse = function () {
            toggleInstanceDirection();
            startTime = 0;
            lastTime = adjustTime(instance.currentTime);
        }

        instance.restart = function () {
            instance.pause();
            instance.reset();
            instance.play();
        }

        instance.finished = promise;

        instance.reset();

        if (instance.autoplay) instance.play();

        return instance;

    }

    // Remove targets from animation

    function removeTargets(targets) {
        const targetsArray = parseTargets(targets);
        for (let i = activeInstances.length; i--;) {
            const instance = activeInstances[i];
            const animations = instance.animations;
            for (let a = animations.length; a--;) {
                if (arrayContains(targetsArray, animations[a].animatable.target)) {
                    animations.splice(a, 1);
                    if (!animations.length) instance.pause();
                }
            }
        }
    }

    // Timeline

    function timeline(params) {
        let tl = anime(params);
        tl.pause();
        tl.duration = 0;
        tl.add = function (instancesParams) {
            tl.children.forEach(i => { i.began = true; i.completed = true; });
            toArray(instancesParams).forEach(instanceParams => {
                let insParams = mergeObjects(instanceParams, replaceObjectProps(defaultTweenSettings, params || {}));
                insParams.targets = insParams.targets || params.targets;
                const tlDuration = tl.duration;
                const insOffset = insParams.offset;
                insParams.autoplay = false;
                insParams.direction = tl.direction;
                insParams.offset = is.und(insOffset) ? tlDuration : getRelativeValue(insOffset, tlDuration);
                tl.began = true;
                tl.completed = true;
                tl.seek(insParams.offset);
                const ins = anime(insParams);
                ins.began = true;
                ins.completed = true;
                if (ins.duration > tlDuration) tl.duration = ins.duration;
                tl.children.push(ins);
            });
            tl.seek(0);
            tl.reset();
            if (tl.autoplay) tl.restart();
            return tl;
        }
        return tl;
    }

    anime.version = '2.2.0';
    anime.speed = 1;
    anime.running = activeInstances;
    anime.remove = removeTargets;
    anime.getValue = getOriginalTargetValue;
    anime.path = getPath;
    anime.setDashoffset = setDashoffset;
    anime.bezier = bezier;
    anime.easings = easings;
    anime.timeline = timeline;
    anime.random = (min, max) => Math.floor(Math.random() * (max - min + 1)) + min;

    return anime;

}));

/*
* anime.js v3.2.1
* (c) 2020 Julian Garnier
* Released under the MIT license
* animejs.com
*/

/*
!function (n, e) { "object" == typeof exports && "undefined" != typeof module ? module.exports = e() : "function" == typeof define && define.amd ? define(e) : n.anime = e() }(this, function () { "use strict"; var n = { update: null, begin: null, loopBegin: null, changeBegin: null, change: null, changeComplete: null, loopComplete: null, complete: null, loop: 1, direction: "normal", autoplay: !0, timelineOffset: 0 }, e = { duration: 1e3, delay: 0, endDelay: 0, easing: "easeOutElastic(1, .5)", round: 0 }, t = ["translateX", "translateY", "translateZ", "rotate", "rotateX", "rotateY", "rotateZ", "scale", "scaleX", "scaleY", "scaleZ", "skew", "skewX", "skewY", "perspective", "matrix", "matrix3d"], r = { CSS: {}, springs: {} }; function a(n, e, t) { return Math.min(Math.max(n, e), t) } function o(n, e) { return n.indexOf(e) > -1 } function u(n, e) { return n.apply(null, e) } var i = { arr: function (n) { return Array.isArray(n) }, obj: function (n) { return o(Object.prototype.toString.call(n), "Object") }, pth: function (n) { return i.obj(n) && n.hasOwnProperty("totalLength") }, svg: function (n) { return n instanceof SVGElement }, inp: function (n) { return n instanceof HTMLInputElement }, dom: function (n) { return n.nodeType || i.svg(n) }, str: function (n) { return "string" == typeof n }, fnc: function (n) { return "function" == typeof n }, und: function (n) { return void 0 === n }, nil: function (n) { return i.und(n) || null === n }, hex: function (n) { return /(^#[0-9A-F]{6}$)|(^#[0-9A-F]{3}$)/i.test(n) }, rgb: function (n) { return /^rgb/.test(n) }, hsl: function (n) { return /^hsl/.test(n) }, col: function (n) { return i.hex(n) || i.rgb(n) || i.hsl(n) }, key: function (t) { return !n.hasOwnProperty(t) && !e.hasOwnProperty(t) && "targets" !== t && "keyframes" !== t } }; function c(n) { var e = /\(([^)]+)\)/.exec(n); return e ? e[1].split(",").map(function (n) { return parseFloat(n) }) : [] } function s(n, e) { var t = c(n), o = a(i.und(t[0]) ? 1 : t[0], .1, 100), u = a(i.und(t[1]) ? 100 : t[1], .1, 100), s = a(i.und(t[2]) ? 10 : t[2], .1, 100), f = a(i.und(t[3]) ? 0 : t[3], .1, 100), l = Math.sqrt(u / o), d = s / (2 * Math.sqrt(u * o)), p = d < 1 ? l * Math.sqrt(1 - d * d) : 0, v = 1, h = d < 1 ? (d * l - f) / p : -f + l; function g(n) { var t = e ? e * n / 1e3 : n; return t = d < 1 ? Math.exp(-t * d * l) * (v * Math.cos(p * t) + h * Math.sin(p * t)) : (v + h * t) * Math.exp(-t * l), 0 === n || 1 === n ? n : 1 - t } return e ? g : function () { var e = r.springs[n]; if (e) return e; for (var t = 0, a = 0; ;)if (1 === g(t += 1 / 6)) { if (++a >= 16) break } else a = 0; var o = t * (1 / 6) * 1e3; return r.springs[n] = o, o } } function f(n) { return void 0 === n && (n = 10), function (e) { return Math.ceil(a(e, 1e-6, 1) * n) * (1 / n) } } var l, d, p = function () { var n = 11, e = 1 / (n - 1); function t(n, e) { return 1 - 3 * e + 3 * n } function r(n, e) { return 3 * e - 6 * n } function a(n) { return 3 * n } function o(n, e, o) { return ((t(e, o) * n + r(e, o)) * n + a(e)) * n } function u(n, e, o) { return 3 * t(e, o) * n * n + 2 * r(e, o) * n + a(e) } return function (t, r, a, i) { if (0 <= t && t <= 1 && 0 <= a && a <= 1) { var c = new Float32Array(n); if (t !== r || a !== i) for (var s = 0; s < n; ++s)c[s] = o(s * e, t, a); return function (n) { return t === r && a === i ? n : 0 === n || 1 === n ? n : o(f(n), r, i) } } function f(r) { for (var i = 0, s = 1, f = n - 1; s !== f && c[s] <= r; ++s)i += e; var l = i + (r - c[--s]) / (c[s + 1] - c[s]) * e, d = u(l, t, a); return d >= .001 ? function (n, e, t, r) { for (var a = 0; a < 4; ++a) { var i = u(e, t, r); if (0 === i) return e; e -= (o(e, t, r) - n) / i } return e }(r, l, t, a) : 0 === d ? l : function (n, e, t, r, a) { for (var u, i, c = 0; (u = o(i = e + (t - e) / 2, r, a) - n) > 0 ? t = i : e = i, Math.abs(u) > 1e-7 && ++c < 10;); return i }(r, i, i + e, t, a) } } }(), v = (l = { linear: function () { return function (n) { return n } } }, d = { Sine: function () { return function (n) { return 1 - Math.cos(n * Math.PI / 2) } }, Circ: function () { return function (n) { return 1 - Math.sqrt(1 - n * n) } }, Back: function () { return function (n) { return n * n * (3 * n - 2) } }, Bounce: function () { return function (n) { for (var e, t = 4; n < ((e = Math.pow(2, --t)) - 1) / 11;); return 1 / Math.pow(4, 3 - t) - 7.5625 * Math.pow((3 * e - 2) / 22 - n, 2) } }, Elastic: function (n, e) { void 0 === n && (n = 1), void 0 === e && (e = .5); var t = a(n, 1, 10), r = a(e, .1, 2); return function (n) { return 0 === n || 1 === n ? n : -t * Math.pow(2, 10 * (n - 1)) * Math.sin((n - 1 - r / (2 * Math.PI) * Math.asin(1 / t)) * (2 * Math.PI) / r) } } }, ["Quad", "Cubic", "Quart", "Quint", "Expo"].forEach(function (n, e) { d[n] = function () { return function (n) { return Math.pow(n, e + 2) } } }), Object.keys(d).forEach(function (n) { var e = d[n]; l["easeIn" + n] = e, l["easeOut" + n] = function (n, t) { return function (r) { return 1 - e(n, t)(1 - r) } }, l["easeInOut" + n] = function (n, t) { return function (r) { return r < .5 ? e(n, t)(2 * r) / 2 : 1 - e(n, t)(-2 * r + 2) / 2 } }, l["easeOutIn" + n] = function (n, t) { return function (r) { return r < .5 ? (1 - e(n, t)(1 - 2 * r)) / 2 : (e(n, t)(2 * r - 1) + 1) / 2 } } }), l); function h(n, e) { if (i.fnc(n)) return n; var t = n.split("(")[0], r = v[t], a = c(n); switch (t) { case "spring": return s(n, e); case "cubicBezier": return u(p, a); case "steps": return u(f, a); default: return u(r, a) } } function g(n) { try { return document.querySelectorAll(n) } catch (n) { return } } function m(n, e) { for (var t = n.length, r = arguments.length >= 2 ? arguments[1] : void 0, a = [], o = 0; o < t; o++)if (o in n) { var u = n[o]; e.call(r, u, o, n) && a.push(u) } return a } function y(n) { return n.reduce(function (n, e) { return n.concat(i.arr(e) ? y(e) : e) }, []) } function b(n) { return i.arr(n) ? n : (i.str(n) && (n = g(n) || n), n instanceof NodeList || n instanceof HTMLCollection ? [].slice.call(n) : [n]) } function M(n, e) { return n.some(function (n) { return n === e }) } function x(n) { var e = {}; for (var t in n) e[t] = n[t]; return e } function w(n, e) { var t = x(n); for (var r in n) t[r] = e.hasOwnProperty(r) ? e[r] : n[r]; return t } function k(n, e) { var t = x(n); for (var r in e) t[r] = i.und(n[r]) ? e[r] : n[r]; return t } function O(n) { return i.rgb(n) ? (t = /rgb\((\d+,\s*[\d]+,\s*[\d]+)\)/g.exec(e = n)) ? "rgba(" + t[1] + ",1)" : e : i.hex(n) ? (r = n.replace(/^#?([a-f\d])([a-f\d])([a-f\d])$/i, function (n, e, t, r) { return e + e + t + t + r + r }), a = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(r), "rgba(" + parseInt(a[1], 16) + "," + parseInt(a[2], 16) + "," + parseInt(a[3], 16) + ",1)") : i.hsl(n) ? function (n) { var e, t, r, a = /hsl\((\d+),\s*([\d.]+)%,\s*([\d.]+)%\)/g.exec(n) || /hsla\((\d+),\s*([\d.]+)%,\s*([\d.]+)%,\s*([\d.]+)\)/g.exec(n), o = parseInt(a[1], 10) / 360, u = parseInt(a[2], 10) / 100, i = parseInt(a[3], 10) / 100, c = a[4] || 1; function s(n, e, t) { return t < 0 && (t += 1), t > 1 && (t -= 1), t < 1 / 6 ? n + 6 * (e - n) * t : t < .5 ? e : t < 2 / 3 ? n + (e - n) * (2 / 3 - t) * 6 : n } if (0 == u) e = t = r = i; else { var f = i < .5 ? i * (1 + u) : i + u - i * u, l = 2 * i - f; e = s(l, f, o + 1 / 3), t = s(l, f, o), r = s(l, f, o - 1 / 3) } return "rgba(" + 255 * e + "," + 255 * t + "," + 255 * r + "," + c + ")" }(n) : void 0; var e, t, r, a } function C(n) { var e = /[+-]?\d*\.?\d+(?:\.\d+)?(?:[eE][+-]?\d+)?(%|px|pt|em|rem|in|cm|mm|ex|ch|pc|vw|vh|vmin|vmax|deg|rad|turn)?$/.exec(n); if (e) return e[1] } function P(n, e) { return i.fnc(n) ? n(e.target, e.id, e.total) : n } function I(n, e) { return n.getAttribute(e) } function D(n, e, t) { if (M([t, "deg", "rad", "turn"], C(e))) return e; var a = r.CSS[e + t]; if (!i.und(a)) return a; var o = document.createElement(n.tagName), u = n.parentNode && n.parentNode !== document ? n.parentNode : document.body; u.appendChild(o), o.style.position = "absolute", o.style.width = 100 + t; var c = 100 / o.offsetWidth; u.removeChild(o); var s = c * parseFloat(e); return r.CSS[e + t] = s, s } function B(n, e, t) { if (e in n.style) { var r = e.replace(/([a-z])([A-Z])/g, "$1-$2").toLowerCase(), a = n.style[e] || getComputedStyle(n).getPropertyValue(r) || "0"; return t ? D(n, a, t) : a } } function T(n, e) { return i.dom(n) && !i.inp(n) && (!i.nil(I(n, e)) || i.svg(n) && n[e]) ? "attribute" : i.dom(n) && M(t, e) ? "transform" : i.dom(n) && "transform" !== e && B(n, e) ? "css" : null != n[e] ? "object" : void 0 } function E(n) { if (i.dom(n)) { for (var e, t = n.style.transform || "", r = /(\w+)\(([^)]*)\)/g, a = new Map; e = r.exec(t);)a.set(e[1], e[2]); return a } } function F(n, e, t, r) { var a, u = o(e, "scale") ? 1 : 0 + (o(a = e, "translate") || "perspective" === a ? "px" : o(a, "rotate") || o(a, "skew") ? "deg" : void 0), i = E(n).get(e) || u; return t && (t.transforms.list.set(e, i), t.transforms.last = e), r ? D(n, i, r) : i } function A(n, e, t, r) { switch (T(n, e)) { case "transform": return F(n, e, r, t); case "css": return B(n, e, t); case "attribute": return I(n, e); default: return n[e] || 0 } } function N(n, e) { var t = /^(\*=|\+=|-=)/.exec(n); if (!t) return n; var r = C(n) || 0, a = parseFloat(e), o = parseFloat(n.replace(t[0], "")); switch (t[0][0]) { case "+": return a + o + r; case "-": return a - o + r; case "*": return a * o + r } } function S(n, e) { if (i.col(n)) return O(n); if (/\s/g.test(n)) return n; var t = C(n), r = t ? n.substr(0, n.length - t.length) : n; return e ? r + e : r } function L(n, e) { return Math.sqrt(Math.pow(e.x - n.x, 2) + Math.pow(e.y - n.y, 2)) } function j(n) { for (var e, t = n.points, r = 0, a = 0; a < t.numberOfItems; a++) { var o = t.getItem(a); a > 0 && (r += L(e, o)), e = o } return r } function q(n) { if (n.getTotalLength) return n.getTotalLength(); switch (n.tagName.toLowerCase()) { case "circle": return o = n, 2 * Math.PI * I(o, "r"); case "rect": return 2 * I(a = n, "width") + 2 * I(a, "height"); case "line": return L({ x: I(r = n, "x1"), y: I(r, "y1") }, { x: I(r, "x2"), y: I(r, "y2") }); case "polyline": return j(n); case "polygon": return t = (e = n).points, j(e) + L(t.getItem(t.numberOfItems - 1), t.getItem(0)) }var e, t, r, a, o } function H(n, e) { var t = e || {}, r = t.el || function (n) { for (var e = n.parentNode; i.svg(e) && i.svg(e.parentNode);)e = e.parentNode; return e }(n), a = r.getBoundingClientRect(), o = I(r, "viewBox"), u = a.width, c = a.height, s = t.viewBox || (o ? o.split(" ") : [0, 0, u, c]); return { el: r, viewBox: s, x: s[0] / 1, y: s[1] / 1, w: u, h: c, vW: s[2], vH: s[3] } } function V(n, e, t) { function r(t) { void 0 === t && (t = 0); var r = e + t >= 1 ? e + t : 0; return n.el.getPointAtLength(r) } var a = H(n.el, n.svg), o = r(), u = r(-1), i = r(1), c = t ? 1 : a.w / a.vW, s = t ? 1 : a.h / a.vH; switch (n.property) { case "x": return (o.x - a.x) * c; case "y": return (o.y - a.y) * s; case "angle": return 180 * Math.atan2(i.y - u.y, i.x - u.x) / Math.PI } } function $(n, e) { var t = /[+-]?\d*\.?\d+(?:\.\d+)?(?:[eE][+-]?\d+)?/g, r = S(i.pth(n) ? n.totalLength : n, e) + ""; return { original: r, numbers: r.match(t) ? r.match(t).map(Number) : [0], strings: i.str(n) || e ? r.split(t) : [] } } function W(n) { return m(n ? y(i.arr(n) ? n.map(b) : b(n)) : [], function (n, e, t) { return t.indexOf(n) === e }) } function X(n) { var e = W(n); return e.map(function (n, t) { return { target: n, id: t, total: e.length, transforms: { list: E(n) } } }) } function Y(n, e) { var t = x(e); if (/^spring/.test(t.easing) && (t.duration = s(t.easing)), i.arr(n)) { var r = n.length; 2 === r && !i.obj(n[0]) ? n = { value: n } : i.fnc(e.duration) || (t.duration = e.duration / r) } var a = i.arr(n) ? n : [n]; return a.map(function (n, t) { var r = i.obj(n) && !i.pth(n) ? n : { value: n }; return i.und(r.delay) && (r.delay = t ? 0 : e.delay), i.und(r.endDelay) && (r.endDelay = t === a.length - 1 ? e.endDelay : 0), r }).map(function (n) { return k(n, t) }) } function Z(n, e) { var t = [], r = e.keyframes; for (var a in r && (e = k(function (n) { for (var e = m(y(n.map(function (n) { return Object.keys(n) })), function (n) { return i.key(n) }).reduce(function (n, e) { return n.indexOf(e) < 0 && n.push(e), n }, []), t = {}, r = function (r) { var a = e[r]; t[a] = n.map(function (n) { var e = {}; for (var t in n) i.key(t) ? t == a && (e.value = n[t]) : e[t] = n[t]; return e }) }, a = 0; a < e.length; a++)r(a); return t }(r), e)), e) i.key(a) && t.push({ name: a, tweens: Y(e[a], n) }); return t } function G(n, e) { var t; return n.tweens.map(function (r) { var a = function (n, e) { var t = {}; for (var r in n) { var a = P(n[r], e); i.arr(a) && 1 === (a = a.map(function (n) { return P(n, e) })).length && (a = a[0]), t[r] = a } return t.duration = parseFloat(t.duration), t.delay = parseFloat(t.delay), t }(r, e), o = a.value, u = i.arr(o) ? o[1] : o, c = C(u), s = A(e.target, n.name, c, e), f = t ? t.to.original : s, l = i.arr(o) ? o[0] : f, d = C(l) || C(s), p = c || d; return i.und(u) && (u = f), a.from = $(l, p), a.to = $(N(u, l), p), a.start = t ? t.end : 0, a.end = a.start + a.delay + a.duration + a.endDelay, a.easing = h(a.easing, a.duration), a.isPath = i.pth(o), a.isPathTargetInsideSVG = a.isPath && i.svg(e.target), a.isColor = i.col(a.from.original), a.isColor && (a.round = 1), t = a, a }) } var Q = { css: function (n, e, t) { return n.style[e] = t }, attribute: function (n, e, t) { return n.setAttribute(e, t) }, object: function (n, e, t) { return n[e] = t }, transform: function (n, e, t, r, a) { if (r.list.set(e, t), e === r.last || a) { var o = ""; r.list.forEach(function (n, e) { o += e + "(" + n + ") " }), n.style.transform = o } } }; function z(n, e) { X(n).forEach(function (n) { for (var t in e) { var r = P(e[t], n), a = n.target, o = C(r), u = A(a, t, o, n), i = N(S(r, o || C(u)), u), c = T(a, t); Q[c](a, t, i, n.transforms, !0) } }) } function _(n, e) { return m(y(n.map(function (n) { return e.map(function (e) { return function (n, e) { var t = T(n.target, e.name); if (t) { var r = G(e, n), a = r[r.length - 1]; return { type: t, property: e.name, animatable: n, tweens: r, duration: a.end, delay: r[0].delay, endDelay: a.endDelay } } }(n, e) }) })), function (n) { return !i.und(n) }) } function R(n, e) { var t = n.length, r = function (n) { return n.timelineOffset ? n.timelineOffset : 0 }, a = {}; return a.duration = t ? Math.max.apply(Math, n.map(function (n) { return r(n) + n.duration })) : e.duration, a.delay = t ? Math.min.apply(Math, n.map(function (n) { return r(n) + n.delay })) : e.delay, a.endDelay = t ? a.duration - Math.max.apply(Math, n.map(function (n) { return r(n) + n.duration - n.endDelay })) : e.endDelay, a } var J = 0; var K = [], U = function () { var n; function e(t) { for (var r = K.length, a = 0; a < r;) { var o = K[a]; o.paused ? (K.splice(a, 1), r--) : (o.tick(t), a++) } n = a > 0 ? requestAnimationFrame(e) : void 0 } return "undefined" != typeof document && document.addEventListener("visibilitychange", function () { en.suspendWhenDocumentHidden && (nn() ? n = cancelAnimationFrame(n) : (K.forEach(function (n) { return n._onDocumentVisibility() }), U())) }), function () { n || nn() && en.suspendWhenDocumentHidden || !(K.length > 0) || (n = requestAnimationFrame(e)) } }(); function nn() { return !!document && document.hidden } function en(t) { void 0 === t && (t = {}); var r, o = 0, u = 0, i = 0, c = 0, s = null; function f(n) { var e = window.Promise && new Promise(function (n) { return s = n }); return n.finished = e, e } var l, d, p, v, h, g, y, b, M = (d = w(n, l = t), p = w(e, l), v = Z(p, l), h = X(l.targets), g = _(h, v), y = R(g, p), b = J, J++, k(d, { id: b, children: [], animatables: h, animations: g, duration: y.duration, delay: y.delay, endDelay: y.endDelay })); f(M); function x() { var n = M.direction; "alternate" !== n && (M.direction = "normal" !== n ? "normal" : "reverse"), M.reversed = !M.reversed, r.forEach(function (n) { return n.reversed = M.reversed }) } function O(n) { return M.reversed ? M.duration - n : n } function C() { o = 0, u = O(M.currentTime) * (1 / en.speed) } function P(n, e) { e && e.seek(n - e.timelineOffset) } function I(n) { for (var e = 0, t = M.animations, r = t.length; e < r;) { var o = t[e], u = o.animatable, i = o.tweens, c = i.length - 1, s = i[c]; c && (s = m(i, function (e) { return n < e.end })[0] || s); for (var f = a(n - s.start - s.delay, 0, s.duration) / s.duration, l = isNaN(f) ? 1 : s.easing(f), d = s.to.strings, p = s.round, v = [], h = s.to.numbers.length, g = void 0, y = 0; y < h; y++) { var b = void 0, x = s.to.numbers[y], w = s.from.numbers[y] || 0; b = s.isPath ? V(s.value, l * x, s.isPathTargetInsideSVG) : w + l * (x - w), p && (s.isColor && y > 2 || (b = Math.round(b * p) / p)), v.push(b) } var k = d.length; if (k) { g = d[0]; for (var O = 0; O < k; O++) { d[O]; var C = d[O + 1], P = v[O]; isNaN(P) || (g += C ? P + C : P + " ") } } else g = v[0]; if ((u.target.closest('.animating-text-rise') && o.property == 'translateY' && g != '50%') || (u.target.closest('.animating-text-fall') && o.property == 'translateY' && g != '-50%')) { if (u.target.closest('.animating-text-rise') && !u.target.classList.contains('text-rise')) { u.target.classList.add('text-rise'); u.target.style.transform = ''; u.target.style.animationDuration = s.duration + 'ms'; } if (u.target.closest('.animating-text-fall') && !u.target.classList.contains('text-fall')) { u.target.classList.add('text-fall'); u.target.style.transform = ''; u.target.style.animationDuration = s.duration + 'ms'; } } else { Q[o.type](u.target, o.property, g, u.transforms), o.currentValue = g } e++ } } function D(n) { M[n] && !M.passThrough && M[n](M) } function B(n) { var e = M.duration, t = M.delay, l = e - M.endDelay, d = O(n); M.progress = a(d / e * 100, 0, 100), M.reversePlayback = d < M.currentTime, r && function (n) { if (M.reversePlayback) for (var e = c; e--;)P(n, r[e]); else for (var t = 0; t < c; t++)P(n, r[t]) }(d), !M.began && M.currentTime > 0 && (M.began = !0, D("begin")), !M.loopBegan && M.currentTime > 0 && (M.loopBegan = !0, D("loopBegin")), d <= t && 0 !== M.currentTime && I(0), (d >= l && M.currentTime !== e || !e) && I(e), d > t && d < l ? (M.changeBegan || (M.changeBegan = !0, M.changeCompleted = !1, D("changeBegin")), D("change"), I(d)) : M.changeBegan && (M.changeCompleted = !0, M.changeBegan = !1, D("changeComplete")), M.currentTime = a(d, 0, e), M.began && D("update"), n >= e && (u = 0, M.remaining && !0 !== M.remaining && M.remaining--, M.remaining ? (o = i, D("loopComplete"), M.loopBegan = !1, "alternate" === M.direction && x()) : (M.paused = !0, M.completed || (M.completed = !0, D("loopComplete"), D("complete"), !M.passThrough && "Promise" in window && (s(), f(M))))) } return M.reset = function () { var n = M.direction; M.passThrough = !1, M.currentTime = 0, M.progress = 0, M.paused = !0, M.began = !1, M.loopBegan = !1, M.changeBegan = !1, M.completed = !1, M.changeCompleted = !1, M.reversePlayback = !1, M.reversed = "reverse" === n, M.remaining = M.loop, r = M.children; for (var e = c = r.length; e--;)M.children[e].reset(); (M.reversed && !0 !== M.loop || "alternate" === n && 1 === M.loop) && M.remaining++, I(M.reversed ? M.duration : 0); for (var e = 0; e < M.animatables.length; e++) { M.animatables[e].target.classList.remove('text-rise', 'text-fall'); } }, M._onDocumentVisibility = C, M.set = function (n, e) { return z(n, e), M }, M.tick = function (n) { i = n, o || (o = i), B((i + (u - o)) * en.speed) }, M.seek = function (n) { B(O(n)) }, M.pause = function () { M.paused = !0, C() }, M.play = function () { M.paused && (M.completed && M.reset(), M.paused = !1, K.push(M), C(), U()) }, M.reverse = function () { x(), M.completed = !M.reversed, C() }, M.restart = function () { M.reset(), M.play() }, M.remove = function (n) { rn(W(n), M) }, M.reset(), M.autoplay && M.play(), M } function tn(n, e) { for (var t = e.length; t--;)M(n, e[t].animatable.target) && e.splice(t, 1) } function rn(n, e) { var t = e.animations, r = e.children; tn(n, t); for (var a = r.length; a--;) { var o = r[a], u = o.animations; tn(n, u), u.length || o.children.length || r.splice(a, 1) } t.length || r.length || e.pause() } return en.version = "3.2.1", en.speed = 1, en.suspendWhenDocumentHidden = !0, en.running = K, en.remove = function (n) { for (var e = W(n), t = K.length; t--;)rn(e, K[t]) }, en.get = A, en.set = z, en.convertPx = D, en.path = function (n, e) { var t = i.str(n) ? g(n)[0] : n, r = e || 100; return function (n) { return { property: n, el: t, svg: H(t), totalLength: q(t) * (r / 100) } } }, en.setDashoffset = function (n) { var e = q(n); return n.setAttribute("stroke-dasharray", e), e }, en.stagger = function (n, e) { void 0 === e && (e = {}); var t = e.direction || "normal", r = e.easing ? h(e.easing) : null, a = e.grid, o = e.axis, u = e.from || 0, c = "first" === u, s = "center" === u, f = "last" === u, l = i.arr(n), d = l ? parseFloat(n[0]) : parseFloat(n), p = l ? parseFloat(n[1]) : 0, v = C(l ? n[1] : n) || 0, g = e.start || 0 + (l ? d : 0), m = [], y = 0; return function (n, e, i) { if (c && (u = 0), s && (u = (i - 1) / 2), f && (u = i - 1), !m.length) { for (var h = 0; h < i; h++) { if (a) { var b = s ? (a[0] - 1) / 2 : u % a[0], M = s ? (a[1] - 1) / 2 : Math.floor(u / a[0]), x = b - h % a[0], w = M - Math.floor(h / a[0]), k = Math.sqrt(x * x + w * w); "x" === o && (k = -x), "y" === o && (k = -w), m.push(k) } else m.push(Math.abs(u - h)); y = Math.max.apply(Math, m) } r && (m = m.map(function (n) { return r(n / y) * y })), "reverse" === t && (m = m.map(function (n) { return o ? n < 0 ? -1 * n : -n : Math.abs(y - n) })) } return g + (l ? (p - d) / y : d) * (Math.round(100 * m[e]) / 100) + v } }, en.timeline = function (n) { void 0 === n && (n = {}); var t = en(n); return t.duration = 0, t.add = function (r, a) { var o = K.indexOf(t), u = t.children; function c(n) { n.passThrough = !0 } o > -1 && K.splice(o, 1); for (var s = 0; s < u.length; s++)c(u[s]); var f = k(r, w(e, n)); f.targets = f.targets || n.targets; var l = t.duration; f.autoplay = !1, f.direction = t.direction, f.timelineOffset = i.und(a) ? l : N(a, l), c(t), t.seek(f.timelineOffset); var d = en(f); c(d), u.push(d); var p = R(u, n); return t.delay = p.delay, t.endDelay = p.endDelay, t.duration = p.duration, t.seek(0), t.reset(), t.autoplay && t.play(), t }, t }, en.easing = h, en.penner = v, en.random = function (n, e) { return Math.floor(Math.random() * (e - n + 1)) + n }, en });
*/

/**
 * Alpha Animated Text Library
 * 
 * @package Alpha FrameWork
 * @version 1.0
 */
window.theme = window.theme || {};

(function ($) {

    if (typeof anime != 'function') {
        return;
    }

    /**
     * Run text animation
     * 
     * @param {string} selector 
     */
    theme.animatedText = (function () {

        function AnimatedText($el, options) {
            return this.init($el, options);
        }

        /**
         * Effects
         */
        AnimatedText.prototype.effects = function (settings) {
            var _settings = settings;
            var effects = {
                'joke': {
                    in: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 1000,
                        delay: function (el, index) { return 75 + index * (_settings.innerDelay != '' ? _settings.innerDelay : 100); },
                        easing: 'easeOutElastic',
                        elasticity: 650,
                        opacity: {
                            value: [0, 1],
                            easing: 'easeOutExpo',
                        },
                        translateY: ['100%', '0%']
                    },
                    out: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 300,
                        delay: function (el, index) { return index * (_settings.innerDelay != '' ? _settings.innerDelay : 40); },
                        easing: 'easeInOutExpo',
                        opacity: 0,
                        translateY: '-100%'
                    }
                },
                'fall': {
                    in: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 250,
                        delay: function (el, index) { return 200 + index * (_settings.innerDelay != '' ? _settings.innerDelay : 25); },
                        easing: 'easeOutCubic',
                        opacity: {
                            value: [0, 1],
                            easing: 'easeOutExpo',
                        },
                        translateY: ['-50%', '0%']
                    },
                    out: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 250,
                        delay: function (el, index) { return index * (_settings.innerDelay != '' ? _settings.innerDelay : 25); },
                        easing: 'easeOutCubic',
                        opacity: 0,
                        translateY: '50%'
                    }
                },
                'rise': {
                    in: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 250,
                        delay: function (el, index) { return 200 + index * (_settings.innerDelay != '' ? _settings.innerDelay : 25); },
                        easing: 'easeOutCubic',
                        opacity: {
                            value: [0, 1],
                            easing: 'easeOutExpo',
                        },
                        translateY: ['50%', '0%'],
                    },
                    out: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 250,
                        delay: function (el, index) { return index * (_settings.innerDelay != '' ? _settings.innerDelay : 25); },
                        easing: 'easeOutCubic',
                        opacity: 0,
                        translateY: '50%'
                    }
                },
                'rotation': {
                    in: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 400,
                        delay: function (el, index) { return index * (_settings.innerDelay != '' ? _settings.innerDelay : 50); },
                        easing: 'easeOutSine',
                        opacity: {
                            value: [0, 1],
                            easing: 'easeOutExpo',
                        },
                        rotateY: [-90, 0]
                    },
                    out: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 200,
                        delay: function (el, index) { return index * (_settings.innerDelay != '' ? _settings.innerDelay : 50); },
                        easing: 'easeOutSine',
                        opacity: 0,
                        rotateY: 45
                    }
                },
                'croco': {
                    in: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 400,
                        delay: function (el, index) { return 200 + index * (_settings.innerDelay != '' ? _settings.innerDelay : 30); },
                        easing: 'easeOutExpo',
                        opacity: 1,
                        rotateX: [90, 0]
                    },
                    out: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 250,
                        delay: function (el, index) { return index * (_settings.innerDelay != '' ? _settings.innerDelay : 30); },
                        easing: 'easeOutExpo',
                        opacity: 0,
                        rotateX: -90
                    }
                },
                'scaling': {
                    in: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 400,
                        delay: function (el, index) { return 100 + index * (_settings.innerDelay != '' ? _settings.innerDelay : 50); },
                        easing: 'easeOutExpo',
                        opacity: {
                            value: [0, 1],
                            easing: 'easeOutExpo',
                        },
                        rotateX: [110, 0]
                    },
                    out: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 250,
                        delay: function (el, index) { return index * (_settings.innerDelay != '' ? _settings.innerDelay : 50); },
                        easing: 'easeOutExpo',
                        opacity: 0,
                        rotateX: -110
                    }
                },
                'typing': {
                    in: {
                        elasticity: false,
                        duration: settings.innerDuration != '' ? settings.innerDuration : 1,
                        delay: function (el, index) {
                            var delay = index * (_settings.innerDelay != '' ? _settings.innerDelay : 100) + anime.random(50, 100);

                            return delay;
                        },
                        width: [0, function (el, i) { return $(el).width(); }]
                    },
                    out: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 1,
                        delay: function (el, index) { return (el.parentNode.children.length - index - 1) * (_settings.innerDelay != '' ? _settings.innerDelay : 20); },
                        easing: 'linear',
                        width: {
                            value: 0
                        }
                    }
                },
                'wave': {
                    in: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 250,
                        delay: function (el, index) { return 200 + index * (_settings.innerDelay != '' ? _settings.innerDelay : 25); },
                        easing: 'easeOutCubic',
                        opacity: {
                            value: [0, 1],
                            easing: 'easeOutExpo',
                        },
                        translateX: ['-.3em', '0em'],
                    },
                    out: {
                        duration: settings.innerDuration != '' ? settings.innerDuration : 250,
                        delay: function (el, index) { return index * (_settings.innerDelay != '' ? _settings.innerDelay : 25); },
                        easing: 'easeOutCubic',
                        opacity: 0,
                        translateX: '-.3em'
                    }
                },
            };

            return effects[settings.effect];
        }

        AnimatedText.prototype.init = function ($el, options) {

            var self = this,
                totalDelay = 0,
                $parent = $el.closest('.elementor-element');
            self.$el = $el;
            self.$items = $el.children('.animating-item');
            self.currentIndex = 0;
            self.settings = $.extend({
                effect: 'joke',
                delay: 3000
            }, $el.data('settings') || {}, options || {});

            if ($parent.hasClass('elementor-invisible')) {
                var settings = $parent.data('settings');
                totalDelay = settings._animation_delay ? parseInt(settings._animation_delay) : 0;
                if ($parent.hasClass('animated-fast')) {
                    totalDelay = totalDelay + 750;
                } else if ($parent.hasClass('animated-slow')) {
                    totalDelay = totalDelay + 2000;
                } else {
                    totalDelay = totalDelay + 1000;
                }
            }

            theme.appear($el.get(0), function () {
                setTimeout(function () {
                    self.showText(
                        self.$items.eq(self.currentIndex),
                        self.settings,
                        null,
                        function () {
                            var currentDelay = self.settings.delay;

                            if (self.timeOut) {
                                clearTimeout(self.timeOut);
                            }

                            self.timeOut = setTimeout(function () {
                                self.textChange();
                            }, currentDelay);
                        }
                    )
                }, totalDelay);
            }, !$parent.hasClass('elementor-invisible') ? { rootMargin: '0px 0px 0px 0px' } : {})
        }

        AnimatedText.prototype.textChange = function () {
            var self = this,
                $prevText = self.$items.eq(self.currentIndex),
                $nextText;

            if (self.currentIndex < self.$items.length - 1) {
                ++self.currentIndex;
            } else {
                var $parent = $prevText.parent();
                if ($parent.hasClass('animating-once')) {
                    return;
                } else {
                    self.currentIndex = 0;
                }
            }

            $nextText = self.$items.eq(self.currentIndex);

            self.hideText($prevText, self.settings.effect, null, function (anime) {
                $prevText.toggleClass('visible');

                var currentDelay = self.settings.delay;

                if (self.timeOut) {
                    clearTimeout(self.timeOut);
                }

                self.showText(
                    $nextText,
                    self.settings,
                    function () {
                        $nextText.toggleClass('active');
                        $prevText.toggleClass('active');

                        $nextText.toggleClass('visible');
                    },
                    function () {
                        self.timeOut = setTimeout(function () {
                            self.textChange();
                        }, currentDelay);
                    }
                );

            });
        };

        AnimatedText.prototype.showText = function ($item, settings, beginCallback, completeCallback) {
            var targets = [];

            $('span.a-text', $item).each(function () {
                $(this).css({
                    'width': 'auto',
                    'opacity': 1,
                    'WebkitTransform': '',
                    'transform': ''
                });
                targets.push(this);
            });

            this.animateText(targets, 'in', settings, beginCallback, completeCallback);
        };

        AnimatedText.prototype.hideText = function ($item, settings, beginCallback, completeCallback) {
            var targets = [];

            $('span.a-text', $item).each(function () {
                targets.push(this);
            });

            this.animateText(targets, 'out', settings, beginCallback, completeCallback);
        };

        AnimatedText.prototype.animateText = function (targets, direction, settings, beginCallback, completeCallback) {
            var effectSettings = this.effects(settings) || {};
            anime($.extend({
                targets: targets,
                begin: beginCallback,
                complete: completeCallback,
            }, effectSettings[direction]));
        }

        return function (selector) {
            theme.$(selector).each(function () {
                var $this = $(this);
                $this.data('theme-animated-text', new AnimatedText($this));
            })
        }
    })();

    if (window.theme && theme.status == 'complete') {
        theme.animatedText('.animating-text');
    } else {
        $(window).on('alpha_complete', function () {
            theme.animatedText('.animating-text');
        });
    }
})(jQuery);