<?php

use Elementor\Plugin;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use ELementor\Group_Control_Box_Shadow;

/**
 * WPForms widget for Elementor page builder.
 *
 * @since 1.6.2
 */
class Alpha_Wpforms_Elementor_Widget extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve shortcode widget name.
	 *
	 * @since 1.6.2
	 *
	 * @return string Widget name.
	 */
	public function get_name() {

		return 'wpforms';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve shortcode widget title.
	 *
	 * @since 1.6.2
	 *
	 * @return string Widget title.
	 */
	public function get_title() {

		return __( 'WPForms', 'wpforms-lite' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve shortcode widget icon.
	 *
	 * @since 1.6.2
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {

		return 'alpha-elementor-widget-icon icon-wpforms';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 1.6.2
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {

		return [
			'form',
			'forms',
			'wpforms',
			'contact form',
			'sullie',
			'the dude',
		];
	}

	/**
	 * Get widget categories.
	 *
	 * @since 1.6.2
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {

		return [
			'alpha_widget',
		];
	}

	/**
	 * Register widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.6.2
	 */
	protected function register_controls() { // phpcs:ignore PSR2.Methods.MethodDeclaration.Underscore

		$this->content_controls();
	}

	public function has_widget_inner_wrapper(): bool {
		return ! Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
	}

	/**
	 * Register content tab controls.
	 *
	 * @since 1.6.2
	 */
	protected function content_controls() {

		$this->start_controls_section(
			'section_form',
			[
				'label' => esc_html__( 'Form', 'wpforms-lite' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			]
		);

			$forms = $this->get_forms();

		if ( empty( $forms ) ) {
			$this->add_control(
				'add_form_notice',
				[
					'show_label'      => false,
					'type'            => Controls_Manager::RAW_HTML,
					'raw'             => wp_kses(
						__( '<b>You haven\'t created a form yet.</b><br> What are you waiting for?', 'wpforms-lite' ),
						[
							'b'  => [],
							'br' => [],
						]
					),
					'content_classes' => 'elementor-panel-alert elementor-panel-alert-info',
				]
			);
		}

			$this->add_control(
				'form_id',
				[
					'label'       => esc_html__( 'Form', 'wpforms-lite' ),
					'type'        => Controls_Manager::SELECT,
					'label_block' => true,
					'options'     => $forms,
					'default'     => '0',
				]
			);

			$this->add_control(
				'edit_form',
				[
					'show_label' => false,
					'type'       => Controls_Manager::RAW_HTML,
					'raw'        => wp_kses( /* translators: %s - WPForms documentation link. */
						__( 'Need to make changes? <a href="#">Edit the selected form.</a>', 'wpforms-lite' ),
						[ 'a' => [] ]
					),
					'condition'  => [
						'form_id!' => '0',
					],
				]
			);

			$this->add_control(
				'test_form_notice',
				[
					'show_label'      => false,
					'type'            => Controls_Manager::RAW_HTML,
					'raw'             => sprintf(
						wp_kses( /* translators: %s - WPForms documentation link. */
							__( '<b>Heads up!</b> Don\'t forget to test your form. <a href="%s" target="_blank" rel="noopener noreferrer">Check out our complete guide!</a>', 'wpforms-lite' ),
							[
								'b'  => [],
								'br' => [],
								'a'  => [
									'href'   => [],
									'rel'    => [],
									'target' => [],
								],
							]
						),
						'https://wpforms.com/docs/how-to-properly-test-your-wordpress-forms-before-launching-checklist/'
					),
					'condition'       => [
						'form_id!' => '0',
					],
					'content_classes' => 'elementor-panel-alert elementor-panel-alert-info',
				]
			);

			$this->add_control(
				'add_form_btn',
				[
					'show_label'  => false,
					'label_block' => false,
					'type'        => Controls_Manager::BUTTON,
					'button_type' => 'default',
					'separator'   => 'before',
					'text'        => '<b>+</b>' . esc_html__( 'New form', 'wpforms-lite' ),
					'event'       => 'elementorWPFormsAddFormBtnClick',
				]
			);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_display',
			[
				'label'     => esc_html__( 'Display Options', 'wpforms-lite' ),
				'tab'       => Controls_Manager::TAB_CONTENT,
				'condition' => [
					'form_id!' => '0',
				],
			]
		);

			$this->add_control(
				'display_form_name',
				[
					'label'        => esc_html__( 'Form Name', 'wpforms-lite' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Show', 'wpforms-lite' ),
					'label_off'    => esc_html__( 'Hide', 'wpforms-lite' ),
					'return_value' => 'yes',
					'condition'    => [
						'form_id!' => '0',
					],
				]
			);

			$this->add_control(
				'display_form_description',
				[
					'label'        => esc_html__( 'Form Description', 'wpforms-lite' ),
					'type'         => Controls_Manager::SWITCHER,
					'label_on'     => esc_html__( 'Show', 'wpforms-lite' ),
					'label_off'    => esc_html__( 'Hide', 'wpforms-lite' ),
					'separator'    => 'after',
					'return_value' => 'yes',
					'condition'    => [
						'form_id!' => '0',
					],
				]
			);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_submit_button_content',
			array(
				'label' => esc_html__( 'Submit Button', 'alpha-core' ),
			)
		);

			$this->add_control(
				'text_hover_effect',
				array(
					'label'       => esc_html__( 'Text Hover Effect', 'alpha-core' ),
					'description' => esc_html__( 'Choose hover effect of button text.', 'alpha-core' ),
					'type'        => Controls_Manager::SELECT,
					'options'     => array(
						''                      => __( 'No Effect', 'alpha-core' ),
						'btn-text-switch-left'  => __( 'Switch Left', 'alpha-core' ),
						'btn-text-marquee-left' => __( 'Marquee Left', 'alpha-core' ),
					),
					'condition'   => array(
						'bg_hover_effect!' => array( 'btn-antiman', 'btn-bubble' ),
					),
				)
			);

			$this->add_control(
				'bg_hover_effect',
				array(
					'label'       => esc_html__( 'Background Hover Effect', 'alpha-core' ),
					'description' => esc_html__( 'Choose hover effect of button background.', 'alpha-core' ),
					'type'        => Controls_Manager::SELECT,
					'default'     => '',
					'options'     => array(
						''                        => __( 'No Effect', 'alpha-core' ),
						'btn-sweep-to-right'      => __( 'Sweep To Right', 'alpha-core' ),
						'btn-sweep-to-left'       => __( 'Sweep To Left', 'alpha-core' ),
						'btn-sweep-to-bottom'     => __( 'Sweep To Bottom', 'alpha-core' ),
						'btn-sweep-to-top'        => __( 'Sweep To Top', 'alpha-core' ),
						'btn-sweep-to-horizontal' => __( 'Sweep To Horizontal', 'alpha-core' ),
						'btn-sweep-to-vertical'   => __( 'Sweep To Vertical', 'alpha-core' ),
						'btn-antiman'             => __( 'Antiman', 'alpha-core' ),
						'btn-bubble'              => __( 'Bubble', 'alpha-core' ),
					),
				)
			);

			$this->add_control(
				'show_icon',
				array(
					'label'       => esc_html__( 'Show Icon', 'alpha-core' ),
					'description' => esc_html__( 'Allows to show icon before or after button text.', 'alpha-core' ),
					'type'        => Controls_Manager::SWITCHER,
				)
			);

			$this->add_control(
				'show_label',
				array(
					'label'     => esc_html__( 'Show Label', 'alpha-core' ),
					'type'      => Controls_Manager::SWITCHER,
					'default'   => 'yes',
					'condition' => array(
						'show_icon'    => 'yes',
						'icon[value]!' => '',
					),
				)
			);

			$this->add_control(
				'icon',
				array(
					'label'       => esc_html__( 'Icon', 'alpha-core' ),
					'description' => esc_html__( 'Choose icon from icon library that will be shown with button text.', 'alpha-core' ),
					'type'        => Controls_Manager::ICONS,
					'default'     => array(
						'value'   => ALPHA_ICON_PREFIX . '-icon-long-arrow-right',
						'library' => 'alpha-icons',
					),
					'condition'   => array(
						'show_icon' => 'yes',
					),
				)
			);

			$this->add_control(
				'icon_pos',
				array(
					'label'       => esc_html__( 'Icon Position', 'alpha-core' ),
					'description' => esc_html__( 'Choose where to show icon with button text. Choose from Before/After.', 'alpha-core' ),
					'type'        => Controls_Manager::SELECT,
					'default'     => 'after',
					'options'     => array(
						'after'  => esc_html__( 'After', 'alpha-core' ),
						'before' => esc_html__( 'Before', 'alpha-core' ),
					),
					'condition'   => array(
						'show_icon' => 'yes',
					),
				)
			);

			$this->add_control(
				'icon_hover_effect',
				array(
					'label'       => esc_html__( 'Icon Hover Effect', 'alpha-core' ),
					'description' => esc_html__( 'Choose hover effect of buttons with icon. Choose from 3 hover effects.', 'alpha-core' ),
					'type'        => Controls_Manager::SELECT,
					'default'     => '',
					'options'     => array(
						''                => esc_html__( 'No Effect', 'alpha-core' ),
						'btn-slide-left'  => esc_html__( 'Slide Left', 'alpha-core' ),
						'btn-slide-right' => esc_html__( 'Slide Right', 'alpha-core' ),
						'btn-slide-up'    => esc_html__( 'Slide Up', 'alpha-core' ),
						'btn-slide-down'  => esc_html__( 'Slide Down', 'alpha-core' ),
						'btn-reveal'      => esc_html__( 'Reveal', 'alpha-core' ),
					),
					'condition'   => array(
						'show_icon' => 'yes',
					),
				)
			);

			$this->add_control(
				'icon_hover_effect_infinite',
				array(
					'label'       => esc_html__( 'Animation Infinite', 'alpha-core' ),
					'description' => esc_html__( 'Determines whether icons should be animated once or several times for buttons with icon.', 'alpha-core' ),
					'type'        => Controls_Manager::SWITCHER,
					'condition'   => array(
						'show_icon'          => 'yes',
						'icon_hover_effect!' => array( '', 'btn-reveal' ),
					),
				)
			);

			$this->add_control(
				'svg_hover_effect',
				array(
					'label'       => esc_html__( 'Icon Draw Effect', 'alpha-core' ),
					'description' => esc_html__( 'Choose hover effect of button svg. Choose from 4 hover effects.', 'alpha-core' ),
					'type'        => Controls_Manager::SELECT,
					'default'     => '',
					'options'     => array(
						''                    => esc_html__( 'No Effect', 'alpha-core' ),
						'btn-draw-left-right' => esc_html__( 'Left to Right', 'alpha-core' ),
						'btn-draw-top-bottom' => esc_html__( 'Top to Bottom', 'alpha-core' ),
						'btn-draw-right-left' => esc_html__( 'Right to Left', 'alpha-core' ),
						'btn-draw-bottom-top' => esc_html__( 'Bottom to Top', 'alpha-core' ),
					),
					'condition'   => array(
						'show_icon'     => 'yes',
						'icon[library]' => 'svg',
					),
				)
			);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_alpha_form_fields',
			array(
				/* translators: %s represents theme name.*/
				'label' => sprintf( esc_html__( 'Form Fields', 'alpha-core' ), ALPHA_DISPLAY_NAME ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

			$this->add_control(
				'control_size',
				array(
					'label'   => esc_html__( 'Size', 'alpha-core' ),
					'type'    => Controls_Manager::SELECT,
					'options' => array(
						'xs' => esc_html__( 'Extra Small', 'alpha-core' ),
						'sm' => esc_html__( 'Small', 'alpha-core' ),
						''   => esc_html__( 'Medium', 'alpha-core' ),
						'lg' => esc_html__( 'Large', 'alpha-core' ),
					),
				)
			);

			$this->add_control(
				'control_rounded',
				array(
					'label'   => esc_html__( 'Border Style', 'alpha-core' ),
					'type'    => Controls_Manager::SELECT,
					'options' => array(
						''        => esc_html__( 'Square', 'alpha-core' ),
						'rounded' => esc_html__( 'Rounded', 'alpha-core' ),
						'ellipse' => esc_html__( 'Ellipse', 'alpha-core' ),
						'custom'  => esc_html__( 'Custom', 'alpha-core' ),
					),
				)
			);

			$this->add_responsive_control(
				'control_border_radius',
				array(
					'label'      => esc_html__( 'Border Radius', 'alpha-core' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => array( 'px', 'em', 'rem' ),
					'selectors'  => array(
						'{{WRAPPER}} .wpforms-form .wpforms-field-container input, {{WRAPPER}} .wpforms-form .wpforms-field-container select' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					),
					'condition'  => array(
						'label_floating'  => '',
						'control_rounded' => 'custom',
					),
				)
			);

			$this->add_responsive_control(
				'control_textarea_border_radius',
				array(
					'label'      => esc_html__( 'Textarea Border Radius', 'alpha-core' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => array( 'px', 'em', 'rem' ),
					'selectors'  => array(
						'{{WRAPPER}} .wpforms-form .wpforms-field-container textarea' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					),
					'condition'  => array(
						'label_floating'  => '',
						'control_rounded' => 'custom',
					),
				)
			);

			$this->add_responsive_control(
				'control_border_width',
				array(
					'label'      => esc_html__( 'Border Width', 'alpha-core' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => array( 'px', 'em', 'rem' ),
					'selectors'  => array(
						'{{WRAPPER}} .wpforms-form .wpforms-field-container input, {{WRAPPER}} .wpforms-form .wpforms-field-container select, {{WRAPPER}} .wpforms-form .wpforms-field-container textarea' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					),
					'condition'  => array(
						'label_floating' => '',
					),
				)
			);

			$this->add_responsive_control(
				'control_padding',
				array(
					'label'      => esc_html__( 'Padding', 'alpha-core' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => array( 'px', 'em', 'rem' ),
					'selectors'  => array(
						'{{WRAPPER}} .wpforms-form .wpforms-field-container input, {{WRAPPER}} .wpforms-form .wpforms-field-container select, {{WRAPPER}} .wpforms-form .wpforms-field-container textarea' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					),
					'condition'  => array(
						'label_floating' => '',
					),
				)
			);

			$this->add_control(
				'textarea_height',
				array(
					'type'       => Controls_Manager::SLIDER,
					'label'      => esc_html__( 'The Height of Textarea', 'alpha-core' ),
					'range'      => array(
						'px'  => array(
							'step' => 1,
							'min'  => 0,
							'max'  => 400,
						),
						'em'  => array(
							'step' => 0.1,
							'min'  => 0,
							'max'  => 40,
						),
						'rem' => array(
							'step' => 0.1,
							'min'  => 0,
							'max'  => 40,
						),
					),
					'default'    => array(
						'unit' => 'px',
					),
					'size_units' => array(
						'px',
						'em',
						'rem',
					),
					'selectors'  => array(
						'{{WRAPPER}} .wpforms-form .wpforms-field-container textarea' => 'height: {{SIZE}}{{UNIT}};',
					),
				)
			);

			$this->add_group_control(
				Group_Control_Typography::get_type(),
				array(
					'name'     => 'control_typography',
					'selector' => '{{WRAPPER}} .wpforms-form .wpforms-field-container input, {{WRAPPER}} .wpforms-form .wpforms-field-container select, {{WRAPPER}} .wpforms-form .wpforms-field-container textarea',
				)
			);

			$this->start_controls_tabs( 'form_tabs_form' );

			$this->start_controls_tab(
				'form_tab_form_normal',
				array(
					'label' => esc_html__( 'Normal', 'alpha-core' ),
				)
			);

				$this->add_control(
					'control_color',
					array(
						'label'     => esc_html__( 'Color', 'alpha-core' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => array(
							'{{WRAPPER}} .wpforms-form .wpforms-field-container input, {{WRAPPER}} .wpforms-form .wpforms-field-container select, {{WRAPPER}} .wpforms-form .wpforms-field-container textarea' => 'color: {{VALUE}};',
						),
					)
				);

				$this->add_control(
					'control_bg',
					array(
						'label'     => esc_html__( 'Background Color', 'alpha-core' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => array(
							'{{WRAPPER}} .wpforms-form .wpforms-field-container input, {{WRAPPER}} .wpforms-form .wpforms-field-container select, {{WRAPPER}} .wpforms-form .wpforms-field-container textarea' => 'background-color: {{VALUE}};',
						),
						'condition' => array(
							'label_floating' => '',
						),
					)
				);

				$this->add_control(
					'control_bd',
					array(
						'label'     => esc_html__( 'Border Color', 'alpha-core' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => array(
							'{{WRAPPER}} .wpforms-form .wpforms-field-container input, {{WRAPPER}} .wpforms-form .wpforms-field-container select, {{WRAPPER}} .wpforms-form .wpforms-field-container textarea' => 'border-color: {{VALUE}};',
						),
					)
				);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'form_tab_form_focus',
				array(
					'label' => esc_html__( 'Focus', 'alpha-core' ),
				)
			);

				$this->add_control(
					'control_active_color',
					array(
						'label'     => esc_html__( 'Color', 'alpha-core' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => array(
							'{{WRAPPER}} .wpforms-form .wpforms-field-container input:focus, {{WRAPPER}} .wpforms-form .wpforms-field-container select:focus, {{WRAPPER}} .wpforms-form .wpforms-field-container textarea:focus' => 'color: {{VALUE}};',
						),
					)
				);

				$this->add_control(
					'control_active_bg',
					array(
						'label'     => esc_html__( 'Background Color', 'alpha-core' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => array(
							'{{WRAPPER}} .wpforms-form .wpforms-field-container input:focus, {{WRAPPER}} .wpforms-form .wpforms-field-container select:focus, {{WRAPPER}} .wpforms-form .wpforms-field-container textarea:focus' => 'background-color: {{VALUE}};',
						),
						'condition' => array(
							'label_floating' => '',
						),
					)
				);

				$this->add_control(
					'control_active_bd',
					array(
						'label'     => esc_html__( 'Border Color', 'alpha-core' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => array(
							'{{WRAPPER}} .wpforms-form .wpforms-field-container input:focus, {{WRAPPER}} .wpforms-form .wpforms-field-container select:focus, {{WRAPPER}} .wpforms-form .wpforms-field-container textarea:focus' => 'border-color: {{VALUE}};',
						),
					)
				);

			$this->end_controls_tab();

			$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_alpha_form_fields_label',
			array(
				/* translators: %s represents theme name.*/
				'label' => sprintf( esc_html__( 'Form Fields Label', 'alpha-core' ), ALPHA_DISPLAY_NAME ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

			$this->add_control(
				'label_color',
				array(
					'label'     => esc_html__( 'Color', 'alpha-core' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .wpforms-field-label' => 'color: {{VALUE}};',
					),
				)
			);

			$this->add_group_control(
				Group_Control_Typography::get_type(),
				array(
					'name'     => 'label_typography',
					'selector' => '{{WRAPPER}} form.wpforms-form .wpforms-field-container .wpforms-field-label, {{WRAPPER}} form.wpforms-form .wpforms-field-container .wpforms-field-label-inline',
				)
			);

			$this->add_control(
				'label_floating',
				array(
					'label'       => esc_html__( 'Floating Label', 'alpha-core' ),
					'type'        => Controls_Manager::SWITCHER,
					'default'     => '',
					'description' => esc_html__( 'Input and textarea form control\'s labels are floating when they are focused. Do not use placeholders for this option.', 'alpha-core' ),
				)
			);

			$this->add_control(
				'label_floating_color',
				array(
					'label'     => esc_html__( 'Floating Active Color', 'alpha-core' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .field-float .wpforms-field-label' => 'color: {{VALUE}};',
						'{{WRAPPER}} .wpforms-field.field-float input, {{WRAPPER}} .wpforms-field.field-float textarea' => 'border-bottom-color: {{VALUE}}; box-shadow: 0 1px {{VALUE}};',
					),
					'condition' => array(
						'label_floating' => 'yes',
					),
				)
			);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_alpha_form_button',
			array(
				/* translators: %s represents theme name.*/
				'label' => sprintf( esc_html__( 'Submit Button', 'alpha-core' ), ALPHA_DISPLAY_NAME ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

			$this->add_group_control(
				Group_Control_Typography::get_type(),
				array(
					'name'     => 'btn_typography',
					'selector' => '{{WRAPPER}} .wpforms-form .wpforms-submit-container button.wpforms-submit',
				)
			);

			$this->add_responsive_control(
				'btn_margin',
				array(
					'label'      => esc_html__( 'Margin', 'alpha-core' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => array( 'px', 'em', 'rem' ),
					'selectors'  => array(
						'.elementor-element-{{ID}} .wpforms-container form.wpforms-form .wpforms-submit-container button.wpforms-submit' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					),
				)
			);

			$this->add_responsive_control(
				'btn_border_radius',
				array(
					'label'      => esc_html__( 'Border Radius', 'alpha-core' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => array( 'px', 'em', 'rem', '%' ),
					'selectors'  => array(
						'.elementor-element-{{ID}} .wpforms-container form.wpforms-form .wpforms-submit-container button.wpforms-submit, .elementor-element-{{ID}} .wpforms-container form.wpforms-form .wpforms-submit-container .wpforms-submit.btn-bubble:before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					),
				)
			);

			$this->add_responsive_control(
				'btn_border_width',
				array(
					'label'      => esc_html__( 'Border Width', 'alpha-core' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => array( 'px', 'em', 'rem' ),
					'selectors'  => array(
						'.elementor-element-{{ID}} .wpforms-container form.wpforms-form .wpforms-submit-container button.wpforms-submit, .elementor-element-{{ID}} .wpforms-container form.wpforms-form .wpforms-submit-container .wpforms-submit.btn-bubble:before' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}; border-style: solid',
					),
				)
			);

			$this->add_responsive_control(
				'btn_padding',
				array(
					'label'      => esc_html__( 'Padding', 'alpha-core' ),
					'type'       => Controls_Manager::DIMENSIONS,
					'size_units' => array( 'px', 'em', 'rem' ),
					'selectors'  => array(
						'{{WRAPPER}} .wpforms-form .wpforms-submit-container button.wpforms-submit' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					),
					'condition'  => array(
						'bg_hover_effect!' => 'btn-bubble',
					),
				)
			);

			$this->add_responsive_control(
				'icon_space',
				array(
					'label'       => esc_html__( 'Icon Spacing (px)', 'alpha-core' ),
					'description' => esc_html__( 'Controls the spacing between label and icon.', 'alpha-core' ),
					'type'        => Controls_Manager::SLIDER,
					'range'       => array(
						'px' => array(
							'step' => 1,
							'min'  => 1,
							'max'  => 30,
						),
					),
					'selectors'   => array(
						'{{WRAPPER}} .wpforms-submit-container .wpforms-submit' => '--alpha-btn-icon-spacing: {{SIZE}}px;',
					),
				)
			);

			$this->add_responsive_control(
				'icon_size',
				array(
					'label'       => esc_html__( 'Icon Size', 'alpha-core' ),
					'description' => esc_html__( 'Controls the size of the icon. In pixels.', 'alpha-core' ),
					'type'        => Controls_Manager::SLIDER,
					'size_units'  => array( 'px', 'rem' ),
					'range'       => array(
						'px' => array(
							'step' => 1,
							'min'  => 1,
							'max'  => 50,
						),
					),
					'selectors'   => array(
						'{{WRAPPER}} .wpforms-submit-container .wpforms-submit' => '--alpha-btn-icon-size: {{SIZE}}{{UNIT}};',
					),
				)
			);

			$this->start_controls_tabs( 'form_tabs_btn_cat' );

			$this->start_controls_tab(
				'form_tab_btn_normal',
				array(
					'label' => esc_html__( 'Normal', 'alpha-core' ),
				)
			);

			$this->add_control(
				'form_btn_color',
				array(
					'label'     => esc_html__( 'Color', 'alpha-core' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .wpforms-form .wpforms-submit-container .wpforms-submit' => '--alpha-btn-color: {{VALUE}};',
					),
				)
			);

			$this->add_control(
				'form_btn_back_color',
				array(
					'label'     => esc_html__( 'Background Color', 'alpha-core' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .wpforms-form .wpforms-submit-container .wpforms-submit' => '--alpha-btn-bg-color: {{VALUE}};',
					),
				)
			);

			$this->add_control(
				'form_btn_border_color',
				array(
					'label'     => esc_html__( 'Border Color', 'alpha-core' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .wpforms-form .wpforms-submit-container .wpforms-submit' => '--alpha-btn-bd-color: {{VALUE}};',
					),
				)
			);

			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				array(
					'name'     => 'form_btn_box_shadow',
					'selector' => '{{WRAPPER}} .wpforms-form .wpforms-submit-container .wpforms-submit',
				)
			);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'form_tab_btn_hover',
				array(
					'label' => esc_html__( 'Hover', 'alpha-core' ),
				)
			);

			$this->add_control(
				'form_btn_color_hover',
				array(
					'label'     => esc_html__( 'Color', 'alpha-core' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .wpforms-form .wpforms-submit-container .wpforms-submit' => '--alpha-btn-hover-color: {{VALUE}}; --alpha-btn-active-color: {{VALUE}};',
					),
				)
			);

			$this->add_control(
				'form_btn_back_color_hover',
				array(
					'label'     => esc_html__( 'Background Color', 'alpha-core' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .wpforms-form .wpforms-submit-container .wpforms-submit' => '--alpha-btn-hover-bg-color: {{VALUE}}; --alpha-btn-active-bg-color: {{VALUE}};',
					),
				)
			);

			$this->add_control(
				'form_btn_border_color_hover',
				array(
					'label'     => esc_html__( 'Border Color', 'alpha-core' ),
					'type'      => Controls_Manager::COLOR,
					'selectors' => array(
						'{{WRAPPER}} .wpforms-form .wpforms-submit-container .wpforms-submit' => '--alpha-btn-hover-bd-color: {{VALUE}}; --alpha-btn-active-bd-color: {{VALUE}};',
					),
				)
			);

			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				array(
					'name'     => 'form_btn_box_shadow_hover',
					'selector' => '{{WRAPPER}} .wpforms-form .wpforms-submit-container .wpforms-submit:hover, {{WRAPPER}} .wpforms-form .wpforms-submit-container .wpforms-submit:focus',
				)
			);

			$this->end_controls_tab();

			$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_alpha_form_comfirm',
			array(
				'label' => esc_html__( 'Confirmation', 'alpha-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

			$this->start_controls_tabs( 'form_tabs_confirm' );

			$this->start_controls_tab(
				'form_tab_confirm_error',
				array(
					'label' => esc_html__( 'Error', 'alpha-core' ),
				)
			);

				$this->add_group_control(
					Group_Control_Typography::get_type(),
					array(
						'name'     => 'error_typography',
						'selector' => '.elementor-element-{{ID}} div.wpforms-container-full .wpforms-form .wpforms-field-container label.wpforms-error',
					)
				);

				$this->add_control(
					'error_color',
					array(
						'label'     => esc_html__( 'Color', 'alpha-core' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => array(
							'.elementor-element-{{ID}} div.wpforms-container-full .wpforms-form .wpforms-field-container label.wpforms-error' => 'color: {{VALUE}};',
							'.elementor-element-{{ID}} div.wpforms-container-full .wpforms-form .wpforms-field-container .wpforms-field .wpforms-has-error .choices__inner, .elementor-element-{{ID}} div.wpforms-container-full .wpforms-form .wpforms-field-container .wpforms-field input.user-invalid, .elementor-element-{{ID}} div.wpforms-container-full .wpforms-form .wpforms-field-container .wpforms-field input.wpforms-error, .elementor-element-{{ID}} div.wpforms-container-full .wpforms-form .wpforms-field-container .wpforms-field select.user-invalid, .elementor-element-{{ID}} div.wpforms-container-full .wpforms-form .wpforms-field-container .wpforms-field select.wpforms-error, .elementor-element-{{ID}} div.wpforms-container-full .wpforms-form .wpforms-field-container .wpforms-field textarea.user-invalid, .elementor-element-{{ID}} div.wpforms-container-full .wpforms-form .wpforms-field-container .wpforms-field textarea.wpforms-error' => 'border-color: {{VALUE}};',
						),
					)
				);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'form_tab_confirm_success',
				array(
					'label' => esc_html__( 'Success', 'alpha-core' ),
				)
			);

				$this->add_group_control(
					Group_Control_Typography::get_type(),
					array(
						'name'     => 'success_typography',
						'selector' => '.elementor-element-{{ID}} .wpforms-confirmation-container-full',
					)
				);

				$this->add_control(
					'success_color',
					array(
						'label'     => esc_html__( 'Color', 'alpha-core' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => array(
							'.elementor-element-{{ID}} .wpforms-confirmation-container-full' => 'color: {{VALUE}};',
						),
					)
				);

				$this->add_control(
					'success_bg_color',
					array(
						'label'     => esc_html__( 'Background Color', 'alpha-core' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => array(
							'.elementor-element-{{ID}} .wpforms-confirmation-container-full' => 'background-color: {{VALUE}};',
						),
					)
				);

				$this->add_control(
					'success_border_color',
					array(
						'label'     => esc_html__( 'Border Color', 'alpha-core' ),
						'type'      => Controls_Manager::COLOR,
						'selectors' => array(
							'.elementor-element-{{ID}} .wpforms-confirmation-container-full' => 'border-color: {{VALUE}};',
						),
					)
				);

			$this->end_controls_tab();

			$this->end_controls_tabs();

		$this->end_controls_section();
	}

	/**
	 * Render widget output.
	 *
	 * @since 1.6.2
	 */
	protected function render() {

		if ( Plugin::$instance->editor->is_edit_mode() ) {
			$this->render_edit_mode();
		} else {
			$this->render_frontend();
		}
	}

	/**
	 * Render widget output in edit mode.
	 *
	 * @since 1.6.3.1
	 */
	protected function render_edit_mode() {

		$form_id = $this->get_settings_for_display( 'form_id' );

		// Popup markup template.
		echo wpforms_render( 'integrations/elementor/popup' ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

		if ( count( $this->get_forms() ) < 2 ) {

			// No forms block.
			echo wpforms_render( 'integrations/elementor/no-forms' ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

			return;
		}

		if ( empty( $form_id ) ) {

			// Render form selector.
			echo wpforms_render( // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				'integrations/elementor/form-selector',
				[
					'forms' => $this->get_form_selector_options(),
				],
				true
			);

			return;
		}

		// Finally, render selected form.
		$this->render_frontend();
	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.6.3.1
	 */
	protected function render_frontend() {

		static $is_root_vars_displayed = false;

		$css_vars_obj = wpforms()->get( 'css_vars' );

		if ( ! empty( $css_vars_obj ) && wpforms_get_render_engine() === 'modern' && ! $is_root_vars_displayed ) {
			$css_vars_obj->output_root( true );

			$is_root_vars_displayed = true;
		}

		// Render selected form.
		echo do_shortcode( $this->render_shortcode() );
	}

	/**
	 * Render widget as plain content.
	 *
	 * @since 1.6.2
	 */
	public function render_plain_content() {

		echo alpha_escaped( $this->render_shortcode() ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}

	/**
	 * Render shortcode.
	 *
	 * @since 1.6.2
	 */
	public function render_shortcode() {

		$settings = $this->get_settings_for_display();

		$icon = array(
			'icon_class'   => '',
			'icon_url'     => '',
			'icon_id'      => '',
			'icon_library' => '',
		);

		if ( $settings['icon'] ) {
			if ( 'svg' == $settings['icon']['library'] ) {
				if ( ! empty( $settings['icon']['value']['url'] ) ) {
					$icon['icon_url']     = $settings['icon']['value']['url'];
					$icon['icon_id']      = $settings['icon']['value']['id'];
					$icon['icon_library'] = 'svg';
				}
			} else {
				$icon['icon_class']   = $settings['icon']['value'];
				$icon['icon_library'] = $settings['icon']['library'];
			}
		}

		return sprintf(
			'[wpforms id="%1$d" title="%2$s" description="%3$s" settings="%4$s"]',
			absint( $this->get_settings_for_display( 'form_id' ) ),
			sanitize_key( $this->get_settings_for_display( 'display_form_name' ) === 'yes' ? 'true' : 'false' ),
			sanitize_key( $this->get_settings_for_display( 'display_form_description' ) === 'yes' ? 'true' : 'false' ),
			implode(
				',',
				array(
					$settings['text_hover_effect'],
					$settings['bg_hover_effect'],
					$settings['show_icon'],
					$settings['show_label'],
					$icon['icon_class'],
					$icon['icon_url'],
					$icon['icon_id'],
					$icon['icon_library'],
					$settings['icon_pos'],
					$settings['icon_hover_effect'],
					$settings['icon_hover_effect_infinite'],
					$settings['svg_hover_effect'],
				)
			)
		);
	}

	/**
	 * Get forms list.
	 *
	 * @since 1.6.2
	 *
	 * @returns array Array of forms.
	 */
	public function get_forms() {

		static $forms_list = [];

		if ( empty( $forms_list ) ) {
			$forms = wpforms()->form->get();

			if ( ! empty( $forms ) ) {
				$forms_list[0] = esc_html__( 'Select a form', 'wpforms-lite' );
				foreach ( $forms as $form ) {
					$forms_list[ $form->ID ] = mb_strlen( $form->post_title ) > 100 ? mb_substr( $form->post_title, 0, 97 ) . '...' : $form->post_title;
				}
			}
		}

		return $forms_list;
	}

	/**
	 * Get form selector options.
	 *
	 * @since 1.6.2
	 *
	 * @returns string Rendered options for the select tag.
	 */
	public function get_form_selector_options() {

		$forms   = $this->get_forms();
		$options = '';

		foreach ( $forms as $form_id => $form ) {
			$options .= sprintf(
				'<option value="%d">%s</option>',
				(int) $form_id,
				esc_html( $form )
			);
		}

		return $options;
	}
}
