<?php
/**
 * Dynamic CSS functions
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Get dynamic CSS
 */
function uncode_get_page_dynamic_css() {
	global $uncode_post_data, $uncode_dynamic_css_selectors, $row_sticky_columns;

	$uncode_dynamic_css_selectors = array();

	// Get an array that contains all the raw content attached to the page
	$content_array = uncode_get_post_data_content_array();

	// Generate custom CSS for the custom colors
	$dynamic_css = uncode_get_dynamic_colors_css( $content_array );

	// Generate custom CSS for the sticky scroll indexes
	$dynamic_css .= uncode_get_dynamic_sticky_scroll_css( $content_array );

	// Generate custom CSS for css grids
	$dynamic_css .= uncode_get_dynamic_css_grid_css( $content_array );

	// Generate custom CSS font size
	$dynamic_css .= uncode_get_dynamic_css_font_size( $content_array );

	// Generate custom CSS text indent
	$dynamic_css .= uncode_get_dynamic_css_text_indent( $content_array );

	// Generate custom CSS nav_carousel
	$dynamic_css .= uncode_get_dynamic_css_nav_carousel( $content_array );

	// Generate custom CSS for linear grids
	$dynamic_css .= uncode_get_dynamic_linear_grid_css( $content_array );

	return $dynamic_css;
}

/**
 * Get dynamic CSS generated by custom colors
 */
function uncode_get_dynamic_colors_css( $content_array ) {
	$css = '';

	foreach ( $content_array as $content ) {
		if ( strpos( $content, 'uncode-gradient' ) !== false || strpos( $content, 'uncode-solid' ) !== false ) {
			$css .= uncode_get_single_dynamic_colors_css( $content );
		}
	}

	return $css;
}

/**
 * Get dynamic colors CSS from a piece of content
 */
function uncode_get_single_dynamic_colors_css( $content ) {
	$css = '';

	$regex = '/\[([^\[]+) [^\s]+_type=\"(uncode-solid|uncode-gradient)\".*?\]/m';
	preg_match_all( $regex, $content, $shortcodes, PREG_SET_ORDER, 0 );

	$shortcodes_with_colors = uncode_get_shortcodes_with_colors( $shortcodes );

	foreach ( $shortcodes_with_colors as $shortcode ) {
		$shortcode_data = uncode_get_shortcodes_with_colors_data( $shortcode );

		if ( count( $shortcode_data ) > 0 ) {
			$css .= uncode_get_dynamic_colors_css_from_shortcode( $shortcode_data );
		}
	}

	return $css;
}

/**
 * Get shortcodes that have a custom color
 */
function uncode_get_shortcodes_with_colors( $shortcodes ) {
	$shortcodes_with_colors = array();

	foreach ( $shortcodes as $key => $value ) {
		if ( isset( $value[0] ) && $value[0] && isset( $value[2] ) && $value[2] ) {
			$shortcodes_with_colors[] = $value[0];
		}
	}

	return $shortcodes_with_colors;
}

/**
 * Get data of each shortcode that has a custom color
 */
function uncode_get_shortcodes_with_colors_data( $shortcode ) {
	$regex_id = '/\[(.+?\s+)/';
	preg_match_all( $regex_id, $shortcode, $shortcode_id, PREG_SET_ORDER, 0 );

	$shortcode_data = array(
		'type'       => '',
		'id'         => '',
		'attributes' => array(),
	);

	if ( isset( $shortcode_id[0] ) && $shortcode_id[0] && isset( $shortcode_id[0][1] ) && $shortcode_id[0][1] ) {
		$shortcode_id = trim( $shortcode_id[0][1] );

		if ( $shortcode_id ) {
			$regex_attr = '/ (.*?)=\"(.*?)\"/';
			preg_match_all( $regex_attr, $shortcode, $attributes, PREG_SET_ORDER, 0 );

			$shortcode_data['type'] = $shortcode_id;

			foreach ( $attributes as $attribute ) {
				if ( isset( $attribute[1] ) && $attribute[1] && isset( $attribute[2] ) && $attribute[2] ) {
					if ( $attribute[1] === 'uncode_shortcode_id' ) {
						$shortcode_data['id'] = absint( $attribute[2] );
					} else {
						$shortcode_data['attributes'][$attribute[1]] = $attribute[2];
					}
				}
			}
		}
	}

	if ( $shortcode_data['id'] > 0 ) {
		return $shortcode_data;
	}

	return array();
}

/**
 * Get CSS from custom color shortcode
 */
function uncode_get_dynamic_colors_css_from_shortcode( $shortcode ) {
	global $uncode_dynamic_css_selectors;

	$css = '';

	if ( isset( $shortcode['type'] ) && isset( $shortcode['id'] ) && isset( $shortcode['attributes'] ) && is_array( $shortcode['attributes'] ) ) {
		// Return early if we don't support this module
		if ( ! in_array( $shortcode['type'], uncode_get_modules_with_dynamic_colors() ) ) {
			return $css;
		}

		$shortcode_id = absint( $shortcode['id'] );

		// Check if we have already processed the same ID
		// (ie. a cloned module)
		if ( is_array( $uncode_dynamic_css_selectors ) && in_array( $shortcode_id, $uncode_dynamic_css_selectors ) ) {
			return $css;
		}

		$custom_color_keys = array();

		foreach ( $shortcode['attributes'] as $attributes_key => $attributes_value ) {
			if ( $attributes_value === 'uncode-solid' || $attributes_value === 'uncode-gradient' ) {
				$custom_color_keys[] = str_replace( '_type', '', $attributes_key );
			}
			if ( $attributes_key === 'text_stroke' && $attributes_value === 'yes' ) {
				$custom_color_keys[] = 'text_stroke';
			}

		}

		$shortcode_function_name = 'uncode_get_dynamic_colors_css_for_shortcode_' . str_replace( '-', '_', $shortcode['type'] );

		if ( function_exists( $shortcode_function_name ) ) {
			$css .= call_user_func_array( $shortcode_function_name, array( $shortcode, $custom_color_keys ) );
		}

		$uncode_dynamic_css_selectors[] = $shortcode['id'];
	}

	return $css;
}

/**
 * List of modules with dynamic colors
 */
function uncode_get_modules_with_dynamic_colors() {
	$modules = array(
		'uncode_author_profile',
		'vc_button',
		'vc_column',
		'vc_column_inner',
		'uncode_consent_notice',
		'vc_accordion',
		'vc_custom_heading',
		'vc_icon',
		'vc_gallery',
		'vc_message',
		'vc_row',
		'vc_row_inner',
		'vc_section',
		'vc_single_image',
		'vc_column_text',
		'vc_separator',
		'vc_gmaps',
		'vc_pie',
		'vc_tabs',
		'uncode_pricing',
		'uncode_pricing_list',
		'uncode_star_rating',
		'uncode_list',
		'uncode_index',
		'vc_gallery',
		'vc_single_image',
		'uncode_counter',
		'uncode_vertical_text',
		'uncode_woocommerce_cart',
		'uncode_woocommerce_checkout',
		'uncode_woocommerce_my_account',
		'uncode_custom_fields'
	);

	return $modules;
}

/**
 * Get dynamic CSS generated by sticky scroll
 */
function uncode_get_dynamic_sticky_scroll_css( $content_array ) {
	$menutype = ot_get_option('_uncode_headers');

	$css = '';

	foreach ( $content_array as $content ) {
		if ( strpos( $content, 'index_type="sticky-scroll"' ) !== false ) {
			$css .= uncode_get_single_dynamic_sticky_scroll_css( $content );
		}
	}

	return $css;
}

/**
 * Get dynamic sticky scroll CSS from a piece of content
 */
function uncode_get_single_dynamic_sticky_scroll_css( $content ) {
	$css = '';

	$regex_index_vw = '/\[[uncode_index|vc_gallery]([^\]]+)type=\"sticky-scroll\".*?\]/m';
	preg_match_all( $regex_index_vw, $content, $indexes_vw, PREG_SET_ORDER, 0 );

	foreach ( $indexes_vw as $index_vw ) {

		$params = $index_vw[0];
		preg_match('/ uncode_shortcode_id=\"(.*?)"/', $params, $id); //row ID
		preg_match('/ sticky_th_grid_lg=\"(.*?)"/', $params, $lg);
		preg_match('/ sticky_th_grid_md=\"(.*?)"/', $params, $md);
		preg_match('/ no_sticky_scroll_tablet=\"(.*?)"/', $params, $no_md);
		preg_match('/ sticky_th_grid_sm=\"(.*?)"/', $params, $sm);
		preg_match('/ no_sticky_scroll_mobile=\"(.*?)"/', $params, $no_sm);

		$shortcode_data = array(
			'size' => 'vw',
			'id' => $id[1],
			'lg' => isset($lg[1]) ? $lg[1] : 3,
			'md' => isset($no_md[1]) ? 1 : (isset($md[1]) ? $md[1] : 3),
			'sm' => isset($no_sm[1]) ? 1 : (isset($sm[1]) ? $sm[1] : 1),
		);
		$css .= uncode_get_dynamic_sticky_scroll_css_from_shortcode( $shortcode_data );

	}

	return $css;
}

/**
 * Get CSS from sticky scroll shortcode
 */
function uncode_get_dynamic_sticky_scroll_css_from_shortcode( $shortcode_data ) {
	global $uncode_dynamic_css_selectors, $row_sticky_columns;

	$css = '';

	if ( isset( $shortcode_data['id'] ) && isset( $shortcode_data['lg'] ) ) {

		$shortcode_id = absint( $shortcode_data['id'] );

		// Check if we have already processed the same ID
		// (ie. a cloned module)
		if ( is_array( $uncode_dynamic_css_selectors ) && in_array( 'hor-scroll-'.$shortcode_id, $uncode_dynamic_css_selectors ) ) {
			return $css;
		}

		if ( function_exists( 'uncode_get_sticky_scroll_css_for_shortcode_index' ) ) {
			$css .= uncode_get_sticky_scroll_css_for_shortcode_index( $shortcode_data );
		}

		$uncode_dynamic_css_selectors[] = 'hor-scroll-'.$shortcode_data['id'];
	}

	return $css;
}

/**
 * Get dynamic CSS generated by linear grids
 */
function uncode_get_dynamic_linear_grid_css( $content_array ) {
	$css = '';

	foreach ( $content_array as $content ) {
		if ( strpos( $content, 'type="linear"' ) !== false ) {
			$css .= uncode_get_single_dynamic_linear_grid_css( $content );
		}
	}

	return $css;
}

/**
 * Get dynamic linear grids CSS from a piece of content
 */
function uncode_get_single_dynamic_linear_grid_css( $content ) {
	$css = '';

	$regex_index = '/\[[uncode_index|vc_gallery]([^\]]+)type=\"linear\".*?\]/m';
	preg_match_all( $regex_index, $content, $indexes, PREG_SET_ORDER, 0 );

	foreach ( $indexes as $index ) {
		$params = $index[0];

		preg_match('/ uncode_shortcode_id=\"(.*?)"/', $params, $id); //row ID
		preg_match('/ size_by=\"(.*?)"/', $params, $by);
		preg_match('/ linear_width=\"(.*?)"/', $params, $width);
		preg_match('/ linear_height=\"(.*?)"/', $params, $height);
		preg_match('/ single_text=\"(.*?)"/', $params, $layout);
		preg_match('/ single_image_size=\"(.*?)"/', $params, $size);

		$shortcode_data = array(
			'id'		=> $id[1],
			'by'		=> isset( $by[1] ) ? $by[1] : '',
			'w'			=> isset( $width[1] ) ? $width[1] : '',
			'h'			=> isset( $height[1] ) ? $height[1] : '',
			'layout'	=> isset( $layout[1] ) ? $layout[1] : '',
			'size'		=> isset( $size[1] ) ? $size[1] : ''
		);
		$css .= uncode_get_dynamic_linear_grids_css_from_shortcode( $shortcode_data );
	}

	return $css;
}

/**
 * Get CSS from custom grids shortcode
 */
function uncode_get_dynamic_linear_grids_css_from_shortcode( $shortcode_data ) {
	global $uncode_dynamic_css_selectors;

	$css = '';

	if ( isset( $shortcode_data['id'] ) && ( isset( $shortcode_data['w'] ) || isset( $shortcode_data['h'] ) ) ) {

		$shortcode_id = absint( $shortcode_data['id'] );

		// Check if we have already processed the same ID
		// (ie. a cloned module)
		if ( is_array( $uncode_dynamic_css_selectors ) && in_array( 'lineargrid-'.$shortcode_id, $uncode_dynamic_css_selectors ) ) {
			return $css;
		}

		if ( function_exists( 'uncode_get_linear_grids_css_for_shortcode_vc_gallery' ) ) {
			$css .= uncode_get_linear_grids_css_for_shortcode_vc_gallery( $shortcode_data );
		}

		$uncode_dynamic_css_selectors[] = 'lineargrid-'.$shortcode_data['id'];
	}

	return $css;
}

/**
 * Get dynamic CSS generated by css grids
 */
function uncode_get_dynamic_css_grid_css( $content_array ) {
	$css = '';

	foreach ( $content_array as $content ) {
		if ( strpos( $content, 'type="css_grid"' ) !== false ) {
			$css .= uncode_get_single_dynamic_css_grid_css( $content );
		}
	}

	return $css;
}

/**
 * Get dynamic css grids CSS from a piece of content
 */
function uncode_get_single_dynamic_css_grid_css( $content ) {
	$css = '';

	$regex_index = '/\[[uncode_index|vc_gallery]([^\]]+)type=\"css_grid\".*?\]/m';
	preg_match_all( $regex_index, $content, $indexes, PREG_SET_ORDER, 0 );

	foreach ( $indexes as $index ) {
		$params = $index[0];

		preg_match('/ uncode_shortcode_id=\"(.*?)"/', $params, $id); //row ID
		preg_match('/ grid_items=\"(.*?)"/', $params, $items);
		preg_match('/ screen_lg_breakpoint=\"(.*?)"/', $params, $lg);
		preg_match('/ screen_lg_items=\"(.*?)"/', $params, $lg_items);
		preg_match('/ screen_md_breakpoint=\"(.*?)"/', $params, $md);
		preg_match('/ screen_md_items=\"(.*?)"/', $params, $md_items);
		preg_match('/ screen_sm_breakpoint=\"(.*?)"/', $params, $sm);
		preg_match('/ screen_sm_items=\"(.*?)"/', $params, $sm_items);

		$shortcode_data = array(
			'id'       => $id[1],
			'items'    => isset( $items[1] ) ? $items[1] : 4,
			'lg'       => isset( $lg[1] ) ? $lg[1] : 1000,
			'lg_items' => isset( $lg_items[1] ) ? $lg_items[1] : 3,
			'md'       => isset( $md[1] ) ? $md[1] : 600,
			'md_items' => isset( $md_items[1] ) ? $md_items[1] : 2,
			'sm'       => isset( $sm[1] ) ? $sm[1] : 480,
			'sm_items' => isset( $sm_items[1] ) ? $sm_items[1] : 1,
		);
		$css .= uncode_get_dynamic_css_grids_css_from_shortcode( $shortcode_data );
	}

	return $css;
}

/**
 * Get CSS from custom grids shortcode
 */
function uncode_get_dynamic_css_grids_css_from_shortcode( $shortcode_data ) {
	global $uncode_dynamic_css_selectors;

	$css = '';

	if ( isset( $shortcode_data['id'] ) && isset( $shortcode_data['lg'] ) ) {

		$shortcode_id = absint( $shortcode_data['id'] );

		// Check if we have already processed the same ID
		// (ie. a cloned module)
		if ( is_array( $uncode_dynamic_css_selectors ) && in_array( 'cssgrid-'.$shortcode_id, $uncode_dynamic_css_selectors ) ) {
			return $css;
		}

		if ( function_exists( 'uncode_get_css_grids_css_for_shortcode_index' ) ) {
			$css .= uncode_get_css_grids_css_for_shortcode_index( $shortcode_data );
		}

		$uncode_dynamic_css_selectors[] = 'cssgrid-'.$shortcode_data['id'];
	}

	return $css;
}

/**
 * Get dynamic font size CSS
 */
function uncode_get_dynamic_css_font_size( $content_array ) {
	$css = '';

	foreach ( $content_array as $content ) {
		if ( strpos( $content, 'custom_size="' ) !== false ) {
			$css .= uncode_get_single_dynamic_css_font_size( $content );
		}
	}

	return $css;
}

/**
 * Get dynamic font size CSS from a piece of content
 */
function uncode_get_single_dynamic_css_font_size( $content ) {
	$css = '';

	$regex_index = '/\[(\w+)([^\[]*?)heading_custom_size="(.*?)"(.*?)\]/m';
	preg_match_all( $regex_index, $content, $indexes, PREG_SET_ORDER, 0 );

	foreach ( $indexes as $index ) {
		$params = $index[0];

		preg_match('/ uncode_shortcode_id=\"(.*?)"/', $params, $id); //row ID
		preg_match('/ heading_custom_size=\"(.*?)"/', $params, $size);

		$shortcode_data = array(
			'id'        => $id[1],
			'font_size' => $size[1],
		);
		$css .= uncode_get_dynamic_css_font_size_shortcode( $shortcode_data );
	}

	$regex_index_2 = '/\[(uncode_star_rating|uncode_custom_fields)([^\[]*?)custom_size="(.*?)"(.*?)\]/m';
	preg_match_all( $regex_index_2, $content, $indexes_2, PREG_SET_ORDER, 0 );

	foreach ( $indexes_2 as $index_2 ) {
		$params = $index_2[0];

		preg_match('/ uncode_shortcode_id=\"(.*?)"/', $params, $id); //row ID
		preg_match('/ custom_size=\"(.*?)"/', $params, $size);

		$shortcode_data = array(
			'id'        => $id[1],
			'font_size' => $size[1],
		);
		$css .= uncode_get_dynamic_css_font_size_shortcode( $shortcode_data );
	}

	return $css;
}

/**
 * Get CSS from font size shortcode
 */
function uncode_get_dynamic_css_font_size_shortcode( $shortcode_data ) {
	global $uncode_dynamic_css_selectors;

	$css = '';

	if ( isset( $shortcode_data['id'] ) ) {

		$shortcode_id = absint( $shortcode_data['id'] );
		$font_size = $shortcode_data['font_size'];

		// Check if we have already processed the same ID
		// (ie. a cloned module)
		if ( is_array( $uncode_dynamic_css_selectors ) && in_array( 'fontsize-'.$shortcode_id . '-custom', $uncode_dynamic_css_selectors ) ) {
			return $css;
		}

		if ( is_numeric($font_size) ) {
			$font_size .= 'px';
		}
		$numeric_val = str_replace('px', '', $font_size);
		if ( is_numeric($numeric_val) && strpos($font_size, 'px') !== false ) {
			$css .= '.fontsize-' . $shortcode_id . '-custom { font-size: ' . $font_size . '; }';
			$first_mquery = $numeric_val / 1.5;
			if ($numeric_val > 35) {
				$css .= '@media (max-width: 959px) { .fontsize-' . $shortcode_id . '-custom { font-size: ' . $first_mquery . 'px; }}';
				if ($first_mquery > 35) {
					$css .= '@media (max-width: 569px) { .fontsize-' . $shortcode_id . '-custom { font-size: 35px; }}';
				}
			}
			if ($first_mquery > 28) {
				$css .= '@media (max-width: 320px) { .fontsize-' . $shortcode_id . '-custom { font-size: 28px; }}';
			}
		} else {
			$css .= '.fontsize-' . $shortcode_id . '-custom { font-size:' . $font_size . ' }';
		}

		$uncode_dynamic_css_selectors[] = 'font_size-'.$shortcode_id . '-custom';
	}

	return $css;
}



/**
 * Get dynamic text indent CSS
 */
function uncode_get_dynamic_css_text_indent( $content_array ) {
	$css = '';

	foreach ( $content_array as $content ) {
		if ( preg_match( '/\[vc_custom_heading[^\]]+text_indent="(.*?)"/', $content ) !== false ) {
			$css .= uncode_get_single_dynamic_css_text_indent( $content );
		}
	}

	return $css;
}

/**
 * Get dynamic text indent CSS from a piece of content
 */
function uncode_get_single_dynamic_css_text_indent( $content ) {
	$css = '';

	$regex_index = '/\[vc_custom_heading[^\]]+text_indent="(.*?)"/m';
	preg_match_all( $regex_index, $content, $indexes, PREG_SET_ORDER, 0 );

	foreach ( $indexes as $index ) {
		$params = $index[0];

		preg_match('/ uncode_shortcode_id=\"(.*?)"/', $params, $id); //row ID
		preg_match('/ text_indent=\"(.*?)"/', $params, $size);

		$shortcode_data = array(
			'id'        => $id[1],
			'text_indent' => $size[1],
		);
		$css .= uncode_get_dynamic_css_text_indent_shortcode( $shortcode_data );
	}

	return $css;
}

/**
 * Get CSS text indent grids shortcode
 */
function uncode_get_dynamic_css_text_indent_shortcode( $shortcode_data ) {
	global $uncode_dynamic_css_selectors;

	$css = '';

	if ( isset( $shortcode_data['id'] ) ) {

		$shortcode_id = absint( $shortcode_data['id'] );
		$text_indent = $shortcode_data['text_indent'];

		// Check if we have already processed the same ID
		// (ie. a cloned module)
		if ( is_array( $uncode_dynamic_css_selectors ) && in_array( 'text-'.$shortcode_id . '-indent', $uncode_dynamic_css_selectors ) ) {
			return $css;
		}

		if ( is_numeric($text_indent) ) {
			$text_indent .= 'px';
		}
		$css .= '@media (min-width: 570px) { ';
		$css .= '.text-' . $shortcode_id . '-indent { text-indent: ' . $text_indent . '; }';
		$css .= '.text-' . $shortcode_id . '-indent .heading-foreword { margin-left: -' . $text_indent . '; margin-right: ' . $text_indent . '; text-indent: 0; width: 0; }';
		$css .= '.text-' . $shortcode_id . '-indent span:not(:first-child) { text-indent: 0; }';
		$css .= ' }';

		$uncode_dynamic_css_selectors[] = 'text-'.$shortcode_id . '-indent:first-child';
	}

	return $css;
}

/**
 * Get dynamic text indent CSS
 */
function uncode_get_dynamic_css_nav_carousel( $content_array ) {
	$css = '';

	foreach ( $content_array as $content ) {
		if ( preg_match( '/\[uncode_carousel_nav[^\]]+[dot_single_width|dot_single_height|dot_single_space|dot_single_radius|dot_single_active|dot_single_boundary|dot_number_align|counter_index_width]="(.*?)"/', $content ) !== false ) {
			$css .= uncode_get_single_dynamic_css_nav_carousel( $content );
		}
	}

	return $css;
}

/**
 * Get dynamic text indent CSS from a piece of content
 */
function uncode_get_single_dynamic_css_nav_carousel( $content ) {
	$css = '';

	$regex_index = '/\[uncode_carousel_nav[^\]]+[dot_single_width|dot_single_height|dot_single_space|dot_single_radius|dot_single_active|dot_single_boundary|dot_number_align|counter_index_width]="(.*?)"/m';
	preg_match_all( $regex_index, $content, $indexes, PREG_SET_ORDER, 0 );

	foreach ( $indexes as $index ) {
		$params = $index[0];

		preg_match('/ uncode_shortcode_id=\"(.*?)"/', $params, $id); //row ID
		preg_match('/ dot_single_width=\"(.*?)"/', $params, $width);
		preg_match('/ dot_single_height=\"(.*?)"/', $params, $height);
		preg_match('/ dot_single_space=\"(.*?)"/', $params, $space);
		preg_match('/ dot_single_radius=\"(.*?)"/', $params, $radius);
		preg_match('/ dot_single_active=\"(.*?)"/', $params, $active);
		preg_match('/ dot_single_boundary=\"(.*?)"/', $params, $boundary);
		preg_match('/ dot_number_align=\"(.*?)"/', $params, $align);
		preg_match('/ counter_index_width=\"(.*?)"/', $params, $counter_w);
		preg_match('/ dots_style=\"(.*?)"/', $params, $style);

		$shortcode_data = array(
			'id'        => $id[1],
			'width' => isset($width[1]) ? $width[1] : '',
			'height' => isset($height[1]) ? $height[1] : '',
			'space' => isset($space[1]) ? $space[1] : '',
			'radius' => isset($radius[1]) ? $radius[1] : '',
			'active' => isset($active[1]) ? $active[1] : '',
			'boundary' => isset($boundary[1]) ? $boundary[1] : '',
			'align' => isset($align[1]) ? $align[1] : '',
			'counter_w' => isset($counter_w[1]) ? $counter_w[1] : '',
			'style' => isset($style[1]) ? $style[1] : '',
		);
		$css .= uncode_get_dynamic_css_nav_carousel_shortcode( $shortcode_data );
	}

	return $css;
}

/**
 * Get CSS text indent grids shortcode
 */
function uncode_get_dynamic_css_nav_carousel_shortcode( $shortcode_data ) {
	global $uncode_dynamic_css_selectors;

	$css = '';

	if ( isset( $shortcode_data['id'] ) ) {

		$shortcode_id = absint( $shortcode_data['id'] );
		$width = isset($shortcode_data['width']) ? $shortcode_data['width'] : '';
		$height = isset($shortcode_data['height']) ? $shortcode_data['height'] : '';
		$space = isset($shortcode_data['space']) ? $shortcode_data['space'] : '';
		$radius = isset($shortcode_data['radius']) ? $shortcode_data['radius'] : '';
		$active = isset($shortcode_data['active']) ? $shortcode_data['active'] : '';
		$boundary = isset($shortcode_data['boundary']) ? $shortcode_data['boundary'] : '';
		$counter_w = isset($shortcode_data['counter_w']) ? $shortcode_data['counter_w'] : '';
		$align = isset($shortcode_data['align']) ? $shortcode_data['align'] : '';
		$style = isset($shortcode_data['style']) ? $shortcode_data['style'] : '';

		// Check if we have already processed the same ID
		// (ie. a cloned module)
		if ( is_array( $uncode_dynamic_css_selectors ) && in_array( 'carousel-nav-'.$shortcode_id . '-custom', $uncode_dynamic_css_selectors ) ) {
			return $css;
		}

		$return = true;

		if ( $width !== '' ) {
			if ( !is_numeric($width) ) {
				$width = floatval($width);
			}
			if ( is_numeric($width) ) {
				$width .= 'px';
				$return = false;
			}
		}

		if ( $height !== '' ) {
			if ( !is_numeric($height) ) {
				$height = floatval($height);
			}
			if ( is_numeric($height) ) {
				$height .= 'px';
				$return = false;
			}
		}

		if ( $space !== '' ) {
			if ( !is_numeric($space) ) {
				$space = floatval($space);
			}
			if ( is_numeric($space) ) {
				$space .= 'px';
				$return = false;
			}
		}

		if ( $radius !== '' ) {
			if ( !is_numeric($radius) ) {
				$radius = floatval($radius);
			}
			if ( is_numeric($radius) ) {
				$radius .= 'px';
				$return = false;
			}
		}

		if ( $boundary !== '' ) {
			if ( !is_numeric($boundary) ) {
				$boundary = floatval($boundary);
			}
			if ( is_numeric($boundary) ) {
				$boundary = ($boundary*2) . 'px';
				$return = false;
			}
		}

		$active_scale = 1;
		if ( $active !== '' ) {
			if ( !is_numeric($active) ) {
				$active = floatval($active);
			}
			if ( is_numeric($active) ) {
				$def_w = $style === '' ? 8 : 20;
				if ( $width !== '' ) {
					$def_w = floatval($width);
				}
				$active_scale = $active / $def_w;
				$active .= 'px';
				$return = false;
			}
		}

		if ( $counter_w !== '' ) {
			if ( !is_numeric($counter_w) ) {
				$counter_w = floatval($counter_w);
			}
			if ( is_numeric($counter_w) ) {
				$counter_w = $counter_w . 'px';
				$return = false;
			}
		}

		if ( $return === false ) {
			//$css .= '@media (min-width: 570px) { ';
			$css_1 = '';
			if ( $width !== '' ) {
				$css_1 .= 'width: ' . $width . '; height: ' . $width . ';';
			}
			if ( $space !== '' ) {
				$css_1 .= 'margin: 0 ' . $space . ';';
			}
			if ( $radius !== '' ) {
				$css_1 .= 'border-radius: ' . $radius . ';';
			}
			if ( $css_1 !== '' ) {
				$css .= '.carousel-nav-' . $shortcode_id . '-custom.dots-default .uncode-nav-dots .uncode-nav-index span { ';
					$css .= $css_1;
				$css .= ' }';
			}

			$css_2 = '';
			if ( $width !== '' ) {
				$css_2 .= 'width: ' . $width . ';';
			}
			if ( $height !== '' ) {
				$css_2 .= 'height: ' . $height . ';';
			}
			if ( $space !== '' ) {
				$css_2 .= 'margin: 0 ' . $space . ';';
			}
			if ( $radius !== '' ) {
				$css_2 .= 'border-radius: ' . $radius . ';';
			}
			if ( $css_2 !== '' ) {
				$css .= '.carousel-nav-' . $shortcode_id . '-custom.dots-lines .uncode-nav-dots .uncode-nav-index span { ';
					$css .= $css_2;
				$css .= ' }';
			}

			$css_3 = '';
			if ( $width !== '' ) {
				$css_3 .= 'min-width: ' . $width . ';';
			}
			if ( $space !== '' ) {
				if ( $align === 'right' ) {
					$css_3 .= 'margin: 0 0 0 ' . $space . ';';
					$css_3 .= 'text-align: right;';
				} elseif ( $align === 'left' ) {
					$css_3 .= 'margin: 0 ' . $space . ' 0 0;';
					$css_3 .= 'text-align: left;';
				} else {
					$css_3 .= 'margin: 0 ' . $space . ';';
					$css_3 .= 'text-align: center;';
				}
			}
			if ( $css_3 !== '' ) {
				$css .= '.carousel-nav-' . $shortcode_id . '-custom.dots-numbers .uncode-nav-dots .uncode-nav-index span { ';
					$css .= $css_3;
				$css .= ' }';
			}

			if ( $active !== '' ) {
				if ( $active_scale !== 1 ) {
					$css .= '.carousel-nav-' . $shortcode_id . '-custom.dots-default .uncode-nav-dots .uncode-nav-index span { ';
						$css .= 'height: ' . $active . ';';
						$css .= 'width: ' . $active . ';';
						$css .= 'transform: scale(' . 1/$active_scale . ');';
					$css .= ' }';
					$css .= '.carousel-nav-' . $shortcode_id . '-custom.dots-default .uncode-nav-dots .uncode-nav-index.active-index span, ';
					$css .= '.carousel-nav-' . $shortcode_id . '-custom.dots-default .uncode-nav-dots .uncode-nav-index:hover span { ';
						$css .= 'transform: scale(1);';
					$css .= ' }';
				}
				$css .= '.carousel-nav-' . $shortcode_id . '-custom.dots-lines .uncode-nav-dots .uncode-nav-index.active-index span { ';
					$css .= 'width: ' . $active . ';';
				$css .= ' }';
			}

			if ( $boundary !== '' ) {
				$css .= '.carousel-nav-' . $shortcode_id . '-custom.dots-look-border:not(.dots-numbers).uncode-owl-nav .uncode-nav-index span:after,';
				$css .= '.carousel-nav-' . $shortcode_id . '-custom.dots-look-shadow:not(.dots-numbers).uncode-owl-nav .uncode-nav-index span:after {';
					$css .= 'width: calc(100% + ' . $boundary . '); height: calc(100% + ' . $boundary . ');';
				$css .= ' }';
			}

			if ( $counter_w !== '' ) {
				$css .= '.carousel-nav-' . $shortcode_id . '-custom .uncode-nav-counter {';
					$css .= 'min-width: ' . $counter_w . ';';
					$css .= 'justify-content: center;';
				$css .= ' }';
			}

			$uncode_dynamic_css_selectors[] = 'carousel-nav-' . $shortcode_id . '-custom';
		} else {
			return $css;
		}
	}

	return $css;
}
