/**
 * Main Woffice object
 *
 * @since 2.5.0
 * @type {{}}
 */
var Woffice = {

    /**
     * Initialize the Woffice's JS
     *
     * @param {jQuery} $
     */
    init: function ($) {

        "use strict";

        var self = this;

        self.$ = (typeof $ === 'undefined') ? jQuery : $;
        self.data = (typeof WOFFICE !== 'undefined') ? WOFFICE : {};

        /*
         * Markup attributes
         */
        self.$body = self.$('body');
        self.$main = self.$("#main-content");
        self.$navigation = self.$("#navigation");
        self.$navbar = self.$('#navbar');
        self.$searchTrigger = self.$("#search-trigger");
        self.$formMembers = self.$("#members-directory-form");
        self.$mainSearch = self.$("#main-search");
        self.$navWrapper = self.$("#nav-languages");
        self.$navTrigger = self.$("a#nav-trigger");
        self.$scrollHelper = self.$('#can-scroll');
        self.$rightSidebar = self.$("#right-sidebar");
        self.$sidebarTrigger = self.$("a#nav-sidebar-trigger");

        // This override the Menu Threshold set in the Theme Settings. It would be a great idea to just remove that option
        if (Woffice.$body.hasClass('menu-is-horizontal')) {
            Woffice.data.menu_threshold = 992;
        } else {
            Woffice.data.menu_threshold = 450;
        }

        self.buddyPress.clearCache();

        /*
         * When the page is starting
         */
        self.$(window).load(function () {
            self.userSidebar.start();
            self.userSidebar.Monitor_sidebars();
            self.alerts.start();
            self.customScrollbars.start();
            self.wooCommerce.start();
            self.tooltips.start();
            self.wiki.start();
            self.frontend.start();
            self.sliders.start();
            self.animatedNumbers.start();
            self.footer.start();
            self.menu.start();
            self.masonryLayout.start();
            self.navigation.start();
            self.sidebar.start();
            self.buddyPress.responsiveMenu();
            self.WpJobManager.start();
            self.ProjectGridListView.start();
            self.TododExtraTabs.watch();
            self.collapseMenu.currentMenuOpen();
            self.select2runner.watch();
        });

        /*
         * When the page is re sized
         */
        self.$(window).resize(function () {

            self.wooCommerce.resize();
            self.menu.submenus();
            self.masonryLayout.refresh();
            self.navigation.resize();
            self.sidebar.resize();
            self.buddyPress.responsiveMenu();

        });

        /*
         * Watchers, binding events after page load
         */
        {
            self.userSidebar.watch();
            self.userSidebar.Monitor_sidebars();
            self.alerts.watch();
            self.searchBar.watch();
            self.advancedSearch.watch();
            self.rolesFilter.watch();
            self.wooCommerce.watch();
            self.languageSwitcher.watch();
            self.sliders.watch();
            self.buddyPress.watchOldActivities();
            self.buddyPress.watchActivityMeta();
            self.buddyPress.watchAdvancedSearch();
            self.scrollTop.watch();
            self.linkEffect.watch();
            self.masonryLayout.watch();
            self.navigation.watch();
            self.navigation.watchResponsiveToggle();
            self.sidebar.watch();
            self.buddyPressNotifications.watch();
            self.buddyPress.BPGirdEdge();
            self.TododExtraTabs.watch();
            self.collapseMenu.slideToggleSubmenu();
            self.collapseMenu.OnHoverArrowIconColor();
        }

    },

    /**
     * Creates a new loader
     *
     * @param {jQuery} $el
     * @param {object} newOptions
     * @return {Spinner}
     */
    loader: function ($el, newOptions) {

        var self = this;

        self.isLoading = false;

        self.options = {
            color: '#757575',
            className: 'woffice-spinner',
            top: '50%',
            speed: 1.6
        };

        // Set default versions
        if (typeof newOptions !== 'undefined')
            Woffice.$.extend(self.options, newOptions);

        /**
         * Creates the loader
         */
        self.create = function () {

            // Loader object
            self.loader = new Spinner(self.options);
            self.loader.spin($el[0]);
            self.loader.element = $el;

            // Element change
            self.loader.element.addClass('has-loader');

            // Watcher variable
            self.isLoading = true;

        };

        /**
         * Removes a loader
         */
        self.remove = function () {

            // Element change
            self.loader.element.removeClass('has-loader');

            // Stop the spinner
            self.loader.stop();

            // Remove the loader
            self.loader = null;

            // Watcher variable
            self.isLoading = false;

        };

        self.create();

        return self;

    },

    /*
     * New Menu Layout
     */
    collapseMenu: {
        slideToggleSubmenu: function () {
            var $ = Woffice.$;
            if ($('#navigation li').find(".click_menutoggle").length === 0) {
                $('#navigation li').prepend('<span class="click_menutoggle"></span>');
            }
            $('#navigation li span.click_menutoggle').on('click', function () {
                var parentModContainer = $(this).parent("li");
                var isMobileMega = parentModContainer.children('.mega-menu').children();
                $(this).toggleClass('caret-up');
                $(this).siblings('a').toggleClass('current-menu-highlight');
                if (parentModContainer.hasClass('menu-item-has-mega-menu')) {
                    parentModContainer.find('> .mega-menu').slideToggle(500);
                    if (isMobileMega.length > 0 || $('#main-modern-menu').hasClass('menu-loop-happened')) {
                        parentModContainer.toggleClass('mobile-megamenu-open');
                    }
                } else {
                    parentModContainer.find('> .sub-menu').slideToggle(500);
                }
            });
        },
        currentMenuOpen: function () {
            var $ = Woffice.$;
            $('#navigation li.current-menu-ancestor').find("> .click_menutoggle").addClass('caret-up');
            $('#navigation li.current-menu-ancestor').find("> a").addClass('current-menu-highlight');

            if ($('body').hasClass('verical-menu-on-hover')) {
                $('.menu-item-has-children .click_menutoggle').unbind('click');
                $('.menu-item-has-children:not(.current-menu-parent)').each(function () {
                    $(this).children('.click_menutoggle').on({
                        mouseenter: function () {
                            $(this).addClass('current-menutoggle-active');
                            $(this).siblings('a').addClass('current-menu-highlight');
                            $(this).siblings('.sub-menu').slideDown(400);
                        }
                    });
                    $(this).mouseleave(function () {
                        $(this).children('.click_menutoggle').removeClass('current-menutoggle-active');
                        $(this).children('a').removeClass('current-menu-highlight');
                        $(this).children('.sub-menu').slideUp(400);
                    });
                });

                $('.menu-item-has-mega-menu').each(function () {
                    $(this).children('.click_menutoggle').on({
                        mouseenter: function () {
                            /*Add class to icon span */
                            $(this).siblings('.mega-menu').slideDown(400);

                        }
                    });
                    $(this).mouseleave(function () {
                        /*Remove class to icon span */
                        $(this).children('.mega-menu').slideUp(400);
                    });
                });
            }

            if ($('body').hasClass('horizontal-menu-on-click')) {
                $('.menu-item-has-children:not(.menu-item-has-mega-menu)').each(function () {
                    $(this).find('.sub-menu').addClass('sub-menu-click');
                    var selector = $(this).find('.menu-icon');
                    $(selector).on('click', function (e) {
                        e.preventDefault();
                        $(this).parent('a').siblings('.sub-menu').addClass('display-submenu');
                    });
                });

                $('.menu-item-has-mega-menu').each(function () {
                    var selector = $(this).find('.menu-icon');
                    $(selector).on('click', function (e) {
                        e.preventDefault();
                        $(this).parent('a').siblings('.mega-menu').addClass('open animated');
                    });
                });
            }

        },

        OnHoverArrowIconColor: function () {
            var $ = Woffice.$;
            $('.main-modern-menu > li.menu-item ').on({
                mouseenter: function () {
                    /*Add class to icon span */
                    $(this).find("> .click_menutoggle").addClass("stellar_heading_color");

                },
                mouseleave: function () {
                    /*Remove class to icon span */
                    $(this).find("> .click_menutoggle").removeClass("stellar_heading_color");
                }
            });
            $('.main-modern-menu > li.menu-item .sub-menu .menu-item').on({
                mouseenter: function () {
                    /*Add class to icon span */
                    $(this).find("> .click_menutoggle").addClass("stellar_sub_menu_color");

                },
                mouseleave: function () {
                    /*Remove class to icon span */
                    $(this).find("> .click_menutoggle").removeClass("stellar_sub_menu_color");
                }
            });
        }
    },

    /*
     * Wiki links bundle
     */
    wiki: {
        start: function () {
            var $ = Woffice.$;

            $('[data-toggle="collapse"]').on('click', function (e) {
                $(e.target).parent().find('.collapse').collapse('toggle');
            });
        }
    },

    /*
     * WP JOB MANAGER
     */
    WpJobManager: {

        toggle: function () {
            var $ = Woffice.$;
            var $content = $('.woffice-job-manager-frontend-wrapper__content #submit-job-form');

            if ($('#job_preview').length !== 0 || $('.job-manager-message').length !== 0) {
                $('#job_preview').parent('div#job-post-create').siblings('#wp-jobmanager-bottom').hide();
                $('.job-manager-message').parent('div#job-post-create').siblings('#wp-jobmanager-bottom').hide();
            }


            if ($content.length === 0) {
                return;
            }

            $content.not('.woffice-job-manager-frontend-wrapper__content--revealed #submit-job-form').hide();

            $('.woffice-job-manager-frontend-wrapper__toggle').on('click', function (e) {
                e.stopImmediatePropagation();

                var loader = new Woffice.loader($('.woffice-job-manager-frontend-wrapper'));

                if ($(e.target).data('action') === 'display') {
                    $('.woffice-job-manager-frontend-wrapper__toggle[data-action="display"]').hide();

                    setTimeout(function () {
                        $('#submit-job-form').slideToggle();
                        loader.remove();
                    }, 1000);
                } else {
                    $('#submit-job-form').slideToggle();
                    setTimeout(function () {
                        $('.woffice-job-manager-frontend-wrapper__toggle[data-action="display"]').show();
                        loader.remove();
                    }, 1000);
                }

                return false;
            });
        },

        start: function () {
            var self = this;

            self.toggle();
        }
    },

    ProjectGridListView: {
        start: function () {
            var $ = Woffice.$;

            $('#list').click(function (event) {
                event.preventDefault();
                $('#project-list .item').addClass('list-group-item');
                $(this).addClass('is-active');
                $('#grid').removeClass('is-active');

                $('#projects-list .item').addClass('list-group-item');
                $('#projects-list').addClass('list-view');
            });

            $('#grid').click(function (event) {
                event.preventDefault();
                $(this).addClass('is-active');
                $('#list').removeClass('is-active');
                $('#projects-list .item').removeClass('list-group-item');
                $('#projects-list .item').addClass('grid-group-item');
                $('#projects-list').removeClass('list-view');

            });
            
            // ------------new js for project new user layout----------
            $(document).on('click', '.wo_remaining_member_toggle', function (e) {
                e.stopPropagation(); // Stop event propagation
                $(this).siblings('.remaining_m_container').toggleClass('wo_show');
            });
            $(document).mouseup(function (e) {
                var remaining_m_container = $(".remaining_m_container");
                var wo_remaining_member = $(".wo_remaining_member_toggle");
                if (!remaining_m_container.is(e.target) && !remaining_m_container.has(e.target).length && !$(e.target).hasClass('wo_remaining_member_toggle')) {
                    remaining_m_container.removeClass('wo_show'); // Remove 'show' class instead of hide()
                }
            });

        }
    },

    /**
     * Main alerts
     */
    alerts: {
        start: function () {

            var $ = Woffice.$,
                self = this;

            var alerts = $('.woffice-main-alert');

            if (alerts.length === 0) {
                return;
            }

            alerts.each(function () {
                var closeTrigger = $(this).find('.woffice-alert-close');
                var alert = $(this);
                var notime = alert.hasClass('no-timeout');

                // We either close it teh button is clicked
                closeTrigger.on('click', function () {
                    alert.slideUp();
                });

                if (!notime) {
                    // Or after x sec
                    setTimeout(function () {
                        alert.slideUp();
                    }, parseInt(Woffice.data.alert_timeout));
                }

            });

        },

        watch: function () {

            var $ = Woffice.$;

            $('.woffice-alert-close ').on('click', function () {

                $(this).closest('.woffice-main-alert').slideUp();

            });

        }

    },

    /**
     * User sidebar functions
     */
    userSidebar: {

        /**
         * Set the global state on the page load
         */
        start: function () {

            var $ = Woffice.$,
                $userSidebar = $("#user-sidebar");

            // Layout
            if ($userSidebar.length > 0) {
                var topbarHeight = $("#topbar").height(),
                    menuHeight = $("#navbar").height(),
                    sidebarTop = 0;
                if ($("topbar").hasClass("topbar-closed")) {
                    sidebarTop = menuHeight + topbarHeight;
                }
                else {
                    sidebarTop = menuHeight;
                }
                $userSidebar.css("padding-top", sidebarTop);
            }

            // Binding events
            $("#user-sidebar nav ul li.menu-parent a, #user-sidebar #dropdown-user-menu li.menu-item-has-children a").bind('click', false);
            $("#user-sidebar nav ul li.menu-child a, #user-sidebar #dropdown-user-menu li.menu-item-has-children ul a").unbind('click', false);

        },

        /**
         * Watch any toggling action
         */
        watch: function () {

            var $ = Woffice.$;

            // This is for the main layout: display the sidebar or not
            $(".bp_is_active #user-thumb, #user-close, #closer-usersidebar").on("click", function () {
                $("#nav-user, #user-sidebar").toggleClass("active");
            });

            $('.user-profile-trigger').on('click', function () {
                $("#nav-user, #user-sidebar").toggleClass("active");
            });

            // Submenus within the sidebar
            $("#user-sidebar nav ul li.menu-parent a, #user-sidebar #dropdown-user-menu li.menu-item-has-children a").on("click", function () {
                $(this).toggleClass("dropdownOn");
                $(this).parent("li").toggleClass("dropdownOn");
                $(this).parent("li").find('ul').slideToggle();
            });

        },

        Monitor_sidebars: function () {
            var self = this,
                $ = Woffice.$;
            var body = $('body');
            $left_bar = $('.is-left-sidebar');
            $center_bar = $('.is-center-content');
            $right_bar = $('.is-right-sidebar');

            $('body').on('click', '.mobile-canvas-header', function () {
                // $left_bar.hide();
                Woffice.$navTrigger.find("i").removeClass("d-none");
                Woffice.$navTrigger.find("svg").addClass("d-none");
            });

            if ($(window).width() < 1025) {
                $('body').removeClass('menu-is-horizontal').addClass('vertical-modern-menu');
            }

            $('body').on('click', '.sidebar-userinfo .mobile-canvas-header', function () {
                $right_bar.hide();
            });

            if ($('body').hasClass('navigation-hidden') && $('body').hasClass('sidebar-hidden')) {
                $('.is-center-content').addClass('w-100');
            }

            $('#nav-trigger').on('click', function () {
                if (body.hasClass('navigation-hidden') && body.hasClass('sidebar-hidden')) {
                    $('.is-center-content').addClass('w-100');
                } else {
                    $('.is-center-content').removeClass('w-100');
                }
            });
        }
    },

    /**
     * Custom scroll bars
     */
    customScrollbars: {

        /**
         * Declares the scroll bars using mCustomScrollbar
         */
        start: function () {

            var $ = Woffice.$;

            /*
             * Navigation scroll bars
             */
            $("body.vertical-modern-menu #navigation").mCustomScrollbar({
                axis: "y",
                theme: "minimal"
            });
            if (window.matchMedia('(max-width: ' + Woffice.data.menu_threshold + 'px)').matches || $('.menu-item mobile-submenu-link').length > 0) {
                $("body.vertical-modern-menu #navigation, body.menu-is-horizontal #navigation").mCustomScrollbar({
                    axis: "y",
                    theme: "minimal"
                });
            }

            /*
             * User menu scroll bar
             */
            $("#user-sidebar").mCustomScrollbar({
                axis: "y",
                theme: "minimal-dark"
            });

            /*
             * WooCommerce box scroll bar
             */
            var $wooCommerceCart = $("#woffice-minicart-top");
            if ($wooCommerceCart.length) {
                $wooCommerceCart.mCustomScrollbar({
                    axis: "y",
                    theme: "minimal-dark"
                });
            }

            /*
             * Right sidebar's custom scroll bar
             * See self.sidebar() for more details
             */
            $("#right-sidebar").mCustomScrollbar({
                axis: "y",
                theme: "minimal-dark",
                mouseWheel: { deltaFactor: 100 },
                callbacks: {
                    onInit: function () {
                        $('#can-scroll').on("click", function () {
                            if (!$('#main-content').hasClass('sidebar-hidden')) {
                                $('#can-scroll').show();
                            }
                            if ($(this).hasClass('clicked')) {
                                $('#right-sidebar').mCustomScrollbar("scrollTo", "top");
                                $(this).removeClass('clicked');
                            }
                            else {
                                $('#right-sidebar').mCustomScrollbar("scrollTo", "bottom");
                                $(this).addClass('clicked');
                            }
                        });
                    },

                    onUpdate: function () {
                        if ($('#main-content').height() >= $('#right-sidebar')[0].scrollHeight) {
                            $('#can-scroll').addClass('clicked');
                        } else {
                            $('#can-scroll').removeClass('clicked');
                        }
                    },
                    onScroll: function () {
                        if ($('#main-content').height() >= $('#right-sidebar')[0].scrollHeight) {
                            $('#can-scroll').addClass('clicked');
                        } else {
                            $('#can-scroll').removeClass('clicked');
                        }
                    }
                }
            });

        }

    },

    /**
     * Search bar
     */
    searchBar: {

        /**
         * Make a search query
         *
         */
        search: _.debounce(function () {
            if (!Woffice.data.has_live_search || Woffice.data.has_live_search !== '1') {
                return;
            }

            var $ = Woffice.$;
            var $results = Woffice.$mainSearch.find('.woffice-search-results');

            var value = Woffice.$mainSearch.find('input#s').val();

            if (!value || (value && value.length < 3)) {
                return false;
            }

            var loader = new Woffice.loader($results, { left: '45%' });

            $.ajax({
                type: "POST",
                url: Woffice.data.ajax_url.toString(),
                data: {
                    action: 'woffice_search',
                    nonce: WOFFICE.nonce,
                    search: value,
                    types: 'all'
                },
                success: function (data) {
                    data = JSON.parse(data);

                    var markup = '';

                    $.each(data, function (key, group) {

                        markup += '<div class="woffice-search-results__group" id="woffice-search-results__group--' + key + '">';

                        markup += '<h3><i class="fa ' + group.icon + '"></i> ' + group.label + '</h3>';

                        if (group.items && group.items.length > 0) {
                            markup += '<ul>';
                            for (var i = 0; i < group.items.length; i++) {
                                var item = group.items[i];

                                markup += '<li class="woffice-search-results__item clearfix" id="woffice-search-results__item--' + item.id + '" data-type="' + key + '">';
                                markup += '<div class="float-left">';
                                markup += '<span class="woffice-search-results__item__title"><a href="' + item.link + '">' + item.title + '</a></span>';
                                if (item.meta) {
                                    var length = 150;
                                    if (item.meta.length > length) {
                                        item.meta = item.meta.substring(0, length) + '...';
                                    }

                                    markup += '<span class="woffice-search-results__item__meta">' + item.meta + '</span>';
                                }
                                markup += '</div>';
                                markup += '<div class="float-right">';
                                markup += '<a href="' + item.link + '" class="btn btn-default"><i class="fa fa-arrow-right"></i></a>';
                                markup += '</div>';
                                markup += '</li>';
                            }
                            markup += '</ul>';
                        }

                        markup += '</div>';

                    });

                    $results.html(markup);

                    loader.remove();
                }
            });
        }, 500),

        /**
         * Set the width of the main search
         */
        setSize: function () {
            if (Woffice.$body.hasClass('menu-is-vertical') && !Woffice.$navigation.hasClass('navigation-hidden')) {
                // Woffice.$mainSearch.css('width', (Woffice.$body.width() - Woffice.$navigation.width()));
            } else {
                // Woffice.$mainSearch.css('width', (Woffice.$body.width()));
            }
        },

        /**
         * Prevent default form submit and trigger ajax search
         *
         * @param {Object} e
         */
        searchSubmit: function (e) {
            e.preventDefault();
            Woffice.searchBar.search();
        },


        /**
         * Whenever someone click on the icon
         */
        watch: function () {

            var $ = Woffice.$;

            var offset = Woffice.$navbar.height();
            offset = (Woffice.$body.hasClass('admin-bar')) ? offset + 32 : offset;

            Woffice.$mainSearch.find('input#s').on('keyup', function () {
                Woffice.searchBar.search();
            });

            Woffice.searchBar.setSize();

            $(window).resize(function () {
                Woffice.searchBar.setSize();
            });

            Woffice.$searchTrigger.on("click", function () {

                Woffice.$mainSearch.toggleClass('opened');

                if (Woffice.$mainSearch.find('.woffice-search-results').length === 0) {
                    Woffice.$mainSearch.append('<div class="woffice-search-results"></div>');
                } else {
                    Woffice.$mainSearch.find('.woffice-search-results').remove();
                }

                if (Woffice.$mainSearch.hasClass('opened')) {
                    Woffice.$searchTrigger.find('i.fa').removeClass('fa-search');
                    Woffice.$searchTrigger.find('i.fa').addClass('fa-times');
                    Woffice.$mainSearch.find('form').on('submit', Woffice.searchBar.searchSubmit);
                } else {
                    Woffice.$searchTrigger.find('i.fa').addClass('fa-search');
                    Woffice.$searchTrigger.find('i.fa').removeClass('fa-times');
                    Woffice.$mainSearch.find('form').off('submit', Woffice.searchBar.searchSubmit);
                }

                $('html,body').animate({ scrollTop: 0 }, 'fast');
                document.getElementById("s").focus();
                return false;

            });

        }

    },

    /**
     * Roles filters for Members
     */
    rolesFilter: {
        /**
         * Whenever someone click on a dropdown's link
         */
        watch: function () {
            var $ = Woffice.$;
            var $dropdownItems = $('#woffice-roles-filter a.dropdown-item');

            var $orderBy = $('#members-order-by');
            var order = ($orderBy.length) ? $orderBy.find('option').first().val() : 'active';

            $dropdownItems.on('click', function (e) {
                e.preventDefault();

                var requestObj = {
                    scope: 'roles',
                    filter: order,
                    action: 'members_filter',
                    object: 'members'
                };

                $dropdownItems.removeClass('active');
                $(e.target).addClass('active');

                requestObj['role'] = $(e.target).data('role');

                Cookies.set('woffice_role', requestObj['role'], { expires: 1, path: '/' });

                window.bp.Nouveau.objectRequest(requestObj);
            });

            var url = new URL(location.href);
            var roleUrl = url.searchParams.get('filterRole');

            if (roleUrl) {
                var $button = $('#woffice-roles-filter a.dropdown-item[data-role="' + roleUrl + '"]');
                if ($button) {
                    $button.trigger('click');
                }
            }
        }
    },

    /**
     * Advanced Search for Members
     */
    advancedSearch: {

        /**
         * Whenever someone click on the icon
         */
        watch: function () {
            var $ = Woffice.$;
            var $orderBy = $('#members-order-by');
            var order = ($orderBy.length) ? $orderBy.val() : 'active';
            var $wrapper = $('#woffice-members-advanced-search');

            $wrapper.find('.clear-value').each(function () {
                $(this).attr('href', 'javascript:void(0)');
                $(this).on('click', function () {
                    $(this).parent('.form-group').find('input[type="radio"]').prop('checked', false);
                    $(this).parent('.form-group').find('option:selected').prop('selected', false);
                });
            });

            $('#advanced-search-submit').on("click", function (e) {
                e.preventDefault();

                var requestObj = {
                    scope: 'advanced-search',
                    filter: order,
                    action: 'members_filter',
                    object: 'members'
                };

                requestObj['advanced-search-submit'] = true;

                var formObj = {};
                var $inputs = $wrapper.find('input, select');
                var data = [];
                var item_list = '';

                $inputs.each(function () {
                    var name = $(this).attr('name');
                    if(name !== undefined){
                        formObj[$(this).attr('name').replace('[]', '')] = $(this).val();
                    }
                });

                if ($wrapper.find('.checkbox-options').length > 0) {
                    $wrapper.find('.checkbox-options').each(function () {
                        item_list = $(this).attr('id');
                        data[item_list] = [];
                        $(this).find('input[type="checkbox"]').each(function () {
                            if ($(this).prop('checked') == true) {
                                data[item_list].push($(this).val());
                            }
                        });
                        formObj[item_list] = '';
                        formObj[item_list] = data[item_list];
                    });
                }

                if ($wrapper.find('.radio-button-options').length > 0) {
                    $wrapper.find('.radio-button-options').each(function () {
                        item_list = $(this).attr('id');
                        data[item_list] = [];
                        $(this).find('input[type="radio"]').each(function () {
                            if ($(this).prop('checked') == true) {
                                data[item_list].push($(this).val());
                            }
                        });
                        formObj[item_list] = '';
                        formObj[item_list] = data[item_list];
                    });
                }

                $.extend(requestObj, { extras: formObj });

                window.bp.Nouveau.objectRequest(requestObj);
            });

            $('#advanced-search-reset').on("click", function (e) {
                location.reload();
            });
        }

    },

    /**
     * WooCommerce
     */
    wooCommerce: {

        flexsliderWoo: null,

        /**
         * Watch for the cart's toggling
         */
        watch: function () {

            var $ = Woffice.$;

            $("#nav-buttons").on("click", "#nav-cart-trigger.active", function () {
                $(this).toggleClass("clicked");
                $("#woffice-minicart-top").slideToggle();
            });

        },

        /**
         * Get the number of items according to the width
         */
        getGridSize: function () {

            return (window.innerWidth < 400) ? 1 :
                (window.innerWidth < 600) ? 2 :
                    (window.innerWidth < 910) ? 3 : 4;

        },

        /**
         * Starts the product carousels
         */
        start: function () {

            var $ = Woffice.$,
                self = this;

            $('.flexslider > .woocommerce').flexslider({
                animation: "slide",
                animationLoop: false,
                selector: ".products > li.product",
                itemWidth: 210,
                itemMargin: 0,
                controlNav: false,
                move: 0,
                slideshow: false,
                minItems: self.getGridSize(), // use function to pull in initial value
                maxItems: self.getGridSize(), // use function to pull in initial value
                start: function (slider) {
                    self.flexsliderWoo = slider; //Initializing flexslider here.
                }
            });

        },

        /**
         * Changes the number of items according to the width
         */
        resize: function () {

            var self = this,
                $ = Woffice.$;

            if ($('.flexslider > .woocommerce').length > 0) {
                var gridSize = self.getGridSize();
                if (self.flexsliderWoo !== null) {
                    self.flexsliderWoo.vars.minItems = gridSize;
                    self.flexsliderWoo.vars.maxItems = gridSize;
                }
            }

        }

    },

    /**
     * Language switcher
     */
    languageSwitcher: {

        watch: function () {

            Woffice.$navWrapper.find('a').on("click", function () {
                Woffice.$navWrapper.find('ul').slideToggle();
            });

        }

    },

    /**
     * Bootstrap tooltips
     */
    tooltips: {

        start: function () {

            var $ = Woffice.$;

            $('[data-toggle="tooltip"]').tooltip();

        }

    },

    /**
     * Theme form actions and frontend actions
     */
    frontend: {

        /**
         * Handle the tab navigation (Projects & Wiki)
         */
        tabNav: function () {
            var $ = Woffice.$;
            var $navWrapper = $('.woffice-tab-layout__nav');
            var $contentWrapper = $('.woffice-tab-layout__content');

            $contentWrapper.find('.woffice-tab-layout__tab').addClass('d-none');
            $contentWrapper.find('.woffice-tab-layout__tab').first().removeClass('d-none');

            $navWrapper.on('click', 'li', function (e) {
                var $item = $(e.target);

                $navWrapper.find('li').removeClass('active');

                if ($(e.target).closest('a').length) {
                    $item = $item.closest('li');
                }

                $item.addClass('active');

                $item.trigger('woffice.tab.change', [$item.data('tab')]);

                if ($('.woffice-todo-extratabs').length > 0) {
                    $('.woffice-todo-extratabs').addClass('d-none');
                    if ($item.data('tab') == 'todo') {
                        $('.woffice-todo-extratabs').removeClass('d-none');
                    }
                }

                var loader = new Woffice.loader($contentWrapper);

                $contentWrapper.find('.woffice-tab-layout__tab').addClass('d-none');

                setTimeout(function () {
                    $contentWrapper.find('.woffice-tab-layout__tab[data-tab="' + $item.data('tab') + '"]').removeClass('d-none');

                    loader.remove();
                }, 1000);
            });

            // Default tab if in the link, use the last part of the div as the unique ID
            var anchor = window.location.hash.substr(1);

            if (anchor.length > 0) {
                var parts = anchor.split('-');

                if (parts.length > 0) {
                    var name = parts[parts.length - 1];

                    $navWrapper.find('li[data-tab="' + name + '"]').trigger('click');
                }
            }

            if ($('.datepicker').length > 0) {
                $('body').on('focus', '.datepicker', function () {
                    $('.datepicker-days').show();
                })
            }

        },

        checkboxes: function () {
            var $ = Woffice.$;

            $('#page-wrapper .wpcf7-checkbox input:checkbox,#page-wrapper .wpcf7-radio input:radio').change(function () {
                if ($(this).is(":checked")) {
                    $(this).parent("label").addClass("checked");
                } else {
                    $(this).parent("label").removeClass("checked");
                }
            });
        },

        toggle: function () {
            var $ = Woffice.$;
            var $content = $('.frontend-wrapper__content');

            if ($content.length === 0) {
                return;
            }

            $content.not('.frontend-wrapper__content--revealed').hide();

            $('.frontend-wrapper__toggle').on('click', function (e) {
                e.stopImmediatePropagation();
                var loader = new Woffice.loader($('.frontend-wrapper'));

                if ($(e.target).data('action') === 'display') {
                    $('.frontend-wrapper__toggle[data-action="display"]').hide();
                    setTimeout(function () {
                        $('.frontend-wrapper__content').slideToggle();
                        loader.remove();
                    }, 1000);
                } else {
                    $('.frontend-wrapper__content').slideToggle();
                    setTimeout(function () {
                        $('.frontend-wrapper__toggle[data-action="display"]').show();
                        loader.remove();
                    }, 1000);
                }

                return false;
            });
        },

        initAutocompleteMap: function () {
            // Create the autocomplete object using Google Maps API, restricting the search to geographical location
            var input = document.getElementById(Woffice.data.input_location_bb);
            // We check that this input exist on the dom to avoid any error
            if (input === null)
                return;
            autocomplete = new google.maps.places.Autocomplete(
                (input),
                { types: ['geocode'] }
            );

            if (navigator.geolocation && Woffice.data.user_id !== '0') {
                navigator.geolocation.getCurrentPosition(function (position) {
                    var geolocation = {
                        lat: position.coords.latitude,
                        lng: position.coords.longitude
                    };
                    var circle = new google.maps.Circle({
                        center: geolocation,
                        radius: position.coords.accuracy
                    });
                    autocomplete.setBounds(circle.getBounds());
                });
            }
        },

        start: function () {
            var self = this;

            self.tabNav();
            self.checkboxes();
            self.toggle();
            if (typeof google !== 'undefined' && google.maps) {
                self.initAutocompleteMap();
            }
        }

    },

    /**
     * Sliders
     */
    sliders: {

        watch: function () {

            var $ = Woffice.$,
                self = this;

            $("#dashboard .widget a, #nav-trigger, #nav-sidebar-trigger").on('click', function () {
                setTimeout(function () {
                    // Flexslider
                    $('#dashboard').find('.widget_woffice_funfacts .flexslider').flexslider();
                    // Rev slider
                    self.refreshRevSlider();
                }, 2000);
            });

        },

        /**
         * Refreshes Revolution slider on layout changes
         */
        refreshRevSlider: function () {

            var $ = Woffice.$,
                $slider = $(".rev_slider"),
                sliderOnPage = (typeof $slider.revredraw === 'function');

            if (sliderOnPage) {
                setTimeout(function () {
                    $slider.revredraw();
                }, 1000);
            }

        },

        /**
         * Creates the fun fact and "familiers" sliders
         * Using the Flexslider plugin
         * We also bind the revolution slider
         */
        start: function () {

            var $ = Woffice.$,
                self = this;

            $('.widget_woffice_funfacts .flexslider').flexslider({
                animation: "slide",
                animationLoop: true,
                slideshow: true,
                directionNav: false,
                selector: ".slides > li",
                smoothHeight: false,
                start: function () {
                    $('.widget_woffice_funfacts .flexslider').resize();
                }
            });

            $('#familiers').find('.flexslider').flexslider({
                animation: "slide",
                animationLoop: true,
                selector: ".slides > li",
                itemWidth: 80,
                itemMargin: 0,
                controlNav: false,
                directionNav: false,
                minItems: 0,
                move: 0,
                slideshow: false
            });

            self.refreshRevSlider();

        }

    },

    /**
     * Animated numbers animation
     */
    animatedNumbers: {

        start: function () {
            var $ = Woffice.$;

            $('.animated-number h1').countTo({
                speed: 4000
            });

        }

    },

    /**
     * BuddyPress customizations
     */
    buddyPress: {

        /**
         * Handle the Responsive menu
         *
         * @return {null}
         */
        responsiveMenu: function () {
            var $ = Woffice.$;
            var threshold = 770;
            var $menu = $('.main-navs').first();
            var handle = 'woffice-bp-menu-toggle';
            var $handle = $('#' + handle);

            if (window.innerWidth > threshold || $menu.length === 0) {
                $handle.remove();

                return null;
            }

            if ($handle.length >= 1) {
                return null;
            }

            $menu.find('> ul').slideToggle();

            var toggle = '<a href="#" class="bb_nav_toggle" id="' + handle + '"><i class="fa fa-bars fa-2x"></i></a>';

            $menu.prepend(toggle);

            $('body').on('click', '#' + handle, function () {
                $menu.find('> ul').slideToggle();
            });
        },

        /**
         * Clear advanced search cache when page reloading
         */
        clearCache: function () {
            var $ = Woffice.$;

            if (!$('body').hasClass('directory members')) {
                return;
            }

            var $orderBy = $('#members-order-by');
            var order = ($orderBy.length) ? $orderBy.find('option').first().val() : 'active';

            var requestObj = {
                scope: 'advanced-search',
                filter: order,
                action: 'members_filter',
                object: 'members'
            };

            $.extend(requestObj, { extras: {} });

            window.bp.Nouveau.objectRequest(requestObj);

            setTimeout(function () {
                var $advancedSearch = $('#members-advanced-search');

                if ($advancedSearch.length) {
                    $advancedSearch.toggleClass('selected');
                }
            }, 1000);
        },

        /**
         * Watch the advanced search button
         */
        watchAdvancedSearch: function () {
            var $ = Woffice.$;

            $("#members-advanced-search").on('click', function () {
                $(this).toggleClass('active');
                $("#woffice-members-advanced-search").slideToggle('slow');
            });
        },

        /**
         * It allows to display old activity comments, it fixes the compatibility issue between BuddyPress and Bootstrap
         */
        watchOldActivities: function () {
            var $ = Woffice.$;
            var url = window.location.href;
            if ($('#user-sidebar').length > 0) {
                $('#menu-bp li').each(function () {
                    var current_url = $(this).children('a').attr('href');

                    if (current_url === url) {
                        if ($(this).parent('ul').parent('.menu-parent').length > 0) {
                            $(this).parent('ul').parent('.menu-parent').addClass('uside-item-active')
                            $(this).addClass('uside-child-active');
                        } else {
                            $(this).addClass('uside-item-active');
                        }
                    }
                });
            }

            $('.activity-comments .show-all').on("click", function () {

                $(this).closest('.activity-comments').find('li').removeClass('hidden');

            });
        },

        /**
         * We bind correctly the icon click for BuddyPress
         */
        watchActivityMeta: function () {
            var $ = Woffice.$;

            $('.activity-meta i').on("click", function (e) {
                e.preventDefault();
                $(this).parent().trigger('click');
            });
        },

        /**
         * We Set the grid edges
         */
        BPGirdEdge: function () {
            var $ = Woffice.$;
            if ($('body').hasClass('is-modern-skin')) {
                $(document).ajaxComplete(function (event, xhr, settings) {
                    $("#members-list").parent('#members-dir-list').addClass("member-moden-grid");
                    $("#groups-list").parent('#groups-dir-list').addClass("member-moden-grid");
                });
            }
        }
    },

    /**
     * Scroll to the top action
     */
    scrollTop: {

        watch: function () {

            var $ = Woffice.$;

            $("#scroll-top").on("click", function () {
                //SCROLL TO TOP
                $('html, body').animate({
                    scrollTop: $($.attr(this, 'href')).offset().top
                }, 500);
                return false;
            });

        }

    },

    /**
     * Material design inspired effect on links click
     */
    linkEffect: {

        elements: [
            '#navbar a',
            '.main-menu li > a',
            'a.btn.btn-default',
            '#content-container #buddypress button',
            '#buddypress .button-nav li a,#main-content button[type="submit"]',
            'input[type="submit"]',
            '#user-sidebar nav ul li a',
            '#buddypress #item-nav div.item-list-tabs ul li a',
            '#woffice-login .login-submit input[type="submit"]',
            '#main-content input[type="button"],#learndash_next_prev_link a',
            '#content-container .ssfa_fileup_wrapper span'
        ],

        watch: function () {

            var self = this,
                selector = self.elements.join(", "),
                $ = Woffice.$;

            var ink, d, x, y;

            $(selector).on("click", function (e) {
                if ($(this).find(".material").length === 0) {
                    $(this).prepend("<span class='material'></span>");
                }

                ink = $(this).find(".material");
                ink.removeClass("animate");

                if (!ink.height() && !ink.width()) {
                    d = Math.max($(this).outerWidth(), $(this).outerHeight());
                    ink.css({ height: d, width: d });
                }

                x = e.pageX - $(this).offset().left - ink.width() / 2;
                y = e.pageY - $(this).offset().top - ink.height() / 2;

                ink.css({ top: y + 'px', left: x + 'px' }).addClass("animate");
            });

        }

    },

    /**
     * Footer functions
     */
    footer: {

        /**
         * Loads the extra footer once the page ready
         */
        loadExtafooterAvatars: function () {

            var $ = Woffice.$,
                $extrafooter = $('#extrafooter[data-woffice-ajax-load=true]');

            if ($extrafooter.length === 0)
                return;

            var loader = new Woffice.loader($extrafooter.find('#familiers'), { left: '45%' });

            $.ajax({
                type: "POST",
                url: Woffice.data.ajax_url.toString(),
                data: {
                    action: 'load_extrafooter_avatars',
                    nonce: WOFFICE.nonce
                },
                success: function (data) {

                    $extrafooter.find('#familiers').html(data);

                    loader.remove();

                }
            });

        },

        /**
         * Build the footer layout (widgets)
         */
        footerWidgetsLayout: function () {

            var $ = Woffice.$;

            if (window.matchMedia("(min-width: 992px)").matches) {

                var $widgets_wrapper = $("#main-footer").find("#widgets"),
                    layout = $widgets_wrapper.attr('data-widgets-layout');

                if (typeof layout !== typeof undefined && layout !== false && $widgets_wrapper.length > 0) {

                    layout = layout.split("-");

                    //For each column assign the width depending on the selected layout
                    $widgets_wrapper.find(".widget").each(function (i) {
                        var layout_length = layout.length;

                        $(this).removeClass("col-md-3");
                        $(this).addClass("col-md-" + layout[i % layout_length]);
                    });

                }

            }

        },

        start: function () {

            var self = this;

            self.loadExtafooterAvatars();
            self.footerWidgetsLayout();

        }

    },

    /**
     * Menu
     * including sub menus and mega menus
     */
    menu: {

        megaMenuTimer: null,

        megaMenuCol: 200,

        edgeLimit: 0,

        /**
         * Mega menu
         */
        megaMenu: function () {

            var self = this,
                $ = Woffice.$;

            if (!Woffice.$body.hasClass('menu-is-horizontal')) {

                setTimeout(function () {
                    $("#main-menu").find("li").each(function () {
                        if ($(this).hasClass("menu-item-has-mega-menu")) {
                            var liheight = $(this).innerHeight();
                            var megamenu = $(this).find('div.mega-menu');
                            $(megamenu).css('margin-top', '-' + (liheight) + 'px');
                        }
                    });
                }, 2500);

                self.megaMenuCol = 180;

            }

            $('.menu-is-horizontal .main-modern-menu > li.menu-item-has-mega-menu').on({
                mouseenter: function () {

                    // COUNT THE NUMBER OF COLUMN
                    var megamenucontainer = $(this).find("div.mega-menu");
                    var numberofrows = megamenucontainer.find("li.mega-menu-col").length;

                    // SIZE -> 180 per rows
                    $mainMenu = $('.main-modern-menu');
                    if (!$mainMenu.hasClass('menu-loop-happened')) {

                        megamenucontainer.width(numberofrows * (self.megaMenuCol) + 20);

                        // SHOW IT

                        self.megaMenuTimer = setTimeout(function () {
                            if (!$('body').hasClass('horizontal-menu-on-click')) {
                                megamenucontainer.addClass('open animated');
                            }

                        }, 0);
                    }

                },
                mouseleave: function () {

                    var megamenucontainer = $(this).find("div.mega-menu");

                    setTimeout(function () {
                        megamenucontainer.removeClass('open');
                    }, 0);

                }
            });

        },

        /**
         * Checks if the submenu elements is placed over the window limit
         * @param who
         */
        calculateEdge: function (who) {

            var $ = Woffice.$,
                self = this,
                elm = $(who).find('ul').first();

            if (elm.length) {

                var off = elm.offset(),
                    l = off.left,
                    w = elm.width(),
                    isEntirelyVisible = (l + w <= self.edgeLimit);

                if (!isEntirelyVisible) {
                    $(who).addClass('edge');
                } else {
                    $(who).removeClass('edge');
                }

            }

        },

        /**
         * Set the submenus
         */
        setSubMenus: function () {

            var $ = Woffice.$;

            $("#main-menu").find("li").each(function () {

                if ($(this).hasClass("menu-item-has-children")) {

                    var lineheight = $(this).height();
                    var submenu = $(this).find('.sub-menu');
                    $(submenu).css('margin-top', '-' + (lineheight) + 'px');

                }

            });

        },

        /**
         * This function is run on the page load and resize
         */
        submenus: function () {

            var self = this,
                $ = Woffice.$;

            self.edgeLimit = $('#page-wrapper').width();

            if (
                window.matchMedia('(max-width: ' + Woffice.data.menu_threshold + 'px)').matches
                ||
                (Woffice.navigation.isTouchDevice() && window.innerWidth < 1100)
            ) {

                $('.main-modern-menu li.menu-item-has-children > a').not(".binded").addClass("binded").on("click", function () {

                    // TODO move this listener out of the first check. In order to avoid the useless nested check
                    if (
                        window.matchMedia('(max-width: ' + Woffice.data.menu_threshold + 'px)').matches
                        ||
                        (Woffice.navigation.isTouchDevice() && window.innerWidth < 1100)
                    ) {
                        $(this).toggleClass("mobile-menu-displayed");
                        var parentContainer = $(this).parent("li");
                        if (parentContainer.hasClass('menu-item-has-mega-menu') && $('#main-modern-menu').hasClass('menu-loop-happened')) {
                            $(parentContainer).toggleClass('mobile-megamenu-open');
                            parentContainer.find('> .mega-menu').slideToggle();
                        } else {
                            parentContainer.find('> .sub-menu').slideToggle();
                        }
                        return false;
                    }

                });

            }
            else {

                $('.menu-is-horizontal .main-modern-menu li.menu-item-has-children:not(.menu-item-has-mega-menu)').on({

                    mouseenter: function () {
                        var self = this,
                            submenu = $(self).find('.sub-menu').first();

                        // Without this timeout, the function handleClicks wouldn't work
                        setTimeout(function () {
                            if (!$('body').hasClass('horizontal-menu-on-click')) {
                                submenu.addClass("display-submenu");
                            }
                        }, 10);


                        var scrollTop = $(window).scrollTop(),
                            elementOffset = submenu.offset().top,
                            distanceFromTop = (elementOffset - scrollTop),
                            megamenu_height = 0;

                        var edgeOffset = (distanceFromTop + submenu.height() - $(window).height());

                        if ($(self).hasClass('mega-menu-col')) {

                            megamenu_height = Math.max.apply(null, $(self).closest('.mega-menu').find('.mega-menu-col').map(function () {
                                return $(this).height();
                            }).get());

                            edgeOffset = (distanceFromTop + megamenu_height - $(window).height());
                        }

                        var isEntirelyVisible = (edgeOffset < 0);

                        if (!isEntirelyVisible) {
                            if ($(self).hasClass('mega-menu-col')) {

                                $(self).closest('.mega-menu')[0].style.removeProperty('margin-top');
                                $(self).closest('.mega-menu')[0].style.setProperty('margin-top', '-' + parseInt(megamenu_height) + 'px', 'important');
                            }
                        }


                        Woffice.menu.calculateEdge(this);

                    },

                    mouseleave: function () {

                        var self = this;
                        var submenu = $(self).find('> .sub-menu');

                        submenu.removeClass("display-submenu");
                        $(self).removeClass('edge');
                    }

                });

                $('body').on({

                    mouseleave: function () {

                        setTimeout(function () {

                            var $navigation = $('#navigation');

                            $navigation.find('.display-submenu').removeClass("display-submenu");
                            $navigation.find('.edge').removeClass('edge');

                        }, 100);

                    }

                });

            }

        },

        /**
         * If a menu item has children, block the link redirection if the submenu isn't displayed
         *
         * This functions assumes that if the submenu isn't displayed when clicked, it means that the mousent hasn't been triggered,
         * which should means that you are using a big touch screen (such as a retina tablet).
         */
        handleClicks: function () {

            var $ = Woffice.$;

            // Todo this only works for the second level menu
            $('.main-modern-menu .menu-item > li.menu-item-has-children > a').on("click", function (event) {
                var $parentContainer = $(this).parent("li");
                if (!$parentContainer.find('> .sub-menu').hasClass('display-submenu')) {
                    event.preventDefault();
                }
            });

        },

        start: function () {

            var self = this,
                $ = Woffice.$;

            self.edgeLimit = $('#page-wrapper').width();
            self.megaMenu();
            self.setSubMenus();
            self.submenus();
            self.handleClicks();
        }

    },

    /**
     * Masonry layouts used across the theme
     */
    masonryLayout: {

        /**
         * Build helper the Masonry layout
         */
        build: function () {

            var $ = Woffice.$,
                $dashboard = $('#dashboard');

            $('.masonry-layout').isotope({
                // options
                itemSelector: '.box',
                layoutMode: 'masonry'
            });

            if ($dashboard.length > 0) {
                if (!$dashboard.hasClass('is-draggie')) {
                    $dashboard.isotope();
                }
                /*
                 * Commented from WOF-92.
                 * It's causing the jQuery memory issue.
                 *
                 * $dashboard.find( '.widget_woffice_funfacts .flexslider' ).resize();
                 */
            }

        },

        /**
         * Refresh layout
         */
        refresh: function () {

            var $ = Woffice.$;

            Woffice.masonryLayout.build();
            var board_divider = 3;
            var outer_winwidth = $(window).width();
            if (outer_winwidth > 1439) {
                board_divider = 4;
            } else if (outer_winwidth < 575) {
                board_divider = 1;
            }
            var board_container = $('.woffice-kanban .kanban-container, .woffice-crm-kanban .kanban-container').outerWidth() / board_divider;
            var board_width = board_container - 20;
            $('.woffice-kanban .kanban-board').css({ 'width': + board_width + 'px', 'flex': '0 0 ' + board_width + 'px' });

            // fix ratios for resizing the calendar size
            $('.eventon_fullcal').each(function () {
                var cal_width = $(this).width();
                var strip = $(this).find('.evofc_months_strip');
                var multiplier = strip.attr('data-multiplier');

                if (multiplier < 0) {
                    strip.width(cal_width * 3).css({ 'margin-left': (multiplier * cal_width) + 'px' });
                }
                $(this).find('.evofc_month').width(cal_width);
            });

        },

        /**
         * We watch several events
         */
        watch: function () {

            var self = this,
                $ = Woffice.$;

            $("#dashboard").on('click', 'a.evcal_list_a, .widget a, p.evo_fc_day', function () {
                self.refresh();
            });

            $("#nav-trigger, #nav-sidebar-trigger").on('click', function () {
                setTimeout(self.refresh, 600);
            });

        },

        /**
         * Starts the layouts
         */
        start: function () {

            var $ = Woffice.$,
                $list = null,
                $wrapper = $('#buddypress [data-bp-list]');

            if (window.location.search.indexOf('members_search') !== -1) {
                $('#dir-members-search-submit').trigger('click');
            }

            $wrapper.bind('bp_ajax_request', function () {
                var loader = new Woffice.loader($wrapper);

                $wrapper.find('.item-list').css({ opacity: 0 });

                setTimeout(function () {
                    $list = $('#groups-list, #members-list').isotope({
                        itemSelector: 'li.item-entry',
                        layoutMode: 'fitRows'
                    });

                    loader.remove();

                    $wrapper.find('.item-list').css({ opacity: 1 });
                }, parseInt(Woffice.data.masonry_refresh_delay));
            });

            $("#nav-trigger, #nav-sidebar-trigger, #item-nav a").on('click', function () {
                setTimeout(function () {
                    $list = $('#groups-list, #members-list').isotope({
                        itemSelector: 'li.item-entry',
                        layoutMode: 'fitRows'
                    });
                }, Woffice.data.masonry_refresh_delay);
            });

            setTimeout(function () {
                Woffice.masonryLayout.build();
            }, 200);

            setInterval(function () {
                Woffice.masonryLayout.build();
                Woffice.masonryLayout.refresh();
            }, Woffice.data.masonry_refresh_delay);

        }

    },

    /**
     * Handles the navigation actions
     */
    navigation: {

        cachedWidth: 0,

        /**
         * Whether we are on a touch device or not
         *
         * @return {boolean}
         */
        isTouchDevice: function () {
            return typeof window.ontouchstart !== 'undefined';
        },

        watch: function () {

            var $ = Woffice.$,
                self = this;

            Woffice.$navTrigger.on("click", function () {
                if ($("#main-content").hasClass("navigation-hidden")) {
                    self.showVerticalMenu(true);
                }
                else {
                    self.hideVerticalMenu(true);
                }
                Woffice.userSidebar.Monitor_sidebars();
                Woffice.searchBar.setSize();
                Woffice.sidebar.setSidebarWidth();
            });

        },

        hideVerticalMenu: function (handleCookie) {

            var $ = Woffice.$;

            // Icon class switching

            $(".is-left-sidebar").addClass("has-navigation-hidden");
            $(".is-left-sidebar").removeClass("has-navigation-show");
            $(".is-left-sidebar").fadeOut("500");

            $("body, #navigation, #main-content, #main-header, #main-footer").addClass("navigation-hidden");

            setTimeout(function () {
                Woffice.$navTrigger.find("svg").addClass("d-none");
                Woffice.$navTrigger.find("i").removeClass("d-none");
            }, 300)

            if (handleCookie) {
                // Create cookies to save user choice :
                Cookies.set('Woffice_nav_position', 'navigation-hidden', { expires: 7, path: '/' });
                Cookies.set('Woffice_hasnav_position', 'has-navigation-hidden', { expires: 7, path: '/' });
            }

            // Rebuild the sliders
            Woffice.sliders.start();

        },

        showVerticalMenu: function (handleCookie) {

            var $ = Woffice.$;

            // Icon class switching
            Woffice.$navTrigger.find("svg").removeClass("d-none");
            Woffice.$navTrigger.find("i").addClass("d-none");
            $(".is-left-sidebar").fadeIn("500");
            $(".is-left-sidebar").removeClass("has-navigation-hidden");
            $(".is-left-sidebar").addClass("has-navigation-show");
            $("body,#navigation, #main-content, #main-header, #main-footer").removeClass("navigation-hidden");

            if (handleCookie) {
                // ERASE COOKIES
                Cookies.remove('Woffice_nav_position', { expires: 7, path: '/' });
                Cookies.remove('Woffice_hasnav_position', { expires: 7, path: '/' });
            }

        },

        responsiveMenu: function (onready) {

            var self = this,
                $ = Woffice.$,
                $mainMenu = $('.main-modern-menu'),
                $notificationAlert = $('#woffice-notifications-menu');

            if ($notificationAlert.length > 0) {
                var $wpadminbar = $('#wpadminbar'),
                    adminbarHeight = ($wpadminbar.length > 0) ? $wpadminbar.height() : 0,
                    topbarHeight = $("#navbar").height();

            }


            if (
                window.matchMedia('(max-width: ' + Woffice.data.menu_threshold + 'px)').matches
                ||
                (Woffice.navigation.isTouchDevice() && window.innerWidth < 1100)
            ) {

                if (onready)
                    self.hideVerticalMenu();
                else {
                    $("body, #navigation, #main-content, #main-header, #main-footer").addClass("navigation-hidden");
                }

                // We create a duplicate of the link
                if (!$mainMenu.hasClass("menu-loop-happened")) {
                    $('.main-modern-menu li.menu-item-has-children').each(function () {
                        var linkElement = $(this).find(" > a");
                        var submenuContainer = $(this).find(" > ul.sub-menu");
                        var linkElement_href = linkElement.attr('href');

                        if (linkElement_href !== '#' && linkElement_href !== 'javascript:void(0)') {
                            if ($(this).hasClass('menu-item-has-mega-menu')) {
                                submenuContainer = $(this).find(".mega-menu .sub-menu:first-child");
                            }
                            var subElement = '<li class="menu-item mobile-submenu-link"><a href="' + linkElement_href + '" class="center"><i class="fa fa-arrow-right"></i></a></li>';

                            submenuContainer.prepend(subElement);
                        }
                    });
                }

                $mainMenu.addClass("menu-loop-happened");
                $mainMenu.addClass('is-touchable');

            } else {

                $mainMenu.removeClass("menu-loop-happened");
                $mainMenu.removeClass('is-touchable');

                $('.main-modern-menu li.menu-item-has-children').each(function () {
                    $(this).find(".mobile-submenu-link").remove();
                });
                if (!$("#page-wrapper").hasClass("menu-is-closed")) {
                    if (!Woffice.$body.hasClass('menu-is-horizontal')) {
                        self.showVerticalMenu();
                    }
                }
            }

        },

        start: function () {

            var self = this,
                $ = Woffice.$;
            self.responsiveMenu(true);
            var $nav_width = $('#navigation').width();
            var menu_is_vertical = $('body').hasClass('menu-is-vertical');
            //Fix the menu display on load
            setTimeout(function () {
                Woffice.$navigation.removeClass("mobile-hidden");
            }, 600)

            if (menu_is_vertical) {
                $(Woffice.$navTrigger).on('click', function () {
                    if ($('.modern-top-menu').length > 0 && $('.navigation-hidden').length > 0) {
                        $('.menu-is-vertical .modern-top-menu').css({ 'margin-left': '0' })
                    } else {
                        $('.menu-is-vertical .modern-top-menu').css({ 'margin-left': $nav_width + 'px' });
                    }
                    if ($('.glass-top-menu').length > 0 && $('.navigation-hidden').length > 0) {
                        $('.menu-is-vertical .glass-top-menu').css({ 'margin-left': '0' })
                    } else {
                        $('.menu-is-vertical .glass-top-menu').css({ 'margin-left': $nav_width + 'px' });
                    }
                });

                if ($(".menu-is-closed").length > 0 || $('.navigation-hidden').length > 0) {
                    $('.menu-is-vertical .modern-top-menu').css({ 'margin-left': '0px' });
                } else {
                    $('.menu-is-vertical .modern-top-menu').css({ 'margin-left': $nav_width + 'px' });
                }
            }

        },

        watchResponsiveToggle: function () {
            var self = this,
                $ = Woffice.$;
            $('body').on('click', '#navigation .close-navmenu', function () {
                $('.is-left-sidebar').slideToggle(500);
                $('#main-content, #navigation, a#nav-sidebar-trigger').addClass('navigation-hidden');
            });
            $('body').on('click', '#right-sidebar .close-navmenu', function () {
                $('#main-content, #right-sidebar').addClass('sidebar-hidden');
            });


            $('.active_search').on('click', function () {
                $('#main-search').toggleClass('active_form');
                $('.active_search').toggleClass('active_search_close');



            });
        },

        resize: function () {
            var self = this,
                $ = Woffice.$;

            var newWidth = $(window).width();

            if (newWidth !== Woffice.navigation.cachedWidth) {
                self.responsiveMenu(false);
                Woffice.navigation.cachedWidth = newWidth;
            }
        }

    },

    /**
     * Sidebar actions
     */
    sidebar: {

        start: function () {

            var self = this,
                $ = Woffice.$;

            // Setting up the layout correctly
            self.setSidebarWidth();
            self.responsiveSidebar(true);

            self.horizontalMenuAuto();

            if (Woffice.$rightSidebar.length === 0 || Woffice.$main.hasClass('sidebar-hidden'))
                Woffice.$body.addClass('sidebar-hidden');

            Woffice.$scrollHelper.hide();

            if (Woffice.$rightSidebar.length > 0) {
                if (Woffice.$main.hasClass('sidebar-hidden')) {
                    Woffice.$scrollHelper.fadeOut();
                }
                else {
                    if (Woffice.$main.height() < Woffice.$rightSidebar[0].scrollHeight) {
                        Woffice.$scrollHelper.fadeIn('slow');
                    }
                }
            }

            // If the Cookies already exists
            if (Cookies.get('Woffice_sidebar_position') && Woffice.data.cookie_allowed.sidebar) {

                Woffice.$sidebarTrigger.addClass("sidebar-hidden");

                // Main Layout changes
                $("#main-content, #main-header, body").addClass("sidebar-hidden");

                // Icon Class
                Woffice.$sidebarTrigger.find('svg').addClass("mk-left-arrow");

                Woffice.$scrollHelper.fadeOut();

                // Rebuild the sliders
                Woffice.sliders.start();

            }
            // For the default position
            if (!self.isOpen()) {
                Woffice.$sidebarTrigger.find('svg').addClass("mk-left-arrow");
            }

        },

        resize: function () {

            var self = this;

            self.setSidebarWidth();
            self.setSidebarTopOffset();
            self.responsiveSidebar(false);

        },

        watch: function () {

            var self = this,
                $ = Woffice.$;

            $("#nav-sidebar-trigger ,  #rsidebar_mobile_close_btn").on("click", function () {

                self.sidebarToggling();
                Woffice.userSidebar.Monitor_sidebars();

            });

        },

        /**
         * Calculates the sidebar's width
         */
        setSidebarWidth: function () {

            var $ = Woffice.$,
                $rightSidebar = $("#right-sidebar");

            if ($rightSidebar.length === 0)
                return;

            var SidebarWidth = $rightSidebar.width();

            $('#can-scroll').width(SidebarWidth);

        },

        /**
         * Makes the sidebar responsive (and the menu)
         *
         * // Todo @antonio I'm lost in those lines, any comments would be appreciated :)
         * // It was in the sidebar block but it's related to the menu right?
         *
         * @param {boolean} on_load if this is during the page loading or the resize event
         */
        responsiveSidebar: function (on_load) {

            var $ = Woffice.$,
                width = $(window).width(),
                height = $(window).height();

            var $horizontalMenuWrapper = $("#horizontal-menu-trigger-container");

            var switcher = $("#horizontal-menu-trigger").length;

            if (window.matchMedia('(max-width: ' + Woffice.data.menu_threshold + 'px)').matches) {
                // Horizontal menu responsive
                if (Woffice.$body.hasClass("menu-is-horizontal")) {

                    Woffice.$navigation.addClass("menu-responsive-horizontal");

                    if (!switcher && $('#horizontal-menu-trigger-container').length === 0) {
                        Woffice.$navigation.find("ul.main-modern-menu").prepend('<li id="horizontal-menu-trigger-container"><a href="#" id="horizontal-menu-trigger"><i class="fa fa-bars"></i></a></li>');
                    }

                }
            }
            else {
                if ($horizontalMenuWrapper.length > 0) {
                    $horizontalMenuWrapper.remove();
                }
            }

            // Don't fire on mobile :
            if ($(window).width() !== width && $(window).height() !== height || on_load) {
                if (window.matchMedia('(max-width: ' + Woffice.data.menu_threshold + 'px)').matches) {


                    // Navigation bar Class
                    Woffice.$sidebarTrigger.addClass("sidebar-hidden");

                    // Icon class switching
                    Woffice.$sidebarTrigger.find("svg").addClass("mk-left-arrow");

                    // Main Layout changes
                    $("#main-content, body").addClass("sidebar-hidden");
                    Woffice.$scrollHelper.fadeOut('fast');

                    // Horizontal menu responsiveness
                    if (Woffice.$body.hasClass("menu-is-horizontal")) {

                        Woffice.$navigation.addClass("menu-responsive-horizontal");

                        if (!switcher && $('#horizontal-menu-trigger-container').length == 0) {
                            Woffice.$navigation.find("ul.main-menu").prepend('<li id="horizontal-menu-trigger-container"><a href="#" id="horizontal-menu-trigger"><i class="fa fa-bars"></i></a></li>');
                        }

                    }

                }
                else {

                    if ($horizontalMenuWrapper.length > 0) {
                        $horizontalMenuWrapper.remove();
                    }

                }
            }

        },

        /**
         * Opens or closes the sidebar and handle the cookies
         */
        sidebarToggling: function () {

            var self = this,
                $ = Woffice.$;

            // Open the sidebar
            if (!self.isOpen()) {

                self.openSidebar();

                // Erase the cookies
                Cookies.remove('Woffice_sidebar_position', { expires: 7, path: '/' });
                Cookies.remove('Woffice_hassidebar_position', { expires: 7, path: '/' });

            }
            // Close the sidebar
            else {

                // If is a mobile device
                if (window.matchMedia('(max-width: ' + Woffice.data.menu_threshold + 'px)').matches) {
                    var $search_trigger = Woffice.$searchTrigger,
                        $notification_trigger = $('#nav-notification-trigger');

                    //if the search bar is opened, then close it
                    if ($search_trigger.length > 0 && $('#main-search').hasClass('opened'))
                        $($search_trigger).click();

                    //if the notifications bar is opened, then close it
                    if ($notification_trigger.length > 0 && $notification_trigger.hasClass('clicked'))
                        $($notification_trigger).click();
                }

                self.closeSidebar();

                // Create cookies to save user choice :
                Cookies.set('Woffice_sidebar_position', 'sidebar-hidden', { expires: 7, path: '/' });
                Cookies.set('Woffice_hassidebar_position', 'has-sidebar-hidden', { expires: 7, path: '/' });
            }

        },

        /**
         * Sets the sidebar offset and animate if required
         */
        setSidebarTopOffset: function () {

            var $ = Woffice.$;

            var $sidebar = $('#right-sidebar'),
                $navbar = $('#navbar'),
                height = 0;

            //If there is no sidebar rendered, then exit from this function
            if ($sidebar.length <= 0)
                return;

            if (window.matchMedia('(max-width: 600px)').matches && $navbar.length > 0) {
                height += $navbar.height();
                if (Woffice.$body.hasClass('has-navigation-fixed'))
                    Woffice.$rightSidebar.css('height', '100%').css('height', '-=' + $navbar.height());
                else
                    Woffice.$rightSidebar.css('height', '100%');
            }

        },

        /**
         * Checks whether the sidebar is open or not
         *
         * @return {boolean}
         */
        isOpen: function () {

            return !(Woffice.$main.hasClass("sidebar-hidden"));

        },

        /**
         * Opens the sidebar
         */
        openSidebar: function () {

            var self = this,
                $ = Woffice.$;

            // Icon class switching
            Woffice.$sidebarTrigger.find("svg").removeClass("mk-left-arrow");

            // Navbar Class
            Woffice.$sidebarTrigger.removeClass("sidebar-hidden");

            // Main Layout changes
            $("#main-content, #main-header, body, #right-sidebar").removeClass("sidebar-hidden");
            $('.is-right-sidebar').removeClass('has-sidebar-hidden');
            //Avoid overlap with header
            self.setSidebarTopOffset();

            Woffice.$scrollHelper.fadeIn('fast');

            if (
                Woffice.$body.hasClass('menu-is-vertical') &&
                !Woffice.$body.hasClass('navigation-hidden') &&
                window.matchMedia('(max-width: 450px)').matches
            ) {
                $('#nav-trigger').click();
            }

        },

        /**
         * Closes the sidebar
         */
        closeSidebar: function () {

            var $ = Woffice.$;

            // Navigation bar class
            Woffice.$sidebarTrigger.addClass("sidebar-hidden");
            $('.is-right-sidebar').addClass('has-sidebar-hidden');
            // Icon class switching
            Woffice.$sidebarTrigger.find("svg").addClass("mk-left-arrow");

            // Main Layout changes
            $("#main-content, #main-header, body").addClass("sidebar-hidden");
            Woffice.$scrollHelper.fadeOut();

        },

        /**
         * Responsive menu class toggling
         * This is here because we need the right scope to close the sidebar automatically
         */
        horizontalMenuAuto: function () {

            var self = this,
                $ = Woffice.$;

            $('body.menu-is-horizontal .main-modern-menu').on("click", "#horizontal-menu-trigger", function () {

                Woffice.$navigation.toggleClass("menu-responsive-horizontal-show");

                Woffice.$body.toggleClass("navigation-hidden");

                if (
                    !Woffice.$body.hasClass('navigation-hidden') &&
                    window.matchMedia('(max-width: ' + Woffice.data.menu_threshold + 'px)').matches
                )
                    self.closeSidebar();

            });

        }

    },

    buddyPressNotifications: {

        /**
         * Close the notification box
         */
        close: function () {

            var $ = Woffice.$;

            $('#woffice-notifications-menu').fadeOut();
            $('#nav-notification-trigger').find('.svg_bell').removeClass('svg_hide').addClass('svg_show');
            $('#nav-notification-trigger').find('i.fa').hide();

        },

        watch: function () {

            var $ = Woffice.$;

            $('#nav-notification-trigger').on('click', function () {

                var $svgicon = $(this).find('.svg_bell');
                var $icon = $(this).find('i.fa');

                // if it wasn't opened in the first place
                if ($svgicon.hasClass('svg_show')) {
                    $svgicon.removeClass('svg_show').addClass('svg_hide');
                    $icon.show();
                    Woffice.buddyPressNotifications.fetch();
                } else {

                    Woffice.buddyPressNotifications.close();
                    $svgicon.removeClass('svg_hide').addClass('svg_show');
                    $icon.hide();
                }

            });

        },

        /**
         * Mark a message as read
         * @param el
         */
        markRead: function (el) {

            var $ = Woffice.$;

            var readLink = $(el),
                component_action = readLink.data('component-action'),
                component_name = readLink.data('component-name'),
                item_id = readLink.data('item-id');

            $.ajax({
                url: Woffice.data.ajax_url.toString(),
                type: 'POST',
                data: {
                    'action': 'wofficeNoticationsMarked',
                    'nonce': WOFFICE.nonce,
                    'component_action': component_action,
                    'component_name': component_name,
                    'item_id': item_id
                },
                success: function () {

                    readLink.parent().closest('div').remove();
                    if ($('#woffice-notifications-content').children().length === 0) {
                        $('#nav-notification-trigger').removeClass('active');
                        Woffice.buddyPressNotifications.close();
                    }

                },
                error: function () {
                    console.error('Ajax marked failed');
                }
            });

        },

        /**
         * Fetch the notifications
         */
        fetch: function () {

            var $ = Woffice.$;

            var $trigger = $('#nav-notification-trigger'),
                $icon = $trigger.find('i.fa');

            var $wrapper = $('#woffice-notifications-menu');
            $wrapper.slideDown();

            var loader = new Woffice.loader($wrapper);

            $.ajax({
                url: Woffice.data.ajax_url.toString(),
                type: 'POST',
                data: { 'action': 'wofficeNoticationsGet', 'nonce': WOFFICE.nonce, 'user': Woffice.data.user_id.toString() },
                success: function (notifications) {
                    $wrapper.find('#woffice-notifications-content').empty();
                    $wrapper.find('#woffice-notifications-content').html(notifications);
                    loader.remove();
                    $('a.mark-notification-read').on('click', function () {
                        Woffice.buddyPressNotifications.markRead(this);
                    });
                },
                error: function () {
                    console.error('Ajax notifications failed');
                }
            });

        }



    },

    /**
     * Todo Extratabs like TODO,Knaban,Timeline
     */
    TododExtraTabs: {

        watch: function () {
            var $ = Woffice.$;
            if ($('.woffice-todo-extratabs').length > 0) {
                $('.todo-extratabs-item').each(function () {
                    $(this).on('click', function (e) {
                        e.preventDefault();
                        var currenttab = $(this).data('extratab');
                        $(this).siblings('.todo-extratabs-item').removeClass('extratabs-item-active');
                        $(this).addClass('extratabs-item-active');
                        $('.extratabs-item').removeClass('extratabs-content-active');
                        $('.project-tabs-wrapper #project-content-' + currenttab).removeClass('d-none');
                        $('.project-tabs-wrapper #project-content-' + currenttab).addClass('extratabs-content-active');
                        $('.extratabs-item:not(.extratabs-content-active)').addClass('d-none');
                        $('.woffice-tab-layout__nav #project-tab-todo').addClass('is-active active');
                    });
                });
            }
        },
    },

    select2runner: {
        watch: function () {
            var $ = Woffice.$;
            if ($('.postform-select').length > 0) {
                $('.postform-select').select2({
                    dropdownParent: $('#postform-select-parent'),
                    placeholder: "Select an option",
                    width: '100%',
                });
            }

            if ($('.multiselectbox-select2').length > 0) {
                $('.multiselectbox-select2').select2({
                    width: '100%',
                    placeholder: {
                        id: "-1",
                        text: "Select an option",
                        selected: 'selected'
                    },
                });
            }

            if ($('.woffice-file-col .woffice-file-specific-members').length > 0) {
                $('.woffice-file-col .woffice-file-specific-members').select2({
                    width: '100%',
                });
            }

            if ($('.horizontal_menu_search_icon').length > 0) {
                $(".horizontal_menu_search_icon").on("click", function () {
                    $(this).toggleClass('horizontal_menu_search_icon_active');
                    $("body.menu-is-horizontal #page-wrapper .is-left-sidebar #navigation .horizontal_menu_icons_wrapper #nav-buttons #main-search form").toggleClass("form_active");
                });
            }

            if ($(window).width() > 1921) {
                var leftwidth = $('#page-wrapper .has-navigation-show').outerWidth();
                $('#page-wrapper .is-center-content').css('padding-left', leftwidth);
            }

        }
    }

};

/**
 * Start it!
 *
 * We give it a jQuery object to play with
 */
Woffice.init(jQuery);
