<?php
/**
 * Wolmart Vendors class
 *
 * Available plugins are: Dokan, WCFM, WC Marketplace, WC Vendors
 *
 * @version 1.0
 */

defined( 'ABSPATH' ) || die;

if ( ! defined( 'ELEMENTOR_VERSION' ) ) {
	return;
}

define( 'WOLMART_CORE_ELEMENTOR', WOLMART_CORE_PLUGINS . '/elementor' );
define( 'WOLMART_CORE_ELEMENTOR_URI', WOLMART_CORE_PLUGINS_URI . '/elementor' );

use Elementor\Frontend;
use Elementor\Plugin;
use Elementor\Controls_Manager;
use Elementor\Wolmart_Controls_Manager;


class Wolmart_Core_Elementor extends Wolmart_Base {

	/**
	 * Check if dom is optimized
	 *
	 * @since 1.0
	 *
	 * @var boolean $is_dom_optimized
	 */
	public static $is_dom_optimized = false;

	/**
	 * Constructor
	 *
	 * @since 1.0
	 */
	public function __construct() {

		// Include Partials
		include_once 'partials/banner.php';
		include_once 'partials/creative.php';
		include_once 'partials/grid.php';
		include_once 'partials/slider.php';
		include_once 'partials/button.php';
		include_once 'partials/tab.php';
		include_once 'partials/products.php';
		include_once 'partials/testimonial.php';

		// Register controls, widgets, elements, icons
		add_action( 'elementor/elements/categories_registered', array( $this, 'register_category' ) );
		add_action( 'elementor/controls/register', array( $this, 'register_control' ) );
		add_action( 'elementor/widgets/register', array( $this, 'register_widget' ) );
		add_action( 'elementor/frontend/after_register_styles', array( $this, 'register_widget_styles' ), 10 );
		add_action( 'elementor/frontend/after_register_scripts', array( $this, 'register_widget_scripts' ), 10 );
		add_action( 'elementor/elements/elements_registered', array( $this, 'register_element' ) );
		add_filter( 'elementor/icons_manager/additional_tabs', array( $this, 'wolmart_add_icon_library' ) );
		add_filter( 'elementor/controls/animations/additional_animations', array( $this, 'add_appear_animations' ), 10, 1 );

		// Load Elementor CSS and JS
		if ( wolmart_is_elementor_preview() ) {
			add_action( 'wp_enqueue_scripts', array( $this, 'load_preview_scripts' ) );
		}

		// Disable elementor resource.
		if ( apply_filters( 'wolmart_resource_disable_elementor', function_exists( 'wolmart_get_option' ) && wolmart_get_option( 'resource_disable_elementor' ) ) && ! current_user_can( 'edit_pages' ) ) {
			add_action( 'wp_enqueue_scripts', array( $this, 'resource_disable_elementor' ), 99 );
			add_action( 'elementor/widget/before_render_content', array( $this, 'enqueue_theme_alternative_scripts' ) );

			// Do not update dynamic css for visitors.
			add_action( 'init', array( $this, 'remove_dynamic_css_update' ) );
		}

		add_action( 'elementor/editor/after_enqueue_styles', array( $this, 'load_admin_styles' ) );

		// Include Elementor Admin JS
		add_action(
			'elementor/editor/after_enqueue_scripts',
			function() {
				if ( defined( 'WOLMART_VERSION' ) ) {
					wp_enqueue_style( 'wolmart-icons', WOLMART_ASSETS . '/vendor/wolmart-icons/css/icons.min.css', array(), WOLMART_VERSION );
				}
				wp_enqueue_script( 'wolmart-elementor-admin-js', WOLMART_CORE_ELEMENTOR_URI . '/assets/elementor-admin' . ( ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '.js' : '.min.js' ), array( 'elementor-editor' ), WOLMART_CORE_VERSION, true );
			}
		);

		// Before page view without page assets
		add_action( 'elementor/frontend/get_builder_content', array( $this, 'after_page_first_print' ), 10, 3 );
		// Update elementor page assets after elementor editor save
		add_action( 'elementor/editor/after_save', array( $this, 'after_elementor_editor_save' ), 10, 2 );

		// Add Elementor Page Custom CSS
		if ( wp_doing_ajax() ) {
			add_action( 'elementor/document/before_save', array( $this, 'save_page_custom_css_js' ), 10, 2 );
			add_action( 'elementor/document/after_save', array( $this, 'save_elementor_page_css_js' ), 10, 2 );
		}

		// Init Elementor Document Config
		add_filter( 'elementor/document/config', array( $this, 'init_elementor_config' ), 10, 2 );

		// Register Document Controls
		add_action( 'elementor/documents/register_controls', array( $this, 'register_document_controls' ) );

		// Add Custom CSS & JS to Wolmart Elementor Addons
		add_filter( 'wolmart_builder_addon_html', array( $this, 'add_custom_css_js_addon_html' ) );

		// Because elementor removes all callbacks, add it again
		add_action( 'elementor/editor/after_enqueue_scripts', 'wolmart_print_footer_scripts' );

		// Add Template Builder Classes
		add_filter( 'body_class', array( $this, 'add_body_class' ) );

		// Compatabilities
		add_filter( 'elementor/widgets/wordpress/widget_args', array( $this, 'add_wp_widget_args' ), 10, 2 );

		// Load Used Block CSS
		/*
		 * Get Dependent Elementor Styles
		 * Includes Kit style and post style
		 */
		add_action( 'elementor/css-file/post/enqueue', array( $this, 'get_dependent_elementor_styles' ) );
		add_action( 'wolmart_before_enqueue_theme_style', array( $this, 'add_global_css' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'add_elementor_css' ), 30 );
		add_action( 'wolmart_before_enqueue_custom_css', array( $this, 'add_elementor_page_css' ), 20 );
		add_action( 'wolmart_before_enqueue_custom_css', array( $this, 'add_block_css' ) );

		// Force generate Wolmart elementor block css
		add_action( 'elementor/core/files/clear_cache', array( $this, 'generate_blocks_css_after_clear_cache' ) );

		if ( wp_doing_ajax() && empty( $_REQUEST['demo'] ) ) {
			add_action( 'save_post', array( $this, 'generate_block_temp_css_onsave' ), 99, 2 );
			add_action( 'elementor/document/after_save', array( $this, 'rename_block_temp_css_onsave' ), 99, 2 );
		}

		// Elementor Custom Control Manager
		require_once WOLMART_CORE_ELEMENTOR . '/restapi/select2.php';
		require_once WOLMART_CORE_ELEMENTOR . '/controls_manager/controls.php';

		// Elementor Custom Advanced Tab Sections
		require_once WOLMART_CORE_ELEMENTOR . '/tabs_advanced/widget_advanced_tabs.php';

		// Alpus flexbox plugin compatibility
		add_filter( 'alpus_modules', array( $this, 'remove_modules' ) );
	}

	// Register new Category
	public function register_category( $self ) {
		$self->add_category(
			'wolmart_widget',
			array(
				'title'  => esc_html__( 'Wolmart Widgets', 'wolmart-core' ),
				'active' => true,
			)
		);
	}

	// Register new Control
	public function register_control( $self ) {

		$controls = array(
			'ajaxselect2',
			'description',
			'image_choose',
			'origin_position',
		);

		foreach ( $controls as $control ) {
			include_once WOLMART_CORE_ELEMENTOR . '/controls/' . $control . '.php';
			$class_name = 'Wolmart_Control_' . ucfirst( $control );
			$self->register( new $class_name() );
		}
	}


	public function register_element() {

		if ( wolmart_elementor_if_container_active() ) {
			include_once WOLMART_CORE_ELEMENTOR . '/elements/container.php';
			Elementor\Plugin::$instance->elements_manager->unregister_element_type( 'container' );
			Elementor\Plugin::$instance->elements_manager->register_element_type( new Wolmart_Element_Container() );
		}

		include_once WOLMART_CORE_ELEMENTOR . '/elements/section.php';
		Elementor\Plugin::$instance->elements_manager->unregister_element_type( 'section' );
		Elementor\Plugin::$instance->elements_manager->register_element_type( new Wolmart_Element_Section() );

		include_once WOLMART_CORE_ELEMENTOR . '/elements/column.php';
		Elementor\Plugin::$instance->elements_manager->unregister_element_type( 'column' );
		Elementor\Plugin::$instance->elements_manager->register_element_type( new Wolmart_Element_Column() );
	}


	// Register new Widget
	public function register_widget( $self ) {

		$widgets = array(
			'heading',
			'posts',
			'block',
			'banner',
			'countdown',
			'button',
			'image-gallery',
			// 'testimonial',
			'testimonial-group',
			'image-box',
			'share',
			'menu',
			'subcategories',
			'hotspot',
			'logo',
			'iconlist',
			'svg-floating',
			'marquee',
		);

		if ( class_exists( 'WooCommerce' ) ) {
			$widgets = array_merge(
				$widgets,
				array(
					'breadcrumb',
					'products',
					'products-tab',
					'products-banner',
					'products-single',
					'categories',
					'brands',
					'singleproducts',
					'filter',
				)
			);

			if ( class_exists( 'WeDevs_Dokan' ) || class_exists( 'WCMp' ) || class_exists( 'WCFM' ) || class_exists( 'WC_Vendors' ) || defined( 'MVX_PLUGIN_VERSION' ) ) {
				$widgets = array_merge(
					$widgets,
					array( 'vendor' )
				);
			}
		}

		array_multisort( $widgets );

		foreach ( $widgets as $widget ) {
			$prefix = $widget;
			if ( 'products' == substr( $widget, 0, 8 ) ) {
				$prefix = 'products';
			} elseif ( 'testimonial' == substr( $widget, 0, 11 ) ) {
				$prefix = 'testimonial';
			}
			wolmart_core_require_once( '/widgets/' . $prefix . '/widget-' . str_replace( '_', '-', $widget ) . '-elementor.php' );
			$class_name = 'Wolmart_' . ucwords( str_replace( '-', '_', $widget ), '_' ) . '_Elementor_Widget';
			$self->register( new $class_name( array(), array( 'widget_name' => $class_name ) ) );
		}
	}



	/**
	 * Register styles and scripts.
	 *
	 * @since 1.0
	 */
	public function register_widget_styles() {
		if ( ! defined( 'WOLMART_ASSETS' ) ) {
			return;
		}

		$styles = array(
			'wolmart-marquee' => array(
				'src' => WOLMART_CORE_FRAMEWORK_URI . '/widgets/marquee/marquee' . ( is_rtl() ? '-rtl' : '' ) . '.min.css',
			),
		);

		// Animation styles
		foreach ( wolmart_get_animations( 'in' ) as $animation => $title ) {
			if ( 'none' == $animation ) {
				continue;
			}
			if ( wp_style_is( 'e-animation-' . $animation, 'registered' ) ) {
				continue;
			}
			$styles[ 'e-animation-' . $animation ] = array(
				'src' => WOLMART_ASSETS . '/css/components/animations/' . $animation . '.min.css',
			);
		}
		foreach ( wolmart_get_animations( 'out' ) as $animation => $title ) {
			$styles[ 'e-animation-' . $animation ] = array(
				'src' => WOLMART_ASSETS . '/css/components/animations/' . $animation . '.min.css',
			);
		}
		foreach ( wolmart_get_animations( 'appear' ) as $group => $animations ) {
			foreach ( $animations as $animation => $animation_name ) {
				$styles[ 'e-animation-' . $animation ] = array(
					'src' => WOLMART_ASSETS . '/css/components/animations/' . $animation . '.min.css',
				);
			}
		}

		$styles = apply_filters( 'wolmart_elementor_styles', $styles );

		foreach ( $styles as $handle => $style ) {
			wp_register_style( $handle, $style['src'], ! empty( $style['deps'] ) ? $style['deps'] : array(), isset( $style['ver'] ) ? $style['ver'] : WOLMART_CORE_VERSION );
		}
	}

	public function register_widget_scripts() {
		$scripts = array();

		$scripts = apply_filters( 'wolmart_elementor_scripts', $scripts );

		foreach ( $scripts as $handle => $script ) {
			wp_register_script( $handle, $script['src'], ! empty( $script['deps'] ) ? $script['deps'] : array(), isset( $script['ver'] ) ? $script['ver'] : WOLMART_CORE_VERSION, isset( $script['in_footer'] ) ? $script['in_footer'] : false );
		}
	}

	public function load_admin_styles() {
		wp_enqueue_style( 'wolmart-elementor-admin-style', WOLMART_CORE_ELEMENTOR_URI . '/assets/elementor-admin' . ( is_rtl() ? '-rtl' : '' ) . '.min.css' );
	}

	public function load_preview_scripts() {

		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '.js' : '.min.js';

		// load needed style file in elementor preview
		wp_enqueue_style( 'wolmart-elementor-preview', WOLMART_CORE_ELEMENTOR_URI . '/assets/elementor-preview' . ( is_rtl() ? '-rtl' : '' ) . '.min.css' );
		wp_enqueue_script( 'wolmart-elementor-js', WOLMART_CORE_ELEMENTOR_URI . '/assets/elementor' . $suffix, array(), WOLMART_CORE_VERSION, true );
		wp_localize_script(
			'wolmart-elementor-js',
			'wolmart_elementor',
			array(
				'ajax_url'      => esc_js( admin_url( 'admin-ajax.php' ) ),
				'wpnonce'       => wp_create_nonce( 'wolmart-elementor-nonce' ),
				'assets_url'    => WOLMART_CORE_FRAMEWORK_URI,
				'text_untitled' => esc_html__( 'Untitled', 'wolmart-core' ),
			)
		);
	}

	/**
	 * Disable elementor resource for high performance
	 *
	 * @since 1.0
	 */
	public function resource_disable_elementor() {
		wp_dequeue_style( 'elementor-icons' );
		wp_dequeue_style( 'e-animations' );
		// wp_dequeue_script( 'elementor-frontend' );
		// wp_dequeue_script( 'elementor-frontend-modules' );
		wp_dequeue_script( 'elementor-waypoints' );
		// wp_dequeue_script( 'elementor-webpack-runtime' );
		wp_deregister_style( 'elementor-icons' );
		// wp_deregister_script( 'elementor-frontend' );
		// wp_deregister_script( 'elementor-frontend-modules' );
		wp_deregister_script( 'elementor-waypoints' );
		// wp_deregister_script( 'elementor-webpack-runtime' );
	}

	/**
	 * Enqueue alternative scripts for disable elementor resource mode.
	 *
	 * @param $widget
	 * @since 1.0
	 */
	public function enqueue_theme_alternative_scripts( $widget ) {
		if ( 'counter' == $widget->get_name() ) {
			wp_enqueue_script( 'jquery-count-to' );
		}
	}

	public function wolmart_add_icon_library( $icons ) {
		if ( defined( 'WOLMART_VERSION' ) ) {
			$icons['wolmart-icons'] = array(
				'name'          => 'wolmart',
				'label'         => esc_html__( 'Wolmart Icons', 'wolmart-core' ),
				'prefix'        => 'w-icon-',
				'displayPrefix' => ' ',
				'labelIcon'     => 'w-icon-gift',
				'fetchJson'     => WOLMART_CORE_ELEMENTOR_URI . '/assets/wolmart-icons.js',
				'ver'           => WOLMART_CORE_VERSION,
				'native'        => false,
			);
		}
		return $icons;
	}

	/**
	 * Checking if Elementor cache is enabled
	 * 
	 * @since 1.9.6
	 */
	public function is_elementor_cache_enabled() {
		if ( version_compare( ELEMENTOR_VERSION, '3.32.0', '>=') ) {
			$option = get_option( 'elementor_element_cache_ttl', '' );
			return 'disable' == $option ? false : true;
		}
		return Elementor\Plugin::$instance->experiments->is_feature_active( 'e_element_cache' );
	}

	public function after_page_first_print( $document, $is_excerpt, $with_css ) {
		if ( version_compare( ELEMENTOR_VERSION, '3.24', '<' ) ) {
			return;
		}

		$post_id  = $document->get_main_id();
		$is_saved = get_post_meta( $post_id, 'wolmart_elementor_page_assets_saved', true );

		if ( $is_saved && ( ! $this->is_elementor_cache_enabled() || get_post_meta( $post_id, Elementor\Core\Base\Document::CACHE_META_KEY, true ) ) ) {
			return;
		}

		$data = $document->get_elements_data();
		$this->create_elementor_page_assets( $post_id, $data, true );
	}

	public function after_elementor_editor_save( $post_id, $editor_data ) {
		if ( ! $this->is_elementor_cache_enabled() ) {
			$this->create_elementor_page_assets( $post_id, $editor_data );
		} else {
			delete_post_meta( $post_id, 'wolmart_elementor_page_assets_saved' );
		}
	}

	public function create_elementor_page_assets( $post_id, $editor_data, $with_css = false ) {
		preg_match_all( '/_animation"[^"]+"([\w]+)"/', json_encode( $editor_data ), $animations );

		if ( $animations ) {
			$page_assets = get_post_meta( $post_id, Elementor\Core\Base\Elements_Iteration_Actions\Assets::ASSETS_META_KEY, true );
			$page_cache  = json_decode( get_post_meta( $post_id, Elementor\Core\Base\Document::CACHE_META_KEY, true ), true );

			if ( empty( $page_assets ) ) {
				$page_assets = array();
			}
			if ( empty( $page_assets['styles'] ) ) {
				$page_assets['styles'] = array();
			}
			foreach ( $animations[1] as $animation ) {
				$style_key = 'e-animation-' . $animation;
				if ( ! in_array( $style_key, $page_assets['styles'] ) ) {
					$page_assets['styles'][] = $style_key;

					if ( isset( $page_cache['value']['styles'] ) ) {
						$page_cache['value']['styles'][] = $style_key;
					}

					if ( $with_css ) {
						wp_enqueue_style( $style_key );
					}
				}
			}
		}

		$animations = array();
		if ( 'popup' == get_post_meta( $post_id, 'wolmart_template_type', true ) ) {
			$popup_settings = get_post_meta( $post_id, '_elementor_page_settings', true );
			$data_json      = json_encode( $popup_settings );
			if ( ! isset( $page_cache ) ) {
				$page_cache = json_decode( get_post_meta( $post_id, Elementor\Core\Base\Document::CACHE_META_KEY, true ), true );
			}
			preg_match_all( '/_animation"[^"]+"([\w]+)"/', $data_json, $animations );

			if ( $animations ) {
				if ( ! isset( $page_assets ) ) {
					$page_assets = get_post_meta( $post_id, Elementor\Core\Base\Elements_Iteration_Actions\Assets::ASSETS_META_KEY, true );
				}
				if ( empty( $page_assets ) ) {
					$page_assets = array();
				}
				if ( empty( $page_assets['styles'] ) ) {
					$page_assets['styles'] = array();
				}
				if ( $animations[1] ) {
					foreach ( $animations[1] as $animation ) {
						$style_key = 'e-animation-' . $animation;
						if ( ! in_array( $style_key, $page_assets['styles'] ) ) {
							$page_assets['styles'][] = $style_key;

							if ( isset( $page_cache['value']['styles'] ) ) {
								$page_cache['value']['styles'][] = $style_key;
							}

							if ( $with_css ) {
								wp_enqueue_style( $style_key );
							}
						}
					}
				} else {
					if ( ! in_array( 'e-animation-fadeIn', $page_assets['styles'] ) ) {
						$page_assets['styles'][] = 'e-animation-fadeIn';

						if ( isset( $page_cache['value']['styles'] ) ) {
							$page_cache['value']['styles'][] = 'e-animation-fadeIn';
						}

						if ( $with_css ) {
							wp_enqueue_style( 'e-animation-fadeIn' );
						}
					}
				}
			}
		}

		// Cache compatiblity of templates
		if ( 'wolmart_template' == get_post_type( $post_id ) && $this->is_elementor_cache_enabled() ) {
			if ( ! isset( $page_assets ) ) {
				$page_assets = get_post_meta( $post_id, Elementor\Core\Base\Elements_Iteration_Actions\Assets::ASSETS_META_KEY, true );
			}
			if ( ! isset( $page_cache ) ) {
				$page_cache = json_decode( get_post_meta( $post_id, Elementor\Core\Base\Document::CACHE_META_KEY, true ), true );
			}

			if ( isset( $page_cache['value']['styles'] ) && ! empty( $page_assets['styles'] ) ) {
				$page_cache['value']['styles'] = array_merge( $page_cache['value']['styles'], array_values( array_diff( $page_assets['styles'], $page_cache['value']['styles'] ) ) );
			}
			if ( isset( $page_cache['value']['scripts'] ) && ! empty( $page_assets['scripts'] ) ) {
				$page_cache['value']['scripts'] = array_merge( $page_cache['value']['scripts'], array_values( array_diff( $page_assets['scripts'], $page_cache['value']['scripts'] ) ) );
			}
		}

		if ( isset( $page_assets ) ) {
			update_post_meta( $post_id, Elementor\Core\Base\Elements_Iteration_Actions\Assets::ASSETS_META_KEY, $page_assets );
		}
		if ( isset( $page_cache ) ) {
			update_post_meta( $post_id, Elementor\Core\Base\Document::CACHE_META_KEY, wp_slash( wp_json_encode( $page_cache ) ) );
		}

		update_post_meta( $post_id, 'wolmart_elementor_page_assets_saved', true );
	}

	public function save_page_custom_css_js( $self, $data ) {
		if ( empty( $data['settings'] ) || empty( $_REQUEST['editor_post_id'] ) ) {
			return;
		}
		$post_id = absint( $_REQUEST['editor_post_id'] );

		// save Wolmart elementor page CSS
		if ( ! empty( $data['settings']['page_css'] ) ) {
			update_post_meta( $post_id, 'page_css', wp_slash( $data['settings']['page_css'] ) );
		} else {
			delete_post_meta( $post_id, 'page_css' );
		}

		if ( current_user_can( 'unfiltered_html' ) ) {
			// save Wolmart elementor page JS
			if ( ! empty( $data['settings']['page_js'] ) ) {
				update_post_meta( $post_id, 'page_js', trim( preg_replace( '#<script[^>]*>(.*)</script>#is', '$1', $data['settings']['page_js'] ) ) );
			} else {
				delete_post_meta( $post_id, 'page_js' );
			}
		}
	}

	public function save_elementor_page_css_js( $self, $data ) {
		if ( current_user_can( 'unfiltered_html' ) || empty( $data['settings'] ) || empty( $_REQUEST['editor_post_id'] ) ) {
			return;
		}
		$post_id = absint( $_REQUEST['editor_post_id'] );
		if ( ! empty( $data['settings']['page_css'] ) ) {
			$elementor_settings = get_post_meta( $post_id, '_elementor_page_settings', true );
			if ( is_array( $elementor_settings ) ) {
				$elementor_settings['page_css'] = wolmart_strip_script_tags( get_post_meta( $post_id, 'page_css', true ) );
				update_post_meta( $post_id, '_elementor_page_settings', $elementor_settings );
			}
		}
		if ( ! empty( $data['settings']['page_js'] ) ) {
			$elementor_settings = get_post_meta( $post_id, '_elementor_page_settings', true );
			if ( is_array( $elementor_settings ) ) {
				$elementor_settings['page_js'] = wolmart_strip_script_tags( get_post_meta( $post_id, 'page_js', true ) );
				update_post_meta( $post_id, '_elementor_page_settings', $elementor_settings );
			}
		}
	}

	public function init_elementor_config( $config = array(), $post_id = 0 ) {

		if ( ! isset( $config['settings'] ) ) {
			$config['settings'] = array();
		}
		if ( ! isset( $config['settings']['settings'] ) ) {
			$config['settings']['settings'] = array();
		}

		$config['settings']['settings']['page_css'] = get_post_meta( $post_id, 'page_css', true );
		$config['settings']['settings']['page_js']  = get_post_meta( $post_id, 'page_js', true );
		return $config;
	}

	/**
	 * Add custom css, js addon html to bottom of elementor editor panel.
	 *
	 * @since 1.0
	 * @param array $html
	 * @return array $html
	 */
	public function add_custom_css_js_addon_html( $html ) {
		$html[] = array(
			'elementor' => '<li id="wolmart-custom-css"><i class="w-icon-css3"></i>' . esc_html__( 'Page CSS', 'wolmart-core' ) . '</li>',
		);
		$html[] = array(
			'elementor' => '<li id="wolmart-custom-js"><i class="w-icon-js"></i>' . esc_html__( 'Page JS', 'wolmart-core' ) . '</li>',
		);
		return $html;
	}

	public function register_document_controls( $document ) {
		if ( ! $document instanceof Elementor\Core\DocumentTypes\PageBase && ! $document instanceof Elementor\Modules\Library\Documents\Page ) {
			return;
		}

		// Add Template Builder Controls
		$id = (int) $document->get_main_id();

		if ( 'wolmart_template' == get_post_type( $id ) ) {
			$category = get_post_meta( get_the_ID(), 'wolmart_template_type', true );

			if ( $id && 'popup' == get_post_meta( $id, 'wolmart_template_type', true ) ) {

				$selector = '.mfp-wolmart-' . $id;

				$document->start_controls_section(
					'wolmart_popup_display_settings',
					array(
						'label' => wolmart_elementor_panel_heading( esc_html__( 'Popup Show Settings', 'wolmart-core' ) ),
						'tab'   => Elementor\Controls_Manager::TAB_SETTINGS,
					)
				);

					$document->add_control(
						'popup_show_on',
						array(
							'label'   => __( 'Show On', 'wolmart-core' ),
							'type'    => Elementor\Controls_Manager::SELECT,
							'default' => 'page_load',
							'options' => array(
								'page_load'      => esc_html__( 'Page Load', 'wolmart-core' ),
								'page_scroll'    => esc_html__( 'Page Scroll', 'wolmart-core' ),
								'scroll_element' => esc_html__( 'Scroll to Element', 'wolmart-core' ),
								'click_counts'   => esc_html__( 'Click', 'wolmart-core' ),
								'click_element'  => esc_html__( 'Element Click', 'wolmart-core' ),
								'exit'           => esc_html__( 'Exit', 'wolmart-core' ),
							),
						)
					);

					$document->add_control(
						'popup_page_load_delay',
						array(
							'type'      => Elementor\Controls_Manager::NUMBER,
							'label'     => esc_html__( 'Delay (s)', 'wolmart-core' ),
							'min'       => 0,
							'step'      => 0.1,
							'condition' => array(
								'popup_show_on' => 'page_load',
							),
						)
					);

					$document->add_control(
						'popup_page_scroll_dir',
						array(
							'label'     => __( 'Scroll Direction', 'wolmart-core' ),
							'type'      => Elementor\Controls_Manager::SELECT,
							'default'   => 'down',
							'options'   => array(
								'down' => esc_html__( 'Down', 'wolmart-core' ),
								'up'   => esc_html__( 'Up', 'wolmart-core' ),
							),
							'condition' => array(
								'popup_show_on' => 'page_scroll',
							),
						)
					);

					$document->add_control(
						'popup_page_scroll_amount',
						array(
							'label'     => esc_html__( 'Amount (%)', 'wolmart-core' ),
							'type'      => Elementor\Controls_Manager::SLIDER,
							'default'   => array(
								'size' => 50,
							),
							'range'     => array(
								'px' => array(
									'step' => 1,
									'min'  => 0,
									'max'  => 100,
								),
							),
							'condition' => array(
								'popup_show_on'         => 'page_scroll',
								'popup_page_scroll_dir' => 'down',
							),
						)
					);

					$document->add_control(
						'popup_scroll_element_selector',
						array(
							'type'        => Elementor\Controls_Manager::TEXT,
							'label'       => esc_html__( 'Css Selectors (#my-id, .my-class)', 'wolmart-core' ),
							'label_block' => true,
							'condition'   => array(
								'popup_show_on' => 'scroll_element',
							),
						)
					);

					$document->add_control(
						'popup_click_counts',
						array(
							'type'      => Elementor\Controls_Manager::NUMBER,
							'label'     => esc_html__( 'Number of Clicks', 'wolmart-core' ),
							'default'   => 1,
							'min'       => 1,
							'step'      => 1,
							'condition' => array(
								'popup_show_on' => 'click_counts',
							),
						)
					);

					$document->add_control(
						'popup_click_element_selector',
						array(
							'type'        => Elementor\Controls_Manager::TEXT,
							'label'       => esc_html__( 'Css Selectors (#my-id, .my-class)', 'wolmart-core' ),
							'label_block' => true,
							'condition'   => array(
								'popup_show_on' => 'click_element',
							),
						)
					);

				$document->end_controls_section();

				$document->start_controls_section(
					'wolmart_popup_settings',
					array(
						'label' => wolmart_elementor_panel_heading( esc_html__( 'Popup Style Settings', 'wolmart-core' ) ),
						'tab'   => Elementor\Controls_Manager::TAB_SETTINGS,
					)
				);

					$document->add_responsive_control(
						'popup_width',
						array(
							'label'      => esc_html__( 'Width', 'wolmart-core' ),
							'type'       => Elementor\Controls_Manager::SLIDER,
							'default'    => array(
								'size' => 600,
							),
							'size_units' => array(
								'px',
								'vw',
							),
							'range'      => array(
								'vw' => array(
									'step' => 1,
									'min'  => 0,
								),
							),
							'selectors'  => array(
								( $selector . ' .popup' ) => 'width: {{SIZE}}{{UNIT}};',
							),
						)
					);

					$document->add_control(
						'popup_height_type',
						array(
							'label'   => __( 'Height', 'wolmart-core' ),
							'type'    => Elementor\Controls_Manager::SELECT,
							'options' => array(
								''       => esc_html__( 'Fit To Content', 'wolmart-core' ),
								'custom' => esc_html__( 'Custom', 'wolmart-core' ),
							),
						)
					);

					$document->add_responsive_control(
						'popup_height',
						array(
							'label'      => esc_html__( 'Custom Height', 'wolmart-core' ),
							'type'       => Elementor\Controls_Manager::SLIDER,
							'default'    => array(
								'size' => 380,
							),
							'size_units' => array(
								'px',
								'vh',
							),
							'range'      => array(
								'vh' => array(
									'step' => 1,
									'min'  => 0,
									'max'  => 100,
								),
							),
							'condition'  => array(
								'popup_height_type' => 'custom',
							),
							'selectors'  => array(
								( $selector . ' .popup' ) => 'height: {{SIZE}}{{UNIT}};',
							),
						)
					);

					$document->add_control(
						'popup_content_pos_heading',
						array(
							'label'     => __( 'Content Position', 'wolmart-core' ),
							'type'      => Elementor\Controls_Manager::HEADING,
							'separator' => 'before',
						)
					);

					$document->add_responsive_control(
						'popup_content_h_pos',
						array(
							'label'     => __( 'Horizontal', 'wolmart-core' ),
							'type'      => Elementor\Controls_Manager::CHOOSE,
							'toggle'    => false,
							'default'   => 'center',
							'options'   => array(
								'flex-start' => array(
									'title' => __( 'Top', 'wolmart-core' ),
									'icon'  => 'eicon-h-align-left',
								),
								'center'     => array(
									'title' => __( 'Middle', 'wolmart-core' ),
									'icon'  => 'eicon-h-align-center',
								),
								'flex-end'   => array(
									'title' => __( 'Bottom', 'wolmart-core' ),
									'icon'  => 'eicon-h-align-right',
								),
							),
							'selectors' => array(
								( $selector . ' .wolmart-popup-content' ) => 'justify-content: {{VALUE}};',
							),
						)
					);

					$document->add_responsive_control(
						'popup_content_v_pos',
						array(
							'label'     => __( 'Vertical', 'wolmart-core' ),
							'type'      => Elementor\Controls_Manager::CHOOSE,
							'toggle'    => false,
							'default'   => 'center',
							'options'   => array(
								'flex-start' => array(
									'title' => __( 'Top', 'wolmart-core' ),
									'icon'  => 'eicon-v-align-top',
								),
								'center'     => array(
									'title' => __( 'Middle', 'wolmart-core' ),
									'icon'  => 'eicon-v-align-middle',
								),
								'flex-end'   => array(
									'title' => __( 'Bottom', 'wolmart-core' ),
									'icon'  => 'eicon-v-align-bottom',
								),
							),
							'selectors' => array(
								( $selector . ' .wolmart-popup-content' ) => 'align-items: {{VALUE}};',
							),
						)
					);

					$document->add_control(
						'popup_pos_heading',
						array(
							'label'     => __( 'Position', 'wolmart-core' ),
							'type'      => Elementor\Controls_Manager::HEADING,
							'separator' => 'before',
						)
					);

					$document->add_responsive_control(
						'popup_h_pos',
						array(
							'label'     => __( 'Horizontal', 'wolmart-core' ),
							'type'      => Elementor\Controls_Manager::CHOOSE,
							'toggle'    => false,
							'default'   => 'center',
							'options'   => array(
								'flex-start' => array(
									'title' => __( 'Left', 'wolmart-core' ),
									'icon'  => 'eicon-h-align-left',
								),
								'center'     => array(
									'title' => __( 'Center', 'wolmart-core' ),
									'icon'  => 'eicon-h-align-center',
								),
								'flex-end'   => array(
									'title' => __( 'Right', 'wolmart-core' ),
									'icon'  => 'eicon-h-align-right',
								),
							),
							'selectors' => array(
								( $selector . ' .mfp-content' ) => 'justify-content: {{VALUE}};',
							),
						)
					);

					$document->add_responsive_control(
						'popup_v_pos',
						array(
							'label'     => __( 'Vertical', 'wolmart-core' ),
							'type'      => Elementor\Controls_Manager::CHOOSE,
							'toggle'    => false,
							'default'   => 'center',
							'options'   => array(
								'flex-start' => array(
									'title' => __( 'Top', 'wolmart-core' ),
									'icon'  => 'eicon-v-align-top',
								),
								'center'     => array(
									'title' => __( 'Middle', 'wolmart-core' ),
									'icon'  => 'eicon-v-align-middle',
								),
								'flex-end'   => array(
									'title' => __( 'Bottom', 'wolmart-core' ),
									'icon'  => 'eicon-v-align-bottom',
								),
							),
							'selectors' => array(
								( $selector . ' .mfp-content' ) => 'align-items: {{VALUE}};',
							),
						)
					);

					$document->add_control(
						'popup_style_heading',
						array(
							'label'     => __( 'Style', 'wolmart-core' ),
							'type'      => Elementor\Controls_Manager::HEADING,
							'separator' => 'before',
						)
					);

					$document->add_control(
						'popup_overlay_color',
						array(
							'label'     => esc_html__( 'Overlay Color', 'wolmart-core' ),
							'type'      => Elementor\Controls_Manager::COLOR,
							'selectors' => array(
								( '.mfp-bg' . $selector ) => 'background-color: {{VALUE}};',
							),
						)
					);

					$document->add_control(
						'popup_content_color',
						array(
							'label'     => esc_html__( 'Content Color', 'wolmart-core' ),
							'type'      => Elementor\Controls_Manager::COLOR,
							'selectors' => array(
								( $selector . ' .popup .wolmart-popup-content' ) => 'background-color: {{VALUE}};',
							),
						)
					);

					$document->add_group_control(
						Elementor\Group_Control_Box_Shadow::get_type(),
						array(
							'name'     => 'popup_box_shadow',
							'selector' => ( $selector . ' .popup' ),
						)
					);

					$document->add_responsive_control(
						'popup_border_radius',
						array(
							'label'      => esc_html__( 'Border Radius', 'wolmart-core' ),
							'type'       => Elementor\Controls_Manager::DIMENSIONS,
							'size_units' => array(
								'px',
								'%',
								'em',
							),
							'selectors'  => array(
								( $selector . ' .popup .wolmart-popup-content' ) => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
							),
						)
					);

					$document->add_responsive_control(
						'popup_margin',
						array(
							'label'      => esc_html__( 'Margin', 'wolmart-core' ),
							'type'       => Elementor\Controls_Manager::DIMENSIONS,
							'size_units' => array(
								'px',
								'%',
								'em',
							),
							'selectors'  => array(
								( $selector . ' .popup' ) => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
							),
						)
					);

					$document->add_control(
						'popup_animation',
						array(
							'type'      => Elementor\Controls_Manager::SELECT,
							'label'     => esc_html__( 'Popup Animation', 'wolmart-core' ),
							'options'   => wolmart_get_animations( 'in' ),
							'separator' => 'before',
							'default'   => 'default',
						)
					);

					$document->add_control(
						'popup_anim_duration',
						array(
							'type'    => Elementor\Controls_Manager::NUMBER,
							'label'   => esc_html__( 'Animation Duration (ms)', 'wolmart-core' ),
							'default' => 400,
						)
					);

				$document->end_controls_section();
			}
		}

		$document->start_controls_section(
			'wolmart_blank_styles',
			array(
				'label' => esc_html__( 'Wolmart Blank Styles', 'wolmart-core' ),
				'tab'   => Elementor\Controls_Manager::TAB_STYLE,
			)
		);

		$document->end_controls_section();

		$document->start_controls_section(
			'wolmart_custom_css_settings',
			array(
				'label' => esc_html__( 'Wolmart Custom CSS', 'wolmart-core' ),
				'tab'   => Elementor\Controls_Manager::TAB_ADVANCED,
			)
		);

			$document->add_control(
				'page_css',
				array(
					'type' => Elementor\Controls_Manager::TEXTAREA,
					'rows' => 20,
				)
			);

		$document->end_controls_section();

		if ( current_user_can( 'unfiltered_html' ) ) {

			$document->start_controls_section(
				'wolmart_custom_js_settings',
				array(
					'label' => esc_html__( 'Wolmart Custom JS', 'wolmart-core' ),
					'tab'   => Elementor\Controls_Manager::TAB_ADVANCED,
				)
			);

			$document->add_control(
				'page_js',
				array(
					'type' => Elementor\Controls_Manager::TEXTAREA,
					'rows' => 20,
				)
			);

			$document->end_controls_section();
		}
	}

	public function add_body_class( $classes ) {
		if ( wolmart_is_elementor_preview() && 'wolmart_template' == get_post_type() ) {
			$template_category = get_post_meta( get_the_ID(), 'wolmart_template_type', true );

			if ( ! $template_category ) {
				$template_category = 'block';
			}

			$classes[] = 'wolmart_' . $template_category . '_template';
		}
		return $classes;
	}

	public function add_appear_animations() {
		return wolmart_get_animations( 'appear' );
	}

	public function add_wp_widget_args( $args, $self ) {
		$args['before_widget'] = '<div class="widget ' . $self->get_widget_instance()->widget_options['classname'] . ' widget-collapsible">';
		$args['after_widget']  = '</div>';
		$args['before_title']  = '<h3 class="widget-title">';
		$args['after_title']   = '</h3>';

		return $args;
	}

	public function get_dependent_elementor_styles( $self ) {
		if ( 'file' == $self->get_meta()['status'] ) { // Re-check if it's not empty after CSS update.
			preg_match( '/post-(\d+).css/', $self->get_url(), $id );
			if ( count( $id ) == 2 ) {
				global $e_post_ids;

				wp_dequeue_style( 'elementor-post-' . $id[1] );

				wp_register_style( 'elementor-post-' . $id[1], $self->get_url(), array( 'elementor-frontend' ), null ); // phpcs:ignore WordPress.WP.EnqueuedResourceParameters.MissingVersion

				if ( ! isset( $e_post_ids ) ) {
					$e_post_ids = array();
				}
				$e_post_ids[] = $id[1];
			}
		}
	}

	public function add_global_css() {
		global $wolmart_layout;
		$wolmart_layout['used_blocks'] = wolmart_get_page_blocks();

		if ( ! empty( $wolmart_layout['used_blocks'] ) ) {
			foreach ( $wolmart_layout['used_blocks'] as $block_id => $enqueued ) {
				if ( $this->is_elementor_block( $block_id ) ) {
					if ( ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_font_icon_svg' ) || \Elementor\Plugin::$instance->preview->is_preview_mode() ) {
						wp_enqueue_style( 'elementor-icons' );
					}

					// enqueue kit css for internal style print method
					if ( isset( \Elementor\Plugin::$instance ) && 'internal' == get_option( 'elementor_css_print_method' ) ) {
						\Elementor\Plugin::$instance->kits_manager->frontend_before_enqueue_styles();
					}

					wp_enqueue_style( 'elementor-frontend' );

					if ( isset( \Elementor\Plugin::$instance ) ) {

						if ( 'internal' !== get_option( 'elementor_css_print_method' ) ) {
							$kit_id = \Elementor\Plugin::$instance->kits_manager->get_active_id();
							if ( $kit_id ) {
								wp_enqueue_style( 'elementor-post-' . $kit_id, wp_upload_dir()['baseurl'] . '/elementor/css/post-' . $kit_id . '.css' );
							}
						}

						add_action(
							'wp_print_footer_scripts',
							function() {
								try {
									$wp_scripts = wp_scripts();
									if ( ! in_array( 'elementor-frontend', $wp_scripts->queue ) ) {
										wp_enqueue_script( 'elementor-frontend' );
										$settings = \Elementor\Plugin::$instance->frontend->get_settings();
										\Elementor\Utils::print_js_config( 'elementor-frontend', 'elementorFrontendConfig', $settings );
									}
								} catch ( Exception $e ) {
									var_dump( $e );
								}
							},
							8
						);
					}

					if ( version_compare( ELEMENTOR_VERSION, '3.28', '<' ) ) {
						$scheme_css_file = Elementor\Core\Files\CSS\Global_CSS::create( 'global.css' );
						$scheme_css_file->enqueue();
					}

					break;
				}
			}
		}

		global $e_post_ids;
		if ( is_array( $e_post_ids ) ) {
			foreach ( $e_post_ids as $id ) {
				if ( get_the_ID() != $id ) {
					wp_enqueue_style( 'elementor-post-' . $id );
				}
			}
		}
	}

	public function is_elementor_block( $id ) {
		$elements_data = get_post_meta( $id, '_elementor_data', true );
		return $elements_data && get_post_meta( $id, '_elementor_edit_mode', true );
	}

	public function add_elementor_css() {
		if ( defined( 'WOLMART_PLUGINS_URI' ) ) {
			// Add Wolmart elementor style
			wp_enqueue_style( 'wolmart-elementor-style', WOLMART_PLUGINS_URI . '/elementor/elementor' . ( is_rtl() ? '-rtl' : '' ) . '.min.css', array( 'elementor-frontend' ) );

			if ( version_compare( ELEMENTOR_VERSION, '3.24', '>=' ) && function_exists( 'wolmart_get_option' ) && 'offcanvas' == wolmart_get_option( 'quickview_type' ) ) {
				$right = is_rtl() ? 'Left' : 'Right';
				wp_enqueue_style( 'e-animation-slideIn' . $right );
				wp_enqueue_style( 'e-animation-slideOut' . $right );
			}

			if ( defined( 'ELEMENTOR_PRO_VERSION' ) ) {
				wp_enqueue_style( 'wolmart-elementor-pro-style', WOLMART_PLUGINS_URI . '/elementor/elementor-pro' . ( is_rtl() ? '-rtl' : '' ) . '.min.css' );
			}
		}
	}

	public function add_elementor_page_css() {
		// Add page css
		wp_enqueue_style( 'elementor-post-' . get_the_ID() );
		// Add inline styles for heading element responsive control
		$breakpoints = wolmart_get_breakpoints();
		$left        = is_rtl() ? 'right' : 'left';
		$right       = 'left' == $left ? 'right' : 'left';

		if ( $breakpoints ) :
			ob_start();
			?>
		<style>
			<?php if ( ! empty( $breakpoints['xxl'] ) ) { ?>
			@media (max-width: <?php echo esc_html( $breakpoints['xxl'] - 1 . 'px' ); ?>) {
				div.title-xxl-center .title {
					margin-<?php echo esc_html( $left ); ?>: auto;
					margin-<?php echo esc_html( $right ); ?>: auto;
					text-align: center;
				}
				div.title-xxl-start .title {
					margin-<?php echo esc_html( $right ); ?>: auto;
					margin-<?php echo esc_html( $left ); ?>: 0;
					text-align: start;
				}
				div.title-xxl-end .title {
					margin-<?php echo esc_html( $left ); ?>: auto;
					margin-<?php echo esc_html( $right ); ?>: 0;
					text-align: end;
				}
			}
			<?php } ?>
			<?php if ( ! empty( $breakpoints['xlg'] ) ) { ?>
			@media (max-width: <?php echo esc_html( $breakpoints['xlg'] - 1 . 'px' ); ?>) {
				div.title-xlg-center .title {
					margin-<?php echo esc_html( $left ); ?>: auto;
					margin-<?php echo esc_html( $right ); ?>: auto;
					text-align: center;
				}
				div.title-xlg-start .title {
					margin-<?php echo esc_html( $right ); ?>: auto;
					margin-<?php echo esc_html( $left ); ?>: 0;
					text-align: start;
				}
				div.title-xlg-end .title {
					margin-<?php echo esc_html( $left ); ?>: auto;
					margin-<?php echo esc_html( $right ); ?>: 0;
					text-align: end;
				}
			}
			<?php } ?>
			<?php if ( ! empty( $breakpoints['xl'] ) ) { ?>
			@media (max-width: <?php echo esc_html( $breakpoints['xl'] - 1 . 'px' ); ?>) {
				div.title-xl-center .title {
					margin-<?php echo esc_html( $left ); ?>: auto;
					margin-<?php echo esc_html( $right ); ?>: auto;
					text-align: center;
				}
				div.title-xl-start .title {
					margin-<?php echo esc_html( $right ); ?>: auto;
					margin-<?php echo esc_html( $left ); ?>: 0;
					text-align: start;
				}
				div.title-xl-end .title {
					margin-<?php echo esc_html( $left ); ?>: auto;
					margin-<?php echo esc_html( $right ); ?>: 0;
					text-align: end;
				}
			}
			<?php } ?>
			<?php if ( ! empty( $breakpoints['lg'] ) ) { ?>
			@media (max-width: <?php echo esc_html( $breakpoints['lg'] - 1 . 'px' ); ?>) {
				div.title-lg-center .title {
					margin-<?php echo esc_html( $left ); ?>: auto;
					margin-<?php echo esc_html( $right ); ?>: auto;
					text-align: center;
				}
				div.title-lg-start .title {
					margin-<?php echo esc_html( $right ); ?>: auto;
					margin-<?php echo esc_html( $left ); ?>: 0;
					text-align: start;
				}
				div.title-lg-end .title {
					margin-<?php echo esc_html( $left ); ?>: auto;
					margin-<?php echo esc_html( $right ); ?>: 0;
					text-align: end;
				}
			}
			<?php } ?>
			<?php if ( ! empty( $breakpoints['md'] ) ) { ?>
			@media (max-width: <?php echo esc_html( $breakpoints['md'] - 1 . 'px' ); ?>) {
				div.title-md-center .title {
					margin-<?php echo esc_html( $left ); ?>: auto;
					margin-<?php echo esc_html( $right ); ?>: auto;
					text-align: center;
				}
				div.title-md-start .title {
					margin-<?php echo esc_html( $right ); ?>: auto;
					text-align: start;
					margin-<?php echo esc_html( $left ); ?>: 0;
				}
				div.title-md-end .title {
					margin-<?php echo esc_html( $left ); ?>: auto;
					margin-<?php echo esc_html( $right ); ?>: 0;
					text-align: end;
				}
			}
			<?php } ?>
			<?php if ( ! empty( $breakpoints['sm'] ) ) { ?>
			@media (max-width: <?php echo esc_html( $breakpoints['sm'] - 1 . 'px' ); ?>) {
				div.title-sm-center .title {
					margin-<?php echo esc_html( $left ); ?>: auto;
					margin-<?php echo esc_html( $right ); ?>: auto;
					text-align: center;
				}
				div.title-sm-start .title {
					margin-<?php echo esc_html( $right ); ?>: auto;
					text-align: start;
					margin-<?php echo esc_html( $left ); ?>: 0;
				}
				div.title-sm-end .title {
					margin-<?php echo esc_html( $left ); ?>: auto;
					margin-<?php echo esc_html( $right ); ?>: 0;
					text-align: end;
				}
			}
			<?php } ?>
		</style>
			<?php
			wolmart_filter_inline_css( ob_get_clean() );
		endif;
	}


	public function generate_block_temp_css_onsave( $post_id, $post, $use_temp = true ) {
		if ( ! isset( $_REQUEST['editor_post_id'] ) ) {
			return;
		}

		if ( 'wolmart_template' == $post->post_type ) {
			if ( 'internal' !== get_option( 'elementor_css_print_method' ) ) {
				$initial_responsive_controls_duplication_mode = Plugin::$instance->breakpoints->get_responsive_control_duplication_mode();
				Plugin::$instance->breakpoints->set_responsive_control_duplication_mode( 'on' );

				$upload        = wp_upload_dir();
				$upload_dir    = $upload['basedir'];
				$post_css_path = wp_normalize_path( $upload_dir . '/elementor/css/post-' . $post_id . ( $use_temp ? '-temp' : '' ) . '.css' );

				$css_file = new Elementor\Core\Files\CSS\Post( $post_id );
				/**
				 * Filters the style for elementor block.
				 *
				 * @since 1.0
				 */
				$block_css = $css_file->get_content();

				// Save block css as elementor post css.
				// filesystem
				global $wp_filesystem;
				// Initialize the WordPress filesystem, no more using file_put_contents function
				if ( empty( $wp_filesystem ) ) {
					require_once ABSPATH . '/wp-admin/includes/file.php';
					WP_Filesystem();
				}

				// Fix elementor's "max-width: auto" error.
				$block_css = str_replace( 'max-width:auto', 'max-width:none', $block_css );

				$wp_filesystem->put_contents( $post_css_path, $block_css, FS_CHMOD_FILE );

				Plugin::$instance->breakpoints->set_responsive_control_duplication_mode( $initial_responsive_controls_duplication_mode );
			}
		}
	}

	/**
	 * Generate blocks' css after clear cache on Elementor -> Tools
	 *
	 * @since 1.1
	 */
	public function generate_blocks_css_after_clear_cache() {
		if ( ! empty( $_REQUEST['demo'] ) ) {
			return;
		}

		$posts = get_posts(
			array(
				'post_type'   => 'wolmart_template',
				'post_status' => 'publish',
				'numberposts' => 100,
			)
		);
		if ( ! empty( $posts ) && is_array( $posts ) ) {
			$mode = get_option( 'elementor_css_print_method' );
			foreach ( $posts as $post ) {
				$this->generate_block_temp_css_onsave( $post->ID, $post, false );
				if ( 'internal' !== $mode ) {
					$css_file = new Elementor\Core\Files\CSS\Post( $post->ID );
					$css_file->update();
				}
			}
		}

		// Remove post metas related to Elementor
		delete_post_meta_by_key( 'wolmart_elementor_page_assets_saved' );
	}

	public function rename_block_temp_css_onsave( $obj, $data ) {
		$post = $obj->get_post();

		if ( 'wolmart_template' == $post->post_type ) {
			if ( 'internal' !== get_option( 'elementor_css_print_method' ) ) {
				$upload      = wp_upload_dir();
				$upload_dir  = $upload['basedir'];
				$origin_path = wp_normalize_path( $upload_dir . '/elementor/css/post-' . $post->ID . '-temp.css' );
				$dest_path   = wp_normalize_path( $upload_dir . '/elementor/css/post-' . $post->ID . '.css' );

				$css_file = new Elementor\Core\Files\CSS\Post( $post->ID );
				$css_file->update();

				// Save block css as elementor post css.
				// filesystem
				global $wp_filesystem;
				// Initialize the WordPress filesystem, no more using file_put_contents function
				if ( empty( $wp_filesystem ) ) {
					require_once ABSPATH . '/wp-admin/includes/file.php';
					WP_Filesystem();
				}

				$wp_filesystem->move( $origin_path, $dest_path, true );
			}
		}
	}

	public function add_block_css() {
		global $wolmart_layout;

		if ( ! empty( $wolmart_layout['used_blocks'] ) ) {
			$upload     = wp_upload_dir();
			$upload_dir = $upload['basedir'];
			$upload_url = $upload['baseurl'];

			foreach ( $wolmart_layout['used_blocks'] as $block_id => $enqueued ) {
				if ( 'internal' !== get_option( 'elementor_css_print_method' ) && ( ! wolmart_is_elementor_preview() || ! isset( $_REQUEST['elementor-preview'] ) || $_REQUEST['elementor-preview'] != $block_id ) && $this->is_elementor_block( $block_id ) ) { // Check if current elementor block is editing

					$block_css = get_post_meta( (int) $block_id, 'page_css', true );
					if ( $block_css ) {
						$block_css = function_exists( 'wolmart_minify_css' ) ? wolmart_minify_css( $block_css ) : $block_css;
					}

					$post_css_path = wp_normalize_path( $upload_dir . '/elementor/css/post-' . $block_id . '.css' );

					if ( file_exists( $post_css_path ) ) {
						wp_enqueue_style( 'elementor-post-' . $block_id, $upload_url . '/elementor/css/post-' . $block_id . '.css' );
						wp_add_inline_style( 'elementor-post-' . $block_id, apply_filters( 'wolmart_elementor_block_style', $block_css ) );
					} else {
						$css_file           = new Elementor\Core\Files\CSS\Post( $block_id );
						$elementor_page_css = $css_file->get_content();
						$block_css          = $elementor_page_css . apply_filters( 'wolmart_elementor_block_style', $block_css );

						// Save block css as elementor post css.
						// filesystem
						global $wp_filesystem;
						// Initialize the WordPress filesystem, no more using file_put_contents function
						if ( empty( $wp_filesystem ) ) {
							require_once ABSPATH . '/wp-admin/includes/file.php';
							WP_Filesystem();
						}
						$wp_filesystem->put_contents( $post_css_path, $elementor_page_css, FS_CHMOD_FILE );

						// Fix elementor's "max-width: auto" error.
						$block_css = str_replace( 'max-width:auto', 'max-width:none', $block_css );
						wp_add_inline_style( 'wolmart-style', $block_css );
					}

					$wolmart_layout['used_blocks'][ $block_id ]['css'] = true;
				}
			}
		}
	}

	/**
	 * Remove elementor action to update dynamic post css.
	 */
	public function remove_dynamic_css_update() {
		remove_action( 'elementor/css-file/post/enqueue', array( Elementor\Plugin::$instance->dynamic_tags, 'after_enqueue_post_css' ) );
	}

	/**
	 * Remove Modules
	 *
	 * @since 4.9.0
	 */
	public function remove_modules( $modules ) {
		$key = array_search( 'sticky', $modules );
		unset( $modules[ $key ] );
		return $modules;
	}
}

/**
 * Create instance
 */
Wolmart_Core_Elementor::get_instance();

if ( ! function_exists( 'wolmart_elementor_if_dom_optimization' ) ) :
	function wolmart_elementor_if_dom_optimization() {
		if ( ! defined( 'ELEMENTOR_VERSION' ) ) {
			return false;
		}
		if ( version_compare( ELEMENTOR_VERSION, '3.18.9', '>' ) ) {
			return true;
		} elseif ( version_compare( ELEMENTOR_VERSION, '3.1.0', '>=' ) ) {
			return \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_dom_optimization' );
		} elseif ( version_compare( ELEMENTOR_VERSION, '3.0', '>=' ) ) {
			return ( ! \Elementor\Plugin::instance()->get_legacy_mode( 'elementWrappers' ) );
		}
		return false;
	}
endif;
