<?php
class WPJ_Orders extends \stdClass {
	public function __construct() {
		add_action( 'wp_ajax_delete_review', [ $this, 'deleteReview' ] );
		add_action( 'wp_ajax_delete_review_response', [ $this, 'deleteReviewResponse' ] );
		add_action( 'wp_ajax_delete_transaction_message', [ $this, 'deleteTransactionMessage' ] );

		add_action( 'wpjobster_taketo_banktransfer_gateway', [ $this, 'initializeBankTransfer' ], 10, 2 );
		add_action( 'wpjobster_processafter_banktransfer_gateway', [ $this, 'processBankTransfer' ], 10, 1 );
	}

	public static function init() {
		$class = __CLASS__; new $class;
	}

	public function changeOrderStatusMessage() {
		$message = [];

		if ( isset( $_GET['payment_status'] ) ) {
			if ( $_GET['payment_status'] == 'success' )
				$message = [ 'type' => 'success', 'message' => __( "Order status changed to Completed", "wpjobster" ) ];

			if ( $_GET['payment_status'] == 'fail' )
				$message = [ 'type' => 'success', 'message' => __( "Order status changed to Cancelled", "wpjobster" ) ];

			if ( $_GET['payment_status'] == 'error' )
				$message = [ 'type' => 'error',   'message' => __( "Something went wrong. Please try again", "wpjobster" ) ];

		}

		return $message;
	}

	public function changeOrderStatusAction() {
		if ( ! is_demo_admin() ) {

			if ( isset( $_GET['order_action'] ) ) {
				$order_id     = WPJ_Form::get( 'order_id', '' );
				$order_action = WPJ_Form::get( 'order_action', '' );
				$payment_type = WPJ_Form::get( 'payment_type', '' );

				if ( $order_id && $payment_type && ( $order_action == 'mark_as_paid' || $order_action == 'mark_as_cancelled' ) ) {
					$class_name = 'WPJ_' . str_replace( '_', '_', ucwords( $this->getClassNameByPaymentType( $payment_type ), '_' ) );

					if ( ! class_exists( $class_name ) ) {
						include_once get_template_directory() . '/lib/gateways/class-' . str_replace( '_', '-', $this->getClassNameByPaymentType( $payment_type ) ) . '.php';
					}

					$payment_type_class = new $class_name( 'free' );

					if ( $order_action == 'mark_as_paid' ) {
						if ( $payment_type == 'subscription' ) {
							do_action( "wpjobster_" . $this->getClassNameByPaymentType( $payment_type ) . "_payment_success", $order_id, 'free', 'Paid by Admin', '', 'no' );

						} elseif ( $payment_type == 'withdrawal' ) {
							$this->acceptWithdrawalRequest( $order_id );

						} elseif ( $payment_type == 'badge' ) {
							do_action( "wpjobster_" . $this->getClassNameByPaymentType( $payment_type ) . "_payment_success", $order_id, 'free', 'Paid by Admin', 'Paid by Admin', 'no' );

							// Set badges
							$badge_order = wpj_get_badge_order( '*', ['id' => $order_id], 'row' );

							$current_badges = get_user_meta( $badge_order->user_id, 'user_badge', true );
							if ( $current_badges ) {
								array_push( $current_badges, $badge_order->badge_nr );

								$badges = array_unique( $current_badges );

							} else $badges = [$badge_order->badge_nr];

							if ( $badges ) update_user_meta( $badge_order->user_id, 'user_badge', $badges );

							// Change payment type
							wpj_update_badge_order( ['payment_type' => 1], ['id' => $order_id] );

						} else {
							do_action( "wpjobster_" . $this->getClassNameByPaymentType( $payment_type ) . "_payment_success", $order_id, 'free', 'Paid by Admin', '', 'no' );

							if ( $payment_type == $this->getClassNameByPaymentType( 'job_purchase' ) )
								do_action( 'wpj_after_job_payment_is_completed', $order_id, 'admin' );

						}
					}

					if ( $order_action == 'mark_as_cancelled' ) {
						if ( $payment_type == 'withdrawal' ) {
							$this->denyWithdrawalRequest( $order_id );

						} elseif ( $payment_type == 'badge' ) {
							do_action( "wpjobster_" . $this->getClassNameByPaymentType( $payment_type ) . "_payment_failed", $order_id, 'free', 'Cancelled by Admin', 'Cancelled by Admin', 'no' );

							// Change payment type
							wpj_update_badge_order( ['payment_type' => 1], ['id' => $order_id] );

						} else {
							do_action( "wpjobster_" . $this->getClassNameByPaymentType( $payment_type ) . "_payment_failed", $order_id, 'free', 'Cancelled by Admin', '', 'no' );

						}
					}

				}
			}
		}
	}

	public function closeJob( $order_id = '' ) {
		if ( ! $order_id && isset( $_GET['idclose'] ) ) $order_id = $_GET['idclose'];

		if ( ! is_demo_admin() ) {
			if ( $order_id ) {

				if ( WPJ_Form::get( 'payment_type' ) == 'subscription' ) { // remove subscription
					wpj_remove_subscription( $order_id );

				} else { // close job order
					if ( ! is_numeric( $order_id ) ) { echo "ERROR!"; die; }

					$row    = wpj_get_job_order_post( '*', ['o.id' => $order_id], 'row' );
					$oid    = $row->id;
					$pid    = $row->pid;
					$buyer  = $row->uid;
					$seller = $row->post_author;

					if ( ( $row->closed != 1 && $row->completed != 1 ) || ( $row->closed != 1 && $row->clearing_period == 3 ) ) {
						// Update order
						wpj_update_job_order(
							['closed' => '1', 'force_cancellation' => '1', 'payment_status' => 'cancelled', 'date_closed' => current_time( 'timestamp', 1 )],
							['id' => $order_id]
						);

						// Insert notification
						$this_notification = wpj_insert_order_notification( ['uid' => -14, 'oid' => $order_id, 'content' => __( 'Closed by Admin', 'wpjobster' )], ['%d', '%d', '%s'] );

						if ( $this_notification > 0 ) {
							// Update notification
							wpj_update_user_notifications([
								'user1'       => $seller,
								'user2'       => $buyer,
								'type'        => 'notifications',
								'number'      => +1,
								'notify_id'   => $this_notification,
								'notify_type' => 'cancel_admin',
								'order_id'    => $oid
							]);

							// Refund the buyer
							wpj_refund_payment( $row, $order_id );

							// Cancel custom offer
							if ( wpj_is_custom_offer( $pid ) ) {
								wpj_cancel_custom_offer( 'cancel_offer_admin', $oid, $seller, $buyer );
								wpj_deactivate_custom_offer( 'closed', $pid, $seller );

							} else {
								// Send emails
								wpj_notify_user_translated( 'cancel_admin', $buyer, [
									'##transaction_number##'    => wpj_camouflage_oid( $oid, $row->date_made ),
									'##transaction_page_link##' => wpj_get_order_link( $oid ),
									'##job_name##'              => $row->post_title,
									'##job_link##'              => urldecode( get_permalink( $pid ) )
								], '', 'email' );

								wpj_notify_user_translated( 'cancel_admin', $seller, [
									'##transaction_number##'    => wpj_camouflage_oid( $oid, $row->date_made ),
									'##transaction_page_link##' => wpj_get_order_link( $oid ),
									'##job_name##'              => $row->post_title,
									'##job_link##'              => urldecode( get_permalink( $pid ) )
								], '', 'email' );

							}

							// Action
							do_action( 'wpj_after_order_is_cancelled', $oid, 'job_purchase' );
						}

					} else { ?>

						<div class="notice notice-error is-dismissible">
							<p><?php _e( 'You can\'t close a completed order.', 'wpjobster' ); ?></p>
						</div>

					<?php }
				}
			}
		}
	}

	public function clearJob() {
		// Clear now
		if ( isset( $_GET['idclear'] ) ) {
			$order = wpj_get_job_order_by_id( $_GET['idclear'] );

			if ( $order->clearing_period != 1 ) {
				// Clear order
				wpj_mark_order_as_cleared( $_GET['idclear'] );

				// Update order
				wpj_update_job_order( ['clearing_period' => 1], ['id' => $_GET['idclear']] );
			}
		}

		// Block clear
		if ( isset( $_GET['idblockclear'] ) ) {
			$order = wpj_get_job_order_by_id( $_GET['idblockclear'] );

			if ( $order->clearing_period == 2 ) {
				// Update order
				wpj_update_job_order( ['clearing_period' => 3], ['id' => $_GET['idblockclear']] );
			}
		}
	}

	public function deleteReview() {
		$orderid = wpj_get_job_rating( 'orderid', ['id' => $_POST['id']], 'var' );

		wpj_delete_job_seller_rating( ['orderid' => $orderid] );

		wpj_delete_job_rating( ['id' => $_POST['id']] );

		wp_die();
	}

	public function deleteReviewResponse() {
		wpj_delete_job_seller_rating( ['id' => $_POST['id']] );

		wp_die();
	}

	public function deleteTransactionMessage() {
		$chatbox_message = wpj_get_order_notification_message( 'oid', ['id' => $_POST['id']], 'row' );

		if ( $chatbox_message ) {
			wpj_delete_order_notification( ['id' => $_POST['id']] );

			wpj_refresh_user_notifications( wpj_get_seller_id( $chatbox_message->oid ), 'notifications' );
			wpj_refresh_user_notifications( wpj_get_buyer_id( $chatbox_message->oid ), 'notifications' );
		}

		wp_die();
	}

	public function initializeBankTransfer( $payment_type, $order_details ) {

		$orderid = $order_details['id'];

		if ( $payment_type == 'job_purchase' ) $uid = $order_details['uid'];
		else $uid = $order_details['user_id'];

		$payment_row = wpj_get_payment( [ 'payment_type_id' => $order_details['id'], 'payment_type' => $payment_type ] );

		if ( $payment_type == 'topup' ) {
			$reason = 'send_bankdetails_to_topup_buyer';

		} elseif ( $payment_type == 'feature' ) {
			$reason = 'send_bankdetails_to_feature_buyer';

		} elseif ( $payment_type == 'custom_extra' ) {
			$reason = 'send_bankdetails_to_custom_extra_buyer';

		} elseif ( $payment_type == 'tips' ) {
			$reason = 'send_bankdetails_to_tips_buyer';

		} else {
			$reason = 'send_bankdetails_to_buyer';

		}

		wpj_notify_user_translated( $reason, $uid, [
			'##bank_details##'          => nl2br( wpj_get_option( 'wpjobster_bank_details' ) ),
			'##job_name##'              => ! empty( $order_details['job_title'] ) ? $order_details['job_title'] : '',
			'##transaction_page_link##' => wpj_get_order_link( $orderid )
		] );

		wpj_notify_user_translated( 'new_bank_transfer_pending', 'admin', [
			'##sender_username##'  => wpj_get_user_display_type( $uid ),
			'##payment_type##'     => $payment_type,
			'##payment_amount##'   => wpj_show_price_classic( $payment_row->final_amount_exchanged ),
			'##admin_orders_url##' => wpj_get_admin_order_link( $payment_type )
		] );

		if ( $orderid ) {
			if ( in_array( $payment_type, ['topup', 'feature', 'custom_extra', 'tips'] ) ) {
				wp_redirect( wpj_get_order_link_by_payment_type( $payment_type, $orderid ) );

			} else {
				wp_redirect( wpj_get_order_link( $orderid ) );

			}

		} else {
			wp_redirect( get_bloginfo( 'url' ) );

		}

		exit;
	}

	public function processBankTransfer( $payment_type ) {
		if ( isset( $_GET ) ) $response_array["get"] = $_GET;
		if ( isset( $_POST ) ) $response_array["post"] = $_POST;

		parse_str( file_get_contents( "php://input" ), $response_array["php_input"] );

		$order_id = WPJ_Form::get( 'order_id', 0 );

		if ( $order_id ) {
			if ( current_user_can( 'administrator' ) ) {
				$cancel_message   = "Cancelled by admin";
				$complete_message = "Completed by admin";

			} else {
				$cancel_message = "Cancelled by buyer";

			}

			if ( isset( $_GET['action'] ) && $_GET['action'] == 'complete' ) {
				if ( current_user_can( 'administrator' ) ) {
					do_action( "wpjobster_" . $payment_type . "_payment_success", $order_id, 'banktransfer', $complete_message, json_encode( $response_array ) );

				} else {
					wp_die(
						__( 'Only admins can complete bank transfers.', 'wpjobster' ),
						__( 'Unauthorized', 'wpjobster' ),
						[ 'response' => 401 ]
					);

				}

			} else {
				do_action( "wpjobster_" . $payment_type . "_payment_failed", $order_id, 'banktransfer', $cancel_message, json_encode( $response_array ) );

			}

			die();

		} else {
			wp_die(
				__( 'Missing order id.', 'wpjobster' ),
				__( 'Bad request', 'wpjobster' ),
				[ 'response' => 400 ]
			);

		}
	}

	public function processPayRequest() {
		if ( ! empty( $_POST['processPayRequest'] ) ) {
			$ids      = isset( $_POST['requests'] ) ? $_POST['requests'] : '';
			$currency = wpj_get_site_default_currency();

			$paypal_appid     = wpj_get_option( 'wpjobster_theme_appid' );
			$paypal_appsecret = wpj_get_option( 'wpjobster_theme_appsecret' );

			if ( $paypal_appid && $paypal_appsecret ) {
				$emails = [];
				$mounts = [];
				$paypal_payout_requests_info = [];

				if ( $ids ) {
					foreach ( $ids as $id ) {
						$row = wpj_get_withdrawal_order( '*', ['id' => $id], 'row' );

						if ( $row->done == 0 ) {
							$paypal_email = wpj_user( $row->uid, 'paypal_email' );
							$emails[] = $paypal_email;
							$mounts[] = $row->amount;
							$paypal_payout_requests_info[$id]['email']    = $paypal_email;
							$paypal_payout_requests_info[$id]['amount']   = $row->amount;
							$paypal_payout_requests_info[$id]['userid']   = $row->uid;
							$paypal_payout_requests_info[$id]['uniqueid'] = $id;

						}
					}
				}

				if ( ! empty( $emails ) ) {
					include_once get_template_directory() . '/lib/plugins/wpjobster-paypal/lib/paypal-withdrawal/vendor/paypal/rest-api-sdk-php/wpj/payment.php';

				} else {
					echo '
						<div class="notice notice-error is-dismissible">
							<p>' . __( 'No Paypal order selected!', 'wpjobster' ) . '</p>
						</div>
					';

				}
			} else {
				echo '
					<div class="notice notice-error is-dismissible">
						<p>' . __( 'PayPal Client ID and PayPal Secret are blank!', 'wpjobster' ) . '</p>
					</div>
				';

			}
		}
	}

	public function acceptWithdrawalRequest( $order_id = '' ) {
		if ( is_numeric( $order_id ) ) {
			$row = wpj_get_withdrawal_order( '*', ['id' => $order_id], 'row' );

			if ( $row->done == 0 || $row->done == -2 ) {
				wpj_update_withdrawal_order( ['done' => 1, 'datedone' => current_time( 'timestamp', 1 )], ['id' => $order_id] );

				$details = wpj_translate_string( $row->methods ) . ': <br>' . $row->payeremail;
				$reason  = __( 'Withdrawal to', 'wpjobster' ) . '<br>' . $details;

				wpj_add_history_log( [ 'tp' => '0', 'reason' => $reason, 'amount' => $row->amount, 'uid' => $row->uid, 'rid' => 9, 'details' => $details ] );

				wpj_notify_user_translated( 'withdraw_compl', $row->uid, [ '##amount_withdrawn##' => wpj_show_price_classic( $row->amount ), '##withdraw_method##' => wpj_translate_string( $row->methods ) ], '', 'email' );
			}
		}
	}

	public function denyWithdrawalRequest( $order_id = '' ) {
		if ( is_numeric( $order_id ) ) {
			$row = wpj_get_withdrawal_order( '*', ['id' => $order_id], 'row' );

			if ( $row->done == 0 ) {
				wpj_update_withdrawal_order(
					['done' => '-1', 'rejected_on' => current_time( 'timestamp', 1 ), 'rejected' => 1, 'datedone' => current_time( 'timestamp', 1 )],
					['id' => $order_id]
				);

				wpj_notify_user_translated( 'withdraw_decl', $row->uid, [ '##amount_withdrawn##' => wpj_show_price_classic( $row->amount ), '##withdraw_method##' => wpj_translate_string( $row->methods ) ], '', 'email' );

				$ucr = wpj_get_user_credit( $row->uid );
				wpj_update_user_credit( $row->uid, $ucr + $row->amount );
			}
		}
	}

	public function getOrderFilters( $payment_type = '' ) {
		$filters = [
			'all'       => __( 'all', 'wpjobster' ),
			'active'    => __( 'active', 'wpjobster' ),
			'delivered' => __( 'delivered', 'wpjobster' ),
			'completed' => __( 'completed', 'wpjobster' ),
			'closed'    => __( 'closed', 'wpjobster' ),
			'pending'   => __( 'pending', 'wpjobster' ),
			'cancelled' => __( 'cancelled', 'wpjobster' ),
			'expired'   => __( 'expired', 'wpjobster' )
		];

		if ( $payment_type != 'job-purchase' && $payment_type != 'subscription' ) {
			unset( $filters['active'], $filters['delivered'], $filters['closed'], $filters['expired'] );

		} elseif ( $payment_type == 'subscription' ) {
			unset( $filters['delivered'], $filters['completed'], $filters['closed'], $filters['expired'] );

		}

		return $filters;
	}

	public function getClassNameByPaymentType( $payment_type = '' ) {
		if ( $payment_type == 'job-purchase' ) {
			return 'job_purchase';

		} elseif ( $payment_type == 'withdrawal' ) {
			return 'job_purchase';

		} elseif ( $payment_type == 'custom-extra' ) {
			return 'custom_extra';

		} else {
			return $payment_type;

		}
	}

	public function getActiveTabInfo( $payment_type = '' ) {
		$tab_name = '';
		$inp_name = '';
		$pgn_name = '';
		$btn_name = '';

		if ( $payment_type == 'job-purchase' ) {
			$tab_name = 'job-purchase';
			$pgn_name = 'pj_job_purchase';
			$inp_name = 'search_user_job_purchase';
			$btn_name = 'wpjobster_save_job_purchase';

		} elseif ( $payment_type == 'topup' ) {
			$tab_name = 'topup';
			$pgn_name = 'pj_topup';
			$inp_name = 'search_user_topup';
			$btn_name = 'wpjobster_save_topup';

		} elseif ( $payment_type == 'feature' ) {
			$tab_name = 'feature';
			$pgn_name = 'pj_feature';
			$inp_name = 'search_user_feature';
			$btn_name = 'wpjobster_save_feature';

		} elseif ( $payment_type == 'withdrawal' ) {
			$tab_name = 'withdrawal';
			$pgn_name = 'pj_withdrawal';
			$inp_name = 'search_user_withdrawal';
			$btn_name = 'wpjobster_save_withdrawal';

		} elseif ( $payment_type == 'custom-extra' ) {
			$tab_name = 'custom-extra';
			$pgn_name = 'pj_custom_extra';
			$inp_name = 'search_user_custom_extra';
			$btn_name = 'wpjobster_save_custom_extra';

		} elseif ( $payment_type == 'tips' ) {
			$tab_name = 'tips';
			$pgn_name = 'pj_tips';
			$inp_name = 'search_user_tips';
			$btn_name = 'wpjobster_save_tips';

		} elseif ( $payment_type == 'subscription' ) {
			$tab_name = 'subscription';
			$pgn_name = 'pj_subscription';
			$inp_name = 'search_user_subscription';
			$btn_name = 'wpjobster_save_subscription';

		} elseif ( $payment_type == 'badge' ) {
			$tab_name = 'badge';
			$pgn_name = 'pj_badge';
			$inp_name = 'search_user_badge';
			$btn_name = 'wpjobster_save_badge';

		}

		return [
			'at'  => $tab_name,
			'inp' => $inp_name,
			'pg'  => $pgn_name,
			'sb'  => $btn_name
		];
	}

	public function getRowValues( $payment_type = '', $row = [] ) {
		$values = [];

		if ( $payment_type == 'job-purchase' || $payment_type == 'custom-extra' || $payment_type == 'tips' ) {
			if ( $payment_type == 'job-purchase' ) {
				$pid               = $row->pid;
				$date_made         = $row->date_made;
				$deciphered_amount = explode( '|', $row->payedamount );
				$currency          = $deciphered_amount[0];
				$total_amount      = $deciphered_amount[1] + wpj_get_custom_extras_amount( $row, 'amount' );
				$amount            = $currency . '|' . apply_filters( 'wpj_admin_order_list_amount_filter', $total_amount, $row, 'job_purchase' );
				$gateway           = $row->payment_gateway;
				$order             = wpj_get_job_order_by_id( $row->id );
				$user_id           = $row->uid;

				if ( $row->done_seller == 1 && $row->done_buyer == 0 ) {
					$payment_status = __( 'Delivered', 'wpjobster' );

				} elseif ( $row->done_buyer == 1 && $row->done_seller == 1 ) {
					$payment_status = __( 'Completed', 'wpjobster' );

				} elseif ( $row->done_seller == 0 && $row->done_buyer == 0 && $row->payment_status == 'completed' ) {
					if ( wpj_get_option( 'wpjobster_seller_order_rejection_enable' ) == 'yes' && $row->seller_confirmation == '0' ) {
						$payment_status = __( 'Waiting for the seller to accept', 'wpjobster' );

					} else {
						$payment_status = __( 'Waiting for the seller to deliver', 'wpjobster' );

					}

				} elseif ( $row->payment_status == 'pending' ) {
					$payment_status = __( 'Pending', 'wpjobster' );

				} elseif ( $row->payment_status == 'cancelled' ) {
					$payment_status = __( 'Cancelled', 'wpjobster' );

				} else {
					$payment_status = $row->payment_status;

				}

			} else {
				$order = wpj_get_job_order_by_id( $row->order_id );

				if ( $payment_type == 'custom-extra' ) {
					$custom_extras = wpj_json_decode( wpj_stripslashes( $order->custom_extras ) );
					$custom_extra  = $custom_extras[$row->custom_extra_id];

					$ce_payedamount = $custom_extra->price;
					if ( isset( $custom_extra->processing_fees ) ) $ce_payedamount += $custom_extra->processing_fees;
					if ( isset( $custom_extra->tax ) ) $ce_payedamount += $custom_extra->tax;
				}

				$pid       = $order->pid;
				$date_made = $row->added_on;
				$amount    = $row->currency . '|' . ( $payment_type == 'custom-extra' ? $ce_payedamount : $row->payable_amount );
				$gateway   = $row->payment_gateway_name;
				$user_id   = $row->user_id;

			}

			$post = get_post( $pid );

			$additional_price_info = apply_filters( 'wpj_admin_order_list_price_info_filter', false, $order, $amount );

			$attachments = explode( ',', get_option( 'bank_transfer_job_proof_attachments_' . $row->id ) );

			$values = [
				$row->id . ' (#' . wpj_camouflage_oid( $row->id, $date_made ) . ')',
				'<a href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&order_id=' . $row->id . '&order_view=order_single_page&payment_type=' . $payment_type . '&active_tab=' . $payment_type . '">' . __( 'View Order Details', 'wpjobster' ) . '</a>',
				'<a href="' . get_permalink( $pid ) . '">' . ( isset( $post->post_title ) ? $post->post_title : '' ) . '</a>',
				wpj_deciphere_amount_classic( $amount ) . $additional_price_info,
				wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $date_made ),
				isset( $user_id ) ? wpj_get_user_display_type( $user_id ) : sprintf( __( 'Deleted User (ID: %d)', 'wpjobster' ), $user_id ),
				isset( $post->post_author ) ? wpj_get_user_display_type( $post->post_author ) : __( 'Deleted User', 'wpjobster' ),
				$gateway,
				$row->payment_status . ( $row->payment_status == 'pending' && $payment_type == 'job-purchase' && $attachments ? '<br><a href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&order_id=' . $row->id . '&order_view=order_single_page&payment_type=' . $payment_type . '&active_tab=' . $payment_type . '">' . __( '(view proof of payment)', 'wpjobster' ) . '</a>' : '' ),
			];

			if ( $payment_type == 'job-purchase' ) { $values[] = $payment_status; }

		} elseif ( $payment_type == 'topup' || $payment_type == 'feature' ) {
			$amount = $payment_type == 'topup' ? wpj_get_exchanged_value( $row->package_cost_without_tax, $row->currency, wpj_get_site_default_currency() ) : $row->featured_amount;

			$values = [
				$row->id . ' (#' . wpj_camouflage_oid( $row->id, $row->added_on ) . ')',
				'<a href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&order_id=' . $row->id . '&order_view=order_single_page&payment_type=' . $payment_type . '&active_tab=' . $payment_type . '">' . __( 'View Order Details', 'wpjobster' ) . '</a>',
				wpj_show_price_classic( $amount ),
				wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->added_on ),
				wpj_get_user_display_type( $row->user_id ),
				$row->payment_gateway_name,
				$row->payment_status
			];

		} elseif ( $payment_type == 'withdrawal' ) {
			if ( $row->done == 1 ) {
				$status = __( 'Completed', 'wpjobster' );
			} elseif ( $row->done == -1 && $row->rejected == 1 ) {
				$status = __( 'Rejected', 'wpjobster' );
			} elseif ( $row->done == -2 ) {
				$status = __( 'Processing', 'wpjobster' );
			} else {
				$status = __( 'Pending', 'wpjobster' );
			}

			$values = [
				$row->id . ' (#' . wpj_camouflage_oid( $row->id, $row->datemade ) . ')',
				'<a href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&order_id=' . $row->id . '&order_view=order_single_page&payment_type=' . $payment_type . '&active_tab=' . $payment_type . '">' . __( 'View Order Details', 'wpjobster' ) . '</a>',
				wpj_deciphere_amount_classic( $row->payedamount ),
				wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->datemade ),
				wpj_get_user_display_type( $row->uid ),
				$row->payeremail,
				$row->methods,
				$status
			];

		} elseif ( $payment_type == 'subscription' ) {
			$level_name = wpj_get_option( 'wpjobster_subscription_name_level' . preg_replace( "/[^0-9\.]/", '', $row->level ) );
			$level_name = $level_name ? ' (' . $level_name . ')' : '';

			$values = [
				$row->id . ' (#' . wpj_camouflage_oid( $row->id, $row->addon_date ) . ')',
				'<a href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&order_id=' . $row->id . '&order_view=order_single_page&payment_type=' . $payment_type . '&active_tab=' . $payment_type . '">' . __( 'View Order Details', 'wpjobster' ) . '</a>',
				wpj_deciphere_amount_classic( $row->mc_currency . '|' . $row->payable_amount ),
				wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->addon_date ),
				wpj_get_user_display_type( $row->user_id ),
				$row->level . $level_name,
				$row->plan,
				$row->payment_gateway_name,
				strtolower( $row->payment_status )
			];

		} elseif ( $payment_type == 'badge' ) {
			$values = [
				$row->id . ' (#' . wpj_camouflage_oid( $row->id, $row->date_made ) . ')',
				'<a href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&order_id=' . $row->id . '&order_view=order_single_page&payment_type=' . $payment_type . '&active_tab=' . $payment_type . '">' . __( 'View Order Details', 'wpjobster' ) . '</a>',
				wpj_get_badge_info( 'wpj_badge_text', $row->badge_nr ),
				wpj_show_price_classic( $row->price ),
				wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->date_made ),
				wpj_get_user_display_type( $row->user_id ),
				$row->payment_gateway,
				$row->payment_status
			];

		}

		return $values;
	}

	public function getRowQuery( $payment_type = '', $filter = '', $rows_per_page = 10, $pageno = 1, $limited = false ) {
		global $wpdb;

		$page_info = $this->getActiveTabInfo( $payment_type );

		$user = WPJ_Form::get( $page_info['inp'] );
		$uid  = ctype_digit( $user ) || is_int( $user ) ? $user : '';

		if ( ! $uid ) {
			$uid = wpj_get_user( 'ID', ['user_login' => $user], 'var' );
		}

		$oid = WPJ_Form::get( $page_info['inp'] . '_id' );

		if ( $oid && ! is_numeric( $oid ) ) {
			$oid = 0;
		}

		$query = '';

		if ( $payment_type == 'job-purchase' ) {
			$query = "SELECT DISTINCT * FROM {$wpdb->prefix}job_orders orders, {$wpdb->prefix}posts posts WHERE posts.ID = orders.pid";

			if ( $filter == 'active' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_orders orders, {$wpdb->prefix}posts posts
					WHERE posts.ID = orders.pid
						AND orders.done_seller = '0'
						AND orders.done_buyer = '0'
						AND orders.date_finished = '0'
						AND orders.closed = '0'
						AND orders.payment_status != 'cancelled'
						AND orders.payment_status != 'pending'
				";

			} elseif( $filter == 'delivered' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_orders orders, {$wpdb->prefix}posts posts
					WHERE posts.ID = orders.pid
					AND orders.done_seller = '1'
					AND orders.done_buyer = '0'
					AND orders.closed = '0'
				";

			} elseif( $filter == 'completed' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_orders orders, {$wpdb->prefix}posts posts
					WHERE posts.ID = orders.pid
						AND orders.done_seller = '1'
						AND orders.done_buyer = '1'
						AND orders.closed = '0'
				";

			} elseif( $filter == 'closed' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_orders orders, {$wpdb->prefix}posts posts
					WHERE posts.ID = orders.pid
						AND orders.closed = '1'
				";

			} elseif( $filter == 'pending' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_orders orders, {$wpdb->prefix}posts posts
					WHERE posts.ID = orders.pid
						AND orders.payment_status = 'pending'
				";

			} elseif( $filter == 'cancelled' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_orders orders, {$wpdb->prefix}posts posts
					WHERE posts.ID = orders.pid
						AND orders.payment_status = 'cancelled'
				";

			} elseif( $filter == 'expired' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_orders orders, {$wpdb->prefix}posts posts
					WHERE posts.ID = orders.pid
						AND orders.payment_status = 'expired'
				";

			}

			if ( is_numeric( $uid ) ) $query .= " AND ( orders.uid = {$uid} OR posts.post_author = {$uid} )";
			if ( is_numeric( $oid ) ) $query .= " AND orders.id = {$oid}";

		} elseif ( $payment_type == 'topup' ) {
			$query = "SELECT DISTINCT * FROM {$wpdb->prefix}job_topup_orders orders WHERE 1 = 1";

			if ( $filter == 'completed' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_topup_orders orders
					WHERE orders.payment_status = 'completed'
				";

			} elseif( $filter == 'pending' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_topup_orders orders
					WHERE orders.payment_status = 'pending'
				";

			} elseif( $filter == 'cancelled' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_topup_orders orders
					WHERE orders.payment_status = 'cancelled'
				";

			}

			if ( is_numeric( $uid ) ) $query .= " AND orders.user_id = {$uid}";
			if ( is_numeric( $oid ) ) $query .= " AND orders.id = {$oid}";

		} elseif ( $payment_type == 'feature' ) {
			$query = "SELECT DISTINCT * FROM {$wpdb->prefix}job_featured_orders orders WHERE 1 = 1";

			if ( $filter == 'completed' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_featured_orders orders
					WHERE orders.payment_status = 'completed'
				";

			} elseif( $filter == 'pending' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_featured_orders orders
					WHERE orders.payment_status = 'pending'
				";

			} elseif( $filter == 'cancelled' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_featured_orders orders
					WHERE orders.payment_status = 'cancelled'
				";

			}

			if ( is_numeric( $uid ) ) $query .= " AND orders.user_id = {$uid}";
			if ( is_numeric( $oid ) ) $query .= " AND orders.id = {$oid}";

		} elseif ( $payment_type == 'withdrawal' ) {
			$query = "SELECT DISTINCT * FROM {$wpdb->prefix}job_withdraw orders WHERE ( activation_key is NULL or activation_key = '' )";

			if ( $filter == 'completed' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_withdraw orders
					WHERE orders.done = '1'
				";

			} elseif( $filter == 'pending' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_withdraw orders
					WHERE ( orders.done = '0' OR orders.done = '-2' )
					AND ( activation_key is NULL or activation_key = '' )
				";

			} elseif( $filter == 'cancelled' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_withdraw orders
					WHERE orders.rejected = '1'
				";

			}

			if ( is_numeric( $uid ) ) $query .= " AND orders.uid = {$uid}";
			if ( is_numeric( $oid ) ) $query .= " AND orders.id = {$oid}";

		} elseif ( $payment_type == 'custom-extra' ) {
			$query = "SELECT DISTINCT * FROM {$wpdb->prefix}job_custom_extra_orders orders WHERE 1 = 1";

			if ( $filter == 'completed' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_custom_extra_orders orders
					WHERE orders.payment_status = 'completed'
				";

			} elseif( $filter == 'pending' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_custom_extra_orders orders
					WHERE orders.payment_status = 'pending'
				";

			} elseif( $filter == 'cancelled' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_custom_extra_orders orders
					WHERE orders.payment_status = 'cancelled'
				";

			}

			if ( is_numeric( $uid ) ) $query .= " AND orders.user_id = {$uid}";
			if ( is_numeric( $oid ) ) $query .= " AND orders.id = {$oid}";

		} elseif ( $payment_type == 'tips' ) {
			$query = "SELECT DISTINCT * FROM {$wpdb->prefix}job_tips_orders orders WHERE 1 = 1";

			if ( $filter == 'completed' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_tips_orders orders
					WHERE orders.payment_status = 'completed'
				";

			} elseif( $filter == 'pending' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_tips_orders orders
					WHERE orders.payment_status = 'pending'
				";

			} elseif( $filter == 'cancelled' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_tips_orders orders
					WHERE orders.payment_status = 'cancelled'
				";

			}

			if ( is_numeric( $uid ) ) $query .= " AND orders.user_id = {$uid}";
			if ( is_numeric( $oid ) ) $query .= " AND orders.id = {$oid}";

		} elseif ( $payment_type == 'subscription' ) {
			$query = "SELECT DISTINCT * FROM {$wpdb->prefix}job_subscription_orders orders WHERE 1 = 1";

			if ( $filter == 'active' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_subscription_orders orders
					WHERE orders.subscription_status = 'active'
				";

			} elseif( $filter == 'pending' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_subscription_orders orders
					WHERE orders.payment_status = 'pending'
				";

			} elseif( $filter == 'cancelled' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_subscription_orders orders
					WHERE orders.payment_status = 'cancelled'
				";

			}

			if ( is_numeric( $uid ) ) $query .= " AND orders.user_id = {$uid}";
			if ( is_numeric( $oid ) ) $query .= " AND orders.id = {$oid}";

		} elseif ( $payment_type == 'badge' ) {
			$query = "SELECT DISTINCT * FROM {$wpdb->prefix}job_badge_orders orders WHERE 1 = 1";

			if ( $filter == 'completed' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_badge_orders orders
					WHERE orders.payment_status = 'completed'
				";

			} elseif( $filter == 'pending' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_badge_orders orders
					WHERE orders.payment_status = 'pending'
				";

			} elseif( $filter == 'cancelled' ) {
				$query = "
					SELECT DISTINCT *
					FROM {$wpdb->prefix}job_badge_orders orders
					WHERE orders.payment_status = 'cancelled'
				";

			}

			if ( is_numeric( $uid ) ) $query .= " AND orders.user_id = {$uid}";
			if ( is_numeric( $oid ) ) $query .= " AND orders.id = {$oid}";

		}

		$query .= " ORDER BY orders.id DESC";

		if ( $limited ) {
			$query .= " LIMIT " . ( $pageno - 1 ) * $rows_per_page . ',' . $rows_per_page;
		}

		return $query;
	}

	public function getCountOfOrders( $payment_type = '', $filter = '' ) {
		global $wpdb;

		$query   = $this->getRowQuery( $payment_type, $filter );
		$results = $wpdb->get_results( $query );
		$count   = $results ? count( $results ) : 0;

		return $count;
	}

	public function getOrderRow( $payment_type = '', $filter = '' ) {
		$r = [ 'all' => '', 'limited' => '', 'lastpage' => '', 'pageno' => '' ];

		global $wpdb;

		$rows_per_page = 10;

		$page_info = $this->getActiveTabInfo( $payment_type );
		$pg = $page_info['pg'];
		if ( isset( $_GET[$pg] ) ) $pageno = $_GET[$pg];
		else $pageno = 1;

		$s1 = $this->getRowQuery( $payment_type, $filter, $rows_per_page, $pageno, false );
		$s  = $this->getRowQuery( $payment_type, $filter, $rows_per_page, $pageno, true );

		$nr       = count( $wpdb->get_results( $s1 ) );
		$lastpage = ceil( $nr / $rows_per_page );

		$r = [
			'all'      => $wpdb->get_results( $s ),
			'limited'  => $wpdb->get_results( $s1 ),
			'lastpage' => $lastpage,
			'pageno'   => $pageno
		];

		return $r;
	}

	public function getActionText( $row = [], $payment_type = '', $filter = '' ) {
		$filter = $filter ? '&status=' . $filter : '';

		$mark_paid = '
			<a title="' . __( 'Mark payment completed', 'wpjobster' ) . '" href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&order_id=' . $row->id . '&order_action=mark_as_paid&payment_type=' . $payment_type . '&active_tab=' . $payment_type . $filter . '" class="awesome tltp_cls' . ( isset( $row->methods ) ? ' ' . sanitize_title( $row->methods ) : '' ) . '">
				<span class=""><i class="check icon"></i></span>
			</a>
		';

		$mark_paid_manual = '
			<a title="' . __( 'Mark payment manual as completed', 'wpjobster' ) . '" href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&order_id=' . $row->id . '&order_action=mark_as_paid&payment_type=' . $payment_type . '&active_tab=' . $payment_type . $filter . '" class="awesome tltp_cls' . ( isset( $row->methods ) ? ' ' . sanitize_title( $row->methods ) : '' ) . '">
				<span class=""><i class="check icon"></i></span>
			</a>
		';

		$mark_paid_automatical = '
			<a title="' . __( 'Mark payment automatically as completed', 'wpjobster' ) . '" href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&order_id=' . $row->id . '&order_action=mark_as_paid&payment_type=' . $payment_type . '&active_tab=' . $payment_type . $filter . '" class="awesome tltp_cls mark-order-completed' . ( isset( $row->methods ) ? ' ' . sanitize_title( $row->methods ) : '' ) . '">
				<span class=""><i class="double check icon"></i></span>
			</a>
		';

		$already_paid = '
			<span title="' . __( 'Payment already completed', 'wpjobster' ) . '" class="tltp_cls" style="color: #ddd;"><i class="check icon"></i></span>
		';

		$order_cancelled = '
			<span title="' . __( 'Payment already cancelled', 'wpjobster' ) . '" class="tltp_cls" style="color: #ddd;"><i class="check icon"></i></span>
		';

		$mark_cancel = '
			<a title="' . __( 'Mark payment cancelled', 'wpjobster' ) . '" href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&order_id=' . $row->id . '&order_action=mark_as_cancelled&payment_type=' . $payment_type . '&active_tab=' . $payment_type . $filter . '" class="awesome tltp_cls">
				<span class=""><i class="minus icon"></i></span>
			</a>';

		$mark_cancel_disabled = '
			<span title="' . __( 'You can\'t cancel this payment', 'wpjobster' ) . '" class="tltp_cls" style="color: #ddd;"><i class="minus icon"></i></span>
		';

		$close_job = '
			<a href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&idclose=' . $row->id . '&payment_type=' . $payment_type . '&active_tab=' . $payment_type . $filter . '" class="tltp_cls awesome" title="' . __( 'Force close order', 'wpjobster' ) . '">
				<span class=""><i class="remove icon"></i></span>
			</a>
		';

		$close_job_disabled = '
			<span title="' . __( 'You can\'t close this order', 'wpjobster' ) . '" class="tltp_cls" style="color: #ddd;"><i class="remove icon"></i></span>
		';

		$mark_cleared = '
			<a href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&idclear=' . $row->id . '&payment_type=' . $payment_type . '&active_tab=' . $payment_type . $filter . '" class="tltp_cls awesome" title="' . __( 'Clear now', 'wpjobster' ) . '">
				<span class=""><i class="clock icon"></i></span>
			</a>
		';

		$already_cleared = '
			<span title="' . __( 'Already cleared', 'wpjobster' ) . '" class="tltp_cls" style="color: #ddd;"><i class="clock icon"></i></span>
		';

		$block_clearing = '
			<a href="' . get_bloginfo( 'url' ) . '/wp-admin/admin.php?page=jobster-orders&idblockclear=' . $row->id . '&payment_type=' . $payment_type . '&active_tab=' . $payment_type . $filter . '" class="tltp_cls awesome" title="' . __( 'Block the clearing', 'wpjobster' ) . '">
				<span class=""><i class="ban icon"></i></span>
			</a>
		';

		$already_blocked = '
			<span title="' . __( 'Clearing already blocked. You can clear the order anytime or you can close the order to refund the payment.', 'wpjobster' ) . '" class="tltp_cls" style="color: #f1990d;"><i class="ban icon"></i></span>
		';

		$block_disabled = '
			<span title="' . __( 'You can\'t block the clearing because the order is already cleared', 'wpjobster' ) . '" class="tltp_cls" style="color: #ff0000;"><i class="ban icon"></i></span>
		';

		$block_order_complete = '
			<span title="' . __( 'The order needs to be completed to be cleared', 'wpjobster' ) . '" class="tltp_cls" style="color: #ddd;"><i class="clock icon"></i></span>
		';

		$clear_order_complete = '
			<span title="' . __( 'The order needs to be completed to block the clearing or the order is already cleared', 'wpjobster' ) . '" class="tltp_cls" style="color: #ddd;"><i class="ban icon"></i></span>
		';

		return [
			'paid'               => $mark_paid,
			'paid_manual'        => $mark_paid_manual, // manual withdrawal
			'paid_automatical'   => $mark_paid_automatical, // automatic withdrawal
			'already_paid'       => $already_paid,
			'order_cancelled'    => $order_cancelled,
			'cancel'             => $mark_cancel,
			'cancel_disabled'    => $mark_cancel_disabled,
			'close_job'          => $close_job,
			'close_job_disabled' => $close_job_disabled,
			'cleared'            => $mark_cleared,
			'already_cleared'    => $already_cleared,
			'block_clearing'     => $block_clearing,
			'already_blocked'    => $already_blocked,
			'block_disabled'     => $block_disabled,
			'block_ord_cmp'      => $block_order_complete,
			'clear_ord_cmp'      => $clear_order_complete
		];
	}

	public function showOrderActions( $payment_type = '', $filter = '', $row = [] ) {
		$action_text = $this->getActionText( $row, $payment_type, $filter );

		if ( $payment_type == 'job-purchase' ) { /* JOB ORDERS */
			if ( $row->payment_status == 'pending' ) {
				echo $action_text['paid']; // mark paid
				echo $action_text['cancel']; // cancel
				echo $action_text['block_ord_cmp']; // block disabled
				echo $action_text['clear_ord_cmp']; // clear disabled
				echo $action_text['close_job']; // close job

			} elseif ( $row->payment_status == 'failed' || $row->payment_status == 'cancelled' || $row->payment_status == 'expired' ) {
				echo $action_text['already_paid']; // already paid
				echo $action_text['cancel_disabled']; // cancel disabled
				echo $action_text['block_ord_cmp']; // block disabled
				echo $action_text['clear_ord_cmp']; // clear disabled
				echo $action_text['close_job_disabled']; // cancel disabled

			} elseif ( $row->clearing_period == 3 ) {
				echo $action_text['already_paid']; // already paid
				echo $action_text['cancel_disabled']; // cancel disabled
				echo $action_text['cleared']; // mark cleared
				echo $action_text['already_blocked']; // already blocked
				echo $action_text['close_job']; // close job

			} else {
				echo $action_text['already_paid']; // already paid
				echo $action_text['cancel_disabled']; // cancel disabled

				if ( $row->done_seller == 1 && $row->done_buyer == 1 && $row->closed != 1 ) {
					if ( $row->clearing_period != 1 ) {
						echo $action_text['cleared']; // mark cleared
					} else {
						echo $action_text['already_cleared']; // already cleared
					}

					if ( $row->clearing_period == 2 ) {
						echo $action_text['block_clearing']; // block the clearing
					} elseif ( $row->clearing_period == 3 ) {
						echo $action_text['already_blocked']; // already blocked
					} else {
						echo $action_text['block_disabled']; // blocked disabled
					}
				} else {
					echo $action_text['block_ord_cmp']; // block disabled
					echo $action_text['clear_ord_cmp']; // clear disabled
				}

				if ( $row->closed != 1 && $row->completed != 1 )
					echo $action_text['close_job']; // close job
				else
					echo $action_text['close_job_disabled']; // cancel disabled

			}

		} elseif ( $payment_type == 'withdrawal' ) { /* WITHDRAWAL ORDERS */
			if ( $filter != 'completed' ) { // if order is not completed or closed
				if ( $row->done == '0' ) {
					$show_manual_btn = apply_filters( 'wpj_show_hide_manual_withdrawal_button_filter', true, $row );
					if ( $show_manual_btn ) {
						echo $action_text['paid_manual']; // mark paid manual
					}

					$configuration_id     = apply_filters( 'wpj_withdrawal_id_filter', wpj_get_option( 'wpjobster_theme_appid' ) );
					$configuration_secret = apply_filters( 'wpj_withdrawal_password_filter', wpj_get_option( 'wpjobster_theme_appsecret' ) );

					$show_automatic_btn = apply_filters( 'wpj_show_hide_automatic_withdrawal_button_filter', ( $configuration_id && $configuration_secret && strtolower( $row->methods ) == 'paypal' ? true : false ), $row );

					if ( $show_automatic_btn && $filter == 'pending' ) {
						echo $action_text['paid_automatical']; // mark paid automatical
					}
					echo $action_text['cancel']; // cancel

				} elseif ( $row->rejected == '1' ) {
					echo $action_text['order_cancelled']; // order cancelled
					echo $action_text['cancel_disabled']; // cancel disabled

				} else {
					echo $action_text['already_paid']; // already paid
					echo $action_text['cancel_disabled']; // cancel disabled

				}

			} else {
				echo $action_text['already_paid']; // already paid
				echo $action_text['cancel_disabled']; // cancel disabled

			}

		} elseif ( $payment_type == 'subscription' ) { /* SUBSCRIPTION ORDERS */
			if ( $row->payment_status != 'failed' && $row->payment_status != 'cancelled' ) {
				if ( $row->payment_status == 'pending' ) {
					echo $action_text['paid']; // mark paid
					echo $action_text['cancel']; // cancel
					echo $action_text['close_job']; // remove subscription
				} else {
					echo $action_text['already_paid']; // already paid
					echo $action_text['cancel_disabled']; // cancel
					echo $action_text['close_job']; // remove subscription
				}

			} else {
				echo $action_text['already_paid']; // already paid
				echo $action_text['cancel_disabled']; // cancel
				echo $action_text['close_job_disabled']; // cancel disabled
			}

		} else {
			if ( $filter != 'completed' ) { // if order is not completed or closed
				if ( $row->payment_status == 'pending' ) {
					echo $action_text['paid']; // mark paid
					echo $action_text['cancel']; // cancel

				} elseif ( $row->payment_status == 'failed' || $row->payment_status == 'cancelled' ) {
					echo $action_text['order_cancelled']; // order cancelled
					echo $action_text['cancel_disabled']; // cancel disabled

				} else {
					echo $action_text['already_paid']; // already paid
					echo $action_text['cancel_disabled']; // cancel

				}

			} else {
				echo $action_text['already_paid']; // already paid
				echo $action_text['cancel_disabled']; // cancel disabled

			}
		}

		do_action( 'wpj_after_admin_order_actions', $row, $payment_type );
	}

	public function listSingleJobValues( $order_id = '', $payment_type = '', $section = '' ) {
		if ( $payment_type == 'job-purchase' ) {
			$values = $this->getJobPurchaseOrderDetails( $order_id );
		} elseif ( $payment_type == 'topup' ) {
			$values = $this->getTopupOrderDetails( $order_id );
		} elseif ( $payment_type == 'feature' ) {
			$values = $this->getFeaturedOrderDetails( $order_id );
		} elseif ( $payment_type == 'withdrawal' ) {
			$values = $this->getWithdrawalOrderDetails( $order_id );
		} elseif ( $payment_type == 'custom-extra' ) {
			$values = $this->getCustomExtraOrderDetails( $order_id );
		} elseif ( $payment_type == 'tips' ) {
			$values = $this->getTipsOrderDetails( $order_id );
		} elseif ( $payment_type == 'subscription' ) {
			$values = $this->getSubscriptionOrderDetails( $order_id );
		} elseif ( $payment_type == 'badge' ) {
			$values = $this->getBadgeOrderDetails( $order_id );
		}

		if ( ! empty( $values[$section] ) ) {

			echo '<table>';
				foreach ( $values[$section] as $key => $value ) {
					if ( $section == 'custom_extra' ) {
						echo '<tr><td colspan="2"><p>' . apply_filters( 'wpj_admin_custom_extra_id_string_filter', sprintf( __( 'Custom extra #%d', 'wpjobster' ), $key ), $key, $order_id ) . '</p></td></tr>';
					} elseif ( $section == 'tips' ) {
						echo '<tr><td colspan="2"><p>' . sprintf( __( 'Tips #%d', 'wpjobster' ), $key ) . '</p></td></tr>';
					} elseif ( $section == 'cancellation' ) {
						echo '<tr><td colspan="2"><p>' . __( 'Request cancellation', 'wpjobster' ) . '</p></td></tr>';
					} elseif ( $section == 'modification' ) {
						echo '<tr><td colspan="2"><p>' . __( 'Request modification', 'wpjobster' ) . '</p></td></tr>';
					} elseif ( $section == 'buyer_review' ) {
						echo '<tr><td colspan="2"><p>' . __( 'Buyer', 'wpjobster' ) . '</p></td></tr>';
					} elseif ( $section == 'seller_review' ) {
						echo '<tr><td colspan="2"><p>' . __( 'Seller', 'wpjobster' ) . '</p></td></tr>';
					} elseif ( $section == 'gateway_responses' ) {
						echo '<tr><td colspan="2"><p>' . sprintf( __( 'Response #%d', 'wpjobster' ), $key + 1 ) . '</p></td></tr>';
					}

					foreach ( $value as $key2 => $value2 ) {
						echo '<tr>';
							echo '<td><b>' . $value2['label'] . ':  </b></td>';
							echo '<td>' . $value2['value'] . '</td>';
						echo '</tr>';
					}
				}
			echo '</table>';

		} else {
			if ( $section == 'payment' || $section == 'order' ) {
				$message = __( 'No informations about this order.', 'wpjobster' );
			} elseif ( $section == 'price' ) {
				$message = __( 'No informations about prices.', 'wpjobster' );
			} elseif ( $section == 'custom_extra' ) {
				$message = __( 'No custom extras for this order.', 'wpjobster' );
			} elseif ( $section == 'tips' ) {
				$message = __( 'No tips for this order.', 'wpjobster' );
			} elseif ( $section == 'buyer_review' ) {
				$message = __( 'No review from buyer.', 'wpjobster' );
			} elseif ( $section == 'seller_review' ) {
				$message = __( 'No review from seller.', 'wpjobster' );
			} elseif ( $section == 'chatbox_messages' ) {
				$message = __( 'No messages for this order.', 'wpjobster' );
			} elseif ( $section == 'gateway_responses' ) {
				$message = __( 'No responses for this order.', 'wpjobster' );
			}

			if ( $section != 'cancellation' && $section != 'modification' ) {
				echo '<p>' . $message . '</p>';
			}

		}
	}

	public function getJobPurchaseOrderDetails( $order_id = '' ) {
		$rows = wpj_get_job_order( '*', ['id' => $order_id], 'results' );

		$payment_arr      = [];
		$order_arr        = [];
		$price_arr        = [];
		$custom_extra_arr = [];
		$tips_arr         = [];
		$extra_arr        = [];
		$cancellation_arr = [];
		$modification_arr = [];

		if ( $rows ) {
			foreach ( $rows as $key => $row ) {
				// Payment
				$payment_arr[$key]['status'] = [
					'label' => __( 'Payment status', 'wpjobster' ),
					'value' => $row->payment_status
				];

				if ( $row->force_cancellation ) {
					$force_cancellation = wpj_get_order_cancellation_message( $row );
					$payment_arr[$key]['reason'] = [
						'label' => __( 'Cancellation details', 'wpjobster' ),
						'value' => $force_cancellation['message'] . ' - ' . $force_cancellation['reason']
					];
				}

				$payment_arr[$key]['gateway'] = [
					'label' => __( 'Payment gateway', 'wpjobster' ),
					'value' => $row->payment_gateway
				];

				if ( $row->payment_details ) {
					$payment_arr[$key]['details'] = [
						'label' => __( 'Payment details', 'wpjobster' ),
						'value' => $row->payment_details
					];
				}

				if ( $row->payment_status == 'pending' ) {
					$attachments = explode( ',', get_option( 'bank_transfer_job_proof_attachments_' . $row->id ) );

					$attachments_values = '';

					foreach ( $attachments as $attachment ) {
						if ( get_attached_file( $attachment ) != '' ) {
							$attachments_values .= '<a href="' . get_bloginfo( 'url' ) . '/?secure_download=' . $attachment . wpj_get_token() . '">'  . substr( get_the_title( $attachment ), 0, 20 ) . '...</a> | ';
						}
					}

					$payment_arr[$key]['attachments'] = [
						'label' => __( 'Proof of payment for bank transfer', 'wpjobster' ),
						'value' => rtrim( $attachments_values, ' | ' )
					];
				}

				// Order
				$buyer  = get_userdata( $row->uid );
				$post   = get_post( $row->pid );
				$seller = get_userdata( $post->post_author );

				if ( $row->clearing_period == 1 ) {
					$cleared = __( 'Yes', 'wpjobster' );
				} elseif ( $row->clearing_period == 3 ) {
					$cleared = __( 'No, blocked by admin', 'wpjobster' );
				} else {
					$cleared = __( 'No', 'wpjobster' );
				}

				$order_arr[] = [
					'post_id'           => [
						'label' => __( 'Post ID', 'wpjobster' ),
						'value' => $row->pid,
					],
					'order_id'          => [
						'label' => __( 'Order ID', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_order_link( $row->id ) . '">' . $row->id . '</a>',
					],
					'job_title'         => [
						'label' => __( 'Job Title', 'wpjobster' ),
						'value' =>  '<a href="' . get_permalink( $row->pid ) . '">' . $row->job_title . '</a>',
					],
					'buyer'             => [
						'label' => __( 'Buyer', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_user_profile_link( $buyer->user_login ) . '">' . wpj_get_user_display_type( $buyer->ID ) . '</a>',
					],
					'seller'            => [
						'label' => __( 'Seller', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_user_profile_link( $seller->user_login ) . '">' . wpj_get_user_display_type( $seller->ID ) . '</a>',
					],
					'date_made'         => [
						'label' => __( 'Date made', 'wpjobster' ),
						'value' => $row->date_made ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->date_made ) : '-',
					],
					'expected_delivery' => [
						'label' => __( 'Expected delivery', 'wpjobster' ),
						'value' => $row->expected_delivery ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->expected_delivery ) : '-',
					],
					'delivered'         => [
						'label' => __( 'Delivered', 'wpjobster' ),
						'value' => $row->done_seller == 1 ? __( 'Yes', 'wpjobster' ) : __( 'No', 'wpjobster' ),
					],
					'date_delivered'    => [
						'label' => __( 'Delivered date', 'wpjobster' ),
						'value' => $row->date_finished ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->date_finished ) : '-',
					],
					'completed'         => [
						'label' => __( 'Completed', 'wpjobster' ),
						'value' => $row->done_buyer == 1 ? __( 'Yes', 'wpjobster' ) : __( 'No', 'wpjobster' ),
					],
					'date_completed'    => [
						'label' => __( 'Completed date', 'wpjobster' ),
						'value' => $row->date_completed ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->date_completed ) : '-',
					],
					'closed'            => [
						'label' => __( 'Closed', 'wpjobster' ),
						'value' => $row->closed == 1 ? __( 'Yes', 'wpjobster' ) : __( 'No', 'wpjobster' ),
					],
					'date_closed'       => [
						'label' => __( 'Closed date', 'wpjobster' ),
						'value' => $row->date_closed ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->date_closed ) : '-',
					],
					'cleared'           => [
						'label' => __( 'Cleared', 'wpjobster' ),
						'value' => $cleared,
					],
					'date_to_clear'     => [
						'label' => __( 'Clearing date', 'wpjobster' ),
						'value' => $row->date_to_clear ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->date_to_clear ) : '-',
					],
				];

				// Order Price
				$final_paid_amount = $row->final_paidamount ? explode( '|', $row->final_paidamount ) : 0;
				if ( isset( $final_paid_amount[0] ) && $final_paid_amount[0] != wpj_get_site_default_currency() ) {
					$paid_amount = wpj_get_exchanged_value( $final_paid_amount[1], $final_paid_amount[0], wpj_get_site_default_currency() );
				} else {
					$paid_amount = $final_paid_amount[1];
				}

				$price_arr[] = [
					'job_price'       => [
						'label' => __( 'Job price', 'wpjobster' ),
						'value' => wpj_show_price_classic( $row->job_price ),
					],
					'processing_fees' => [
						'label' => __( 'Processing fees', 'wpjobster' ),
						'value' => wpj_show_price_classic( $row->processing_fees ),
					],
					'site_fees'       => [
						'label' => __( 'Site fees', 'wpjobster' ),
						'value' => wpj_show_price_classic( $row->site_fees ),
					],
					'tax_amount'      => [
						'label' => __( 'Tax', 'wpjobster' ),
						'value' => wpj_show_price_classic( $row->tax_amount ),
					],
					'admin_fee'       => [
						'label' => __( 'Admin fees', 'wpjobster' ),
						'value' => wpj_show_price_classic( $row->admin_fee ),
					],
					'shipping'        => [
						'label' => __( 'Shipping', 'wpjobster' ),
						'value' => wpj_show_price_classic( $row->shipping ),
					],
					'total_paid'      => [
						'label' => __( 'Total paid', 'wpjobster' ),
						'value' => wpj_show_price_classic( $paid_amount ),
					],
				];

				$price_arr = apply_filters( 'wpj_admin_order_details_price_info_filter', $price_arr, $order_id );

				// Extra
				for ( $i = 1; $i <= 10; $i++ ) {
					$extra_arr[] = $this->getExtraSectionValues( $row, $i );
				}
				$extra_arr[] = $this->getExtraSectionValues( $row, '_fast' );
				$extra_arr[] = $this->getExtraSectionValues( $row, '_revision' );

				// Custom extra
				$custom_extras = wpj_json_decode( wpj_stripslashes( $row->custom_extras ) );
				if ( $custom_extras ) {
					foreach ( $custom_extras as $ce_key => $ce ) {
						$custom_extra_arr[] = $this->getTipsAndCeSectionValues( $ce, 'custom_extra' );
					}
				}

				// Tips
				$tips = wpj_json_decode( wpj_stripslashes( $row->tips ) );
				if ( $tips ) {
					foreach ( $tips as $tip_key => $tip ) {
						$tips_arr[] = $this->getTipsAndCeSectionValues( $tip, 'tips' );
					}
				}

				// Cancellation and Modification
				if ( $row->request_cancellation_from_seller == 1 || $row->request_cancellation_from_buyer == 1 ) {
					$cancellation_accepted = $row->accept_cancellation_request == 1 ? __( 'Yes', 'wpjobster' ) : __( 'No', 'wpjobster' );
				}

				$cancellation_arr[] = [
					'cancelled_by_buyer'         => [
						'label' => __( 'Buyer request cancellation', 'wpjobster' ),
						'value' => $row->request_cancellation_from_buyer == 1 ? __( 'Yes', 'wpjobster' ) : __( 'No', 'wpjobster' ),
					],
					'cancelled_by_seller'        => [
						'label' => __( 'Seller request cancellation', 'wpjobster' ),
						'value' => $row->request_cancellation_from_seller == 1 ? __( 'Yes', 'wpjobster' ) : __( 'No', 'wpjobster' ),
					],
					'cancelled_by_admin'         => [
						'label' => __( 'Admin cancelled the order', 'wpjobster' ),
						'value' => $row->force_cancellation == 1 ? __( 'Yes', 'wpjobster' ) : __( 'No', 'wpjobster' ),
					],
					'cancellation_date'          => [
						'label' => __( 'Request cancellation date', 'wpjobster' ),
						'value' => $row->date_request_cancellation ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->date_request_cancellation ) : '-',
					],
					'cancellation_accepted'      => [
						'label' => __( 'Cancellation accepted', 'wpjobster' ),
						'value' => isset( $cancellation_accepted ) ? $cancellation_accepted : '-',
					],
					'cancellation_accepted_date' => [
						'label' => __( 'Cancellation accepted date', 'wpjobster' ),
						'value' => $row->date_accept_cancellation ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->date_accept_cancellation ) : '-',
					],
				];

				$modification_arr[] = [
					'request_modification'         => [
						'label' => __( 'Request modification', 'wpjobster' ),
						'value' => $row->request_modification == 1 ? __( 'Yes', 'wpjobster' ) : __( 'No', 'wpjobster' ),
					],
					'request_modification_date'    => [
						'label' => __( 'Request modification date', 'wpjobster' ),
						'value' => $row->date_request_modification ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->date_request_modification ) : '-',
					],
					'request_modification_message' => [
						'label' => __( 'Request modification message', 'wpjobster' ),
						'value' => $row->message_request_modification ? $row->message_request_modification : '-',
					],
				];

				// Gateway responses
				$responses_arr     = [];
				$responses_results = wpj_get_webhook( 'DISTINCT *', ['order_id' => $row->id, 'payment_type' => 'job_purchase'], 'results' );

				foreach ( $responses_results as $row ) {
					$responses_arr[] = [
						'status' => [
							'label' => __( 'Status', 'wpjobster' ),
							'value' => $row->status,
						],
						'payment_id' => [
							'label' => __( 'Payment ID', 'wpjobster' ),
							'value' => $row->payment_id,
						],
						'event_type' => [
							'label' => __( 'Event Type', 'wpjobster' ),
							'value' => $row->type,
						],
						'description' => [
							'label' => __( 'Description', 'wpjobster' ),
							'value' => $row->description,
						],
						'amount' => [
							'label' => __( 'Amount', 'wpjobster' ),
							'value' => $row->amount . ' ' . $row->amount_currency,
						],
						'fees' => [
							'label' => __( 'Fees', 'wpjobster' ),
							'value' => $row->fees . ' ' . $row->fees_currency,
						],
						'date' => [
							'label' => __( 'Date', 'wpjobster' ),
							'value' => wpj_date( '', $row->create_time ),
						]
					];
				}
			}

			// Review
			$buyer_ratings_arr    = [];
			$seller_ratings_arr   = [];
			$chatbox_messages_arr = [];

			$chatbox_results = wpj_get_order_notification_message( '*', ['oid' => $order_id], 'results' );

			foreach ( $chatbox_results as $row ) {
				// Buyer rating
				if ( $row->uid == -18 ) {
					$buyer_rating_row = wpj_get_job_rating( '*', ['orderid' => $order_id], 'row' );

					if ( $buyer_rating_row ) {
						ob_start();
						wpj_display_rating_stars( $buyer_rating_row->grade );
						$stars = ob_get_contents();
						ob_end_clean();

						$buyer_ratings_arr[] = [
							'buyer_stars' => [
								'label' => __( 'Buyer stars', 'wpjobster' ),
								'value' => $stars,
							],
							'buyer_message' => [
								'label' => __( 'Buyer review', 'wpjobster' ),
								'value' => $buyer_rating_row->reason,
							],
							'buyer_awarded' => [
								'label' => __( 'Awarded On', 'wpjobster' ),
								'value' => wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $buyer_rating_row->datemade ),
							],
							'buyer_action' => [
								'label' => __( 'Delete buyer review', 'wpjobster' ),
								'value' => '<a id="delete_review" data-id="' . $buyer_rating_row->id . '" class="button-secondary">' . __( 'Delete', 'wpjobster' ) . '</a>'
							]
						];
					}
				}

				// Seller rating
				if ( $row->uid == -19 ) {
					$seller_rating_row = wpj_get_job_seller_rating( '*', ['orderid' => $order_id], 'row' );

					if ( $seller_rating_row ) {
						ob_start();
						wpj_display_rating_stars( $seller_rating_row->grade );
						$stars = ob_get_contents();
						ob_end_clean();

						$seller_ratings_arr[] = [
							'seller_stars' => [
								'label' => __( 'Seller stars', 'wpjobster' ),
								'value' => $stars,
							],
							'seller_message' => [
								'label' => __( 'Seller review', 'wpjobster' ),
								'value' => $seller_rating_row->reason,
							],
							'seller_awarded' => [
								'label' => __( 'Awarded On', 'wpjobster' ),
								'value' => wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $seller_rating_row->datemade ),
							],
							'seller_action' => [
								'label' => __( 'Delete seller review', 'wpjobster' ),
								'value' => '<a id="delete_review_response" data-id="' . $seller_rating_row->id . '" class="button-secondary">' . __( 'Delete response', 'wpjobster' ) . '</a>'
							]
						];
					}
				}

				// Chatbox messages
				if ( $row->uid > 0 ) {
					$usr_dt = get_userdata( $row->uid );

					$chatbox_messages_arr[] = [
						'messages' => [
							'label' => '<a href="' . wpj_get_user_profile_link( $usr_dt->user_login ) . '">' . wpj_get_user_display_type( $usr_dt->ID ) . '</a>',
							'value' => $row->content . '&nbsp;<a class="cursor-pointer" id="delete_order_message" data-id="' . $row->id . '">' . __( 'Delete', 'wpjobster' ) . '</a>'
						]
					];
				}
			}
		}

		if ( count( array_filter( $extra_arr ) ) == 0 ) {
			$extra_arr = [];
		}

		return [
			'payment'           => $payment_arr,
			'order'             => $order_arr,
			'price'             => $price_arr,
			'custom_extra'      => $custom_extra_arr,
			'tips'              => $tips_arr,
			'extra'             => $extra_arr,
			'cancellation'      => $cancellation_arr,
			'modification'      => $modification_arr,
			'buyer_review'      => $buyer_ratings_arr,
			'seller_review'     => $seller_ratings_arr,
			'chatbox_messages'  => $chatbox_messages_arr,
			'gateway_responses' => $responses_arr
		];

	}

	public function getTopupOrderDetails( $order_id = '' ) {
		$rows = wpj_get_topup_order( 'DISTINCT *', ['id' => $order_id], 'results' );

		$payment_arr      = [];
		$order_arr        = [];
		$price_arr        = [];

		if ( $rows ) {
			foreach ( $rows as $key => $row ) {

				// Payment
				$payment_arr[] = [
					'status'       => [
						'label' => __( 'Payment status', 'wpjobster' ),
						'value' => $row->payment_status,
					],
					'gateway'      => [
						'label' => __( 'Payment gateway', 'wpjobster' ),
						'value' => $row->payment_gateway_name,
					],
					'payment_date' => [
						'label' => __( 'Payment date', 'wpjobster' ),
						'value' => $row->paid_on ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->paid_on ) : '-',
					],
				];

				// Order
				$user = get_userdata( $row->user_id );
				$order_arr[] = [
					'order_id'   => [
						'label' => __( 'Order ID', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_order_link_by_payment_type( 'topup', $row->id ) . '">' . $row->id . '</a>',
					],
					'added_date' => [
						'label' => __( 'Date made', 'wpjobster' ),
						'value' => $row->added_on ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->added_on ) : '-',
					],
					'user_id'    => [
						'label' => __( 'User', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_user_profile_link( $user->user_login ) . '">' . wpj_get_user_display_type( $user->ID ) . '</a>',
					],
				];

				// Price
				$price_arr[] = [
					'package_cost'    => [
						'label' => __( 'Package cost', 'wpjobster' ),
						'value' => wpj_show_price_classic ( wpj_get_exchanged_value( $row->package_cost_without_tax, $row->currency, wpj_get_site_default_currency() ) ),
					],
					'package_credits' => [
						'label' => __( 'Package credits', 'wpjobster' ),
						'value' => wpj_show_price_classic( wpj_get_exchanged_value( $row->package_credit_without_tax, $row->currency, wpj_get_site_default_currency() ) ),
					],
					'tax'             => [
						'label' => __( 'Tax', 'wpjobster' ),
						'value' => wpj_show_price_classic( wpj_get_exchanged_value( $row->tax, $row->currency, wpj_get_site_default_currency() ) ),
					],
				];

				// Gateway responses
				$responses_arr     = [];
				$responses_results = wpj_get_webhook( 'DISTINCT *', ['order_id' => $row->id, 'payment_type' => 'topup'], 'results' );

				foreach ( $responses_results as $row ) {
					$responses_arr[] = [
						'status' => [
							'label' => __( 'Status', 'wpjobster' ),
							'value' => $row->status,
						],
						'payment_id' => [
							'label' => __( 'Payment ID', 'wpjobster' ),
							'value' => $row->payment_id,
						],
						'event_type' => [
							'label' => __( 'Event Type', 'wpjobster' ),
							'value' => $row->type,
						],
						'description' => [
							'label' => __( 'Description', 'wpjobster' ),
							'value' => $row->description,
						],
						'amount' => [
							'label' => __( 'Amount', 'wpjobster' ),
							'value' => $row->amount . ' ' . $row->amount_currency,
						],
						'fees' => [
							'label' => __( 'Fees', 'wpjobster' ),
							'value' => $row->fees . ' ' . $row->fees_currency,
						],
						'date' => [
							'label' => __( 'Date', 'wpjobster' ),
							'value' => wpj_date( '', $row->create_time ),
						]
					];
				}
			}
		}

		return [
			'payment'           => $payment_arr,
			'order'             => $order_arr,
			'price'             => $price_arr,
			'gateway_responses' => $responses_arr
		];
	}

	public function getFeaturedOrderDetails( $order_id = '' ) {
		$rows = wpj_get_feature_order( '*', ['id' => $order_id], 'results' );

		$payment_arr      = [];
		$order_arr        = [];
		$price_arr        = [];

		if ( $rows ) {
			foreach ( $rows as $key => $row ) {

				// Payment
				$payment_arr[] = [
					'status'       => [
						'label' => __( 'Payment status', 'wpjobster' ),
						'value' => $row->payment_status,
					],
					'gateway'      => [
						'label' => __( 'Payment gateway', 'wpjobster' ),
						'value' => $row->payment_gateway_name,
					],
					'payment_date' => [
						'label' => __( 'Payment date', 'wpjobster' ),
						'value' => $row->paid_on ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->paid_on ) : '-',
					],
				];

				// Order
				$user   = get_userdata( $row->user_id );
				$f_page = $this->getOrderFeaturedPages( $row );
				$order_arr[] = [
					'order_id'       => [
						'label' => __( 'Order ID', 'wpjobster' ),
						'value' => $row->id,
					],
					'job_id'         => [
						'label' => __( 'Job ID', 'wpjobster' ),
						'value' => $row->job_id,
					],
					'job_title'      => [
						'label' => __( 'Job Title', 'wpjobster' ),
						'value' =>  '<a href="' . get_permalink( $row->job_id ) . '">' . get_the_title( $row->job_id ) . '</a>',
					],
					'added_date'     => [
						'label' => __( 'Date made', 'wpjobster' ),
						'value' => $row->added_on ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->added_on ) : '-',
					],
					'user_id'        => [
						'label' => __( 'User', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_user_profile_link( $user->user_login ) . '">' . wpj_get_user_display_type( $user->ID ) . '</a>',
					],
					'featured_pages' => [
						'label' => __( 'Featured job', 'wpjobster' ),
						'value' => $f_page,
					],
				];

				// Price
				$featured_amount = $row->featured_amount;
				$tax             = wpj_get_exchanged_value( $row->tax, $row->currency, wpj_get_site_default_currency() );
				$paid_amount     = wpj_get_exchanged_value( $row->payable_amount, $row->currency, wpj_get_site_default_currency() );
				$processing_fees = $paid_amount - $tax - $featured_amount;

				$price_arr[] = [
					'featured_amount' => [
						'label' => __( 'Feature price', 'wpjobster' ),
						'value' => wpj_show_price_classic( $featured_amount ),
					],
					'processing_fees' => [
						'label' => __( 'Processing fees', 'wpjobster' ),
						'value' => wpj_show_price_classic( $processing_fees ),
					],
					'tax'             => [
						'label' => __( 'Tax', 'wpjobster' ),
						'value' => wpj_show_price_classic( $tax ),
					],
					'paid_amount'     => [
						'label' => __( 'Total paid amount', 'wpjobster' ),
						'value' => wpj_show_price_classic( $paid_amount ),
					],
				];

				// Gateway responses
				$responses_arr     = [];
				$responses_results = wpj_get_webhook( 'DISTINCT *', ['order_id' => $row->id, 'payment_type' => 'feature'], 'results' );

				foreach ( $responses_results as $row ) {
					$responses_arr[] = [
						'status' => [
							'label' => __( 'Status', 'wpjobster' ),
							'value' => $row->status,
						],
						'payment_id' => [
							'label' => __( 'Payment ID', 'wpjobster' ),
							'value' => $row->payment_id,
						],
						'event_type' => [
							'label' => __( 'Event Type', 'wpjobster' ),
							'value' => $row->type,
						],
						'description' => [
							'label' => __( 'Description', 'wpjobster' ),
							'value' => $row->description,
						],
						'amount' => [
							'label' => __( 'Amount', 'wpjobster' ),
							'value' => $row->amount . ' ' . $row->amount_currency,
						],
						'fees' => [
							'label' => __( 'Fees', 'wpjobster' ),
							'value' => $row->fees . ' ' . $row->fees_currency,
						],
						'date' => [
							'label' => __( 'Date', 'wpjobster' ),
							'value' => wpj_date( '', $row->create_time ),
						]
					];
				}
			}
		}

		return [
			'payment'           => $payment_arr,
			'order'             => $order_arr,
			'price'             => $price_arr,
			'gateway_responses' => $responses_arr
		];
	}

	public function getWithdrawalOrderDetails( $order_id = '' ) {
		$rows = wpj_get_withdrawal_order( '*', ['id' => $order_id], 'results' );

		$payment_arr      = [];
		$order_arr        = [];
		$price_arr        = [];

		if ( $rows ) {
			foreach ( $rows as $key => $row ) {

				// Payment
				if ( $row->done == 1 ) {
					$status = __( 'Completed by Admin', 'wpjobster' );
				} elseif ( $row->done == -1 && $row->rejected == 1 ) {
					$status = __( 'Rejected By Admin', 'wpjobster' );
				} elseif ( $row->done == -2 ) {
					$status = __( 'Processing', 'wpjobster' );
				} else {
					$status = __( 'Pending', 'wpjobster' );
				}
				$payment_arr[] = [
					'status'        => [
						'label' => __( 'Payment status', 'wpjobster' ),
						'value' => $status,
					],
					'gateway'       => [
						'label' => __( 'Payment gateway', 'wpjobster' ),
						'value' => $row->methods,
					],
					'date_accepted' => [
						'label' => __( 'Date accepted', 'wpjobster' ),
						'value' => $row->datedone ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->datedone ) : '-',
					],
					'date_rejected' => [
						'label' => __( 'Date rejected', 'wpjobster' ),
						'value' => $row->rejected_on ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->rejected_on ) : '-',
					],
				];

				// Order
				$user = get_userdata( $row->uid );
				$order_arr[] = [
					'order_id'   => [
						'label' => __( 'Order ID', 'wpjobster' ),
						'value' => $row->id,
					],
					'added_date' => [
						'label' => __( 'Date made', 'wpjobster' ),
						'value' => $row->datemade ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->datemade ) : '-',
					],
					'user_id'    => [
						'label' => __( 'User', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_user_profile_link( $user->user_login ) . '">' . wpj_get_user_display_type( $user->ID ) . '</a>',
					],
					'email'      => [
						'label' => __( 'User info', 'wpjobster' ),
						'value' => $row->payeremail,
					],
				];

				// Price
				$final_paid_amount = $row->payedamount ? explode( '|', $row->payedamount ) : 0;
				if ( isset( $final_paid_amount[0] ) && $final_paid_amount[0] != wpj_get_site_default_currency() ) {
					$paid_amount = wpj_get_exchanged_value( $final_paid_amount[1], $final_paid_amount[0], wpj_get_site_default_currency() );
				} else {
					$paid_amount = $final_paid_amount[1];
				}
				$price_arr[] = [
					'amount' => [
						'label' => __( 'Amount', 'wpjobster' ),
						'value' => wpj_show_price_classic( $paid_amount ),
					],
				];

				// Gateway responses
				$responses_arr     = [];
				$responses_results = wpj_get_webhook( 'DISTINCT *', ['order_id' => $row->id, 'payment_type' => 'withdrawal'], 'results' );

				foreach ( $responses_results as $row ) {
					$responses_arr[] = [
						'status' => [
							'label' => __( 'Status', 'wpjobster' ),
							'value' => $row->status,
						],
						'payment_id' => [
							'label' => __( 'Payment ID', 'wpjobster' ),
							'value' => $row->payment_id,
						],
						'event_type' => [
							'label' => __( 'Event Type', 'wpjobster' ),
							'value' => $row->type,
						],
						'description' => [
							'label' => __( 'Description', 'wpjobster' ),
							'value' => $row->description,
						],
						'amount' => [
							'label' => __( 'Amount', 'wpjobster' ),
							'value' => $row->amount . ' ' . $row->amount_currency,
						],
						'fees' => [
							'label' => __( 'Fees', 'wpjobster' ),
							'value' => $row->fees . ' ' . $row->fees_currency,
						],
						'date' => [
							'label' => __( 'Date', 'wpjobster' ),
							'value' => wpj_date( '', $row->create_time ),
						]
					];
				}
			}
		}

		return [
			'payment'           => $payment_arr,
			'order'             => $order_arr,
			'price'             => $price_arr,
			'gateway_responses' => $responses_arr
		];
	}

	public function getCustomExtraOrderDetails( $order_id = '' ) {
		$rows = wpj_get_custom_extra_order( '*', ['id' => $order_id], 'results' );

		$payment_arr      = [];
		$order_arr        = [];
		$price_arr        = [];

		if ( $rows ) {
			foreach ( $rows as $key => $row ) {

				// Payment
				$payment_arr[] = [
					'status'       => [
						'label' => __( 'Payment status', 'wpjobster' ),
						'value' => $row->payment_status,
					],
					'gateway'      => [
						'label' => __( 'Payment gateway', 'wpjobster' ),
						'value' => $row->payment_gateway_name,
					],
					'payment_date' => [
						'label' => __( 'Payment date', 'wpjobster' ),
						'value' => $row->paid_on ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->paid_on ) : '-',
					],
				];

				// Order
				$order_info = wpj_get_job_order_by_id( $row->order_id );
				$buyer      = get_userdata( $row->user_id );
				$post       = get_post( $order_info->pid );
				$seller     = get_userdata( $post->post_author );

				$order_arr[] = [
					'order_id'        => [
						'label' => __( 'Order ID', 'wpjobster' ),
						'value' => $row->id,
					],
					'ce_id'           => [
						'label' => __( 'Custom Extra ID', 'wpjobster' ),
						'value' => $row->custom_extra_id,
					],
					'parent_order_id' => [
						'label' => __( 'Parent Order ID', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_order_link( $row->order_id ) . '">' . $row->order_id . '</a>',
					],
					'job_title'       => [
						'label' => __( 'Job Title', 'wpjobster' ),
						'value' =>  '<a href="' . get_permalink( $order_info->pid ) . '">' . $order_info->job_title . '</a>',
					],
					'buyer'           => [
						'label' => __( 'Buyer', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_user_profile_link( $buyer->user_login ) . '">' . wpj_get_user_display_type( $buyer->ID ) . '</a>',
					],
					'seller'          => [
						'label' => __( 'Seller', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_user_profile_link( $seller->user_login ) . '">' . wpj_get_user_display_type( $seller->ID ) . '</a>',
					],
					'added_date'      => [
						'label' => __( 'Date made', 'wpjobster' ),
						'value' => $row->added_on ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->added_on ) : '-',
					],
				];

				// Price
				$custom_extras = wpj_json_decode( wpj_stripslashes( $order_info->custom_extras ) );
				$custom_extra  = $custom_extras[$row->custom_extra_id];

				$ce_payedamount = $custom_extra->price;
				if ( isset( $custom_extra->processing_fees ) ) $ce_payedamount += $custom_extra->processing_fees;
				if ( isset( $custom_extra->tax ) ) $ce_payedamount += $custom_extra->tax;

				$ce_tax = isset( $custom_extra->tax ) ? $custom_extra->tax : 0;

				$price_arr[] = [
					'ce_amount'   => [
						'label' => __( 'Custom Extra Amount', 'wpjobster' ),
						'value' => wpj_show_price_classic( wpj_get_exchanged_value( $custom_extra->price, $row->currency, wpj_get_site_default_currency() ) ),
					],
					'paid_amount' => [
						'label' => __( 'Paid amount', 'wpjobster' ),
						'value' => wpj_show_price_classic( wpj_get_exchanged_value( $ce_payedamount, $row->currency, wpj_get_site_default_currency() ) ),
					],
					'tax'         => [
						'label' => __( 'Tax', 'wpjobster' ),
						'value' => wpj_show_price_classic( wpj_get_exchanged_value( $ce_tax, $row->currency, wpj_get_site_default_currency() ) ),
					],
				];

				// Gateway responses
				$responses_arr     = [];
				$responses_results = wpj_get_webhook( 'DISTINCT *', ['order_id' => $row->id, 'payment_type' => 'custom_extra'], 'results' );

				foreach ( $responses_results as $row ) {
					$responses_arr[] = [
						'status' => [
							'label' => __( 'Status', 'wpjobster' ),
							'value' => $row->status,
						],
						'payment_id' => [
							'label' => __( 'Payment ID', 'wpjobster' ),
							'value' => $row->payment_id,
						],
						'event_type' => [
							'label' => __( 'Event Type', 'wpjobster' ),
							'value' => $row->type,
						],
						'description' => [
							'label' => __( 'Description', 'wpjobster' ),
							'value' => $row->description,
						],
						'amount' => [
							'label' => __( 'Amount', 'wpjobster' ),
							'value' => $row->amount . ' ' . $row->amount_currency,
						],
						'fees' => [
							'label' => __( 'Fees', 'wpjobster' ),
							'value' => $row->fees . ' ' . $row->fees_currency,
						],
						'date' => [
							'label' => __( 'Date', 'wpjobster' ),
							'value' => wpj_date( '', $row->create_time ),
						]
					];
				}
			}
		}

		return [
			'payment'           => $payment_arr,
			'order'             => $order_arr,
			'price'             => $price_arr,
			'gateway_responses' => $responses_arr
		];
	}

	public function getTipsOrderDetails( $order_id = '' ) {
		$rows = wpj_get_tips_order( '*', ['id' => $order_id], 'results' );

		$payment_arr      = [];
		$order_arr        = [];
		$price_arr        = [];

		if ( $rows ) {
			foreach ( $rows as $key => $row ) {

				// Payment
				$payment_arr[] = [
					'status'       => [
						'label' => __( 'Payment status', 'wpjobster' ),
						'value' => $row->payment_status,
					],
					'gateway'      => [
						'label' => __( 'Payment gateway', 'wpjobster' ),
						'value' => $row->payment_gateway_name,
					],
					'payment_date' => [
						'label' => __( 'Payment date', 'wpjobster' ),
						'value' => $row->paid_on ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->paid_on ) : '-',
					],
				];

				// Order
				$order_info = wpj_get_job_order_by_id( $row->order_id );
				$buyer      = get_userdata( $row->user_id );
				$post       = get_post( $order_info->pid );
				$seller     = get_userdata( $post->post_author );

				$order_arr[] = [
					'order_id'        => [
						'label' => __( 'Order ID', 'wpjobster' ),
						'value' => $row->id,
					],
					'ce_id'           => [
						'label' => __( 'Tips ID', 'wpjobster' ),
						'value' => $row->tips_id,
					],
					'parent_order_id' => [
						'label' => __( 'Parent order ID', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_order_link( $row->order_id ) . '">' . $row->order_id . '</a>',
					],
					'job_title'       => [
						'label' => __( 'Job title', 'wpjobster' ),
						'value' =>  '<a href="' . get_permalink( $order_info->pid ) . '">' . $order_info->job_title . '</a>',
					],
					'buyer'           => [
						'label' => __( 'Buyer', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_user_profile_link( $buyer->user_login ) . '">' . wpj_get_user_display_type( $buyer->ID ) . '</a>',
					],
					'seller'          => [
						'label' => __( 'Seller', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_user_profile_link( $seller->user_login ) . '">' . wpj_get_user_display_type( $seller->ID ) . '</a>',
					],
					'added_date'      => [
						'label' => __( 'Date made', 'wpjobster' ),
						'value' => $row->added_on ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->added_on ) : '-',
					],
				];

				// Price
				$price_arr[] = [
					'ce_amount'   => [
						'label' => __( 'Tips amount', 'wpjobster' ),
						'value' => wpj_show_price_classic ( wpj_get_exchanged_value( $row->tips_amount, $row->currency, wpj_get_site_default_currency() ) ),
					],
					'paid_amount' => [
						'label' => __( 'Paid amount', 'wpjobster' ),
						'value' => wpj_show_price_classic( wpj_get_exchanged_value( $row->payable_amount, $row->currency, wpj_get_site_default_currency() ) ),
					],
					'tax'         => [
						'label' => __( 'Tax', 'wpjobster' ),
						'value' => wpj_show_price_classic( wpj_get_exchanged_value( $row->tax, $row->currency, wpj_get_site_default_currency() ) ),
					],
				];

				// Gateway responses
				$responses_arr     = [];
				$responses_results = wpj_get_webhook( 'DISTINCT *', ['order_id' => $row->id, 'payment_type' => 'tips'], 'results' );

				foreach ( $responses_results as $row ) {
					$responses_arr[] = [
						'status' => [
							'label' => __( 'Status', 'wpjobster' ),
							'value' => $row->status,
						],
						'payment_id' => [
							'label' => __( 'Payment ID', 'wpjobster' ),
							'value' => $row->payment_id,
						],
						'event_type' => [
							'label' => __( 'Event Type', 'wpjobster' ),
							'value' => $row->type,
						],
						'description' => [
							'label' => __( 'Description', 'wpjobster' ),
							'value' => $row->description,
						],
						'amount' => [
							'label' => __( 'Amount', 'wpjobster' ),
							'value' => $row->amount . ' ' . $row->amount_currency,
						],
						'fees' => [
							'label' => __( 'Fees', 'wpjobster' ),
							'value' => $row->fees . ' ' . $row->fees_currency,
						],
						'date' => [
							'label' => __( 'Date', 'wpjobster' ),
							'value' => wpj_date( '', $row->create_time ),
						]
					];
				}
			}
		}

		return [
			'payment'           => $payment_arr,
			'order'             => $order_arr,
			'price'             => $price_arr,
			'gateway_responses' => $responses_arr
		];
	}

	public function getSubscriptionOrderDetails( $order_id = '' ) {
		$rows = wpj_get_subscription_order( '*', ['id' => $order_id], 'results' );

		$payment_arr      = [];
		$order_arr        = [];
		$price_arr        = [];

		if ( $rows ) {
			foreach ( $rows as $key => $row ) {

				// Payment
				$payment_arr[] = [
					'status'       => [
						'label' => __( 'Payment status', 'wpjobster' ),
						'value' => $row->payment_status,
					],
					'gateway'      => [
						'label' => __( 'Payment gateway', 'wpjobster' ),
						'value' => $row->payment_gateway_name,
					],
					'payment_date' => [
						'label' => __( 'Payment date', 'wpjobster' ),
						'value' => $row->payment_date ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->payment_date ) : '-',
					],
				];

				// Order
				$user = get_userdata( $row->user_id );
				$order_arr[] = [
					'order_id'            => [
						'label' => __( 'Order ID', 'wpjobster' ),
						'value' => $row->id,
					],
					'subscription_status' => [
						'label' => __( 'Subscription status', 'wpjobster' ),
						'value' => $row->subscription_status,
					],
					'subscription_plan'   => [
						'label' => __( 'Subscription plan', 'wpjobster' ),
						'value' => $row->plan,
					],
					'subscription_level'  => [
						'label' => __( 'Subscription level', 'wpjobster' ),
						'value' => str_replace( [ '-', '–' ], '', $row->level ),
					],
					'user'                => [
						'label' => __( 'User', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_user_profile_link( $user->user_login ) . '">' . wpj_get_user_display_type( $user->ID ) . '</a>',
					],
					'added_date'          => [
						'label' => __( 'Date made', 'wpjobster' ),
						'value' => $row->addon_date ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->addon_date ) : '-',
					]
				];

				// Next billing
				$subscription = wpj_get_subscription_by_uid( $row->user_id );
				if ( $subscription && $subscription->subscription_status == 'active' && $row->subscription_status == 'active' ) {
					$next_billing_info[] = [
						'next_billing_date'       => [
							'label' => __( 'Next billing date', 'wpjobster' ),
							'value' => $row->plan == 'lifetime' ? __( 'NA', 'wpjobster' ) : $subscription->next_billing_date
						],
						'next_subscription_level'  => [
							'label' => __( 'Next subscription level', 'wpjobster' ),
							'value' => $subscription->next_subscription_level
						],
						'next_subscription_type'   => [
							'label' => __( 'Next subscription type', 'wpjobster' ),
							'value' => $subscription->next_subscription_type
						],
						'next_subscription_amount' => [
							'label' => __( 'Next subscription amount', 'wpjobster' ),
							'value' => wpj_show_price_classic( $subscription->next_subscription_amount )
						]
					];

				} else {
					$next_billing_info[] = [
						'inactive_subscription' => [
							'label' => __( 'Status', 'wpjobster' ),
							'value' => __( 'This subscription is inactive', 'wpjobster' )
						]
					];
				}

				// Price
				$price_arr[] = [
					'amount'   => [
						'label' => __( 'Amount', 'wpjobster' ),
						'value' => wpj_show_price_classic ( wpj_get_exchanged_value( $row->amount, $row->mc_currency, wpj_get_site_default_currency() ) ),
					],
					'paid_amount' => [
						'label' => __( 'Paid amount', 'wpjobster' ),
						'value' => wpj_show_price_classic( wpj_get_exchanged_value( $row->payable_amount, $row->mc_currency, wpj_get_site_default_currency() ) ),
					],
					'tax'         => [
						'label' => __( 'Tax', 'wpjobster' ),
						'value' => wpj_show_price_classic( wpj_get_exchanged_value( $row->tax, $row->mc_currency, wpj_get_site_default_currency() ) ),
					],
				];

				// Gateway responses
				$responses_arr     = [];
				$responses_results = wpj_get_webhook( 'DISTINCT *', ['order_id' => $row->id, 'payment_type' => 'subscription'], 'results' );

				foreach ( $responses_results as $row ) {
					$responses_arr[] = [
						'status' => [
							'label' => __( 'Status', 'wpjobster' ),
							'value' => $row->status,
						],
						'payment_id' => [
							'label' => __( 'Payment ID', 'wpjobster' ),
							'value' => $row->payment_id,
						],
						'event_type' => [
							'label' => __( 'Event Type', 'wpjobster' ),
							'value' => $row->type,
						],
						'description' => [
							'label' => __( 'Description', 'wpjobster' ),
							'value' => $row->description,
						],
						'amount' => [
							'label' => __( 'Amount', 'wpjobster' ),
							'value' => $row->amount . ' ' . $row->amount_currency,
						],
						'fees' => [
							'label' => __( 'Fees', 'wpjobster' ),
							'value' => $row->fees . ' ' . $row->fees_currency,
						],
						'date' => [
							'label' => __( 'Date', 'wpjobster' ),
							'value' => wpj_date( '', $row->create_time ),
						]
					];
				}
			}
		}

		return [
			'payment'           => $payment_arr,
			'order'             => $order_arr,
			'billing'           => $next_billing_info,
			'price'             => $price_arr,
			'gateway_responses' => $responses_arr
		];
	}

	public function getBadgeOrderDetails( $order_id = '' ) {
		$rows = wpj_get_badge_order( '*', ['id' => $order_id], 'results' );

		$payment_arr      = [];
		$order_arr        = [];
		$price_arr        = [];

		if ( $rows ) {
			foreach ( $rows as $key => $row ) {

				// Payment
				$payment_arr[] = [
					'status'           => [
						'label' => __( 'Payment status', 'wpjobster' ),
						'value' => $row->payment_status
					],
					'gateway'          => [
						'label' => __( 'Payment gateway', 'wpjobster' ),
						'value' => $row->payment_gateway
					],
					'payment_date'     => [
						'label' => __( 'Payment date', 'wpjobster' ),
						'value' => $row->payment_date ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->payment_date ) : '-'
					],
					'payment_amount'   => [
						'label' => __( 'Payment amount', 'wpjobster' ),
						'value' => wpj_show_price_classic( $row->payment_amount )
					],
					'payment_currency' => [
						'label' => __( 'Payment currency', 'wpjobster' ),
						'value' => $row->payment_currency
					],
					'payment_type'     => [
						'label' => __( 'Payment type', 'wpjobster' ),
						'value' => $row->payment_type == 1 ? __( 'By Admin', 'wpjobster' ) : __( 'By Seller', 'wpjobster' )
					],
				];

				// Order
				$user = get_userdata( $row->user_id );
				$order_arr[] = [
					'order_id'   => [
						'label' => __( 'Order ID', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_order_link_by_payment_type( 'badge', $row->id ) . '">' . $row->id . '</a>',
					],
					'added_date' => [
						'label' => __( 'Date made', 'wpjobster' ),
						'value' => $row->date_made ? wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->date_made ) : '-',
					],
					'user_id'    => [
						'label' => __( 'User', 'wpjobster' ),
						'value' => '<a href="' . wpj_get_user_profile_link( $user->user_login ) . '">' . wpj_get_user_display_type( $user->ID ) . '</a>',
					],
					'description' => [
						'label' => __( 'Description', 'wpjobster' ),
						'value' => $row->description
					],
					'icon' => [
						'label' => __( 'Icon', 'wpjobster' ),
						'value' => $row->icon
					],
					'price' => [
						'label' => __( 'Price', 'wpjobster' ),
						'value' => wpj_show_price_classic( $row->price )
					],
					'min_rating'  => [
						'label' => __( 'Min rating', 'wpjobster' ),
						'value' => $row->min_rating
					],
					'min_reviews' => [
						'label' => __( 'Min reviews', 'wpjobster' ),
						'value' => $row->min_reviews
					],
				];

				// Gateway responses
				$responses_arr     = [];
				$responses_results = wpj_get_webhook( 'DISTINCT *', ['order_id' => $row->id, 'payment_type' => 'badge'], 'results' );

				foreach ( $responses_results as $row ) {
					$responses_arr[] = [
						'status' => [
							'label' => __( 'Status', 'wpjobster' ),
							'value' => $row->status,
						],
						'payment_id' => [
							'label' => __( 'Payment ID', 'wpjobster' ),
							'value' => $row->payment_id,
						],
						'event_type' => [
							'label' => __( 'Event Type', 'wpjobster' ),
							'value' => $row->type,
						],
						'description' => [
							'label' => __( 'Description', 'wpjobster' ),
							'value' => $row->description,
						],
						'amount' => [
							'label' => __( 'Amount', 'wpjobster' ),
							'value' => $row->amount . ' ' . $row->amount_currency,
						],
						'fees' => [
							'label' => __( 'Fees', 'wpjobster' ),
							'value' => $row->fees . ' ' . $row->fees_currency,
						],
						'date' => [
							'label' => __( 'Date', 'wpjobster' ),
							'value' => wpj_date( '', $row->create_time ),
						]
					];
				}
			}
		}

		return [
			'payment'           => $payment_arr,
			'order'             => $order_arr,
			'gateway_responses' => $responses_arr
		];
	}

	public function getTipsAndCeSectionValues( $row = '', $type = '' ) {
		if ( isset( $row->completed ) && $row->completed == 1 )     $status = __( 'completed', 'wpjobster' );
		elseif ( isset( $row->delivered ) && $row->delivered == 1 ) $status = __( 'delivered', 'wpjobster' );
		elseif ( isset( $row->paid ) && $row->paid == 1 )           $status = __( 'paid', 'wpjobster' );
		elseif ( isset( $row->declined ) && $row->declined == 1 )   $status = __( 'declined', 'wpjobster' );
		elseif ( isset( $row->cancelled ) && $row->cancelled == 1 ) $status = __( 'cancelled', 'wpjobster' );
		else $status = __( 'pending', 'wpjobster' );

		$amount = $type == 'tips' ? $row->amount : $row->price;

		$processing_fees = isset( $row->processing_fees ) ? $row->processing_fees : 0;
		$tax = isset( $row->tax ) ? $row->tax : 0;

		return [
			'description' => [
				'label' => $type == 'tips' ? __( 'Reason', 'wpjobster' ) : __( 'Description', 'wpjobster' ),
				'value' => $type == 'tips' ? $row->reason : $row->description,
			],
			'price' => [
				'label' => __( 'Price', 'wpjobster' ),
				'value' => $amount,
			],
			'processing_fees' => [
				'label' => __( 'Processing fees', 'wpjobster' ),
				'value' => wpj_show_price_classic( $processing_fees ),
			],
			'tax' => [
				'label' => __( 'Tax', 'wpjobster' ),
				'value' => wpj_show_price_classic( $tax ),
			],
			'paid_amount' => [
				'label' => __( 'Paid amount', 'wpjobster' ),
				'value' => wpj_show_price_classic( $amount + $processing_fees + $tax ),
			],
			'date' => [
				'label' => __( 'Date', 'wpjobster' ),
				'value' => wpj_date( wpj_get_option( 'date_format' ) . ' ' . wpj_get_option( 'time_format' ), $row->time ),
			],
			'status' => [
				'label' => __( 'Status', 'wpjobster' ),
				'value' => $status,
			],
		];
	}

	public function getExtraSectionValues( $row = '', $nr = '' ) {
		if ( $row->{'extra' . $nr } == 1 ) {
			$extra['extra' . $nr]['enable'] = $row->{'extra' . $nr };

			if ( $nr != '_fast' && $nr != '_revision' ) {
				$extra['extra' . $nr]['title'] = [
					'label' => __( 'Title', 'wpjobster' ),
					'value' => $row->{'extra' . $nr . '_title' },
				];
			}

			$extra['extra' . $nr]['price'] = [
				'label' => __( 'Price', 'wpjobster' ),
				'value' => wpj_show_price_classic( $row->{'extra' . $nr . '_price' } ),
			];

			$extra['extra' . $nr]['days'] = [
				'label' => __( 'Max days to deliver', 'wpjobster' ),
				'value' => sprintf( _n( '%d day', '%d days', $row->{'extra' . $nr . '_days' }, 'wpjobster' ), $row->{'extra' . $nr . '_days' } ),
			];

			return $extra;
		} else {
			return false;
		}
	}

	public function getOrderFeaturedPages( $row = [] ) {
		$f_page = [];

		$pages = str_split( $row->feature_pages );

		if ( isset( $pages[0] ) ) {
			if ( $pages[0] == 'h' ) $f_page[] = [ __( 'homepage', 'wpjobster' ), $row->h_date_start, wpj_get_featured_end_date( $row->h_date_start ) ];
			elseif ( $pages[0] == 'c' ) $f_page[] = [ __( 'category', 'wpjobster' ), $row->c_date_start, wpj_get_featured_end_date( $row->c_date_start ) ];
			elseif ( $pages[0] == 's' ) $f_page[] = [ __( 'subcategory', 'wpjobster' ), $row->s_date_start, wpj_get_featured_end_date( $row->s_date_start ) ];
		}

		if ( isset( $pages[1] ) ) {
			if ( $pages[1] == 'h' ) $f_page[] = [ __( 'homepage', 'wpjobster' ), $row->h_date_start, wpj_get_featured_end_date( $row->h_date_start ) ];
			elseif ( $pages[1] == 'c' ) $f_page[] = [ __( 'category', 'wpjobster' ), $row->c_date_start, wpj_get_featured_end_date( $row->c_date_start ) ];
			elseif ( $pages[1] == 's' ) $f_page[] = [ __( 'subcategory', 'wpjobster' ), $row->s_date_start, wpj_get_featured_end_date( $row->s_date_start ) ];
		}

		if ( isset( $pages[2] ) ) {
			if ( $pages[2] == 'h' ) $f_page[] = [ __( 'homepage', 'wpjobster' ), $row->h_date_start, wpj_get_featured_end_date( $row->h_date_start ) ];
			elseif ( $pages[2] == 'c' ) $f_page[] = [ __( 'category', 'wpjobster' ), $row->c_date_start, wpj_get_featured_end_date( $row->c_date_start ) ];
			elseif ( $pages[2] == 's' ) $f_page[] = [ __( 'subcategory', 'wpjobster' ), $row->s_date_start, wpj_get_featured_end_date( $row->s_date_start ) ];
		}

		$elements = [];
		foreach ( $f_page as $f_key => $f_value ) {
			$elements[] = sprintf( __( 'Featured on %s between: <strong>%s - %s</strong>' ), $f_value[0], wpj_date( wpj_get_option( 'date_format' ), $f_value[1] ), wpj_date( wpj_get_option( 'date_format' ), $f_value[2] ) );
		}

		return implode( ', ', $elements );
	}
}

add_action( 'after_setup_theme', [ 'WPJ_Orders', 'init' ] );