<?php class WPJ_Job_Purchase {

	private $payment_gateway;

	public function __construct($payment_gateway = '' ) {
		if ( ! $payment_gateway ) $this->payment_gateway = WPJ_Form::get( 'pay_for_item' );
		else $this->payment_gateway = $payment_gateway;

		add_action( 'wpjobster_job_purchase_payment_success', [$this, 'process'], 10, 5 );
		add_action( 'wpjobster_job_purchase_payment_failed',  [$this, 'failed'],  10, 5 );
		add_action( 'wpjobster_job_purchase_payment_other',   [$this, 'other'],   10, 6 );
	}

	public function insert( $payment_gateway = '' ) {
		// Get payment gateway
		if ( ! $payment_gateway ) $payment_gateway = $this->payment_gateway;

		// Get current order ID
		if ( ! empty( $_GET['oid'] ) && isset( $_GET['process_pending'] ) ) $oid = $_GET['oid'];
		else $oid = 0;

		// Insert order if is new
		if ( $oid == 0 ) $oid = wpj_insert_job_purchase_order(['payment_gateway' => $payment_gateway]);

		// Get the new order data
		$order = wpj_get_job_order_by_id( $oid );

		// Action
		do_action( 'wpj_after_new_job_order', $order );

		// Return
		return $order;
	}

	public function process( $oid, $payment_gateway, $details, $response, $redirect = 'yes' ) {
		// Get job purchase order
		$order = wpj_get_job_order_by_id( $oid );

		// Get payment gateway
		if ( ! $payment_gateway ) $payment_gateway = $this->payment_gateway;

		// Get details
		if ( ! $details ) $details = $payment_gateway;

		// Start the payment process
		if ( isset( $order->payment_status ) && $order->payment_status != 'completed' ) {
			// Insert notification
			$this_notification = wpj_insert_order_notification( ['uid' => 0, 'oid' => $oid, 'content' => sprintf(__( 'Paid with %s', 'wpjobster' ), $payment_gateway)], ['%d', '%d', '%s'] );

			if ( $this_notification > 0 ) {
				// Get seller ID
				$seller_id = get_post_field( 'post_author', $order->pid );

				// Accept custom offer
				if ( wpj_is_custom_offer( $order->pid ) )
					wpj_accept_custom_offer( $order->pid, $oid, $order->uid, $seller_id );

				// Insert rating
				wpj_insert_job_rating(['orderid' => $oid, 'uid' => $seller_id, 'pid' => $order->pid], ['%d', '%d', '%d']);

				// Update notification
				wpj_update_user_notifications([
					'user1'       => $seller_id,
					'user2'       => $order->uid,
					'type'        => wpj_is_custom_offer( $order->pid ) ? 'messages' : 'notifications',
					'number'      => +1,
					'notify_id'   => $this_notification,
					'notify_type' => wpj_is_custom_offer( $order->pid ) ? 'custom_offer_paid_new' : 'purchased_job',
					'order_id'    => $oid
				]);

				// Save log
				wpj_maintain_log( $oid );

				// Send emails
				$email_data = [
					'##mc_gross##'              => $order->mc_gross,
					'##processing_fees##'       => $order->processing_fees,
					'##tax_amount##'            => $order->tax_amount,
					'##transaction_number##'    => wpj_camouflage_oid( $oid, current_time( 'timestamp', 1 ) ),
					'##transaction_page_link##' => wpj_get_order_link( $oid ),
					'##job_name##'              => $order->job_title,
					'##job_link##'              => urldecode( get_permalink( $order->pid ) ),
					'##sender_username##'       => get_current_user_id() == $order->uid ? wpj_get_user_display_type( $order->uid ) : wpj_get_user_display_type( $seller_id )
				];

				$email_data_buyer  = $email_data + ['##sender_username##' => wpj_get_user_display_type( $seller_id )];
				$email_data_seller = $email_data + ['##sender_username##' => wpj_get_user_display_type( $order->uid )];

				wpj_notify_user_translated( 'purchased_buyer', $order->uid, $email_data_buyer );
				wpj_notify_user_translated( 'purchased_seller', $seller_id, $email_data_seller );

				// Update payment
				wpj_update_payment(
					['payment_status' => 'completed', 'payment_details' => $details, 'payment_response' => $response],
					['payment_type_id' => $oid, 'payment_type' => 'job_purchase']
				);

				// Update order
				wpj_update_job_order(
					['payment_status' => 'completed', 'payment_details' => $details, 'payment_response' => $response, 'closed' => 0],
					['id' => $oid]
				);

				// Action
				do_action( 'wpj_after_job_payment_is_completed', $oid, 'process' );
			}
		}

		// Redirect
		$this->redirect( $redirect, $oid, 'success' );
	}

	public function failed( $oid, $payment_gateway, $details = '', $response = '', $redirect = 'yes' ) {
		// Get payment gateway
		if ( ! $payment_gateway ) $payment_gateway = $this->payment_gateway;

		// Get details
		if ( ! $details ) $details = $payment_gateway;

		// Update payment
		wpj_update_payment(
			['payment_status' => 'cancelled', 'payment_details' => $details, 'payment_response' => $response],
			['payment_type_id' => $oid, 'payment_type' => 'job_purchase']
		);

		// Update order
		wpj_update_job_order(
			['payment_status' => 'cancelled', 'payment_details' => $details, 'payment_response' => $response, 'closed' => 1, 'date_closed' => current_time( 'timestamp', 1 )],
			['id' => $oid]
		);

		// Actions
		do_action( 'wpj_after_order_is_cancelled', $oid, 'job_purchase' );
		do_action( 'wpjobster_job_payment_failed', $oid );

		// Redirect
		$this->redirect( $redirect, $oid, 'failure' );
	}

	public function other( $oid, $payment_gateway, $details, $response, $status = 'processing', $redirect = 'yes' ) {
		// Get payment gateway
		if ( ! $payment_gateway ) $payment_gateway = $this->payment_gateway;

		// Get details
		if ( ! $details ) $details = $payment_gateway;

		// Update payment
		wpj_update_payment(
			['payment_status' => $status, 'payment_details' => $details, 'payment_response' => $response],
			['payment_type_id' => $oid, 'payment_type' => 'job_purchase']
		);

		// Update order
		wpj_update_job_order(
			['payment_status' => $status, 'payment_details' => $details, 'payment_response' => $response],
			['id' => $oid]
		);

		// Action
		do_action( 'wpjobster_job_payment_other', $oid );

		// Redirect
		$this->redirect( $redirect, $oid, 'success' );
	}

	private function redirect( $redirect, $oid, $status ) {
		if ( $redirect == 'yes' ) {
			$page_id = wpj_get_option( 'wpjobster_' . $this->payment_gateway . '_' . $status . '_page' );

			if ( $page_id ) wp_redirect( get_permalink( $page_id ) . $oid );
			else wp_redirect( wpj_get_order_link( $oid ) );

			exit;
		}
	}

}